﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_AbstractedPadUsb.h"
#include "hid_CommonStateUtility.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< Usb コントローラー共通の電源状態
const system::PowerInfo StaticUsbPowerInfo = { false, false, system::BatteryLevel_High };

//!< Usb の InputReport の長さ
const size_t UsbInputReportLengthMin = 8;

//!< AnalogStick のセンター値
const AnalogStickState StickCenter = { 128, 128 };

//!< AnalogStick の値のとるレンジ Max 値
const AnalogStickState StickMax = { 127, 128 };

//!< AnalogStick の値のとるレンジ Min 値
const AnalogStickState StickMin = { 128, 127 };

//!< USBPadのレポートをデコードします。
bool DecodeUsbPadReport(AbstractedPadState* pOutValue, const uint8_t* input, const size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(input);

    if (length < UsbInputReportLengthMin)
    {
        return false;
    }

    pOutValue->buttons.Reset();

    struct buttonPattern
    {
        uint8_t offset;
        uint8_t pattern;
    };

    const buttonPattern bPattern[] = {
        { 0,0x04 },{ 0,0x02 },{ 0,0x08 },{ 0,0x01 },//ABXY
        { 1,0x04 },{ 1,0x08 },{ 0,0x10 },{ 0,0x20 },//SLB SRB LR
        { 0,0x40 },{ 0,0x80 },{ 1,0x02 },{ 1,0x01 }//ZL ZE Plus Minus
    };

    //ABXYL----PlusMinusまでをパース
    for (int i = 0; i < 12; i++)
    {
        uint8_t button = input[bPattern[i].offset] & bPattern[i].pattern;
        pOutValue->buttons.Set(i, button != 0);
    }

    //HOMEとCAPTUREをパース　TODO
    pOutValue->buttons.Set<AbstractedPadButton::Home>((input[1] & 0x10) != 0);
    pOutValue->buttons.Set<AbstractedPadButton::Shot>((input[1] & 0x20) != 0);

    //方向キーをパース
    {
        uint8_t button = input[2] & 0x0F;

        //無入力でない
        if (button != 0x0F)
        {

            switch (button)
            {
            case 0:
                pOutValue->buttons.Set(13, 1);//上
                break;
            case 1:
                pOutValue->buttons.Set(13, 1);//上
                pOutValue->buttons.Set(14, 1);//右
                break;
            case 2:
                pOutValue->buttons.Set(14, 1);//右
                break;
            case 3:
                pOutValue->buttons.Set(14, 1);//右
                pOutValue->buttons.Set(15, 1);//下
                break;
            case 4:
                pOutValue->buttons.Set(15, 1);//下

                break;
            case 5:
                pOutValue->buttons.Set(12, 1);//左
                pOutValue->buttons.Set(15, 1);//下
                break;
            case 6:
                pOutValue->buttons.Set(12, 1);//左
                break;
            case 7:
                pOutValue->buttons.Set(12, 1);//左
                pOutValue->buttons.Set(13, 1);//上
                break;
            default:
                break;
            }
        }

    }

    //アナログスティックをパース
    pOutValue->analogStickL = ClampAnalogStick({ input[3] - StickCenter.x, 0 - (input[4] - StickCenter.y) }, 0, StickMin, StickMax);
    pOutValue->analogStickR = ClampAnalogStick({ input[5] - StickCenter.x, 0 - (input[6] - StickCenter.y) }, 0, StickMin, StickMax);

    // アナログスティックの十字ボタンエミュレーション
    SetCrossStickEmulationButtonsOnAbstractedPad(pOutValue);

    return true;
}

}

AbstractedPadUsb::AbstractedPadUsb() NN_NOEXCEPT
    : m_IsConnected(false)
{
    // Usb 接続
    m_InterfaceType = system::InterfaceType_Usb;
    // Switch Pro Controller 互換
    m_DeviceType = system::DeviceType::SwitchProController::Mask;

    // FeatureSet
    m_FeatureSet.Reset();
    m_FeatureSet.Set<AbstractedPadFeature::AnalogStick>();
}

AbstractedPadUsb::~AbstractedPadUsb() NN_NOEXCEPT
{
    // 何もしない
}

AbstractedPadType AbstractedPadUsb::GetType() NN_NOEXCEPT
{
    return s_Type;
}

bool AbstractedPadUsb::IsConnected() NN_NOEXCEPT
{
    return m_IsConnected;
}

system::PowerInfo AbstractedPadUsb::GetPowerInfo() NN_NOEXCEPT
{
    return StaticUsbPowerInfo;
}

AbstractedPadState AbstractedPadUsb::GetPadState() NN_NOEXCEPT
{
    return m_PadState;
}

bool AbstractedPadUsb::GetButtonTriggerElapsedTime(nn::os::Tick* pOutTick, AbstractedPadButtonSet button) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(button.CountPopulation(), 1);
    NN_UNUSED(pOutTick);
    NN_UNUSED(button);

    // ボタン押しをさかのぼる機能は提供しない
    return false;
}

void AbstractedPadUsb::SetIndicator(uint8_t pattern, bool blink) NN_NOEXCEPT
{
    NN_UNUSED(blink);

    if (m_IsConnected)
    {
        m_Pattern = pattern;
    }
}

uint8_t AbstractedPadUsb::GetIndicator() NN_NOEXCEPT
{
    if (m_IsConnected)
    {
        return m_Pattern;
    }

    return 0;
}

bool AbstractedPadUsb::IsWired() NN_NOEXCEPT
{
    if (m_IsConnected == false)
    {
        return false;
    }
    // USB コントローラーは常時 Wired 状態
    return true;
}

bool AbstractedPadUsb::IsUsbConnected() NN_NOEXCEPT
{
    if (m_IsConnected == false)
    {
        return false;
    }
    // USB コントローラーは常時 USB 接続状態
    return true;
}

void AbstractedPadUsb::Connect() NN_NOEXCEPT
{
    if (m_IsAttached == false)
    {
        return;
    }

    // 接続状態に変更
    if (m_IsConnected == false)
    {
        m_IsConnected = true;
    }
}

void AbstractedPadUsb::Detach() NN_NOEXCEPT
{
    m_IsConnected = false;
}

bool AbstractedPadUsb::HasBattery() NN_NOEXCEPT
{
    return false;
}

bool AbstractedPadUsb::SetDeviceInfoOnPlayReportControllerUsage(system::PlayReportControllerUsage* pOutValue) NN_NOEXCEPT
{
    pOutValue->deviceType = system::PlayReportDeviceType_UsbController;
    pOutValue->usbDevice.vidHigh = static_cast<uint8_t>((m_Vid >> 8) & 0x00ff);
    pOutValue->usbDevice.vidLow  = static_cast<uint8_t>(m_Vid & 0x00ff);
    pOutValue->usbDevice.pidHigh = static_cast<uint8_t>((m_Pid >> 8) & 0x00ff);
    pOutValue->usbDevice.pidLow  = static_cast<uint8_t>(m_Pid & 0x00ff);
    return true;
}

void AbstractedPadUsb::AttachDevice(size_t port, AbstractedPadId id, uint16_t vid, uint16_t pid) NN_NOEXCEPT
{
    if(m_IsAttached == true)
    {
        return;
    }
    m_Id = id;
    m_Port = port;
    m_Vid = vid;
    m_Pid = pid;
    m_IsAttached = true;
    m_IsNewSampleReceived = false;
    m_ButtonMask.Reset();
}

void AbstractedPadUsb::RemoveDevice() NN_NOEXCEPT
{
    m_Id = AbstractedPadId();
    m_Port = 0;
    m_IsConnected = false;
    m_IsAttached = false;
}

size_t AbstractedPadUsb::GetPort() NN_NOEXCEPT
{
    if(m_IsAttached == false)
    {
        return 0;
    }

    return m_Port;
}

void AbstractedPadUsb::SetReport(const uint8_t* pReport, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pReport);

    if (m_IsAttached == false)
    {
        return;
    }

    auto oldButtons = m_PadState.buttons;

    DecodeUsbPadReport(&m_PadState, pReport, length);

    // 離されたボタンは無効化を解除
    m_ButtonMask |= ~m_PadState.buttons;
    // 無効化
    m_PadState.buttons &= m_ButtonMask;

    m_IsNewSampleReceived = true;

    // 何かボタンが押されたら接続状態へと変更する
    if (m_IsConnected == false &&
        ((oldButtons ^ m_PadState.buttons) & m_PadState.buttons).IsAnyOn())
    {
        m_IsConnected = true;
        // 入力無効化するボタンを追加
        m_ButtonMask &= ~m_PadState.buttons;
    }
}

}}} // namespace nn::hid::detail
