﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/os/os_TimerEventTypes.h>

#include "hid_AbstractedPadIdPublisher.h"
#include "hid_ActivationCount.h"
#include "hid_IAbstractedPad.h"
#include "hid_WindowsGenericPad-os.win.h"
#include "hid_WindowsKeyboard-os.win.h"
#include "hid_XpadMappingManager.h"

namespace nn { namespace hid { namespace detail {

//!< GenericPadの扱える最大数
static const int AbstractedGenericPadCountMax = 4;

class AbstractedGenericPad final : public IAbstractedPad
{
public:
    //!< サンプリング間隔
    static const ::nn::TimeSpan SamplingInterval;

private:
    static const AbstractedPadType s_Type = AbstractedPadType_Generic;

    //!< タイマーイベント
    ::nn::os::TimerEventType* m_pTimerEvent;

    //!< アクティブ化した回数
    ActivationCount m_ActivationCount;

    //!< AbstractedId パブリッシャー
    AbstractedPadIdPublisher* m_pIdPublisher;

    //!< コントローラーの接続状態
    bool m_IsConnected;

    //!< Windows の汎用コントローラ
    WindowsGenericPad m_WindowsGenericPad;

    //!< Windows のキーボード
    WindowsKeyboard m_WindowsKeyboard;

    //!< 汎用ゲームパッドの割り当て
    ::nn::settings::BasicXpadGenericPadMap m_GenericPadMap;

    //!< キーボードの割り当て
    ::nn::settings::BasicXpadKeyboardMap m_KeyboardMap;

    //!< HidShell のポート番号
    int m_HidShellPort;

    //!< マッピングマネージャを指すポインタ
    XpadMappingManager* m_pMappingManager;

    //!< コントローラーの入力状態
    AbstractedPadState m_PadState;

    //!< インジケーターパターン
    uint8_t m_Pattern;

    //!< 入力無効化するボタン状態
    AbstractedPadButtonSet m_ButtonMask;

public:
    AbstractedGenericPad() NN_NOEXCEPT;
    virtual ~AbstractedGenericPad() NN_NOEXCEPT NN_OVERRIDE;

    //!< AbstractedPadType を取得する
    virtual AbstractedPadType GetType() NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスの接続状態を取得する
    virtual bool IsConnected() NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスの電源状態を取得する
    virtual system::PowerInfo GetPowerInfo() NN_NOEXCEPT NN_OVERRIDE;

    //!< デバイスのボタン/アナログスティックの入力状態を取得する
    virtual AbstractedPadState GetPadState() NN_NOEXCEPT NN_OVERRIDE;

    //!< 最後にボタンが押下された時間を取得する
    virtual bool GetButtonTriggerElapsedTime(nn::os::Tick* pOutTick, AbstractedPadButtonSet button) NN_NOEXCEPT NN_OVERRIDE;

    //!< 6軸センサーの値を取得する。暫定的に xcd の構造体を利用する
    virtual int GetSensorStates(nn::xcd::SixAxisSensorState* pOutValue, int count) NN_NOEXCEPT NN_OVERRIDE;

    //!< コントローラーのインジケーターを制御する
    virtual void SetIndicator(uint8_t pattern, bool blink) NN_NOEXCEPT NN_OVERRIDE;

    //!< コントローラーのインジケーターパターンを取得する
    virtual uint8_t GetIndicator() NN_NOEXCEPT NN_OVERRIDE;

    //!< ケーブル接続されているかどうか
    virtual bool IsWired() NN_NOEXCEPT NN_OVERRIDE;

    //!< Usb の接続状態を取得します
    virtual bool IsUsbConnected() NN_NOEXCEPT NN_OVERRIDE;

    //!< コントローラーを接続状態に変更します
    virtual void Connect() NN_NOEXCEPT NN_OVERRIDE;

    //!< コントローラーを切断する
    virtual void Detach() NN_NOEXCEPT NN_OVERRIDE;

    //!< バッテリーを搭載しているかどうか取得します
    virtual bool HasBattery() NN_NOEXCEPT NN_OVERRIDE;

    //!< プレイレポート用のデフォルトの情報を取得します
    virtual bool SetDeviceInfoOnPlayReportControllerUsage(system::PlayReportControllerUsage* pOutValue) NN_NOEXCEPT NN_OVERRIDE;

    //!< タイマーイベントを設定します。
    void SetTimerEvent(::nn::os::TimerEventType* pTimerEvent) NN_NOEXCEPT;

    //!< タイマーイベントをアクティブ化します。
    void ActivateTimerEvent() NN_NOEXCEPT;

    //!< タイマーイベントを非アクティブ化します。
    void DeactivateTimerEvent() NN_NOEXCEPT;

    //!< AbstractedId パブリッシャーをセットします
    void SetAbstractedPadIdPublisher(AbstractedPadIdPublisher* pPublisher) NN_NOEXCEPT;

    //!< HidShell のポートをセットする
    void SetHidShellPort(int port) NN_NOEXCEPT;

    //!< Xpad のマッピングマネージャをセットする
    void SetMappingManager(XpadMappingManager* pMappingManager) NN_NOEXCEPT;

    //!< アクティベートする
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< ディアクティベートする
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< アクティブ化されているかどうか
    bool IsActivated() NN_NOEXCEPT;

    //!< 内部状態を更新する
    void Update() NN_NOEXCEPT;

private:
    //!< デバイスを接続状態を更新する
    void UpdateConnectionState(bool isConnected, bool isKeyOn) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
