﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_AbstractedAutoPilotPad.h"
#include "hid_CommonStateUtility.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< Usb コントローラー共通の電源状態
const system::PowerInfo StaticAutoPilotPowerInfo = { false, false, system::BatteryLevel_High };
}

AbstractedAutoPilotPad::AbstractedAutoPilotPad() NN_NOEXCEPT
    : m_IsConnected(false)
    , m_FixedNpadAssignmentEnabled(true)
    , m_PadState()
    , m_Pattern(0)
{
    // Usb 接続
    m_InterfaceType = system::InterfaceType_Bluetooth;

    // Switch Pro Controller 互換
    m_DeviceType = system::DeviceType::SwitchProController::Mask;

    // FeatureSet
    m_FeatureSet.Reset();
    m_FeatureSet.Set<AbstractedPadFeature::AnalogStick>();
}

AbstractedAutoPilotPad::~AbstractedAutoPilotPad() NN_NOEXCEPT
{
    // 何もしない
}

AbstractedPadType AbstractedAutoPilotPad::GetType() NN_NOEXCEPT
{
    return s_Type;
}

bool AbstractedAutoPilotPad::IsConnected() NN_NOEXCEPT
{
    return m_IsConnected;
}

system::PowerInfo AbstractedAutoPilotPad::GetPowerInfo() NN_NOEXCEPT
{
    return StaticAutoPilotPowerInfo;
}

AbstractedPadState AbstractedAutoPilotPad::GetPadState() NN_NOEXCEPT
{
    return m_PadState;
}

bool AbstractedAutoPilotPad::GetButtonTriggerElapsedTime(nn::os::Tick* pOutTick, AbstractedPadButtonSet button) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(button.CountPopulation(), 1);
    NN_UNUSED(pOutTick);
    NN_UNUSED(button);

    // ボタン押しをさかのぼる機能は提供しない
    return false;
}

int AbstractedAutoPilotPad::GetSensorStates(nn::xcd::SixAxisSensorState* pOutValue, int count) NN_NOEXCEPT
{
    NN_UNUSED(pOutValue);
    NN_UNUSED(count);
    return 0;
}

void AbstractedAutoPilotPad::SetIndicator(uint8_t pattern, bool blink) NN_NOEXCEPT
{
    NN_UNUSED(blink);

    if (m_IsConnected)
    {
        m_Pattern = pattern;
    }
}

uint8_t AbstractedAutoPilotPad::GetIndicator() NN_NOEXCEPT
{
    if (m_IsConnected)
    {
        return m_Pattern;
    }

    return 0;
}

bool AbstractedAutoPilotPad::IsWired() NN_NOEXCEPT
{
    return false;
}

bool AbstractedAutoPilotPad::IsUsbConnected() NN_NOEXCEPT
{
    return false;
}

void AbstractedAutoPilotPad::Connect() NN_NOEXCEPT
{
    // 接続状態に変更
    if (m_IsConnected == false)
    {
        m_IsConnected = true;
    }
}

void AbstractedAutoPilotPad::Detach() NN_NOEXCEPT
{
    m_IsConnected = false;
}

bool AbstractedAutoPilotPad::IsFixedNpadAssignmentEnabled(NpadIdType* pOutValue) NN_NOEXCEPT
{
    if (m_FixedNpadAssignmentEnabled)
    {
        *pOutValue = m_NpadId;
    }

    return m_FixedNpadAssignmentEnabled;
}

bool AbstractedAutoPilotPad::HasBattery() NN_NOEXCEPT
{
    return false;
}

bool AbstractedAutoPilotPad::SetDeviceInfoOnPlayReportControllerUsage(system::PlayReportControllerUsage* pOutValue) NN_NOEXCEPT
{
    pOutValue->deviceType = system::PlayReportDeviceType_SwitchProController;
    for (auto& byte : pOutValue->raw)
    {
        byte = 0;
    }
    return true;
}

void AbstractedAutoPilotPad::SetAbstractedPadIdPublisher(AbstractedPadIdPublisher* pPublisher) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPublisher);
    m_pIdPublisher = pPublisher;
}

void AbstractedAutoPilotPad::SetNpadIdType(const NpadIdType& id) NN_NOEXCEPT
{
    m_NpadId = id;
}

::nn::Result AbstractedAutoPilotPad::SetAutoPilotState(
    const ::nn::hid::debug::BasicXpadAutoPilotState& value) NN_NOEXCEPT
{
    {
        if (value.powerState != PowerState_Disconnected && value.powerState != PowerState_NoBattery)
        {
            if (m_IsConnected == false)
            {
                m_Id = m_pIdPublisher->PublishId();
                m_IsConnected = true;
            }
            m_IsNewSampleReceived = true;

           m_PadState.buttons = ConvertBasicXpadButtonSetToAbstracted(value.buttons);

            m_PadState.analogStickL = LimitAnalogStick(
                value.analogStickL, 0, AnalogStickMax, AnalogStickMax);

            m_PadState.analogStickR = LimitAnalogStick(
                value.analogStickR, 0, AnalogStickMax, AnalogStickMax);

            // アナログスティックの十字ボタンエミュレーション
            SetCrossStickEmulationButtonsOnAbstractedPad(&m_PadState);
        }
        else
        {
            m_IsConnected = false;
            m_IsNewSampleReceived = false;
            m_PadState = AbstractedPadState();
        }

    }

    NN_RESULT_SUCCESS;
}

::nn::Result AbstractedAutoPilotPad::UnsetAutoPilotState() NN_NOEXCEPT
{
    m_IsConnected = false;
    m_IsNewSampleReceived = false;
    m_PadState = AbstractedPadState();
    NN_RESULT_SUCCESS;
}


}}} // namespace nn::hid::detail
