﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrv_SubTaskThread.h"

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Exchange.h>
#include <memory>
#include <atomic>
#include <mutex>
#include <nn/util/util_Optional.h>
#include <nn/grc/grc_Result.h>

namespace nn { namespace grcsrv { namespace detail {

Result SubTaskThread::StartThread(void* stack, size_t stackSize, int priority, const char* name) NN_NOEXCEPT
{
    auto f = [](void* p)
    {
        static_cast<SubTaskThread*>(p)->DoTaskImpl();
    };
    NN_RESULT_DO(os::CreateThread(&m_Thread, f, this, stack, stackSize, priority));
    this->m_IsThreadExitRequesting = false;
    os::SetThreadNamePointer(&m_Thread, name);
    this->m_IsThreadCreated = true;
    this->m_IsRequesting = false;
    this->m_IsRunning = false;
    this->m_NotRunningEvent.Signal();
    os::StartThread(&m_Thread);
    NN_RESULT_SUCCESS;
}

void SubTaskThread::RequestThreadExitImpl() NN_NOEXCEPT
{
    {
        std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
        this->m_IsThreadExitRequesting = true;
        m_Cv.Signal();
    }
    DoCancel();
}

void SubTaskThread::StopThread() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_IsThreadCreated);
    RequestThreadExitImpl();
    os::WaitThread(&m_Thread);
    os::DestroyThread(&m_Thread);
    this->m_IsThreadCreated = false;
}

void SubTaskThread::DoTaskImpl() NN_NOEXCEPT
{
    std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
    for (;;)
    {
        if (m_IsThreadExitRequesting)
        {
            return;
        }
        if (util::Exchange(&m_IsRequesting, false))
        {
            lk.unlock();
            this->DoTask();
            lk.lock();
            this->m_IsRunning = false;
            m_NotRunningEvent.Signal();
            continue;
        }
        m_Cv.Wait(m_Mutex);
    }
}

Result SubTaskThread::Start(void (*onStart)(void* p), void* p) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsThreadCreated, grc::ResultInvalidState());
    std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
    NN_RESULT_THROW_UNLESS(!util::Exchange(&m_IsRunning, true), grc::ResultFlushIsInProcess());
    if (onStart)
    {
        onStart(p);
    }
    this->m_IsThreadExitRequesting = false;
    this->m_IsRequesting = true;
    m_NotRunningEvent.Clear();
    m_Cv.Signal();
    NN_RESULT_SUCCESS;
}

Result SubTaskThread::Cancel() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsThreadCreated, grc::ResultInvalidState());
    DoCancel();
    NN_RESULT_SUCCESS;
}

Result SubTaskThread::GetNotRunningEvent(sf::Out<sf::NativeHandle> pOutSystemEventHandle) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsThreadCreated, grc::ResultInvalidState());
    *pOutSystemEventHandle = sf::NativeHandle(m_NotRunningEvent.GetReadableHandle(), false);
    NN_RESULT_SUCCESS;
}

bool SubTaskThread::IsRunning() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsThreadCreated);
    std::unique_lock<decltype(m_Mutex)> lk(m_Mutex);
    return m_IsRunning;
}

}}}
