﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrv_CapsrvMovieFileReader.h"

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <utility>
#include <nn/grc/grc_Result.h>

#include <nn/capsrv/capsrv_AlbumFileDescription.h>
#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nn/capsrv/capsrv_ScreenShotControl.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_File.h>

namespace nn { namespace grcsrv {

class CapsrvMovieFileReader::CapsrvMovieFileReadableHandle
    : public MovieFileReadableHandle
{
private:

    bool m_Initialized = false;
    capsrv::movie::MovieReaderFileSystem m_Mrfs;
    capsrv::AlbumMovieReadStreamHandle m_StreamHandle;
    std::unique_ptr<char[]> m_MrfsBuffer;
    fs::FileHandle m_Handle;

    virtual fs::FileHandle GetFileHandle() const NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT(m_Initialized);
        return m_Handle;
    }

    void FinalizeImpl() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_Initialized);
        fs::CloseFile(m_Handle);
        m_Mrfs.UnregisterMovieReadStreamDataSection(m_StreamHandle);
        capsrv::CloseAlbumMovieReadStream(m_StreamHandle);
        m_MrfsBuffer.reset();
        m_Mrfs.Finalize();
    }

public:

    Result Initialize(const capsrv::AlbumFileId& fileId) NN_NOEXCEPT
    {
        auto success = false;

        capsrv::AlbumFileId outFileId;
        NN_ABORT_UNLESS_RESULT_SUCCESS(capsrv::GenerateCurrentAlbumFileId(&outFileId, fileId.applicationId, capsrv::AlbumFileContents_Movie));

        #define MOUNT_NAME "trim"
        #define FILE_NAME "a.tmp"
        const auto FileName = FILE_NAME;
        const auto FilePath = MOUNT_NAME ":/" FILE_NAME;
        m_Mrfs.Initialize("trim");
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                m_Mrfs.Finalize();
            }
        };

        const auto ChunkCacheCount = 3;
        auto mrfsBufferSize = m_Mrfs.GetRequiredMemorySizeForFile(capsrv::movie::MovieDataChunkUnitSize, ChunkCacheCount);
        auto mrfsBuffer = std::unique_ptr<char[]>(new char[mrfsBufferSize]);
        NN_RESULT_THROW_UNLESS(mrfsBuffer, grc::ResultOutOfMemory());

        capsrv::AlbumMovieReadStreamHandle streamHandle;
        NN_RESULT_DO(capsrv::OpenAlbumMovieReadStream(&streamHandle, fileId));
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                capsrv::CloseAlbumMovieReadStream(streamHandle);
            }
        };

        NN_RESULT_DO(m_Mrfs.RegisterMovieReadStreamDataSection(FileName, streamHandle, capsrv::movie::MovieDataChunkUnitSize, ChunkCacheCount, mrfsBuffer.get(), mrfsBufferSize));
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                m_Mrfs.UnregisterMovieReadStreamDataSection(FileName);
            }
        };

        fs::FileHandle handle;
        NN_RESULT_DO(fs::OpenFile(&handle, FilePath, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                fs::CloseFile(handle);
            }
        };

        success = true;
        this->m_Initialized = true;
        this->m_StreamHandle = streamHandle;
        this->m_MrfsBuffer = std::move(mrfsBuffer);
        this->m_Handle = handle;
        NN_RESULT_SUCCESS;
    }

    ~CapsrvMovieFileReadableHandle() NN_NOEXCEPT
    {
        if (m_Initialized)
        {
            FinalizeImpl();
        }
    }

};

Result CapsrvMovieFileReader::Create(std::unique_ptr<MovieFileReadableHandle>* pOut, const capsrv::AlbumFileId& fileId) NN_NOEXCEPT
{
    auto p = std::unique_ptr<CapsrvMovieFileReadableHandle>(new CapsrvMovieFileReadableHandle);
    NN_RESULT_THROW_UNLESS(p, grc::ResultOutOfMemory());
    NN_RESULT_DO(p->Initialize(fileId));
    *pOut = std::move(p);
    NN_RESULT_SUCCESS;
}

}}
