﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Macro.h>

#include <nn/gfx/gfx_SwapChainInfo.h>
#include <nn/gfx/gfx_DataAccessorConverter.h>
#include <nn/gfx/gfx_Texture.h>
#include <nn/gfx/gfx_Device.h>
#include <nn/gfx/gfx_SyncInfo.h>

#include <nn/gfx/detail/gfx_SwapChain-api.vk.1.h>
#include <nn/gfx/detail/gfx_Device-api.vk.1.h>
#include <nn/gfx/detail/gfx_Texture-api.vk.1.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.vk.1.h>
#include <nn/gfx/detail/gfx_Sync-api.vk.1.h>

#include "gfx_CommonHelper.h"
#include "gfx_VkHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationVk1 Target;

void InitializeSwapChainCommonImpl( SwapChainImpl< Target >* pThis, DeviceImpl< Target >* pDevice,
    const SwapChainInfo& info, MemoryPoolImpl< Target >* pMemoryPool,
    ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    NN_UNUSED( pMemoryPool );
    NN_UNUSED( memoryPoolSize );
    NN_UNUSED( memoryPoolOffset );

    SwapChainImpl< Target >::DataType& data = pThis->ToData();

    VkBool32 queueSupportsPresent = VK_FALSE;
    VkPhysicalDevice physicalDevice = CastToVkDispatchableObject< VkPhysicalDevice >(
        data.pGfxDevice->ToData()->hPhysicalDevice );
    VkSurfaceKHR vkSurfaceKHR = CastToVkNonDispatchableObject< VkSurfaceKHR >( data.hSurfaceKHR );
    NN_GFX_CALL_VK_FUNCTION( vkGetPhysicalDeviceSurfaceSupportKHR( physicalDevice, 0, vkSurfaceKHR,
        &queueSupportsPresent ) );
    NN_SDK_ASSERT_NOT_EQUAL( queueSupportsPresent, static_cast< VkBool32 >( VK_FALSE ) );  // L"Default queue doesn't support present!" );

    uint32_t formatCount = 0;
    VkResult result;
    NN_GFX_CALL_VK_FUNCTION( result = vkGetPhysicalDeviceSurfaceFormatsKHR( physicalDevice,
        vkSurfaceKHR, &formatCount, NULL ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    // Also somewhat dubious of this whole approach of letting the implementation select a preferred
    // format.  We could attempt that if the user sent in VK_FORMAT_UNDEFINED.  But if the user selected a
    // format, then we should make sure the physicalDevice can do it and then apply it
    VkSurfaceFormatKHR* pSurfaceFormatKHR = reinterpret_cast< VkSurfaceFormatKHR* >(
        Vk::AllocDriverMemory( sizeof( VkSurfaceFormatKHR ) * formatCount, 8 ) );
    NN_SDK_ASSERT_NOT_NULL( pSurfaceFormatKHR );

    NN_GFX_CALL_VK_FUNCTION( result = vkGetPhysicalDeviceSurfaceFormatsKHR( physicalDevice, vkSurfaceKHR,
        &formatCount, pSurfaceFormatKHR ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    VkFormat format = VK_FORMAT_B8G8R8A8_SRGB;
    ImageFormat scanBufferFormat = ImageFormat_B8_G8_R8_A8_UnormSrgb;
    if ( formatCount != 1 || pSurfaceFormatKHR[ 0 ].format != VK_FORMAT_UNDEFINED )
    {
        // FIXME: We should make sure format matches the requested format. Currently it does not.
        // pSurfaceFormatKHR[ 0 ].format = VK_FORMAT_B8G8R8A8_UNORM
        // image.GetFormat() = ImageFormat_R8_G8_B8_A8_UnormSrgb => VK_FORMAT_R8G8B8A8_SRGB
        NN_SDK_ASSERT_NOT_EQUAL( formatCount, 0U );  // L"No VkSurfaceFormatKHRs found!" );

                                                     // At least pick SRGB vs Unorm
        for ( int idxFormat = 0; idxFormat < static_cast< int >( formatCount ); ++idxFormat )
        {
            if ( GetTypeFormat( info.GetFormat() ) == TypeFormat_UnormSrgb &&
                ( pSurfaceFormatKHR[ idxFormat ].format == VK_FORMAT_R8G8B8A8_SRGB ||
                    pSurfaceFormatKHR[ idxFormat ].format == VK_FORMAT_B8G8R8A8_SRGB ) )
            {
                format = pSurfaceFormatKHR[ idxFormat ].format;
                if ( pSurfaceFormatKHR[ idxFormat ].format == VK_FORMAT_R8G8B8A8_SRGB )
                {
                    scanBufferFormat = ImageFormat_R8_G8_B8_A8_UnormSrgb;
                }
                else if ( pSurfaceFormatKHR[ idxFormat ].format == VK_FORMAT_B8G8R8A8_SRGB )
                {
                    scanBufferFormat = ImageFormat_B8_G8_R8_A8_UnormSrgb;
                }
                break;
            }

            if ( GetTypeFormat( info.GetFormat() ) == TypeFormat_Unorm &&
                ( pSurfaceFormatKHR[ idxFormat ].format == VK_FORMAT_R8G8B8A8_UNORM ||
                    pSurfaceFormatKHR[ idxFormat ].format == VK_FORMAT_B8G8R8A8_UNORM ) )
            {
                format = pSurfaceFormatKHR[ idxFormat ].format;
                if ( pSurfaceFormatKHR[ idxFormat ].format == VK_FORMAT_R8G8B8A8_UNORM )
                {
                    scanBufferFormat = ImageFormat_R8_G8_B8_A8_Unorm;
                }
                else if ( pSurfaceFormatKHR[ idxFormat ].format == VK_FORMAT_B8G8R8A8_UNORM )
                {
                    scanBufferFormat = ImageFormat_B8_G8_R8_A8_Unorm;
                }
                break;
            }
        }
    }
    Vk::FreeDriverMemory( pSurfaceFormatKHR );
    pSurfaceFormatKHR = NULL;

    VkSurfaceCapabilitiesKHR surfCapabilities;
    NN_GFX_CALL_VK_FUNCTION( result = vkGetPhysicalDeviceSurfaceCapabilitiesKHR( physicalDevice,
        vkSurfaceKHR, &surfCapabilities ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    uint32_t presentModeCount = 0;
    NN_GFX_CALL_VK_FUNCTION( result = vkGetPhysicalDeviceSurfacePresentModesKHR( physicalDevice,
        vkSurfaceKHR, &presentModeCount, NULL ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    VkPresentModeKHR pPresentModes[ 4 ];
    NN_SDK_ASSERT_LESS_EQUAL( presentModeCount, 4U );
    NN_GFX_CALL_VK_FUNCTION( result = vkGetPhysicalDeviceSurfacePresentModesKHR( physicalDevice,
        vkSurfaceKHR, &presentModeCount, pPresentModes ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    VkExtent2D * pSwapChainExtent = reinterpret_cast< VkExtent2D * >( &data.extent2D );
    if ( surfCapabilities.currentExtent.width == UINT32_MAX )
    {
        pSwapChainExtent->width = 1280;
        pSwapChainExtent->height = 720;
    }
    else
    {
        *pSwapChainExtent = surfCapabilities.currentExtent;
    }

    VkPresentModeKHR swapChainPresentMode = VK_PRESENT_MODE_FIFO_KHR;

    uint32_t desiredSwapImages = data.totalScanBuffers;
    NN_SDK_ASSERT( desiredSwapImages >= surfCapabilities.minImageCount );
    if ( surfCapabilities.maxImageCount > 0 && desiredSwapImages > surfCapabilities.maxImageCount )
    {
        desiredSwapImages = surfCapabilities.maxImageCount;
    }

    VkSwapchainCreateInfoKHR swapChainInfo = { };
    swapChainInfo.sType = VK_STRUCTURE_TYPE_SWAPCHAIN_CREATE_INFO_KHR;
    swapChainInfo.pNext = NULL;
    swapChainInfo.surface = vkSurfaceKHR;
    swapChainInfo.minImageCount = desiredSwapImages;
    swapChainInfo.imageFormat = format;
    swapChainInfo.imageExtent = *pSwapChainExtent;
    swapChainInfo.preTransform = ( surfCapabilities.supportedTransforms & VK_SURFACE_TRANSFORM_IDENTITY_BIT_KHR )
        ? VK_SURFACE_TRANSFORM_IDENTITY_BIT_KHR : surfCapabilities.currentTransform;
    swapChainInfo.compositeAlpha = VK_COMPOSITE_ALPHA_OPAQUE_BIT_KHR;
    swapChainInfo.imageArrayLayers = 1;
    swapChainInfo.presentMode = swapChainPresentMode;
    swapChainInfo.oldSwapchain = VK_NULL_HANDLE;
    swapChainInfo.clipped = true;
    swapChainInfo.imageColorSpace = VK_COLORSPACE_SRGB_NONLINEAR_KHR;
    swapChainInfo.imageUsage = VK_IMAGE_USAGE_COLOR_ATTACHMENT_BIT | VK_IMAGE_USAGE_TRANSFER_DST_BIT;
    swapChainInfo.imageSharingMode = VK_SHARING_MODE_EXCLUSIVE;
    swapChainInfo.queueFamilyIndexCount = 0;
    swapChainInfo.pQueueFamilyIndices = NULL;

    VkDevice device = CastToVkDispatchableObject< VkDevice >( data.pGfxDevice->ToData()->hDevice );
    VkSwapchainKHR swapChainKHR;
    NN_GFX_CALL_VK_FUNCTION( result = vkCreateSwapchainKHR( device, &swapChainInfo,
        pDevice->ToData()->pAllocationCallback, &swapChainKHR ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    data.hSwapChainKHR = CastFromVkNonDispatchableObject< VkSwapchainKHR >( swapChainKHR );

    // Query the total images
    uint32_t swapChainImageCount;
    NN_GFX_CALL_VK_FUNCTION( result = vkGetSwapchainImagesKHR( device, swapChainKHR,
        &swapChainImageCount, NULL ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    NN_SDK_ASSERT_LESS_EQUAL( swapChainImageCount, SwapChainImplData< ApiVariationVk1 >::VkMaxScanBufferCount );

    // Grab the VkImage handles for the swap buffers
    VkImage swapChainImages[ SwapChainImplData< ApiVariationVk1 >::VkMaxScanBufferCount ];
    NN_GFX_CALL_VK_FUNCTION( result = vkGetSwapchainImagesKHR( device,
        swapChainKHR, &swapChainImageCount, swapChainImages ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    TextureInfo textureInfo;
    Vk::SetupScanBufferTextureInfo( &textureInfo, info );

    textureInfo.SetImageFormat( scanBufferFormat );

    // Setup the scan buffer Textures and ColorTargetViews
    for ( int index = 0; index < info.GetBufferCount(); index++ )
    {
        TextureImpl< Target >* pScanBuffer =
            nn::gfx::DataToAccessor( data.scanBufferTextures[ index ] );
        ColorTargetViewImpl< Target >* pScanBufferView =
            nn::gfx::DataToAccessor( data.scanBufferViews[ index ] );
        SemaphoreImpl< Target >* pSemaphore =
            nn::gfx::DataToAccessor( data.presentSemaphore[ index ] );

        // ダミーでTextureを生成し、保持するvkImageを置き換えます。
        pScanBuffer->Initialize( pDevice, textureInfo, NULL, 0, 0 );
        NN_GFX_CALL_VK_FUNCTION( vkDestroyImage( device,
            CastToVkNonDispatchableObject< VkImage >( pScanBuffer->ToData()->hImage ),
            pDevice->ToData()->pAllocationCallback ) );
        NN_GFX_CALL_VK_FUNCTION( vkFreeMemory( device,
            CastToVkNonDispatchableObject< VkDeviceMemory >( pScanBuffer->ToData()->hMemory ),
            pDevice->ToData()->pAllocationCallback ) );
        pScanBuffer->ToData()->hImage = CastFromVkNonDispatchableObject< VkImage >( swapChainImages[ index ] );
        pScanBuffer->ToData()->hMemory = 0;

        ClearColorValue clearColor;
        for ( int idxComponent = 0; idxComponent < 4; ++idxComponent )
        {
            clearColor.valueFloat[ idxComponent ] = 0;
        }
        Vk::ClearColorTexture( pDevice, swapChainImages[ index ], clearColor );

        if ( detail::Vk::IsLayoutManagementEnabled() )
        {
            TextureManageData* p_Manager = pScanBuffer->ToData()->pManager;
            p_Manager->SetImageHandle( swapChainImages[ index ] );
            p_Manager->ResetImageLayout( VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL );
            p_Manager->SetImageAspectFlags( Vk::GetImageAspect( textureInfo.GetImageFormat() ) );
        }

        ColorTargetViewInfo colorTargetViewInfo;
        colorTargetViewInfo.SetImageDimension( nn::gfx::ImageDimension_2d );
        colorTargetViewInfo.SetMipLevel( 0 );
        colorTargetViewInfo.SetImageFormat( textureInfo.GetImageFormat() );
        colorTargetViewInfo.EditArrayRange().SetBaseArrayIndex( 0 );
        colorTargetViewInfo.EditArrayRange().SetArrayLength( 1 );
        colorTargetViewInfo.SetTexturePtr( static_cast< TTexture< Target >* >( pScanBuffer ) );
        pScanBufferView->Initialize( pDevice, colorTargetViewInfo );

        SemaphoreInfo semaphoreInfo;
        semaphoreInfo.SetDefault();
        pSemaphore->Initialize( pDevice, semaphoreInfo );
    }

}//NOLINT(impl/function_size)

size_t SwapChainImpl< Target >::GetScanBufferAlignment( DeviceImpl< Target >*, const SwapChainInfo& ) NN_NOEXCEPT
{
    return 1;
}

size_t SwapChainImpl< Target >::CalculateScanBufferSize(
    DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    TextureInfo textureInfo;
    Vk::SetupScanBufferTextureInfo( &textureInfo, info );

    return TextureImpl< Target >::CalculateMipDataSize( pDevice, textureInfo ) * info.GetBufferCount();
}

SwapChainImpl< Target >::SwapChainImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

SwapChainImpl< Target >::~SwapChainImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

int SwapChainImpl< Target >::GetScanBufferViews(
    TColorTargetView< Target >** ppOutScanBufferArray, int maxScanBufferCount ) NN_NOEXCEPT
{
    if ( ppOutScanBufferArray )
    {
        int count = std::min NN_PREVENT_MACRO_FUNC( maxScanBufferCount,
            static_cast< int >( this->totalScanBuffers ) );
        for ( int idxScanBuffer = 0; idxScanBuffer < count; ++idxScanBuffer )
        {
            ppOutScanBufferArray[ idxScanBuffer ] =
                nn::gfx::DataToAccessor( this->scanBufferViews[ idxScanBuffer ] );
        }
        return count;
    }
    else
    {
        return this->totalScanBuffers;
    }
}

int SwapChainImpl< Target >::GetScanBuffers(
    TTexture< Target >** ppOutScanBufferTextureArray, int maxScanBufferTextureCount ) NN_NOEXCEPT
{
    if ( ppOutScanBufferTextureArray )
    {
        int count = std::min NN_PREVENT_MACRO_FUNC( maxScanBufferTextureCount,
            static_cast< int >( this->totalScanBuffers ) );
        for ( int idxScanBuffer = 0; idxScanBuffer < count; ++idxScanBuffer )
        {
            ppOutScanBufferTextureArray[ idxScanBuffer ] =
                nn::gfx::DataToAccessor( this->scanBufferTextures[ idxScanBuffer ] );
        }
        return count;
    }
    else
    {
        return this->totalScanBuffers;
    }
}

AcquireScanBufferResult SwapChainImpl< Target >::AcquireNextScanBufferIndex(
    int* pOutScanBufferIndex, SemaphoreImpl< Target >* pSemaphore,
    FenceImpl< Target >* pFence ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_SDK_REQUIRES_NOT_NULL( pOutScanBufferIndex );
    NN_SDK_REQUIRES( pSemaphore != NULL || pFence != NULL );
    NN_SDK_REQUIRES( pSemaphore == NULL || IsInitialized( *pSemaphore ) );
    NN_SDK_REQUIRES( pFence == NULL || IsInitialized( *pFence ) );

    VkDevice device = CastToVkDispatchableObject< VkDevice >( this->pGfxDevice->ToData()->hDevice );

    VkFence fence;
    if ( pFence == NULL )
    {
        fence = VK_NULL_HANDLE;
    }
    else
    {
        fence = CastToVkNonDispatchableObject< VkFence >( pFence->ToData()->hFence );
    }
    VkSemaphore semaphore;
    if ( pSemaphore == nullptr )
    {
        semaphore = VK_NULL_HANDLE;
    }
    else
    {
        semaphore = CastToVkNonDispatchableObject< VkSemaphore >( pSemaphore->ToData()->hSemaphore );
    }

    VkResult result = vkAcquireNextImageKHR( device,
        CastToVkNonDispatchableObject< VkSwapchainKHR >( this->hSwapChainKHR ), UINT64_MAX,
        semaphore, fence, &this->currentScanBufferIndex );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );

    *pOutScanBufferIndex = this->currentScanBufferIndex;

    return result == VK_SUCCESS ?
        AcquireScanBufferResult_Success : AcquireScanBufferResult_Failed;
}

int SwapChainImpl< Target >::AcquireNextScanBufferIndex() NN_NOEXCEPT
{
    return this->currentScanBufferIndex;
}

ColorTargetViewImpl< Target >* SwapChainImpl< Target >::AcquireNextScanBufferView() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    return nn::gfx::DataToAccessor( this->scanBufferViews[ this->currentScanBufferIndex ] );
}

TextureImpl< Target >* SwapChainImpl< Target >::AcquireNextScanBuffer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    return nn::gfx::DataToAccessor( this->scanBufferTextures[ this->currentScanBufferIndex ] );
}

}
}
}
