﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/gfx_SamplerInfo.h>

#include <nn/gfx/detail/gfx_Sampler-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Device-api.nvn.8.h>

#include "gfx_CommonHelper.h"
#include "gfx_NvnHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationNvn8 Target;

SamplerImpl< Target >::SamplerImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

SamplerImpl< Target >::~SamplerImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void SamplerImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_STATIC_ASSERT( sizeof( this->nvnSampler ) == sizeof( NVNsampler ) );

    static const float s_BorderColors[][ 4 ] =
    {
        { 1.0f, 1.0f, 1.0f, 1.0f }, // TextureBorderColorType_White
        { 0.0f, 0.0f, 0.0f, 0.0f }, // TextureBorderColorType_TransparentBlack
        { 0.0f, 0.0f, 0.0f, 1.0f }, // TextureBorderColorType_OpaqueBlack
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_BorderColors ) == TextureBorderColorType_End );

    static const NVNsamplerReduction s_ReductionTable[] =
    {
        NVN_SAMPLER_REDUCTION_AVERAGE,
        NVN_SAMPLER_REDUCTION_MIN,
        NVN_SAMPLER_REDUCTION_MAX
    };

    NVNsamplerBuilder builder;

    NN_GFX_CALL_NVN_FUNCTION( nvnSamplerBuilderSetDefaults( &builder ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnSamplerBuilderSetDevice(
        &builder, pDevice->ToData()->pNvnDevice ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnSamplerBuilderSetMinMagFilter(
        &builder, Nvn::GetMinFilter( info.GetFilterMode() ), Nvn::GetMagFilter( info.GetFilterMode() ) ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnSamplerBuilderSetWrapMode(
        &builder, Nvn::GetWrapMode( info.GetAddressU() ),
        Nvn::GetWrapMode( info.GetAddressV() ), Nvn::GetWrapMode( info.GetAddressW() ) ) );
    NN_GFX_CALL_NVN_FUNCTION(
        nvnSamplerBuilderSetLodClamp( &builder, info.GetMinLod(), info.GetMaxLod() ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnSamplerBuilderSetLodBias( &builder, info.GetLodBias() ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnSamplerBuilderSetCompare( &builder,
        IsFilterModeComparison( info.GetFilterMode() ) ? NVN_COMPARE_MODE_COMPARE_R_TO_TEXTURE
        : NVN_COMPARE_MODE_NONE, Nvn::GetRComparisonFunction( info.GetComparisonFunction() ) ) );
    NN_GFX_CALL_NVN_FUNCTION(
        nvnSamplerBuilderSetBorderColor( &builder, s_BorderColors[ info.GetBorderColorType() ] ) );
    if( IsFilterModeAnisotropic( info.GetFilterMode() ) )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnSamplerBuilderSetMaxAnisotropy(
            &builder, static_cast< float >( info.GetMaxAnisotropy() ) ) );
    }
    NVNsamplerReduction reductionMode = NVN_SAMPLER_REDUCTION_AVERAGE;
    if( pDevice->ToData()->supportedFeatures.GetBit( NvnDeviceFeature_SupportMinMaxFiltering ) )
    {
        reductionMode = s_ReductionTable[ ( info.GetFilterMode() >> 8 ) & 0x03 ];
    }
    NN_GFX_CALL_NVN_FUNCTION( nvnSamplerBuilderSetReductionFilter( &builder, reductionMode ) );

    // This is a convenience pointer that is used throught the code
    this->pNvnSampler = &this->nvnSampler;

    NVNboolean isSamplerOK = NN_GFX_CALL_NVN_FUNCTION(
        nvnSamplerInitialize( this->pNvnSampler, &builder ) );
    NN_SDK_ASSERT( isSamplerOK );
    NN_UNUSED( isSamplerOK );

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void SamplerImpl< Target >::Finalize( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnSamplerFinalize( this->pNvnSampler ) );
    this->pNvnSampler = NULL;

    this->state = State_NotInitialized;
}

}
}
}
