﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/gfx_SamplerInfo.h>

#include <nn/gfx/detail/gfx_Sampler-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Device-api.d3d.11.h>

#include "gfx_D3dHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationD3d11 Target;

SamplerImpl< Target >::SamplerImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

SamplerImpl< Target >::~SamplerImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void SamplerImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_ASSERT( !IsD3dHandleValid( this->pSampler ) );

    D3D11_SAMPLER_DESC desc;

    desc.Filter = D3d::GetFilterMode( info.GetFilterMode() );
    desc.AddressU = D3d::GetTextureAddressMode( info.GetAddressU() );
    desc.AddressV = D3d::GetTextureAddressMode( info.GetAddressV() );
    desc.AddressW = D3d::GetTextureAddressMode( info.GetAddressW() );
    desc.MipLODBias = info.GetLodBias();
    desc.MaxAnisotropy = info.GetMaxAnisotropy();
    desc.ComparisonFunc = D3d::GetComparisonFunction( info.GetComparisonFunction() );

    switch ( info.GetBorderColorType() )
    {
    case TextureBorderColorType_White:
        {
            desc.BorderColor[ 0 ] = desc.BorderColor[ 1 ]
                = desc.BorderColor[ 2 ] = desc.BorderColor[ 3 ] = 1.f;
        }
        break;
    case TextureBorderColorType_TransparentBlack:
        {
            desc.BorderColor[ 0 ] = desc.BorderColor[ 1 ]
                = desc.BorderColor[ 2 ] = desc.BorderColor[ 3 ] = 0.f;
        }
        break;
    case TextureBorderColorType_OpaqueBlack:
        {
            desc.BorderColor[ 0 ] = desc.BorderColor[ 1 ] = desc.BorderColor[ 2 ] = 0.f;
            desc.BorderColor[ 3 ] = 1.f;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    desc.MinLOD = info.GetMinLod();
    desc.MaxLOD = info.GetMaxLod();

    ID3D11Device* pD3dDevice = static_cast< ID3D11Device* >( pDevice->ToData()->renderingContext.hD3dDevice );
    ID3D11SamplerState* pSamplerState;

    NN_GFX_CALL_D3D_FUNCTION( pD3dDevice->CreateSamplerState( &desc, &pSamplerState ) );
    NN_SDK_ASSERT( IsD3dHandleValid( pSamplerState ) );

    this->pSampler = pSamplerState;
    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void SamplerImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );
    NN_UNUSED( pDevice );

    if( this->pSampler )
    {
        ID3D11SamplerState* pSamplerState = static_cast< ID3D11SamplerState* >( this->pSampler );
        NN_GFX_CALL_D3D_FUNCTION( pSamplerState->Release() );
        this->pSampler = NULL;
    }

    this->state = State_NotInitialized;
}

}
}
}
