﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_Common.h>
#include <nn/gfx/gfx_VariationBase.h>
#include <nn/gfx/gfx_RootSignatureInfo.h>

#include <nn/gfx/detail/gfx_RootSignature-api.common.h>

#include "gfx_CommonHelper.h"

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS

namespace nn {
namespace gfx {
namespace detail {

template< typename TTarget >
size_t RootSignatureImplCommon< TTarget >::GetRequiredMemorySize( const InfoType& info ) NN_NOEXCEPT
{
    size_t ret = 0;
    int descriptorTableCount = info.GetDescriptorTableCount();
    int dynamicDescriptorCount = info.GetDynamicDescriptorCount();
    ret += sizeof( RootSignatureInfo::DataType );
    ret += sizeof( DescriptorTableInfo::DataType ) * descriptorTableCount;
    ret += sizeof( DynamicDescriptorInfo::DataType ) * dynamicDescriptorCount;

    int sumDescriptorRange = 0;
    for( int idxDescriptorTable = 0; idxDescriptorTable < descriptorTableCount; ++idxDescriptorTable )
    {
        sumDescriptorRange += info.GetDescriptorTableInfoArray()[ idxDescriptorTable ].GetDescriptorRangeCount();
    }
    ret += sizeof( DescriptorRangeInfo::DataType ) * sumDescriptorRange;

    return ret;
}

template< typename TTarget >
RootSignatureImplCommon< TTarget >::RootSignatureImplCommon() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

template< typename TTarget >
RootSignatureImplCommon< TTarget >::~RootSignatureImplCommon() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

template< typename TTarget >
void RootSignatureImplCommon< TTarget >::SetMemory( void* value, size_t size ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( value || size == 0 );

    this->pWorkMemory = value;
    this->memorySize = static_cast< uint32_t >( size );
}

template< typename TTarget >
void* RootSignatureImplCommon< TTarget >::GetMemory() NN_NOEXCEPT
{
    return this->pWorkMemory;
}

template< typename TTarget >
const void* RootSignatureImplCommon< TTarget >::GetMemory() const NN_NOEXCEPT
{
    return this->pWorkMemory;
}

template< typename TTarget >
void RootSignatureImplCommon< TTarget >::Initialize( DeviceImpl< TTarget >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pDevice );
    NN_UNUSED( pDevice );

    NN_SDK_ASSERT( this->pWorkMemory );
    NN_SDK_ASSERT( this->memorySize >= GetRequiredMemorySize( info ) );

    nn::util::BytePtr current( this->pWorkMemory.ptr );
    RootSignatureInfo::DataType* pRootSignature = current.Get< RootSignatureInfo::DataType >();
    *pRootSignature = info.ToData();
    current.Advance( sizeof( RootSignatureInfo::DataType ) );

    int descriptorTableCount = info.GetDescriptorTableCount();
    int descriptorTableSize = sizeof( DescriptorTableInfo::DataType ) * descriptorTableCount;
    memcpy( current.Get(), info.GetDescriptorTableInfoArray(), descriptorTableSize );
    DescriptorTableInfo::DataType* pDescriptorTableArray = current.Get< DescriptorTableInfo::DataType >();
    pRootSignature->pDescriptorTableArray = pDescriptorTableArray;
    current.Advance( descriptorTableSize );
    for( int idxDescriptorTable = 0; idxDescriptorTable < descriptorTableCount; ++idxDescriptorTable )
    {
        const DescriptorTableInfo& srcDescriptorTable = info.GetDescriptorTableInfoArray()[ idxDescriptorTable ];
        int descriptorRangeSize = sizeof( DescriptorRangeInfo::DataType ) * srcDescriptorTable.GetDescriptorRangeCount();
        memcpy( current.Get(), srcDescriptorTable.GetDescriptorRangeInfoArray(), descriptorRangeSize );
        pDescriptorTableArray[ idxDescriptorTable ].pDescriptorRangeArray = current.Get< DescriptorRangeInfo::DataType >();
        current.Advance( descriptorRangeSize );
    }

    int dynamicDescriptorSize = sizeof( DynamicDescriptorInfo::DataType ) * info.GetDynamicDescriptorCount();
    memcpy( current.Get(), info.GetDynamicDescriptorInfoArray(), dynamicDescriptorSize );
    pRootSignature->pDynamicDescriptorArray = current.Get< DynamicDescriptorInfo::DataType >();

    NN_SDK_ASSERT( this->memorySize >= static_cast< size_t >( nn::util::BytePtr(
        this->pWorkMemory.ptr ).Distance( current.Advance( dynamicDescriptorSize ).Get() ) ) );

    this->state = State_Initialized;
}

template< typename TTarget >
void RootSignatureImplCommon< TTarget >::Finalize( DeviceImpl< TTarget >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pDevice );
    NN_UNUSED( pDevice );

    this->state = State_NotInitialized;
}

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
template class RootSignatureImplCommon< ApiVariation< ApiType< 4 >, ApiVersion< 8 > > >;
template class RootSignatureImplCommon< ApiVariation< ApiType< 5 >, ApiVersion< 1 > > >;
#endif

#if defined( NN_BUILD_CONFIG_OS_WIN32 )
template class RootSignatureImplCommon< ApiVariation< ApiType< 1 >, ApiVersion< 4 > > >;
template class RootSignatureImplCommon< ApiVariation< ApiType< 3 >, ApiVersion< 11 > > >;
template class RootSignatureImplCommon< ApiVariation< ApiType< 4 >, ApiVersion< 8 > > >;
template class RootSignatureImplCommon< ApiVariation< ApiType< 5 >, ApiVersion< 1 > > >;
#endif

#if defined( NN_BUILD_CONFIG_OS_COS )
template class RootSignatureImplCommon< ApiVariation< ApiType< 2 >, ApiVersion< 2 > > >;
#endif

}
}
}

NN_PRAGMA_POP_WARNINGS
