﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>

#include <nn/os.h>

#include <nn/gfx/detail/gfx_Queue-api.gl.4.h>
#include <nn/gfx/detail/gfx_Device-api.gl.4.h>
#include <nn/gfx/detail/gfx_CommandBuffer-api.gl.4.h>
#include <nn/gfx/detail/gfx_Sync-api.gl.4.h>
#include <nn/gfx/detail/gfx_SwapChain-api.gl.4.h>
#include <nn/gfx/detail/gfx_Texture-api.gl.4.h>

#include "gfx_CommonHelper.h"
#include "gfx_GlHelper.h"
#include "gfx_GlCommand.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGl4 Target;

template< bool IsDstBackBuffer, bool IsSrcSwapChain >
void CopyToScanbufferImpl( QueueImpl< Target >* pThis,
    SwapChainImpl< Target >* pSwapChain, const ColorTargetViewImpl< Target >* pSrc ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pThis );
    NN_SDK_ASSERT_NOT_NULL( pSwapChain );
    NN_SDK_ASSERT_NOT_NULL( pSrc );

    NN_SDK_ASSERT( IsInitialized( *pSwapChain ) );
    NN_SDK_ASSERT( IsInitialized( *pSrc ) );
    NN_SDK_ASSERT( IsInitialized( *pThis ) );

    QueueImpl< Target >::DataType& data = pThis->ToData();

    SwapChainImpl< Target >::DataType& swapChain = pSwapChain->ToData();

    GlDeviceActivator activator( pSwapChain, &data.renderingContext );

    GLboolean tmpSrgb = NN_GFX_CALL_GL_FUNCTION( ::glIsEnabled( GL_FRAMEBUFFER_SRGB ) );

    GLint tmpFboDraw;
    GLint tmpFboRead;
    NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv( GL_DRAW_FRAMEBUFFER_BINDING, &tmpFboDraw ) );
    NN_GFX_CALL_GL_FUNCTION( ::glGetIntegerv( GL_READ_FRAMEBUFFER_BINDING, &tmpFboRead ) );

    GLboolean tmpScissor = NN_GFX_CALL_GL_FUNCTION( ::glIsEnabled( GL_SCISSOR_TEST ) );

    GLint srcWidth;
    GLint srcHeight;
    const ColorTargetViewImpl< Target >::DataType& src = pSrc->ToData();
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetTextureLevelParameteriv )(
        src.hTexture, src.target, 0, GL_TEXTURE_WIDTH, &srcWidth ) );
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetTextureLevelParameteriv )(
        src.hTexture, src.target, 0, GL_TEXTURE_HEIGHT, &srcHeight ) );
    NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glNamedFramebufferTexture )(
        data.commandContext.hTmpFbo[ 1 ], GL_COLOR_ATTACHMENT0, src.hTexture, 0 ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_READ_FRAMEBUFFER, data.commandContext.hTmpFbo[ 1 ] ) );

    int dstWidth;
    int dstHeight;
    GLint tmpAttachment0 = GlInvalidHandle;
    if( NN_STATIC_CONDITION( IsDstBackBuffer ) )
    {
        Gl::GetSurfaceSize( &dstWidth, &dstHeight, pSwapChain );
        NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_DRAW_FRAMEBUFFER, 0 ) );
    }
    else
    {
        const ColorTargetViewImpl< Target >::DataType& dst =
            swapChain.scanBufferViews[ swapChain.currentScanBufferIndex ];
        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetTextureLevelParameteriv )(
            dst.hTexture, dst.target, 0, GL_TEXTURE_WIDTH, &dstWidth ) );
        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetTextureLevelParameteriv )(
            dst.hTexture, dst.target, 0, GL_TEXTURE_HEIGHT, &dstHeight ) );
        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glGetNamedFramebufferAttachmentParameteriv )(
            data.commandContext.hTmpFbo[ 0 ], GL_COLOR_ATTACHMENT0, GL_FRAMEBUFFER_ATTACHMENT_OBJECT_NAME, &tmpAttachment0 ) );
        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glNamedFramebufferTexture )(
            data.commandContext.hTmpFbo[ 0 ], GL_COLOR_ATTACHMENT0, dst.hTexture, 0 ) );
        NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_DRAW_FRAMEBUFFER, data.commandContext.hTmpFbo[ 0 ] ) );
    }
    Gl::SetEnable( GL_FRAMEBUFFER_SRGB, !IsSrcSwapChain );

    NN_GFX_CALL_GL_FUNCTION( ::glDisable( GL_SCISSOR_TEST ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBlitFramebuffer( 0, 0, srcWidth, srcHeight,
        0, 0, dstWidth, dstHeight, GL_COLOR_BUFFER_BIT, GL_LINEAR ) );

    if( IsValid( static_cast< GlHandle >( tmpAttachment0 ) ) )
    {
        NN_GFX_CALL_GL_FUNCTION( NN_GFX_GL_DSA( ::glNamedFramebufferTexture )(
            data.commandContext.hTmpFbo[ 0 ], GL_COLOR_ATTACHMENT0, tmpAttachment0, 0 ) );
    }

    Gl::SetEnable( GL_SCISSOR_TEST, tmpScissor );
    Gl::SetEnable( GL_FRAMEBUFFER_SRGB, tmpSrgb );
    NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_DRAW_FRAMEBUFFER, tmpFboDraw ) );
    NN_GFX_CALL_GL_FUNCTION( ::glBindFramebuffer( GL_READ_FRAMEBUFFER, tmpFboRead ) );

    NN_GFX_GL_ASSERT();
}
template void CopyToScanbufferImpl< true, true >( QueueImpl< Target >*,
    SwapChainImpl< Target >*, const ColorTargetViewImpl< Target >* );

void InitializeQueueCommonImpl( QueueImpl< Target >* pThis,
    DeviceImpl< Target >*, const QueueInfo& ) NN_NOEXCEPT
{
    QueueImpl< Target >::DataType& data = pThis->ToData();

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::InitializeMutex(
        &data.renderingContext.mutex, true, nn::os::MutexLockLevelMax ) );

    GlDeviceActivator activator( pThis );

    // GL バージョンチェック
    NN_SDK_ASSERT( Gl::CheckRequiredVersion( GlRequiredVersionMajor, GlRequiredVersionMinor ) );

    NN_GFX_CALL_GL_FUNCTION( ::glGenFramebuffers( 1, &data.commandContext.hFbo ) );
    NN_GFX_CALL_GL_FUNCTION( ::glGenFramebuffers( 2, data.commandContext.hTmpFbo ) );
    NN_GFX_CALL_GL_FUNCTION( ::glGenVertexArrays( 1, &data.commandContext.hVao ) );
    NN_GFX_CALL_GL_FUNCTION( ::glGenProgramPipelines( 1, &data.commandContext.hPipeline ) );
    NN_GFX_CALL_GL_FUNCTION( ::glGenQueries( QueryTarget_End, data.commandContext.hQuery ) );

    NN_GFX_GL_ASSERT();
}

void FinalizeQueueCommonImpl( QueueImpl< Target >* pThis, DeviceImpl< Target >* ) NN_NOEXCEPT
{
    QueueImpl< Target >::DataType& data = pThis->ToData();

    GlDeviceActivator activator( pThis );

    if( IsValid( data.commandContext.hPipeline ) )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDeleteProgramPipelines( 1, &data.commandContext.hPipeline ) );
        data.commandContext.hPipeline = GlInvalidHandle;
    }
    if( IsValid( data.commandContext.hVao ) )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDeleteVertexArrays( 1, &data.commandContext.hVao ) );
        data.commandContext.hVao = GlInvalidHandle;
    }
    if( IsValid( data.commandContext.hFbo ) )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDeleteFramebuffers( 1, &data.commandContext.hFbo ) );
        data.commandContext.hFbo = GlInvalidHandle;
    }
    if( IsValid( data.commandContext.hTmpFbo[ 0 ] ) )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDeleteFramebuffers( 1, data.commandContext.hTmpFbo ) );
        data.commandContext.hTmpFbo[ 0 ] = GlInvalidHandle;
    }
    if( IsValid( data.commandContext.hTmpFbo[ 1 ] ) )
    {
        NN_GFX_CALL_GL_FUNCTION( ::glDeleteFramebuffers( 1, data.commandContext.hTmpFbo + 1 ) );
        data.commandContext.hTmpFbo[ 1 ] = GlInvalidHandle;
    }
    for( int idxQueryTarget = 0; idxQueryTarget < QueryTarget_End; ++idxQueryTarget )
    {
        if( IsValid( data.commandContext.hQuery[ idxQueryTarget ] ) )
        {
            NN_GFX_CALL_GL_FUNCTION( ::glDeleteQueries( 1, data.commandContext.hQuery + idxQueryTarget ) );
        }
    }
    NN_GFX_GL_ASSERT();
}

QueueImpl< Target >::QueueImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

QueueImpl< Target >::~QueueImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void QueueImpl< Target >::Flush() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    GlDeviceActivator activator( this );
    NN_GFX_CALL_GL_FUNCTION( ::glFlush() );
}

void QueueImpl< Target >::Sync() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    GlDeviceActivator activator( this );
    NN_GFX_CALL_GL_FUNCTION( ::glFinish() );
}

void QueueImpl< Target >::ExecuteCommand( const CommandBufferImpl< Target >* pCommandBuffer,
    FenceImpl< Target >* pFence ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pCommandBuffer );
    NN_SDK_REQUIRES( pCommandBuffer->ToData()->state == CommandBufferImpl< Target >::DataType::State_Initialized );
    NN_SDK_REQUIRES( pCommandBuffer->ToData()->commandBufferType == CommandBufferType_Direct );
    NN_SDK_REQUIRES( pFence == NULL || IsInitialized( *pFence ) );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    GlDeviceActivator activator( this );

    NN_SDK_ASSERT_NOT_NULL( pCommandBuffer->ToData()->pCommandList.ptr );
    ExecuteGlCommandList( pCommandBuffer->ToData()->pCommandList, &this->commandContext );
    if( pFence )
    {
        if ( pFence->ToData()->glSync )
        {
            NN_GFX_CALL_GL_FUNCTION(
                ::glDeleteSync( static_cast< GLsync >( pFence->ToData()->glSync ) ) );
        }
        pFence->ToData()->glSync = NN_GFX_CALL_GL_FUNCTION(
            ::glFenceSync( GL_SYNC_GPU_COMMANDS_COMPLETE, 0 ) );
        pFence->ToData()->flags.SetBit( FenceImpl< Target >::DataType::Flag_Signaled, false );
        NN_SDK_ASSERT_NOT_NULL( pFence->ToData()->glSync.ptr );
        pFence->ToData()->pQueue = this;
    }
    NN_GFX_GL_ASSERT();
}

void QueueImpl< Target >::SetSemaphore( SemaphoreImpl< ApiVariationGl4 >* pSemaphore ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pSemaphore );
    NN_SDK_REQUIRES( IsInitialized( *pSemaphore ) );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    GlDeviceActivator activator( this );
    NN_GFX_CALL_GL_FUNCTION( pSemaphore->ToData()->glSync =
        ::glFenceSync( GL_SYNC_GPU_COMMANDS_COMPLETE, 0 ) );
    NN_GFX_GL_ASSERT();
    NN_SDK_ASSERT_NOT_NULL( pSemaphore->ToData()->glSync.ptr );
    pSemaphore->ToData()->pQueue = this;
    pSemaphore->ToData()->flags.SetBit( SemaphoreImpl< Target >::DataType::Flag_Signaled, false );
}

void QueueImpl< Target >::SyncSemaphore( const SemaphoreImpl< ApiVariationGl4 >* pSemaphore ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pSemaphore );
    NN_SDK_REQUIRES( IsInitialized( *pSemaphore ) );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    if( !pSemaphore->ToData()->flags.GetBit( SemaphoreImpl< Target >::DataType::Flag_Signaled ) )
    {
        GlDeviceActivator activator( this );
        NN_GFX_CALL_GL_FUNCTION( ::glWaitSync( static_cast<GLsync>(
            pSemaphore->ToData()->glSync ), 0, GL_TIMEOUT_IGNORED ) );
        NN_GFX_GL_ASSERT();
    }
}

void QueueImpl< Target >::CopyToScanBuffer( SwapChainImpl< Target >* pSwapChain,
    const ColorTargetViewImpl< Target >* pColorTarget ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pSwapChain );
    NN_SDK_REQUIRES_NOT_NULL( pColorTarget );

    SwapChainImpl< Target >::DataType& swapChain = pSwapChain->ToData();

    if( swapChain.flag.GetBit( SwapChainImpl< Target >::DataType::Flag_Acquired ) )
    {
        CopyToScanbufferImpl< false, false >( this, pSwapChain, pColorTarget );
    }
    else
    {
        CopyToScanbufferImpl< true, false >( this, pSwapChain, pColorTarget );
    }

    swapChain.flag.SetBit( SwapChainImpl< Target >::DataType::Flag_Copied, true );
}

}
}
}
