﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <GL/glew.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

#include <nn/gfx/gfx_Enum.h>
#include <nn/gfx/gfx_Variation-api.gl.h>
#include <nn/gfx/gfx_DeviceInfo.h>

#include <nn/gfx/detail/gfx_Common-api.gl.h>
#include <nn/gfx/detail/gfx_Declare.h>
#include <nn/gfx/detail/gfx_Misc.h>

#define NN_GFX_CALL_GL_FUNCTION(func) (func)

#if defined( NN_SDK_BUILD_DEBUG )
    #define NN_GFX_GL_ASSERT() \
    do \
    { \
        GLenum error; NN_UNUSED( error ); \
        NN_SDK_ASSERT( GL_NO_ERROR == ( error = ::glGetError() ), "GL Error: %s (0x%x)\n", \
            nn::gfx::detail::Gl::ErrorToString( error ), error ); \
    } while( NN_STATIC_CONDITION( 0 ) )
#else
    #define NN_GFX_GL_ASSERT()
#endif

#define NN_GFX_GL_DSA( func ) \
    func ## EXT

namespace nn {
namespace gfx {
namespace detail {

const GLint GlRequiredVersionMajor = 4;
const GLint GlRequiredVersionMinor = 5;

struct GlTextureFormat
{
    GLenum imageFormat;
    GLenum pixelFormat;
    GLenum pixelType;
};

struct GlAttributeFormat
{
    GLint size;
    GLenum type;
    GLboolean normalized;
};

class Gl
{
public:
    static void PolygonMode( GLenum face, GLenum mode ) NN_NOEXCEPT;

    static void LogicOp( GLenum opcode ) NN_NOEXCEPT;

    static const char* ErrorToString( GLenum error ) NN_NOEXCEPT;

    static const char* FramebufferStatusToString( GLenum status ) NN_NOEXCEPT;

    static bool CheckFramebufferStatus( GLuint hFramebuffer ) NN_NOEXCEPT;

    static void SetEnable( GLenum cap, bool value ) NN_NOEXCEPT
    {
        if( value )
        {
            NN_GFX_CALL_GL_FUNCTION( ::glEnable( cap ) );
        }
        else
        {
            NN_GFX_CALL_GL_FUNCTION( ::glDisable( cap ) );
        }
    }
    static void SetEnable( GLenum cap, GLboolean value ) NN_NOEXCEPT
    {
        return SetEnable( cap, value == GL_TRUE );
    }

    static bool CheckShaderStatus( GlHandle hShader ) NN_NOEXCEPT;

    static bool CheckProgramStatus( GlHandle hProgram ) NN_NOEXCEPT;

    static GLenum GetTextureTarget( ImageDimension dimension ) NN_NOEXCEPT;
    static bool IsLayeredTarget( GLenum target ) NN_NOEXCEPT;
    static GlTextureFormat GetTextureFormat( ImageFormat format ) NN_NOEXCEPT;

    static size_t TextureSubImage( GlHandle texture, GLenum target, GLint mipLevel, bool isCompressed,
        GLenum imageFormat, GLenum pixelFormat, GLenum pixelType, GLint xOffset, GLint yOffset,
        GLint zOffset, GLsizei width, GLsizei height, GLsizei depth, GLsizei imageSize, const void* pMemory ) NN_NOEXCEPT;
    static size_t CopyCpuMemoryToTexture( GlHandle texture, GLenum target, GLint mipLevel,
        ImageFormat format, GLsizei width, GLsizei height, GLsizei depth, const void* pMemory ) NN_NOEXCEPT;

    static void FramebufferTexture( GlHandle hFbo, GLenum attachment, GlHandle hTexture, GLint mipLevel, GLint layer = -1 ) NN_NOEXCEPT;

    static GlAttributeFormat GetAttributeFormat( AttributeFormat format ) NN_NOEXCEPT;

    static bool IsSamplerType( GLenum type ) NN_NOEXCEPT;

    static bool IsImageType( GLenum type ) NN_NOEXCEPT;

    static bool IsArrayTarget( GLenum target ) NN_NOEXCEPT;

    static bool IsCompatibleBinaryFormat( GLenum binaryFormat ) NN_NOEXCEPT;

    static GLenum GetQueryTarget( QueryTarget target ) NN_NOEXCEPT;

    static bool CheckRequiredVersion(GLint requiredVersionMajor, GLint requiredVersionMinor) NN_NOEXCEPT;

    // デバイス初期化の共通処理。とりあえずここに置いている。
    static void InitializeDeviceCommonImpl(
        DeviceImpl< ApiVariationGl4 >* pDevice, const DeviceInfo& ) NN_NOEXCEPT;

    static void GetSurfaceSize( int* pOutWidth, int* pOutHeight,
        SwapChainImpl< ApiVariationGl4 >* pSwapChain ) NN_NOEXCEPT;
};

class GlDeviceActivator
{
    NN_DISALLOW_COPY( GlDeviceActivator );

public:
    template< typename TTarget >
    GlDeviceActivator( DeviceImpl< TTarget >* pTarget, bool overwrite = false ) NN_NOEXCEPT
    {
        Activate( pTarget, overwrite );
    }

    template< typename TTarget >
    GlDeviceActivator( QueueImpl< TTarget >* pTarget, bool overwrite = false ) NN_NOEXCEPT
    {
        Activate( pTarget, overwrite );
    }

    template< typename TTarget >
    GlDeviceActivator( SwapChainImpl< TTarget >* pSwapChain,
        GlRenderingContext* pRenderingContext, bool overwrite = false ) NN_NOEXCEPT
    {
        Activate( pSwapChain, pRenderingContext, overwrite );
    }

    // win 用
    GlDeviceActivator( void* hDc, GlRenderingContext* pRenderingContext, bool overwrite = false ) NN_NOEXCEPT
    {
        Activate( hDc, pRenderingContext, overwrite );
    }

    // egl 用
    GlDeviceActivator( void* display, void* surface,
        GlRenderingContext* pRenderingContext, bool overwrite = false ) NN_NOEXCEPT
    {
        Activate( display, surface, pRenderingContext, overwrite );
    }

    ~GlDeviceActivator() NN_NOEXCEPT;

private:
    template< typename TTarget >
    void Activate( DeviceImpl< TTarget >* pTarget, bool overwrite ) NN_NOEXCEPT;

    template< typename TTarget >
    void Activate( QueueImpl< TTarget >* pTarget, bool overwrite ) NN_NOEXCEPT;

    template< typename TTarget >
    void Activate( SwapChainImpl< TTarget >* pSwapChain,
        GlRenderingContext* pRenderingContext, bool overwrite = false ) NN_NOEXCEPT;

    // win 用
    void Activate( void* hDc, GlRenderingContext* pRenderingContext, bool overwrite ) NN_NOEXCEPT;

    // egl 用
    void Activate( void* display, void* surface,
        GlRenderingContext* pRenderingContext, bool overwrite ) NN_NOEXCEPT;

    nn::os::MutexType* m_pMutex;
    union
    {
        struct
        {
            void* hGlRc;
            void* hDc;
        } m_Wgl;
        struct
        {
            void* hDisplay;
            void* hDrawSurface;
            void* hReadSurface;
            void* hContext;
        } m_Egl;
    };
};

}
}
}
