﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>

#if !defined( NN_BUILD_CONFIG_OS_HORIZON )
#error
#endif

#include <nn/nn_SdkAssert.h>

#include <nn/os/os_MutexApi.h>

#include <nn/gfx/gfx_DeviceInfo.h>

#include <nn/gfx/detail/gfx_Device-api.gl.4-os.horizon.h>

#include "gfx_CommonHelper.h"
#include "gfx_GlHelper.h"
#include "gfx_EglHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGl4 Target;

DeviceImpl< Target >::DeviceImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

DeviceImpl< Target >::~DeviceImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );

    if( this->flags.GetBit( Flag_Shared ) )
    {
        Egl::MakeCurrent( this->hDisplay, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT );
        NN_GFX_CALL_NNOS_FUNCTION( nn::os::FinalizeMutex( &this->renderingContext.mutex ) );
    }
}

void DeviceImpl< Target >::Initialize( const InfoType& info ) NN_NOEXCEPT
{
    UseMiddleWare();

    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( info );

    NN_SDK_ASSERT( this->renderingContext.hGlRc == EGL_NO_CONTEXT );

    NN_GFX_CALL_EGL_FUNCTION_WITH_CHECK( this->hDisplay = eglGetDisplay( EGL_DEFAULT_DISPLAY ) );
    NN_SDK_ASSERT_NOT_EQUAL( this->hDisplay, EGL_NO_DISPLAY );

    EGLint major = EGLint();
    EGLint minor = EGLint();
    NN_GFX_CALL_EGL_FUNCTION_WITH_CHECK_RESULT( eglInitialize( this->hDisplay, &major, &minor ) );

    EGLConfig hConfig = Egl::GetConfig( this->hDisplay );

    // 初期値は EGL_OPENGL_ES_API
    // eglCreateContext についてのみ EGL_OPENGL_API と EGL_OPENGL_ES_API で挙動が異なる（EGL 1.4 以前とで少し違う）
    NN_GFX_CALL_EGL_FUNCTION_WITH_CHECK_RESULT( eglBindAPI( EGL_OPENGL_API ) );

    this->renderingContext.hGlRc = Egl::CreateGlRc( this->hDisplay,
        hConfig, EGL_NO_CONTEXT, GlRequiredVersionMajor, GlRequiredVersionMinor );
    NN_SDK_ASSERT_NOT_EQUAL( this->renderingContext.hGlRc, EGL_NO_CONTEXT );

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::InitializeMutex(
        &this->renderingContext.mutex, true, nn::os::MutexLockLevelMax ) );

    const EGLint pbufferAttribs[] =
    {
        EGL_NONE
    };
    NN_GFX_CALL_EGL_FUNCTION_WITH_CHECK( this->hSurface =
        eglCreatePbufferSurface( this->hDisplay, hConfig, pbufferAttribs ) );
    NN_SDK_ASSERT_NOT_EQUAL( this->hSurface, EGL_NO_SURFACE );

    Gl::InitializeDeviceCommonImpl( this, info );

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void DeviceImpl< Target >::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    if( this->hSurface != EGL_NO_SURFACE )
    {
        NN_GFX_CALL_EGL_FUNCTION_WITH_CHECK_RESULT(
            eglDestroySurface( this->hDisplay, this->hSurface ) );
        this->hSurface = EGL_NO_SURFACE;
    }

    Egl::MakeCurrent( this->hDisplay, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT );
    if( this->renderingContext.hGlRc != EGL_NO_CONTEXT )
    {
        NN_GFX_CALL_EGL_FUNCTION_WITH_CHECK_RESULT(
            eglDestroyContext( this->hDisplay, this->renderingContext.hGlRc ) );
        this->renderingContext.hGlRc = EGL_NO_CONTEXT;
    }

    NN_GFX_CALL_EGL_FUNCTION_WITH_CHECK_RESULT( eglReleaseThread() );

    NN_GFX_CALL_EGL_FUNCTION_WITH_CHECK_RESULT( eglTerminate( this->hDisplay ) );
    this->hDisplay = EGL_NO_DISPLAY;

    NN_GFX_CALL_NNOS_FUNCTION( nn::os::FinalizeMutex( &this->renderingContext.mutex ) );

    this->state = State_NotInitialized;
}

}
}
}
