﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_SdkAssert.h>

#include <nn/gfx/gfx_Common.h>
#include <nn/gfx/gfx_Enum.h>
#include <nn/gfx/gfx_DescriptorSlot.h>
#include <nn/gfx/gfx_RootSignatureInfo.h>

#include <nn/gfx/detail/gfx_Declare.h>

#if defined( NN_GFX_CONFIG_BUILD_FOR_PCTOOL )
    #define NN_GFX_CALL_NNOS_FUNCTION( func )
    #define NN_GFX_CALL_NNVI_FUNCTION( func )
    #define NN_GFX_CALL_NNRO_FUNCTION( func )
#else
    #define NN_GFX_CALL_NNOS_FUNCTION( func ) func
    #define NN_GFX_CALL_NNVI_FUNCTION( func ) func
    #define NN_GFX_CALL_NNRO_FUNCTION( func ) func
#endif

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS

namespace nn {
namespace gfx {

class RootSignatureInfo;
class GpuAddress;
class TextureInfo;
struct ResShaderContainerData;

namespace detail {

void UseMiddleWare() NN_NOEXCEPT;

inline ChannelFormat GetChannelFormat( ImageFormat format ) NN_NOEXCEPT
{
    return static_cast< ChannelFormat >( format >> TypeFormat_Bits );
}
inline ChannelFormat GetChannelFormat( AttributeFormat format ) NN_NOEXCEPT
{
    return static_cast< ChannelFormat >( format >> TypeFormat_Bits );
}

inline TypeFormat GetTypeFormat( ImageFormat format ) NN_NOEXCEPT
{
    return static_cast< TypeFormat >( format & ( ( 0x01 << TypeFormat_Bits ) - 1 ) );
}
inline TypeFormat GetTypeFormat( AttributeFormat format ) NN_NOEXCEPT
{
    return static_cast< TypeFormat >( format & ( ( 0x01 << TypeFormat_Bits ) - 1 ) );
}

bool IsCompressedFormat( ChannelFormat format ) NN_NOEXCEPT;

bool IsSrgbFormat( TypeFormat format ) NN_NOEXCEPT;

int GetBlockWidth( ChannelFormat format ) NN_NOEXCEPT;

int GetBlockHeight( ChannelFormat format ) NN_NOEXCEPT;

int GetBytePerPixel( ChannelFormat format ) NN_NOEXCEPT;

size_t CalculateImageSize( ChannelFormat imageFormat, uint32_t width, uint32_t height, uint32_t depth ) NN_NOEXCEPT;

int GetChannelCount( ChannelFormat format ) NN_NOEXCEPT;

size_t CalculateRowSize( uint32_t width, ChannelFormat format ) NN_NOEXCEPT;

inline bool IsMinFilterNearest( FilterMode filterMode ) NN_NOEXCEPT
{
    return ( ( static_cast< int >( filterMode ) >> 4 ) & 0x3 ) == 0;
}

inline bool IsMagFilterNearest( FilterMode filterMode ) NN_NOEXCEPT
{
    return ( ( static_cast< int >( filterMode ) >> 4 ) & 0x3 ) == 0;
}

inline bool IsMipFilterNearest( FilterMode filterMode ) NN_NOEXCEPT
{
    return ( ( static_cast< int >( filterMode ) >> 4 ) & 0x3 ) == 0;
}

inline bool IsFilterModeAnisotropic( FilterMode filterMode ) NN_NOEXCEPT
{
    return ( filterMode & FilterModeBit_Anisotropic ) != 0;
}

inline bool IsFilterModeComparison( FilterMode filterMode ) NN_NOEXCEPT
{
    return ( ( ( static_cast< int >( filterMode ) >> 7 ) & 0x1 ) == 1 );
}

bool IsValidMemoryPoolProperty( int value ) NN_NOEXCEPT;

ImageDimension GetImageDimension(
    ImageStorageDimension imageStorageDimension, bool isArray, bool isMultisample ) NN_NOEXCEPT;

bool CheckBinaryTarget( const ResShaderContainerData& resShaderContainerData,
    int lowLevelApi, int apiVersion ) NN_NOEXCEPT;

// RootSignature 共通処理
template< typename TTarget >
const RootSignatureInfo::DataType& ToInfoData( const RootSignatureImpl< TTarget >* pRootSignature ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pRootSignature );
    NN_SDK_ASSERT( pRootSignature->ToData()->pWorkMemory );
    return *static_cast< const RootSignatureInfo::DataType* >( pRootSignature->ToData()->pWorkMemory );
}

template< typename TTarget >
const DescriptorTableInfo::DataType& GetDescriptorTableInfoData(
    const RootSignatureImpl< TTarget >* pRootSignature, int indexDescriptorTable ) NN_NOEXCEPT
{
    const RootSignatureInfo::DataType& rootSignature = ToInfoData( pRootSignature );
    NN_SDK_ASSERT( indexDescriptorTable >= 0 );
    NN_SDK_ASSERT( indexDescriptorTable < static_cast< int >( rootSignature.descriptorTableCount ) );
    const DescriptorTableInfoData* pDescriptorTable = rootSignature.pDescriptorTableArray;
    return pDescriptorTable[ indexDescriptorTable ];
}

template< typename TTarget >
const DynamicDescriptorInfo::DataType& GetDynamicDescriptorInfoData(
    const RootSignatureImpl< TTarget >* pRootSignature, int indexDynamicDescriptor ) NN_NOEXCEPT
{
    const RootSignatureInfo::DataType& rootSignature = ToInfoData( pRootSignature );
    NN_SDK_ASSERT( indexDynamicDescriptor >= 0 );
    NN_SDK_ASSERT( indexDynamicDescriptor < static_cast< int >( rootSignature.dynamicDescriptorCount ) );
    const DynamicDescriptorInfoData* pDescriptorTable = rootSignature.pDynamicDescriptorArray;
    return pDescriptorTable[ indexDynamicDescriptor ];
}

template< typename TTarget >
void SetRootBufferDescriptorTable( CommandBufferImpl< TTarget >* pThis, PipelineType,
    int indexDescriptorTable, const DescriptorSlot& startBufferDescriptorSlot, ptrdiff_t descriptorSlotIncrementSize ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pThis );
    NN_SDK_ASSERT( pThis->ToData()->state == CommandBufferImpl< TTarget >::DataType::State_Begun );
    const DescriptorTableInfo::DataType& descriptorTable =
        GetDescriptorTableInfoData< TTarget >( pThis->ToData()->pGfxRootSignature, indexDescriptorTable );
    ShaderStage stage = static_cast< ShaderStage >( descriptorTable.shaderStage );
    for( int idxDescriptorRange = 0; idxDescriptorRange < static_cast< int >(
        descriptorTable.descriptorRangeCount ); ++idxDescriptorRange )
    {
        const DescriptorRangeInfoData* pDescriptorRange = descriptorTable.pDescriptorRangeArray;
        const DescriptorRangeInfo::DataType& descriptorRange = pDescriptorRange[ idxDescriptorRange ];
        DescriptorSlot slot = startBufferDescriptorSlot;
        slot.Offset( descriptorSlotIncrementSize * descriptorRange.bufferDescriptorSlotOffset );
        for( int idxDescriptor = 0; idxDescriptor < static_cast< int >(
            descriptorRange.descriptorSlotCount ); ++idxDescriptor, slot.Offset( descriptorSlotIncrementSize ) )
        {
            switch( descriptorRange.descriptorSlotType )
            {
            case DescriptorSlotType_ConstantBuffer:
                {
                    pThis->SetConstantBuffer( descriptorRange.baseShaderSlot + idxDescriptor, stage, slot );
                }
                break;
            case DescriptorSlotType_UnorderedAccessBuffer:
                {
                    pThis->SetUnorderedAccessBuffer( descriptorRange.baseShaderSlot + idxDescriptor, stage, slot );
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    }
}

template< typename TTarget >
void SetRootTextureAndSamplerDescriptorTable( CommandBufferImpl< TTarget >* pThis,
    PipelineType, int indexDescriptorTable, const DescriptorSlot& startTextureDescriptorSlot,
    const DescriptorSlot& startSamplerDescriptorSlot, ptrdiff_t textureDescriptorSlotIncrementSize,
    ptrdiff_t samplerDescriptorSlotIncrementSize ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pThis );
    NN_SDK_ASSERT( pThis->ToData()->state == CommandBufferImpl< TTarget >::DataType::State_Begun );
    const DescriptorTableInfo::DataType& descriptorTable =
        GetDescriptorTableInfoData< TTarget >( pThis->ToData()->pGfxRootSignature, indexDescriptorTable );
    ShaderStage stage = static_cast< ShaderStage >( descriptorTable.shaderStage );
    for( int idxDescriptorRange = 0; idxDescriptorRange < static_cast< int >(
        descriptorTable.descriptorRangeCount ); ++idxDescriptorRange )
    {
        const DescriptorRangeInfoData* pDescriptorRange = descriptorTable.pDescriptorRangeArray;
        const DescriptorRangeInfo::DataType& descriptorRange = pDescriptorRange[ idxDescriptorRange ];
        DescriptorSlot textureSlot = startTextureDescriptorSlot;
        DescriptorSlot samplerSlot = startSamplerDescriptorSlot;
        textureSlot.Offset( textureDescriptorSlotIncrementSize *
            descriptorRange.textureSamplerDescriptorSlotOffset.textureDescriptorSlotOffset );
        samplerSlot.Offset( samplerDescriptorSlotIncrementSize *
            descriptorRange.textureSamplerDescriptorSlotOffset.samplerDescriptorSlotOffset );
        for( int idxDescriptor = 0; idxDescriptor < static_cast< int >( descriptorRange.descriptorSlotCount );
            ++idxDescriptor, textureSlot.Offset( textureDescriptorSlotIncrementSize ),
            samplerSlot.Offset( samplerDescriptorSlotIncrementSize ) )
        {
            pThis->SetTextureAndSampler( descriptorRange.baseShaderSlot
                + idxDescriptor, stage, textureSlot, samplerSlot );
        }
    }
}

template< typename TTarget >
void SetRootUnorderedAccessBuffer( CommandBufferImpl< TTarget >* pThis, PipelineType,
    int indexDynamicDescriptor, const GpuAddress& unorderedAccessBufferAddress, size_t size ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pThis );
    NN_SDK_ASSERT( pThis->ToData()->state == CommandBufferImpl< TTarget >::DataType::State_Begun );
    const DynamicDescriptorInfo::DataType& dynamicDescriptor =
        GetDynamicDescriptorInfoData< TTarget >( pThis->ToData()->pGfxRootSignature, indexDynamicDescriptor );
    pThis->SetUnorderedAccessBuffer( dynamicDescriptor.shaderSlot,
        static_cast< ShaderStage >( dynamicDescriptor.shaderStage ), unorderedAccessBufferAddress, size );
}

template< typename TTarget >
void SetRootConstantBuffer( CommandBufferImpl< TTarget >* pThis, PipelineType,
    int indexDynamicDescriptor, const GpuAddress& constantBufferAddress, size_t size ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pThis );
    NN_SDK_ASSERT( pThis->ToData()->state == CommandBufferImpl< TTarget >::DataType::State_Begun );
    const DynamicDescriptorInfo::DataType& dynamicDescriptor =
        GetDynamicDescriptorInfoData< TTarget >( pThis->ToData()->pGfxRootSignature, indexDynamicDescriptor );
    pThis->SetConstantBuffer( dynamicDescriptor.shaderSlot,
        static_cast< ShaderStage >( dynamicDescriptor.shaderStage ), constantBufferAddress, size );
}

template< typename TTarget >
void SetRootTextureAndSampler( CommandBufferImpl< TTarget >* pThis, PipelineType, int indexDynamicDescriptor,
    const TextureViewImpl< TTarget >* pTextureView, const SamplerImpl< TTarget >* pSampler ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pThis );
    NN_SDK_ASSERT( pThis->ToData()->state == CommandBufferImpl< TTarget >::DataType::State_Begun );
    const DynamicDescriptorInfo::DataType& dynamicDescriptor =
        GetDynamicDescriptorInfoData< TTarget >( pThis->ToData()->pGfxRootSignature, indexDynamicDescriptor );
    pThis->SetTextureAndSampler( dynamicDescriptor.shaderSlot,
        static_cast< ShaderStage >( dynamicDescriptor.shaderStage ), pTextureView, pSampler );
}

}
}
}

NN_PRAGMA_POP_WARNINGS
