﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_StaticAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Middleware.h>
#include <nn/nn_Version.h>

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_Common.h>
#include <nn/gfx/gfx_Config.h>
#include <nn/gfx/gfx_RootSignatureInfo.h>
#include <nn/gfx/gfx_GpuAddress.h>
#include <nn/gfx/gfx_TextureInfo.h>
#include <nn/gfx/gfx_ResShader.h>

#include <nn/gfx/detail/gfx_Misc.h>
#include <nn/gfx/detail/gfx_Log.h>

#include "gfx_CommonHelper.h"

#define NW_MIDDLEWARE_SYMBOL(buildOption) "NintendoSDK_gfx-" NN_MACRO_STRINGIZE(NN_NX_ADDON_VERSION_MAJOR) "_" NN_MACRO_STRINGIZE(NN_NX_ADDON_VERSION_MINOR) "_" NN_MACRO_STRINGIZE(NN_NX_ADDON_VERSION_MICRO) "-" #buildOption

#if defined(NN_SDK_BUILD_DEBUG)
NN_DEFINE_MIDDLEWARE( g_MiddlewareInfo, "Nintendo", NW_MIDDLEWARE_SYMBOL( Debug ) );
#elif defined(NN_SDK_BUILD_DEVELOP)
NN_DEFINE_MIDDLEWARE( g_MiddlewareInfo, "Nintendo", NW_MIDDLEWARE_SYMBOL( Develop ) );
#elif defined(NN_SDK_BUILD_RELEASE)
NN_DEFINE_MIDDLEWARE( g_MiddlewareInfo, "Nintendo", NW_MIDDLEWARE_SYMBOL( Release ) );
#endif

namespace nn {
namespace gfx {
namespace detail {

void UseMiddleWare() NN_NOEXCEPT
{
    NN_USING_MIDDLEWARE( g_MiddlewareInfo );
}

int GetBlockWidth( ChannelFormat format ) NN_NOEXCEPT
{
    switch( format )
    {
    case ChannelFormat_Pvrtc1_2Bpp:
    case ChannelFormat_Pvrtc1_Alpha_2Bpp:
        return 16;
    case ChannelFormat_Pvrtc1_4Bpp:
    case ChannelFormat_Pvrtc1_Alpha_4Bpp:
    case ChannelFormat_Pvrtc2_Alpha_2Bpp:
        return 8;
    case ChannelFormat_Astc_5x4:
    case ChannelFormat_Astc_5x5:
        return 5;
    case ChannelFormat_Astc_6x5:
    case ChannelFormat_Astc_6x6:
        return 6;
    case ChannelFormat_Astc_8x5:
    case ChannelFormat_Astc_8x6:
    case ChannelFormat_Astc_8x8:
        return 8;
    case ChannelFormat_Astc_10x5:
    case ChannelFormat_Astc_10x6:
    case ChannelFormat_Astc_10x8:
    case ChannelFormat_Astc_10x10:
        return 10;
    case ChannelFormat_Astc_12x10:
    case ChannelFormat_Astc_12x12:
        return 12;
    default:
        return 4;
    }
}

int GetBlockHeight( ChannelFormat format ) NN_NOEXCEPT
{
    switch( format )
    {
    case ChannelFormat_Astc_5x5:
    case ChannelFormat_Astc_6x5:
    case ChannelFormat_Astc_8x5:
    case ChannelFormat_Astc_10x5:
        return 5;
    case ChannelFormat_Astc_6x6:
    case ChannelFormat_Astc_8x6:
    case ChannelFormat_Astc_10x6:
        return 6;
    case ChannelFormat_Pvrtc1_2Bpp:
    case ChannelFormat_Pvrtc1_Alpha_2Bpp:
    case ChannelFormat_Pvrtc1_4Bpp:
    case ChannelFormat_Pvrtc1_Alpha_4Bpp:
    case ChannelFormat_Astc_8x8:
    case ChannelFormat_Astc_10x8:
        return 8;
    case ChannelFormat_Astc_10x10:
    case ChannelFormat_Astc_12x10:
        return 10;
    case ChannelFormat_Astc_12x12:
        return 12;
    default:
        return 4;
    }
}

bool IsCompressedFormat( ChannelFormat format ) NN_NOEXCEPT
{
    switch( format )
    {
    case ChannelFormat_Bc1:
    case ChannelFormat_Bc2:
    case ChannelFormat_Bc3:
    case ChannelFormat_Bc4:
    case ChannelFormat_Bc5:
    case ChannelFormat_Bc6:
    case ChannelFormat_Bc7:
    case ChannelFormat_Eac_R11:
    case ChannelFormat_Eac_R11_G11:
    case ChannelFormat_Etc1:
    case ChannelFormat_Etc2:
    case ChannelFormat_Etc2_Mask:
    case ChannelFormat_Etc2_Alpha:
    case ChannelFormat_Pvrtc1_2Bpp:
    case ChannelFormat_Pvrtc1_4Bpp:
    case ChannelFormat_Pvrtc1_Alpha_2Bpp:
    case ChannelFormat_Pvrtc1_Alpha_4Bpp:
    case ChannelFormat_Pvrtc2_Alpha_2Bpp:
    case ChannelFormat_Pvrtc2_Alpha_4Bpp:
    case ChannelFormat_Astc_4x4:
    case ChannelFormat_Astc_5x4:
    case ChannelFormat_Astc_5x5:
    case ChannelFormat_Astc_6x5:
    case ChannelFormat_Astc_6x6:
    case ChannelFormat_Astc_8x5:
    case ChannelFormat_Astc_8x6:
    case ChannelFormat_Astc_8x8:
    case ChannelFormat_Astc_10x5:
    case ChannelFormat_Astc_10x6:
    case ChannelFormat_Astc_10x8:
    case ChannelFormat_Astc_10x10:
    case ChannelFormat_Astc_12x10:
    case ChannelFormat_Astc_12x12:
        return true;
    default:
        return false;
    }
}

bool IsSrgbFormat( TypeFormat format ) NN_NOEXCEPT
{
    return format == TypeFormat_UnormSrgb;
}

int GetBytePerPixel( ChannelFormat format ) NN_NOEXCEPT
{
    switch( format )
    {
    case ChannelFormat_Bc1:
    case ChannelFormat_Bc4:
    case ChannelFormat_Eac_R11:
    case ChannelFormat_Etc1:
    case ChannelFormat_Etc2:
    case ChannelFormat_Etc2_Mask:
    case ChannelFormat_Pvrtc2_Alpha_2Bpp:
    case ChannelFormat_Pvrtc2_Alpha_4Bpp:
        return 8;
    case ChannelFormat_Bc2:
    case ChannelFormat_Bc3:
    case ChannelFormat_Bc5:
    case ChannelFormat_Bc6:
    case ChannelFormat_Bc7:
    case ChannelFormat_Eac_R11_G11:
    case ChannelFormat_Etc2_Alpha:
    case ChannelFormat_Astc_4x4:
    case ChannelFormat_Astc_5x4:
    case ChannelFormat_Astc_5x5:
    case ChannelFormat_Astc_6x5:
    case ChannelFormat_Astc_6x6:
    case ChannelFormat_Astc_8x5:
    case ChannelFormat_Astc_8x6:
    case ChannelFormat_Astc_8x8:
    case ChannelFormat_Astc_10x5:
    case ChannelFormat_Astc_10x6:
    case ChannelFormat_Astc_10x8:
    case ChannelFormat_Astc_10x10:
    case ChannelFormat_Astc_12x10:
    case ChannelFormat_Astc_12x12:
        return 16;
    case ChannelFormat_Pvrtc1_2Bpp:
    case ChannelFormat_Pvrtc1_4Bpp:
    case ChannelFormat_Pvrtc1_Alpha_2Bpp:
    case ChannelFormat_Pvrtc1_Alpha_4Bpp:
        return 32;
    case ChannelFormat_B5_G5_R5_A1:
        return 2;
    default:
        {
            if( format <= ChannelFormat_R8 )
            {
                return 1;
            }
            if( format <= ChannelFormat_R16 )
            {
                return 2;
            }
            if( format <= ChannelFormat_R32 )
            {
                return 4;
            }
            if( format <= ChannelFormat_R32_G32 )
            {
                return 8;
            }
            if( format <= ChannelFormat_R32_G32_B32 )
            {
                return 12;
            }
        }
        return 16;
    }
}

size_t CalculateImageSize( ChannelFormat format, uint32_t width, uint32_t height, uint32_t depth ) NN_NOEXCEPT
{
    if( IsCompressedFormat( format ) )
    {
        int blockWidth = GetBlockWidth( format );
        int blockHeight = GetBlockHeight( format );
        width = ( width + blockWidth - 1 ) / blockWidth;
        height = ( height + blockHeight - 1 ) / blockHeight;
    }

    return width * height * depth * GetBytePerPixel( format );
}

int GetChannelCount( ChannelFormat format ) NN_NOEXCEPT
{
    static const int s_ChannelCountTable[] =
    {
        0, // ChannelFormat_Undefined
        2, // ChannelFormat_R4_G4
        1, // ChannelFormat_R8
        4, // ChannelFormat_R4_G4_B4_A4
        4, // ChannelFormat_A4_B4_G4_R4
        4, // ChannelFormat_R5_G5_B5_A1
        4, // ChannelFormat_A1_B5_G5_R5
        3, // ChannelFormat_R5_G6_B5
        3, // ChannelFormat_B5_G6_R5
        2, // ChannelFormat_R8_G8
        1, // ChannelFormat_R16
        4, // ChannelFormat_R8_G8_B8_A8
        4, // ChannelFormat_B8_G8_R8_A8
        4, // ChannelFormat_R9_G9_B9_E5
        4, // ChannelFormat_R10_G10_B10_A2
        3, // ChannelFormat_R11_G11_B10
        3, // ChannelFormat_B10_G11_R11
        3, // ChannelFormat_R10_G11_B11
        2, // ChannelFormat_R16_G16
        2, // ChannelFormat_R24_G8
        1, // ChannelFormat_R32
        4, // ChannelFormat_R16_G16_B16_A16
        2, // ChannelFormat_R32_G8_X24
        2, // ChannelFormat_R32_G32
        3, // ChannelFormat_R32_G32_B32
        4, // ChannelFormat_R32_G32_B32_A32
        4, // ChannelFormat_Bc1
        4, // ChannelFormat_Bc2
        4, // ChannelFormat_Bc3
        1, // ChannelFormat_Bc4
        2, // ChannelFormat_Bc5
        3, // ChannelFormat_Bc6
        4, // ChannelFormat_Bc7
        1, // ChannelFormat_Eac_R11
        2, // ChannelFormat_Eac_R11_G11
        3, // ChannelFormat_Etc
        3, // ChannelFormat_Etc2
        4, // ChannelFormat_Etc2_Mask
        4, // ChannelFormat_Etc2_Alpha
        3, // ChannelFormat_Pvrtc_2bpp
        3, // ChannelFormat_Pvrtc_4bpp
        4, // ChannelFormat_Pvrtc_Alpha_2bpp
        4, // ChannelFormat_Pvrtc_Alpha_4bpp
        4, // ChannelFormat_Pvrtc2_Alpha_2bpp
        4, // ChannelFormat_Pvrtc2_Alpha_4bpp
        4, // ChannelFormat_Astc_4x4
        4, // ChannelFormat_Astc_5x4
        4, // ChannelFormat_Astc_5x5
        4, // ChannelFormat_Astc_6x5
        4, // ChannelFormat_Astc_6x6
        4, // ChannelFormat_Astc_8x5
        4, // ChannelFormat_Astc_8x6
        4, // ChannelFormat_Astc_8x8
        4, // ChannelFormat_Astc_10x5
        4, // ChannelFormat_Astc_10x6
        4, // ChannelFormat_Astc_10x8
        4, // ChannelFormat_Astc_10x10
        4, // ChannelFormat_Astc_12x10
        4, // ChannelFormat_Astc_12x12
        4 // ChannelFormat_B5_G5_R5_A1
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_ChannelCountTable ) == ChannelFormat_End );

    return s_ChannelCountTable[ format ];
}

size_t CalculateRowSize( uint32_t width, ChannelFormat format ) NN_NOEXCEPT
{
    if( IsCompressedFormat( format ) )
    {
        int blockWidth = GetBlockWidth( format );
        width = ( width + blockWidth - 1 ) / blockWidth;
    }

    return width * GetBytePerPixel( format );
}

bool IsValidMemoryPoolProperty( int value ) NN_NOEXCEPT
{
    int cpuPageProperty = value & 0x07;
    int gpuPageProperty = value & 0x38;
    return ( cpuPageProperty == MemoryPoolProperty_CpuInvisible
        || cpuPageProperty == MemoryPoolProperty_CpuUncached
        || cpuPageProperty == MemoryPoolProperty_CpuCached )
        && ( gpuPageProperty == MemoryPoolProperty_GpuInvisible
        || gpuPageProperty == MemoryPoolProperty_GpuUncached
        || gpuPageProperty == MemoryPoolProperty_GpuCached );
}

ImageDimension GetImageDimension(
    ImageStorageDimension imageStorageDimension, bool isArray, bool isMultisample ) NN_NOEXCEPT
{
    ImageDimension ret;
    switch( imageStorageDimension )
    {
    case ImageStorageDimension_1d:
        {
            ret = isArray ? ImageDimension_1dArray : ImageDimension_1d;
        }
        break;
    case ImageStorageDimension_2d:
        {
            if( isArray )
            {
                ret = isMultisample ? ImageDimension_2dMultisampleArray : ImageDimension_2dArray;
            }
            else
            {
                ret = isMultisample ? ImageDimension_2dMultisample : ImageDimension_2d;
            }
        }
        break;
    case ImageStorageDimension_3d:
        {
            ret = ImageDimension_3d;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    return ret;
}

bool CheckBinaryTarget( const ResShaderContainerData& resShaderContainer,
    int lowLevelApi, int apiVersion ) NN_NOEXCEPT
{
    NN_UNUSED( apiVersion );

    static const char* s_LowLevelApiToStringTable[] =
    {
        "Common",
        "Gl",
        "Gx",
        "D3d",
        "Nvn",
        "Vk"
    };
    NN_SDK_ASSERT_RANGE( lowLevelApi, 0, static_cast< int >(
        NN_GFX_ARRAY_LENGTH( s_LowLevelApiToStringTable ) ) );

    int targetApi = resShaderContainer.targetApiType;
    if( targetApi != lowLevelApi )
    {
        const char* pTargetApiString = targetApi >= 0 &&
            targetApi < NN_GFX_ARRAY_LENGTH( s_LowLevelApiToStringTable )
            ? s_LowLevelApiToStringTable[ targetApi ] : "Unknown";
        NN_UNUSED( pTargetApiString );
        NN_DETAIL_GFX_ERROR( "API mismatch:\n\tShader Binary: %s\n\tRuntime: %s\n",
            pTargetApiString, s_LowLevelApiToStringTable[ lowLevelApi ] );
        return false;
    }

    return true;
}

}
}
}
