﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/ens_SimpleMessagePackWriter.h>
#include <nn/ens/detail/ens_Common.h>
#include <nn/ens/detail/util/ens_MessagePackWriter.h>
#include <nn/ens/detail/util/ens_MessagePackReader.h>

namespace nn { namespace ens {

SimpleMessagePackWriter::SimpleMessagePackWriter(void* pBuffer, size_t size) NN_NOEXCEPT
    : m_pBuffer(reinterpret_cast<nn::Bit8*>(pBuffer))
    , m_Size(size)
    , m_Written(0)
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_ASSERT_GREATER(size, 0u);
}

bool SimpleMessagePackWriter::WriteMap(size_t num) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(num, UINT32_MAX);

    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteMap(num));
    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteArray(size_t num) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(num, UINT32_MAX);

    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteArray(num));
    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteValue(bool value) NN_NOEXCEPT
{
    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteValue(int8_t value) NN_NOEXCEPT
{
    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteValue(int16_t value) NN_NOEXCEPT
{
    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteValue(int32_t value) NN_NOEXCEPT
{
    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteValue(int64_t value) NN_NOEXCEPT
{
    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteValue(uint8_t value) NN_NOEXCEPT
{
    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteValue(uint16_t value) NN_NOEXCEPT
{
    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteValue(uint32_t value) NN_NOEXCEPT
{
    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteValue(uint64_t value) NN_NOEXCEPT
{
    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteValue(double value) NN_NOEXCEPT
{
    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteValue(const char* pValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pValue);

    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    size_t valueLength = static_cast<size_t>(nn::util::Strnlen(pValue, INT_MAX));

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(pValue, valueLength));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteValue(const void* pValue, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pValue);
    NN_SDK_REQUIRES_LESS_EQUAL(size, UINT32_MAX);

    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(pValue, size));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteKeyMap(const char* pKey, size_t num) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKey);
    NN_SDK_REQUIRES_LESS_EQUAL(num, UINT32_MAX);

    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteKey(pKey));
    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteMap(num));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteKeyArray(const char* pKey, size_t num) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKey);
    NN_SDK_REQUIRES_LESS_EQUAL(num, UINT32_MAX);

    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteKey(pKey));
    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteArray(num));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteKeyValue(const char* pKey, bool value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKey);

    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteKey(pKey));
    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteKeyValue(const char* pKey, int8_t value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKey);

    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteKey(pKey));
    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteKeyValue(const char* pKey, int16_t value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKey);

    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteKey(pKey));
    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteKeyValue(const char* pKey, int32_t value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKey);

    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteKey(pKey));
    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteKeyValue(const char* pKey, int64_t value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKey);

    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteKey(pKey));
    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteKeyValue(const char* pKey, uint8_t value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKey);

    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteKey(pKey));
    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteKeyValue(const char* pKey, uint16_t value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKey);

    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteKey(pKey));
    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteKeyValue(const char* pKey, uint32_t value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKey);

    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteKey(pKey));
    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteKeyValue(const char* pKey, uint64_t value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKey);

    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteKey(pKey));
    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteKeyValue(const char* pKey, double value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKey);

    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteKey(pKey));
    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(value));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteKeyValue(const char* pKey, const char* pValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKey);
    NN_SDK_REQUIRES_NOT_NULL(pValue);

    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    size_t valueLength = static_cast<size_t>(nn::util::Strnlen(pValue, INT_MAX));

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteKey(pKey));
    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(pValue, valueLength));

    m_Written += writer.GetWrittenSize();

    return true;
}

bool SimpleMessagePackWriter::WriteKeyValue(const char* pKey, const void* pValue, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKey);
    NN_SDK_REQUIRES_NOT_NULL(pValue);
    NN_SDK_REQUIRES_LESS_EQUAL(size, UINT32_MAX);

    size_t remainSize = m_Size - m_Written;

    if (remainSize == 0)
    {
        return false;
    }

    detail::util::MessagePackWriter writer;

    writer.SetBuffer(m_pBuffer + m_Written, remainSize);

    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteKey(pKey));
    NN_DETAIL_ENS_RETURN_IF_FALSE(writer.WriteValue(pValue, size));

    m_Written += writer.GetWrittenSize();

    return true;
}

size_t SimpleMessagePackWriter::GetWrittenSize() const NN_NOEXCEPT
{
    return m_Written;
}

bool SimpleMessagePackWriter::IsValid() const NN_NOEXCEPT
{
    detail::util::MessagePackReader reader;

    return reader.Read(m_pBuffer, m_Written);
}

}}
