﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/ens_SearchQueryBuilder.h>
#include <nn/ens/detail/util/ens_HttpUtility.h>
#include <cstring>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_Utf8StringUtil.h>

namespace nn { namespace ens {

namespace
{
    // キーに指定できる文字かどうかを判定する。
    bool IsAcceptableKeyChar(char c) NN_NOEXCEPT
    {
        return (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') || (c >= '0' && c <= '9') || (c == '-') || (c == '_');
    }
}

SearchQueryBuilder::SearchQueryBuilder(char* pBuffer, size_t size) NN_NOEXCEPT
    : m_pBuffer(pBuffer)
    , m_Size(size)
    , m_Length(0)
{
    NN_SDK_REQUIRES_NOT_NULL(m_pBuffer);
    NN_SDK_REQUIRES_GREATER(m_Size, 0u);

    m_pBuffer[0] = '\0';
}

bool SearchQueryBuilder::Set(const char* pKey, const char* pValue) NN_NOEXCEPT
{
    return SetImpl(pKey, pValue, false);
}

bool SearchQueryBuilder::SetMulti(const char* pKey, const char* pValue) NN_NOEXCEPT
{
    return SetImpl(pKey, pValue, true);
}

const char* SearchQueryBuilder::GetString() const NN_NOEXCEPT
{
    return m_pBuffer;
}

bool SearchQueryBuilder::SetImpl(const char* pKey, const char* pValue, bool isMulti) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKey);
    NN_SDK_REQUIRES_NOT_NULL(pValue);

    size_t terminal = m_Length;

    NN_UTIL_SCOPE_EXIT
    {
        m_Length = terminal;
        m_pBuffer[m_Length] = '\0';
    };

    if (!WriteKey(pKey, isMulti))
    {
        return false;
    }
    if (!WriteValue(pValue))
    {
        return false;
    }

    terminal = m_Length;

    return true;
}

bool SearchQueryBuilder::WriteKey(const char* pKey, bool isMulti) NN_NOEXCEPT
{
    size_t keyLength = 0;

    for (; pKey[keyLength] != '\0'; keyLength++)
    {
        if (!IsAcceptableKeyChar(pKey[keyLength]))
        {
            return false;
        }
    }
    if (keyLength == 0)
    {
        return false;
    }

    size_t remainSize = m_Size - m_Length;

    // isMulti ? "&q[<key>][]=" : "&q[<key>]="
    size_t written = static_cast<size_t>(nn::util::SNPrintf(&m_pBuffer[m_Length], remainSize,
        isMulti ? "&q%%5B%s%%5D%%5B%%5D=" : "&q%%5B%s%%5D=", pKey));

    if (written >= remainSize)
    {
        return false;
    }

    if (!isMulti)
    {
        // 同一キーが既に存在していたら、 std::strstr は &m_pBuffer[m_Length] より前を返す。
        NN_SDK_REQUIRES_EQUAL(std::strstr(m_pBuffer, &m_pBuffer[m_Length]), &m_pBuffer[m_Length]);
    }

    m_Length += written;

    return true;
}

bool SearchQueryBuilder::WriteValue(const char* pValue) NN_NOEXCEPT
{
    size_t length = static_cast<size_t>(nn::util::Strnlen(pValue, INT_MAX));

    if (length == 0)
    {
        return false;
    }

    size_t escapedLength = 0;

    if (!detail::util::EscapeUriDataString(&escapedLength, &m_pBuffer[m_Length], m_Size - m_Length, pValue, length))
    {
        return false;
    }

    m_Length += escapedLength;

    return true;
}

}}
