﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/util/ens_JsonReader.h>
#include <nn/ens/detail/util/rapidjson/ens_RapidJsonMemoryInputStream.h>

namespace nn { namespace ens { namespace detail { namespace util {

JsonReader::JsonReader() NN_NOEXCEPT
{
}

void JsonReader::SetCallback(ReadCallback pCallback, void* pParam) NN_NOEXCEPT
{
    m_EventHandler.SetCallback(pCallback, pParam);
}

bool JsonReader::Read(const void* pData, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);

    if (size == 0)
    {
        return false;
    }

    m_EventHandler.Clear();

    char stringBuffer[512];
    char streamBuffer[64];

    rapidjson::RapidJsonMemoryInputStream stream;
    rapidjson::NullAllocator allocator;

    stream.SetBuffer(stringBuffer, sizeof (stringBuffer), streamBuffer, sizeof (streamBuffer));
    stream.Open(pData, size);

    const unsigned ParseFlags = RAPIDJSON_NAMESPACE::kParseInsituFlag | RAPIDJSON_NAMESPACE::kParseValidateEncodingFlag;

    RAPIDJSON_NAMESPACE::ParseResult result = rapidjson::Reader(&allocator, 1).Parse<ParseFlags>(stream, m_EventHandler);

    if (stream.IsStringOverflowDetected())
    {
        return false;
    }
    if (result.Code() != RAPIDJSON_NAMESPACE::kParseErrorNone)
    {
        return false;
    }

    return true;
}

JsonReader::EventHandler::EventHandler() NN_NOEXCEPT
    : m_pCallback(nullptr)
    , m_pCallbackParam(nullptr)
{
}

void JsonReader::EventHandler::Clear() NN_NOEXCEPT
{
    m_JsonPath.Clear();
}

void JsonReader::EventHandler::SetCallback(ReadCallback pCallback, void* pParam) NN_NOEXCEPT
{
    m_pCallback = pCallback;
    m_pCallbackParam = pParam;
}

bool JsonReader::EventHandler::StartObject() NN_NOEXCEPT
{
    NN_DETAIL_ENS_RETURN_IF_FALSE(m_JsonPath.PushObject());

    DataHolder holder = {DataHolderType_BeginMap};

    NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForNode(holder));

    return true;
}

bool JsonReader::EventHandler::EndObject(size_t num) NN_NOEXCEPT
{
    m_JsonPath.Pop();

    DataHolder holder = {DataHolderType_EndMap};
    holder.integer.u64 = num;

    NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForNode(holder));

    m_JsonPath.PopIfParentNodeIsKey();

    return true;
}

bool JsonReader::EventHandler::StartArray() NN_NOEXCEPT
{
    NN_DETAIL_ENS_RETURN_IF_FALSE(m_JsonPath.PushArray());

    DataHolder holder = {DataHolderType_BeginArray};

    NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForNode(holder));

    return true;
}

bool JsonReader::EventHandler::EndArray(size_t num) NN_NOEXCEPT
{
    m_JsonPath.Pop();

    DataHolder holder = {DataHolderType_EndArray};
    holder.integer.u64 = num;

    NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForNode(holder));

    m_JsonPath.PopIfParentNodeIsKey();

    return true;
}

bool JsonReader::EventHandler::Key(const char* pKey, size_t length, bool copy) NN_NOEXCEPT
{
    NN_UNUSED(copy);

    NN_DETAIL_ENS_RETURN_IF_FALSE(m_JsonPath.PushKey(pKey, length));

    return true;
}

bool JsonReader::EventHandler::Null() NN_NOEXCEPT
{
    DataHolder holder = {DataHolderType_Nil};

    NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));

    return true;
}

bool JsonReader::EventHandler::Bool(bool value) NN_NOEXCEPT
{
    DataHolder holder = {DataHolderType_Boolean};
    holder.boolean = value;

    NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));

    return true;
}

bool JsonReader::EventHandler::Int(int32_t value) NN_NOEXCEPT
{
    DataHolder holder = {DataHolderType_Integer};
    holder.integer.s64 = value;
    holder.integer.isSigned = true;

    NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));

    return true;
}

bool JsonReader::EventHandler::Uint(uint32_t value) NN_NOEXCEPT
{
    DataHolder holder = {DataHolderType_Integer};
    holder.integer.s64 = value;
    holder.integer.isSigned = true;

    NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));

    return true;
}

bool JsonReader::EventHandler::Int64(int64_t value) NN_NOEXCEPT
{
    DataHolder holder = {DataHolderType_Integer};
    holder.integer.s64 = value;
    holder.integer.isSigned = true;

    NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));

    return true;
}

bool JsonReader::EventHandler::Uint64(uint64_t value) NN_NOEXCEPT
{
    DataHolder holder = {DataHolderType_Integer};
    holder.integer.u64 = value;
    holder.integer.isSigned = false;

    NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));

    return true;
}

bool JsonReader::EventHandler::Double(double value) NN_NOEXCEPT
{
    DataHolder holder = {DataHolderType_Float};
    holder.float64 = value;

    NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));

    return true;
}

bool JsonReader::EventHandler::String(const char* pValue, size_t length, bool copy) NN_NOEXCEPT
{
    NN_UNUSED(copy);

    DataHolder holder = {DataHolderType_String};
    holder.string.pValue = pValue;
    holder.string.length = length;

    NN_DETAIL_ENS_RETURN_IF_FALSE(DoCallbackForLeaf(holder));

    return true;
}

bool JsonReader::EventHandler::DoCallbackForNode(const DataHolder& holder) NN_NOEXCEPT
{
    if (m_pCallback)
    {
        NN_DETAIL_ENS_RETURN_IF_FALSE(m_pCallback(m_JsonPath, holder, m_pCallbackParam));
    }

    return true;
}

bool JsonReader::EventHandler::DoCallbackForLeaf(const DataHolder& holder) NN_NOEXCEPT
{
    NN_DETAIL_ENS_RETURN_IF_FALSE(m_JsonPath.IncrementArrayCountIfParentNodeIsArray());

    if (m_pCallback)
    {
        NN_DETAIL_ENS_RETURN_IF_FALSE(m_pCallback(m_JsonPath, holder, m_pCallbackParam));
    }

    m_JsonPath.PopIfParentNodeIsKey();

    return true;
}

}}}}
