﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/util/ens_JsonPath.h>

namespace nn { namespace ens { namespace detail { namespace util {

NN_STATIC_ASSERT(JsonPath::PathSize <= UINT16_MAX);

JsonPath::JsonPath() NN_NOEXCEPT
{
    Clear();
}

void JsonPath::Clear() NN_NOEXCEPT
{
    m_Depth = 1;
    m_PathLength = 1;

    m_Nodes[0].type = NodeType_Root;
    m_Nodes[0].positionInPath = 0;
    m_Nodes[0].arrayCount = 0;

    m_Path[0] = '$';
    m_Path[1] = '\0';
}

bool JsonPath::Compare(const char* pPath) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPath);

    return (nn::util::Strncmp(m_Path, pPath, sizeof (m_Path)) == 0);
}

bool JsonPath::Match(const char* pPath) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPath);

    const char* p1 = pPath;
    const char* p2 = m_Path;

    while (*p1)
    {
        const char* s1 = p1;
        const char* s2 = p2;

        const char* e1 = s1;
        const char* e2 = s2;
        const char* w = nullptr;

        // キー名の末尾まで移動する。
        while (*e1 && *e1 != '.')
        {
            if (*e1 == '*')
            {
                // 1 階層にワイルドカード文字が複数回出てはいけない。
                NN_SDK_REQUIRES(!w);
                w = e1;
            }
            e1++;
        }
        while (*e2 && *e2 != '.')
        {
            e2++;
        }

        // 次のノードの先頭位置を取得する。
        p1 = e1 + ((*e1 == '.') ? 1 : 0);
        p2 = e2 + ((*e2 == '.') ? 1 : 0);

        if (w)
        {
            // ワイルドカード文字出現まで文字列比較を行う。
            if (s1 < w)
            {
                while (s1 != w && s2 != e2 && *s1 == *s2)
                {
                    s1++;
                    s2++;
                }
                if (s1 != w)
                {
                    return false;
                }
            }
            if (e1 > w)
            {
                while (e1 != w && s2 != e2 && *e1 == *e2)
                {
                    e1--;
                    e2--;
                }
                if (e1 != w)
                {
                    return false;
                }
            }
        }
        else
        {
            // ワイルドカードが指定されていない場合、単純比較する。
            while (s1 != e1 && s2 != e2 && *s1 == *s2)
            {
                s1++;
                s2++;
            }
            if (s1 != e1 || s2 != e2)
            {
                return false;
            }
        }
    }

    return (*p2 == '\0');
}

const char* JsonPath::GetString() const NN_NOEXCEPT
{
    return m_Path;
}

bool JsonPath::PushObject() NN_NOEXCEPT
{
    if (m_Depth >= NN_ARRAY_SIZE(m_Nodes))
    {
        return false;
    }
    if (!IncrementArrayCountIfParentNodeIsArray())
    {
        return false;
    }

    m_Nodes[m_Depth].type = NodeType_Object;
    m_Nodes[m_Depth].positionInPath = m_PathLength;
    m_Nodes[m_Depth].arrayCount = 0;
    m_Depth++;

    return true;
}

bool JsonPath::PushArray() NN_NOEXCEPT
{
    if (m_Depth >= NN_ARRAY_SIZE(m_Nodes))
    {
        return false;
    }
    if (!IncrementArrayCountIfParentNodeIsArray())
    {
        return false;
    }

    if (GetPathRemainSize() < sizeof ("[]"))
    {
        return false;
    }

    m_Nodes[m_Depth].type = NodeType_Array;
    m_Nodes[m_Depth].positionInPath = m_PathLength;
    m_Nodes[m_Depth].arrayCount = 0;
    m_Depth++;

    WriteArrayBrackets();

    return true;
}

bool JsonPath::PushKey(const char* pKey, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKey);

    if (m_Depth >= NN_ARRAY_SIZE(m_Nodes))
    {
        return false;
    }
    if (!IncrementArrayCountIfParentNodeIsArray())
    {
        return false;
    }

    if (GetPathRemainSize() < length + 1 + 1) // .<key> + null 終端
    {
        return false;
    }

    m_Nodes[m_Depth].type = NodeType_Key;
    m_Nodes[m_Depth].positionInPath = m_PathLength;
    m_Nodes[m_Depth].arrayCount = 0;
    m_Depth++;

    m_Path[m_PathLength++] = '.';

    std::memcpy(&m_Path[m_PathLength], pKey, length);
    m_PathLength += static_cast<uint16_t>(length);

    m_Path[m_PathLength] = '\0';

    return true;
}

bool JsonPath::IncrementArrayCountIfParentNodeIsArray() NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(m_Depth, 1u);

    uint16_t parent = m_Depth - 1;

    if (m_Nodes[parent].type != NodeType_Array)
    {
        return true;
    }

    char index[sizeof ("[4294967295]")] = {};

    uint16_t indexLength = static_cast<uint16_t>(nn::util::SNPrintf(index, sizeof (index), "[%u]", m_Nodes[parent].arrayCount));

    if (GetPathRemainSize(parent) < static_cast<size_t>(indexLength + 1))
    {
        return false;
    }

    m_PathLength = m_Nodes[parent].positionInPath;

    std::memcpy(&m_Path[m_Nodes[parent].positionInPath], index, indexLength);
    m_PathLength += indexLength;

    m_Path[m_PathLength] = '\0';

    m_Nodes[parent].arrayCount++;

    return true;
}

void JsonPath::Pop() NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(m_Depth, 1u);

    m_Depth--;

    if (m_Nodes[m_Depth].type == NodeType_Array)
    {
        m_PathLength = m_Nodes[m_Depth].positionInPath;

        // 配列の開始時点で "[]" が書き込めることは確認しているため、残りサイズの確認はしない。
        WriteArrayBrackets();
    }
    else if (m_Nodes[m_Depth].type != NodeType_Object)
    {
        m_PathLength = m_Nodes[m_Depth].positionInPath;

        m_Path[m_PathLength] = '\0';
    }
}

void JsonPath::PopIfParentNodeIsKey() NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(m_Depth, 1u);

    if (m_Nodes[m_Depth - 1].type == NodeType_Key)
    {
        Pop();
    }
}

size_t JsonPath::GetPathRemainSize() const NN_NOEXCEPT
{
    return sizeof (m_Path) - m_PathLength;
}

size_t JsonPath::GetPathRemainSize(uint16_t depth) const NN_NOEXCEPT
{
    return sizeof (m_Path) - m_Nodes[depth].positionInPath;
}

void JsonPath::WriteArrayBrackets() NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(GetPathRemainSize(), sizeof ("[]"));

    m_Path[m_PathLength++] = '[';
    m_Path[m_PathLength++] = ']';
    m_Path[m_PathLength] = '\0';
}

}}}}
