﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/core/ens_HttpTask.h>
#include <nn/ens/detail/util/ens_MessagePackReader.h>
#include <nn/ens/detail/util/ens_ResponseStructureReader.h>
#include <nn/ens/detail/util/ens_ServiceErrorHandler.h>
#include <nn/nn_Version.h>
#include <nn/socket.h>
#include <nn/ssl/ssl_Context.h>

#define USER_AGENT_ENGINE \
    "libcurl/"                                      \
    NN_MACRO_STRINGIZE(LIBCURL_VERSION_MAJOR) "."   \
    NN_MACRO_STRINGIZE(LIBCURL_VERSION_MINOR) "."   \
    NN_MACRO_STRINGIZE(LIBCURL_VERSION_PATCH)

#if defined(NN_BUILD_CONFIG_SPEC_NX)
#define USER_AGENT_PLATFORM "HAC"
#else
#define USER_AGENT_PLATFORM "Windows"
#endif

#define USER_AGENT_MODULE "nnEns"

#define USER_AGENT_SDK \
    "SDK "                                          \
    NN_MACRO_STRINGIZE(NN_SDK_VERSION_MAJOR) "."    \
    NN_MACRO_STRINGIZE(NN_SDK_VERSION_MINOR) "."    \
    NN_MACRO_STRINGIZE(NN_SDK_VERSION_MICRO) "."    \
    NN_MACRO_STRINGIZE(NN_SDK_VERSION_RELSTEP)

namespace nn { namespace ens { namespace detail { namespace core {

namespace
{
    const char UserAgent[] = USER_AGENT_ENGINE " (" USER_AGENT_PLATFORM "; " USER_AGENT_MODULE "; " USER_AGENT_SDK ")";
}

HttpTask::HttpTask() NN_NOEXCEPT
    : m_ReceivedSize(0)
    , m_pConnectionProvider(nullptr)
    , m_pToken(nullptr)
{
}

HttpTask::~HttpTask() NN_NOEXCEPT
{
}

void HttpTask::SetHttpConnectionProvider(HttpConnectionProvider* pConnectionProvider) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConnectionProvider);

    m_pConnectionProvider = pConnectionProvider;
}

void HttpTask::SetNetworkServiceAccountIdToken(const char* pToken) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pToken);

    m_pToken = pToken;
}

HttpConnectionProvider* HttpTask::GetHttpConnectionProvider() NN_NOEXCEPT
{
    return m_pConnectionProvider;
}

const char* HttpTask::GetNetworkServiceAccountIdToken() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsAsyncContextRegistered());

    return m_pToken;
}

nn::Result HttpTask::SetHttpCommonOptions(CURL* pCurl) NN_NOEXCEPT
{
    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_SOCKOPTFUNCTION, SockOptFunction));

    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_SSL_CTX_FUNCTION, SslCtxFunction));

    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_WRITEFUNCTION, HttpWriteFunction));
    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_WRITEDATA, this));

    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_CONNECTTIMEOUT, 10));

    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_LOW_SPEED_LIMIT, 1));
    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_LOW_SPEED_TIME, 30));

    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_USERAGENT, UserAgent));

    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_VERBOSE, LibrarySettings::GetHttpVerboseInformationOption()));

    NN_RESULT_SUCCESS;
}

nn::Result HttpTask::Perform() NN_NOEXCEPT
{
    m_ReceivedSize = 0;

    NN_RESULT_DO(m_pConnectionProvider->GetConnection().Perform(&GetCancelEvent()));

    CURL* pHandle = m_pConnectionProvider->GetConnection().GetHandle();

    DumpCommunicationPerformance(pHandle);

    long statusCode = 0;
    curl_easy_getinfo(pHandle, CURLINFO_RESPONSE_CODE, &statusCode);

    if (statusCode / 100 == 2)
    {
        NN_RESULT_SUCCESS;
    }

    detail::util::ResponseStructureReader<2> reader;

    char category[32] = {};
    char code[5] = {};

    reader.Add("$.category",
        category, sizeof (category));
    reader.Add("$.code",
        code, sizeof (code));

    NN_DETAIL_ENS_DUMP_RESPONSE(detail::util::MessagePackReader, GetResponse(), GetResponseSize());

    do
    {
        if (!reader.Read<detail::util::MessagePackReader>(GetResponse(), GetResponseSize()))
        {
            break;
        }
        if (nn::util::Strnlen(code, sizeof (code)) != 4)
        {
            break;
        }

        char* pEnd = nullptr;
        int32_t serverCode = static_cast<int32_t>(std::strtol(code, &pEnd, 10));

        if (!(*pEnd == '\0' && serverCode >= 0 && serverCode <= 9999))
        {
            break;
        }

        if (nn::util::Strncmp(category, "lib", sizeof (category)) == 0)
        {
            return detail::util::HandleServiceError(serverCode);
        }
        if (nn::util::Strncmp(category, "app", sizeof (category)) == 0)
        {
            SetErrorDetail(serverCode);
            NN_RESULT_THROW(ResultApplicationSpecificError());
        }
    }
    while (NN_STATIC_CONDITION(false));

    // エラーレスポンスが解析できなかった場合、ステータスコードを nn::Result に変換する。
    return detail::util::HandleHttpStatusCode(statusCode);
}

void* HttpTask::GetRequestBuffer() NN_NOEXCEPT
{
    return GetWorkBuffer();
}

size_t HttpTask::GetRequestBufferSize() const NN_NOEXCEPT
{
    return GetWorkBufferSize();
}

void* HttpTask::GetResponse() NN_NOEXCEPT
{
    return GetWorkBuffer();
}

size_t HttpTask::GetResponseSize() const NN_NOEXCEPT
{
    return m_ReceivedSize;
}

void HttpTask::DumpCommunicationPerformance(CURL* pCurl) NN_NOEXCEPT
{
#if defined(NN_SDK_BUILD_RELEASE)

    NN_UNUSED(pCurl);

#else

    if (!nn::ens::detail::LibrarySettings::IsCommunicationPerformanceDumpEnabled())
    {
        return;
    }

    double nameLookupTime = 0.0;
    curl_easy_getinfo(pCurl, CURLINFO_NAMELOOKUP_TIME, &nameLookupTime);
    double connectTime = 0.0;
    curl_easy_getinfo(pCurl, CURLINFO_CONNECT_TIME, &connectTime);
    double appConnectTime = 0.0;
    curl_easy_getinfo(pCurl, CURLINFO_APPCONNECT_TIME, &appConnectTime);
    double preTransferTime = 0.0;
    curl_easy_getinfo(pCurl, CURLINFO_PRETRANSFER_TIME, &preTransferTime);
    double totalTime = 0.0;
    curl_easy_getinfo(pCurl, CURLINFO_TOTAL_TIME, &totalTime);

    double uploadSize = 0.0;
    curl_easy_getinfo(pCurl, CURLINFO_SIZE_UPLOAD, &uploadSize);
    double downloadSize = 0.0;
    curl_easy_getinfo(pCurl, CURLINFO_SIZE_DOWNLOAD, &downloadSize);

    NN_DETAIL_ENS_INFO("==================================================\n");
    NN_DETAIL_ENS_INFO("Communication Performance\n");
    NN_DETAIL_ENS_INFO("==================================================\n");
    NN_DETAIL_ENS_INFO("Time Measurement:\n");
    NN_DETAIL_ENS_INFO("    Pre-Transfer(Name Lookup)   = %.2f ms\n", nameLookupTime * 1000.0);
    NN_DETAIL_ENS_INFO("    Pre-Transfer(Connect)       = %.2f ms\n", (connectTime - nameLookupTime) * 1000.0);
    NN_DETAIL_ENS_INFO("    Pre-Transfer(SSL Handshake) = %.2f ms\n", (appConnectTime - connectTime) * 1000.0);
    NN_DETAIL_ENS_INFO("    Pre-Transfer(Other)         = %.2f ms\n", (preTransferTime - appConnectTime) * 1000.0);
    NN_DETAIL_ENS_INFO("    Transfer                    = %.2f ms\n", (totalTime - preTransferTime) * 1000.0);
    NN_DETAIL_ENS_INFO("    Total                       = %.2f ms\n", totalTime * 1000.0);
    NN_DETAIL_ENS_INFO("Transfer:\n");
    NN_DETAIL_ENS_INFO("    Upload Size                 = %.2f KiB\n", uploadSize / 1024.0);
    NN_DETAIL_ENS_INFO("    Download Size               = %.2f KiB\n", downloadSize / 1024.0);
    NN_DETAIL_ENS_INFO("==================================================\n");

#endif
}

int HttpTask::SockOptFunction(void* pParam, curl_socket_t sock, curlsocktype purpose) NN_NOEXCEPT
{
    NN_UNUSED(pParam);
    NN_UNUSED(purpose);

    // 長時間通信を行わないことで NAT テーブルがクリアされ通信エラーが発生してしまうことを防ぐため、 TCP Keep-Alive を行う。
    // NAT テーブルのクリア時間を 30 秒と仮定し、25 秒後から開始する。
    const int TcpKeepAliveIdle = 25;
    const int TcpKeepAliveInterval = 10;
    const int Enable = 1;

    int httpKeepAliveTimeout = static_cast<int>(LibrarySettings::GetHttpKeepAliveTimeout().GetSeconds());

    if (httpKeepAliveTimeout > TcpKeepAliveIdle)
    {
        int probeCount = (httpKeepAliveTimeout - TcpKeepAliveIdle) / TcpKeepAliveInterval + 1;

        nn::socket::SetSockOpt(sock, nn::socket::Level::Sol_Socket, nn::socket::Option::So_KeepAlive,
            &Enable, sizeof (Enable));
        nn::socket::SetSockOpt(sock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepIdle,
            &TcpKeepAliveIdle, sizeof (TcpKeepAliveIdle));
        nn::socket::SetSockOpt(sock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepIntvl,
            &TcpKeepAliveInterval, sizeof (TcpKeepAliveInterval));
        nn::socket::SetSockOpt(sock, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_KeepCnt,
            &probeCount, sizeof (probeCount));
    }

    return CURL_SOCKOPT_OK;
}

CURLcode HttpTask::SslCtxFunction(CURL* pCurl, void* pSsl, void* pParam) NN_NOEXCEPT
{
    NN_UNUSED(pCurl);
    NN_UNUSED(pParam);

    nn::ssl::Context* pContext = reinterpret_cast<nn::ssl::Context*>(pSsl);

    if (pContext->Create(nn::ssl::Context::SslVersion_Auto).IsFailure())
    {
        return CURLE_ABORTED_BY_CALLBACK;
    }

    return CURLE_OK;
}

size_t HttpTask::HttpWriteFunction(char* pBuffer, size_t size, size_t count, void* pParam) NN_NOEXCEPT
{
    HttpTask* pThis = static_cast<HttpTask*>(pParam);

    size_t received = size * count;

    if (received + pThis->m_ReceivedSize > pThis->GetWorkBufferSize())
    {
        return 0;
    }

    Bit8* pOutBuffer = reinterpret_cast<Bit8*>(pThis->GetWorkBuffer());

    std::memcpy(&pOutBuffer[pThis->m_ReceivedSize], pBuffer, received);
    pThis->m_ReceivedSize += received;

    return received;
}

}}}}
