﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/app/acbaa/task/ens_TaskReceiveMessageHeaderList.h>
#include <nn/ens/detail/core/ens_HttpRequestHeader.h>
#include <nn/ens/detail/util/ens_MessagePackWriter.h>
#include <nn/ens/detail/util/ens_ResponseStructureReader.h>

namespace nn { namespace ens { namespace detail { namespace app_acbaa { namespace task {

TaskReceiveMessageHeaderList::TaskReceiveMessageHeaderList() NN_NOEXCEPT
    : m_pOutCount(nullptr)
    , m_pOutTotalCount(nullptr)
    , m_pOutHeaderList(nullptr)
    , m_Offset(0)
    , m_ListCount(0)
    , m_SetIndex(0)
    , m_IsHeaderListParsing(false)
{
}

TaskReceiveMessageHeaderList::~TaskReceiveMessageHeaderList() NN_NOEXCEPT
{
}

void TaskReceiveMessageHeaderList::SetParameter(int* pOutCount, int* pOutTotalCount, MessageHeader pOutHeaderList[],
    int count, int offset) NN_NOEXCEPT
{
    m_pOutCount = pOutCount;
    m_pOutTotalCount = pOutTotalCount;
    m_pOutHeaderList = pOutHeaderList;

    m_Offset = offset;
    m_ListCount = count;
}

nn::Result TaskReceiveMessageHeaderList::ExecuteSteps() NN_NOEXCEPT
{
    NN_RESULT_DO(StepReceive());

    NN_RESULT_SUCCESS;
}

nn::Result TaskReceiveMessageHeaderList::StepReceive() NN_NOEXCEPT
{
    NN_DETAIL_ENS_SCOPED_ACQUIRE_HTTP_CONNECTION(pCurl);

    char url[256] = {};
    detail::core::MakeUrl(url, sizeof (url), "api", "/api/v1/message_cards?offset=%d&limit=%d",
        m_Offset, m_ListCount);

    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_CUSTOMREQUEST, "GET"));
    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_URL, url));

    detail::core::HttpRequestHeader header;

    NN_RESULT_DO(header.Add("Authorization", "Bearer %s", GetAuthToken().value));
    NN_RESULT_DO(header.Add("Content-Type", "application/x-msgpack"));

    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_HTTPHEADER, header.Get()));

    NN_RESULT_DO(Perform());

    NN_DETAIL_ENS_DUMP_RESPONSE(detail::util::MessagePackReader, GetResponse(), GetResponseSize());

    // 出力パラメータへのアクセス
    {
        NN_UTIL_LOCK_GUARD(GetParameterLock());

        NN_RESULT_THROW_UNLESS(IsAsyncContextRegistered(), ResultInvalidState());

        detail::util::MessagePackReader reader;

        reader.SetCallback(ResponseReadCallback, this);

        std::memset(&m_ListParseBits, 0, sizeof (m_ListParseBits));

        NN_RESULT_THROW_UNLESS(reader.Read(GetResponse(), GetResponseSize()), ResultInvalidResponse());
    }

    NN_RESULT_SUCCESS;
}

bool TaskReceiveMessageHeaderList::ResponseReadCallbackImpl(const nn::ens::detail::util::JsonPath& jsonPath,
    const detail::util::DataHolder& holder) NN_NOEXCEPT
{
    if (!m_ListParseBits.total &&
        jsonPath.Compare("$.total"))
    {
        m_ListParseBits.total = true;

        NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(m_pOutTotalCount));
    }
    else if (!m_ListParseBits.count &&
        jsonPath.Compare("$.count"))
    {
        m_ListParseBits.count = true;

        NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(m_pOutCount));
    }
    else if ((holder.type == detail::util::DataHolderType_BeginArray || holder.type == detail::util::DataHolderType_EndArray) &&
        jsonPath.Match("$.headers[]"))
    {
        m_IsHeaderListParsing = (holder.type == detail::util::DataHolderType_BeginArray);
    }
    else if ((holder.type == detail::util::DataHolderType_BeginMap || holder.type == detail::util::DataHolderType_EndMap) &&
        jsonPath.Match("$.headers[*]"))
    {
        if (holder.type == detail::util::DataHolderType_BeginMap)
        {
            NN_DETAIL_ENS_RETURN_IF_FALSE(m_ListCount > m_SetIndex);

            MessageHeader& header = m_pOutHeaderList[m_SetIndex];

            header.id = InvalidMessageId;
            header.sender = InvalidUserId;
            header.sentAt.value = 0;

            std::memset(&m_HeaderParseBits, 0, sizeof (m_HeaderParseBits));
        }
        else
        {
            const MessageHeader& header = m_pOutHeaderList[m_SetIndex];

            NN_DETAIL_ENS_RETURN_IF_FALSE(header.id != InvalidMessageId);
            NN_DETAIL_ENS_RETURN_IF_FALSE(header.sender != InvalidUserId);
            NN_DETAIL_ENS_RETURN_IF_FALSE(header.sentAt.value != 0);

            m_SetIndex++;
        }
    }
    else if (m_IsHeaderListParsing)
    {
        MessageHeader& header = m_pOutHeaderList[m_SetIndex];

        if (!m_HeaderParseBits.messageId &&
            jsonPath.Match("$.headers[*].id"))
        {
            m_HeaderParseBits.messageId = true;

            NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(&header.id.value));
        }
        else if (!m_HeaderParseBits.userId &&
            jsonPath.Match("$.headers[*].user_id"))
        {
            m_HeaderParseBits.userId = true;

            NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(&header.sender.value));
        }
        else if (!m_HeaderParseBits.sentAt &&
            jsonPath.Match("$.headers[*].sent_at"))
        {
            m_HeaderParseBits.sentAt = true;

            NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(&header.sentAt.value));
        }
        else if (!m_HeaderParseBits.digest &&
            jsonPath.Match("$.headers[*].digest"))
        {
            m_HeaderParseBits.digest = true;

            if (!(holder.type == detail::util::DataHolderType_String && holder.string.length == Digest::StringLength))
            {
                return false;
            }

            NN_DETAIL_ENS_RETURN_IF_FALSE(Digest::FromString(&header.digest, holder.string.pValue));
        }
        else if (!m_HeaderParseBits.metadata &&
            jsonPath.Match("$.headers[*].meta"))
        {
            m_HeaderParseBits.metadata = true;

            if (header.metadata.IsNull())
            {
                // メタデータバッファの空指定は許可するが、サイズ 0 のバイナリ以外だったらエラーとする。
                if (!(holder.type == detail::util::DataHolderType_Binary && holder.binary.length == 0))
                {
                    return false;
                }

                header.metadata.receivedSize = 0;
            }
            else
            {
                NN_DETAIL_ENS_RETURN_IF_FALSE(holder.Get(&header.metadata));
            }
        }
    }

    return true;
} // NOLINT(impl/function_size)

bool TaskReceiveMessageHeaderList::ResponseReadCallback(const nn::ens::detail::util::JsonPath& jsonPath,
    const detail::util::DataHolder& holder, void* pParam) NN_NOEXCEPT
{
    TaskReceiveMessageHeaderList* pThis = reinterpret_cast<TaskReceiveMessageHeaderList*>(pParam);

    return pThis->ResponseReadCallbackImpl(jsonPath, holder);
}

}}}}}
