﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_Misc.h>
#include <nw/eft/eftvw2_ModelPreview.h>
#include <nw/eft/eftvw2_ViewerSystem.h>
#include <nw/eft/eftvw2_UsrModelPreview.h>

#if EFT_IS_WIN
#include <Windows.h>
#include <stdio.h>
#endif

namespace nw   {
namespace eftvw2 {

#define SEND_BUFFER_SIZE        (1024*1024)
#define READ_BUFFER_SIZE        (1024*1024*5)

#define RESOURCE_ENTRY_NUM      32

//---------------------------------------------------------------------------
//  コンストラクタ
//---------------------------------------------------------------------------
 ViewerSystem::ViewerSystem( nw::eft2::Heap* heap, nw::eft2::Heap* connectionHeap, nw::eft2::System* system, bool noUseInternalThread, int priority )
{
    mViewerHeap                  = heap;
    mConnectionHeap              = connectionHeap;
    mTime                        = 0.0f;
    mFrameRate                   = 1.0f;
    mEndTime                     = 120.f;
    mIsLoop                      = false;
    mIsPaused                    = false;
    mStep                        = false;
    mIsAppInitialized            = false;
    mIsConnected                 = false;
    mIsResetTime                 = false;
    mCreateModelPreviewCallback  = NULL;
    mDestroyModelPreviewCallback = NULL;

    mViewerCenter.SetIdentity();

    // エフェクトシステムのポインタを設定する
    mEftSystem = system;

    // コマンド受信クラスを生成する
    mCmdReceiver = CreateCmdReceiver( mConnectionHeap );

    // コマンド送信クラスを生成する
    mCmdSender = CreateCmdSender( mConnectionHeap, SEND_BUFFER_SIZE );

    // 通信スレッドの初期化
    ToolConnecter::Initialize( mConnectionHeap, READ_BUFFER_SIZE, mCmdReceiver, mCmdSender, noUseInternalThread, priority );

    // パケットプロシジャーの初期化
    mPacketProcedure.Initialize( this, mConnectionHeap );

    // リソースマネージャーの初期化
    mResManager.Initialize( heap, RESOURCE_ENTRY_NUM, system->GetResourceMaxNum() - nw::eft2::Config::EFT_DEFAULT_RESOURCE_VIEWER_RESERVE_NUM );

    // プレビュー先頭を初期化
    eftcom::Guid guid;
    for( u32 i = 0; i < 16; i++ ) guid.data[i] = 0;
    mPreviewHead.Initialize( heap, Preview::EFT_VWR_PRV_TOP, guid );

}

//---------------------------------------------------------------------------
//  デストラクタです。
//---------------------------------------------------------------------------
ViewerSystem::~ViewerSystem()
{
    // プレビューの破棄
    DestroyAllPreview();

    // リソース破棄
    DestroyAllResource();
    mResManager.Finalize();

    // 通信スレッドの終了処理
    nw::eftvw2::ToolConnecter::Finalize();

    // コマンド送信クラスの終了処理
    DestroyCmdSender( mConnectionHeap );

    // コマンド受信クラスの終了処理
    DestroyCmdReceiver( mConnectionHeap );
}

//---------------------------------------------------------------------------
//  アプリケーションの初期化完了を通知します。
//---------------------------------------------------------------------------
void ViewerSystem::SetAppIsInitialized()
{
    // 接続中のとき初期化完了通知メッセージを送る
    if (mIsConnected)
    {
        SendAppIsInitialized();
    }

    mIsAppInitialized = true;
}

//---------------------------------------------------------------------------
// コマンド処理を実行します。
//---------------------------------------------------------------------------
void ViewerSystem::ExecuteCommand()
{
    // EffectMakerからのコマンド要求
    mCmdReceiver->BeginProc();
    Cmd* cmd = mCmdReceiver->GetEffectCommand();
    while ( cmd )
    {
        switch ( cmd->GetCommand() )
        {
        // Packet
        case 101:
            // パケットを処理します。
            mPacketProcedure.DoPacket( (char *)cmd->GetCommandBuffer(), (size_t)cmd->GetCommandBufferSize() );
            break;
        }

        // コマンドをpop
        mCmdReceiver->PopEffectCommand();
        cmd = mCmdReceiver->GetEffectCommand();
    }
    mCmdReceiver->EndProc();
}

//---------------------------------------------------------------------------
//  全てのプレビューを破棄します。
//---------------------------------------------------------------------------
void ViewerSystem::DestroyAllPreview()
{
    Preview* preview = mPreviewHead.GetNextPreview();
    while (preview)
    {
        Preview* temp = preview;
        preview = preview->GetNextPreview();
        {
            // リストから削除
            temp->RemovePreview();

            // エフェクトプレビューの破棄
            if ( temp->GetPreviewType() == Preview::EFT_VWR_PRV_EFFECT )
            {
                EffectPreview::DestroyEffectPreview( mViewerHeap, (EffectPreview *)temp );
            }
            // モデルプレビューの破棄
            else if ( temp->GetPreviewType() == Preview::EFT_VWR_PRV_MODEL )
            {
                mDestroyModelPreviewCallback( mViewerHeap, temp );
            }
        }
    }
}

//---------------------------------------------------------------------------
//  全てのリソースを破棄します。
//---------------------------------------------------------------------------
void ViewerSystem::DestroyAllResource()
{
    for( u32 i = 0; i < nw::eft2::Config::EFT_DEFAULT_RESOURCE_VIEWER_RESERVE_NUM; i++ )
    {
        u32 resId = i + ( mEftSystem->GetResourceMaxNum() - nw::eft2::Config::EFT_DEFAULT_RESOURCE_VIEWER_RESERVE_NUM );

        if ( mEftSystem->GetResource( resId ) )
        {
            mEftSystem->ClearResource( mViewerHeap, resId );
        }
    }
}

//---------------------------------------------------------------------------
//  システムの定期処理
//---------------------------------------------------------------------------
bool ViewerSystem::ProcCalc( f32 frameRate, const nw::math::MTX34& viewMatrix )
{
    // 切断された。
    if ( mIsConnected && !ToolConnecter::IsConnected() )
    {
        int resCount = mResManager.GetResourceCount();
        for ( int i = 0; i < resCount; i++ )
        {
            UnLoadEffectData( mResManager.GetResource(i)->guid );
        }

        mIsConnected = ToolConnecter::IsConnected();

        return true;
    }

    // 接続時にアプリケーションが初期化済みだったとき初期化完了メッセージを送信
    if ( !mIsConnected && ToolConnecter::IsConnected() && mIsAppInitialized)
    {
        SendAppIsInitialized();
    }

    // 接続状態を保持
    mIsConnected = ToolConnecter::IsConnected();
    if ( !mIsConnected )
    {
        Preview* preview = &mPreviewHead;
        while (NULL != preview)
        {
            preview->Calc( mIsPaused, mFrameRate, mViewerCenter, viewMatrix );
            preview = preview->GetNextPreview();
        }
        return true;
    }

    // 今フレームに受信されたコマンドの実行
    ExecuteCommand();

    bool addFrame = false;
    if ( !mIsPaused )
    {
        addFrame = true;
    }
    else
    {
        if ( mStep ) { addFrame = true; }
    }

    // ステップ再生時は1.0f固定
    mFrameRate = mStep ? 1.0f : frameRate;
    bool isPausedPreviewCalc = mIsPaused && !mStep;
    mStep = false;

    if ( addFrame )
    {
        if ( mIsResetTime )
        {
            ResetTime( true );
        }

        // ループ処理を行います
        if ( mIsLoop && mTime >= mEndTime - FLT_EPSILON )
        {
            mTime = 0.0f;
            mIsResetTime = true;
        }
        else
        {
            mTime += mFrameRate;
            mIsResetTime = false;
        }
    }

    // プレビュー定期処理を行います。
    // プレビューのリセットをコール
    Preview* preview = &mPreviewHead;
    while (NULL != preview)
    {
        preview->Calc( isPausedPreviewCalc, mFrameRate, mViewerCenter, viewMatrix );
        preview = preview->GetNextPreview();
    }

//    mTopPreview.CalcTree( m_IsPaused, view );

    // ビューアセンターを利用する
//    SetViewerCenter2Preview();

    // リソースのフラッシュ
    mResManager.Flush( mEftSystem, mViewerHeap );

    return (!addFrame);
}

//------------------------------------------------------------------------------
//  タイムをリセットします。
//------------------------------------------------------------------------------
void ViewerSystem::ResetTime( bool bFade )
{
    // フェードで追えないエフェクトを削除
    if( !bFade )
    {
        mEftSystem->KillEmitterGroup( EFT_VWR_USE_GROUP_ID );
    }

    // プレビューのリセットをコール
    Preview* preview = &mPreviewHead;
    while (NULL != preview)
    {
        preview->ResetPreview( bFade );
        preview = preview->GetNextPreview();
    }

    mTime = 0.0f;
}


//------------------------------------------------------------------------------
//  強制的にフェードさせます。
//------------------------------------------------------------------------------
void ViewerSystem::ForceFade()
{
    Preview* preview = &mPreviewHead;

    while ( preview )
    {
        if ( preview->GetPreviewType() == Preview::EFT_VWR_PRV_EFFECT )
        {
            EffectPreview* effectPreview = reinterpret_cast<EffectPreview*>( preview );
            effectPreview->Fade();
        }

        preview = preview->GetNextPreview();
    }
}


//---------------------------------------------------------------------------
//  指定リソースIDのエフェクトプレビューを削除。
//---------------------------------------------------------------------------
bool ViewerSystem::RemoveEffectPreviewFromResID( s32 resID )
{
    bool ret = false;
    Preview* preview = &mPreviewHead;

    while ( preview )
    {
        Preview* next = preview->GetNextPreview();
        if ( preview->GetPreviewType() == Preview::EFT_VWR_PRV_EFFECT )
        {
            EffectPreview* effectPreview = reinterpret_cast<EffectPreview*>( preview );

            if ( effectPreview->GetResourceID() == resID )
            {
                preview->RemovePreview();
                EffectPreview::DestroyEffectPreview( mViewerHeap, preview );
                ret = true;
            }
        }
        preview = next;
    }

    return ret;
}


//---------------------------------------------------------------------------
//  エミッタセットバイナリの読み込みを行います。
//---------------------------------------------------------------------------
bool ViewerSystem::LoadEffectData( const char* pPath, eftcom::Guid guid )
{
    ResourceManager::EffectResource* res = NULL;

    s32 oldResId = -1;
    s32 newResId = -1;

    // リソースが既登録であれば、それに伴うプレビューを削除する。
    if ( mResManager.GetResource( guid ) )
    {
        oldResId = mResManager.GetResource( guid )->resId;
        if ( oldResId != -1 )
        {
            UnLoadEffectData( mResManager.GetResource( guid )->guid );
        }
    }

    // リソースを登録する。
    if ( mResManager.Entry( pPath, guid ) )
    {
        res = mResManager.GetResource( guid );
        if ( !res ) return false;
        newResId = res->resId;
        if ( newResId == -1 ) return false;
    }
    else
    {
        return false;
    }

    // リソース登録
    mEftSystem->EntryResource( mViewerHeap, res->binary, newResId, true );
    char text[nw::eftcom::Guid::GUID_STRING_BUFF_SIZE];
    NW_UNUSED_VARIABLE(text)
        NW_LOG("Entry Resource %x : %d / guid = %s\n", res->binary, newResId, guid.GetString( (char*)text, nw::eftcom::Guid::GUID_STRING_BUFF_SIZE) );
    mEftSystem->OutputResourceInfo( newResId );

    // バインド対象のリソースがあればバインドする。
    BindResource( guid );

    // oldResIdに紐づいたエミッタセットは、
    // newResIdで再生成する
    if ( oldResId != -1 )
    {
        const char* name = mEftSystem->GetResource( newResId )->GetEmitterSetName( 0 );
        mEftSystem->RecreateEmitterSet2( name, oldResId, newResId );
    }

    return true;
}

//---------------------------------------------------------------------------
//  エミッタセットバイナリの破棄を行います。
//---------------------------------------------------------------------------
bool ViewerSystem::UnLoadEffectData( eftcom::Guid guid )
{
    ResourceManager::EffectResource* resource = mResManager.GetResource( guid );

    if ( resource )
    {
        if ( resource->resId == -1 ) return false;

        // リソースに紐づいたプレビューを削除
        RemoveEffectPreviewFromResID( resource->resId );

        // 対象のリソースがあればunbindする。
        UnbindResource( guid );

        // それ以外にも紐づいたエフェクトがあれば削除
        const char* name = mEftSystem->GetResource( resource->resId )->GetEmitterSetName( 0 );
        mEftSystem->KillEmitterSet( name, resource->resId );

        // リソースの利用を不可に。
        mEftSystem->GetResource( resource->resId )->SetKillMark();

        // リソースの削除
        mResManager.Release( guid );

        char text[nw::eftcom::Guid::GUID_STRING_BUFF_SIZE];
        NW_UNUSED_VARIABLE(text);
        NW_LOG("Clear Resource guid = %s\n", guid.GetString( (char*)text, nw::eftcom::Guid::GUID_STRING_BUFF_SIZE ) );
    }

    return true;
}

//---------------------------------------------------------------------------
// エフェクトプレビューを生成します。
//---------------------------------------------------------------------------
bool ViewerSystem::CreateEffectPreview( eftcom::Guid resGuid, eftcom::Guid prevGuid )
{
    // TODO:ビューアではVisible機能の為、0Emitterのデータの再生要求があるのでリソースのエミッタ数を確認
    //      エミッタセットを探す方法は後程考える。

    //再生対象となるリソースを取得
    ResourceManager::EffectResource* res = mResManager.GetResource( resGuid );
    if ( !res ) return false;

    // プレビューを生成
    EffectPreview* eftPreview = EffectPreview::CreateEffectPreview( mViewerHeap, mEftSystem, prevGuid );
    if ( !eftPreview ) return false;

    // エミッタセット生成します。
    u32 forceCalcFrame = 0;
    u32 startFrame     = 0;
    eftPreview->CreateEmitterSet( res->resId, startFrame, forceCalcFrame );

    // 生成したプレビューをリストに追加
    mPreviewHead.AddPreview( eftPreview );

    ResetTime( false );
    return true;
}

//---------------------------------------------------------------------------
// モデルプレビューを生成します。
//---------------------------------------------------------------------------
bool ViewerSystem::CreateModelPreview( eftcom::Guid guid, void* pData, u32 dataSize )
{
    if ( mCreateModelPreviewCallback )
    {
        Preview* modelPreview = mCreateModelPreviewCallback( mViewerHeap, this, pData, dataSize, guid );
        if ( !modelPreview ) return false;

        // 生成したプレビューをリストに追加
        mPreviewHead.AddPreview( modelPreview );

        ResetTime( false );
    }
    return true;
}

//---------------------------------------------------------------------------
// モデルプレビューを破棄します。
//---------------------------------------------------------------------------
bool ViewerSystem::DestroyModelPreview( eftcom::Guid guid )
{
    if ( mDestroyModelPreviewCallback )
    {
        // 破棄するプレビューを取得
        Preview* modelPreview = mPreviewHead.GetPreview(guid);
        if ( !modelPreview ) return false;

        // 取得したプレビューをリストから削除
        modelPreview->RemovePreview();

        mDestroyModelPreviewCallback( mViewerHeap, modelPreview );
    }
    return true;
}

//---------------------------------------------------------------------------
//! @brief        指定エミッタセットをバインドします。
//---------------------------------------------------------------------------
bool ViewerSystem::BindResource( eftcom::Guid resGuid )
{
    // 変更元リソース と 変更先リソース の特定
    nw::eft2::Resource* srcResource = NULL;
    nw::eft2::Resource* dstResource = NULL;
    s32 dstEmitterSetId = nw::eft2::EFT_INVALID_EMITTER_SET_ID;

    // 変更元リソース
    {
        ResourceManager::EffectResource* res = mResManager.GetResource( resGuid );
        if ( !res ) return false;
        if ( res->resId == -1 ) return false;

        // 変更元リソースは、EffectMakerから送られてきたリソースなので、
        // EmitterSetは１つしか含まない。
        srcResource = mEftSystem->GetResource( res->resId );
        if ( !srcResource ) return false;
    }

    // バインドするエミッタセット名
    const char* emitterSetName = srcResource->GetEmitterSetName(0);

    // 変更先リソース
    {
        // 全リソースの中から同名のエミッタセットを検索する
        for ( u32 i = 0; i < (u32)mEftSystem->GetResourceMaxNum()/* - nw::eft2::Config::EFT_DEFAULT_RESOURCE_VIEWER_RESERVE_NUM*/; ++i )
        {
            dstResource = mEftSystem->GetResource( i );
            if ( !dstResource ) continue;

            dstEmitterSetId = dstResource->SearchEmitterSetID( emitterSetName );
            if ( dstEmitterSetId == nw::eft2::EFT_INVALID_EMITTER_SET_ID ) continue;

            // リソースを差し替え
            if ( dstResource->BindResource( dstEmitterSetId, srcResource->GetEmitterSetResource(0) ) )
            {
                // 繋ぎ換えたことをEffectMakerに送信
                SendLinkedEmitterSetV2E( emitterSetName );

                // バインドされたリソースであることをマーキング
//                mResMng.SetBindResourceFlag( emitterSetName );
                break;
            }
            else
            {
                // 差し替え失敗
                break;
            }
        }
    }

    return true;
}

//---------------------------------------------------------------------------
//! @brief        指定リソースをアンバインドします。
//---------------------------------------------------------------------------
bool ViewerSystem::UnbindResource( eftcom::Guid resGuid )
{
    // 変更元リソース と 変更先リソース の特定
    nw::eft2::Resource* srcResource = NULL;
    nw::eft2::Resource* dstResource = NULL;
    s32 dstEmitterSetId = nw::eft2::EFT_INVALID_EMITTER_SET_ID;

    // 変更元リソース
    {
        ResourceManager::EffectResource* res = mResManager.GetResource( resGuid );
        if ( !res ) return false;
        if ( res->resId == -1 ) return false;

        // 変更元リソースは、EffectMakerから送られてきたリソースなので、
        // EmitterSetは１つしか含まない。
        srcResource = mEftSystem->GetResource( res->resId );
        if ( !srcResource ) return false;
    }

    // アンバインドするエミッタセット名
    const char* emitterSetName = srcResource->GetEmitterSetName(0);

    // 変更先リソース
    {
        // 全リソースの中から同名のエミッタセットを検索する
        for ( u32 i = 0; i < (u32)mEftSystem->GetResourceMaxNum() - nw::eft2::Config::EFT_DEFAULT_RESOURCE_VIEWER_RESERVE_NUM; ++i )
        {
            dstResource = mEftSystem->GetResource( i );
            if ( !dstResource ) continue;

            dstEmitterSetId = dstResource->SearchEmitterSetID( emitterSetName );
            if ( dstEmitterSetId == nw::eft2::EFT_INVALID_EMITTER_SET_ID ) continue;

            // リソースを差し替え
            if ( dstResource->UnbindResource( dstEmitterSetId ) )
            {
                // EM4Fに指定エミッタセットがユーザ再生済みエフェクトがバインドが解消されたことを送信する。
                SendUnLinkedEmitterSetV2E( emitterSetName );
                break;
            }
            else
            {
                // 差し替え失敗
                break;
            }
        }
    }

    return true;
}

//---------------------------------------------------------------------------
//! @brief  指定リソースをアンバインドします。
//---------------------------------------------------------------------------
bool ViewerSystem::UnbindResource( s32 resID )
{
    // 変更元リソース と 変更先リソース の特定
    nw::eft2::Resource* srcResource = NULL;
    nw::eft2::Resource* dstResource = NULL;
    s32 dstEmitterSetId = nw::eft2::EFT_INVALID_EMITTER_SET_ID;

    // 変更元リソース
    {
        // 変更元リソースは、EffectMakerから送られてきたリソースなので、
        // EmitterSetは１つしか含まない。
        srcResource = mEftSystem->GetResource( resID );
        if ( !srcResource ) return false;
    }

    // アンバインドするエミッタセット名
    const char* emitterSetName = srcResource->GetBinaryName();

    // 変更先リソース
    {
        // 全リソースの中から同名のエミッタセットを検索する
        for ( u32 i = 0; i < (u32)mEftSystem->GetResourceMaxNum() - nw::eft2::Config::EFT_DEFAULT_RESOURCE_VIEWER_RESERVE_NUM; ++i )
        {
            dstResource = mEftSystem->GetResource( i );
            if ( !dstResource ) continue;

            dstEmitterSetId = dstResource->SearchEmitterSetID( emitterSetName );
            if ( dstEmitterSetId == nw::eft2::EFT_INVALID_EMITTER_SET_ID ) continue;

            // リソースを差し替え
            if ( dstResource->UnbindResource( dstEmitterSetId ) )
            {
                // TODO:EM4Fに指定エミッタセットがユーザ再生済みエフェクトがバインドが解消されたことを送信する。
                SendUnLinkedEmitterSetV2E( emitterSetName );
                break;
            }
            else
            {
                // 差し替え失敗
                break;
            }
        }
    }

    return true;
}


//---------------------------------------------------------------------------
//  エフェクトプレビューリソースをエフェクトプレビューに設定します。
//---------------------------------------------------------------------------
bool ViewerSystem::SetResEffectPreview( eftcom::Guid prevGuid, ResEffectPreviewPack* resource )
{
    Preview* preview = mPreviewHead.GetPreview( prevGuid );
    if ( !preview ) return false;
    if ( preview->GetPreviewType() != Preview::EFT_VWR_PRV_EFFECT ) return false;

    EffectPreview* effectPreview = reinterpret_cast<EffectPreview*>( preview );
    effectPreview->SetResPreview( &resource->preview );
    effectPreview->SetResEffectPreview( &resource->effectPreview );

    return true;
}


//---------------------------------------------------------------------------
//! @brief モデルプレビューリソースをモデルプレビューに設定します。
//---------------------------------------------------------------------------
bool ViewerSystem::SetResModelPreview( eftcom::Guid prevGuid, ResModelPreviewPack* resource )
{
    Preview* preview = mPreviewHead.GetPreview( prevGuid );
    if ( !preview ) return false;
    if ( preview->GetPreviewType() != Preview::EFT_VWR_PRV_MODEL ) return false;

    ModelPreview* modelPreview = reinterpret_cast<ModelPreview*>( preview );
    modelPreview->SetResPreview( &resource->preview );
    modelPreview->SetResModelPreview( &resource->modelPreview );

    return true;
}


//---------------------------------------------------------------------------
//  エフェクトプレビューリソースを更新します。
//---------------------------------------------------------------------------
bool ViewerSystem::UpdateResEffectPreview( eftcom::Guid   prevGuid,
                                           u32            offset,
                                           void*          overrideData,
                                           u32            overrideDataSize,
                                           u8             endian )
{
    Preview* preview = mPreviewHead.GetPreview( prevGuid );
    if ( !preview ) return false;
    if ( preview->GetPreviewType() != Preview::EFT_VWR_PRV_EFFECT ) return false;

    // ResEffectPreviewを書き換える
    if ( offset < sizeof( ResEffectPreview ) )
    {
        EffectPreview* effectPreview = reinterpret_cast<EffectPreview*>( preview );
        u8* pDst = (u8*)( (u32)effectPreview->GetResEffectPreview() + offset );

        // フリップ専用のデータを用意.
        nw::eftvw2::ResEffectPreview temp;
        u8* pSrc = (u8*)( &temp ) + offset;

        // 一時データに書き込み
        memcpy( pSrc, overrideData, overrideDataSize );

#ifdef EFT_BI_ENDIAN
        if ( endian != EFT_ENDIAN_TYPE )
        {
            // 一時データをフリップ.
            temp.FlipEndian();
        }
#endif

        // フリップしたデータのみをコピる.
        memcpy( pDst, pSrc, overrideDataSize );
    }
    // ResPreviewを書き換える
    else
    {
        offset -= sizeof( ResEffectPreview );
        u8* pDst = (u8*)( (u32)preview->GetResPreview() + offset );

        // フリップ専用のデータを用意.
        nw::eftvw2::ResPreview temp;
        u8* pSrc = (u8*)( &temp ) + offset;

        // 一時データに書き込み
        memcpy( pSrc, overrideData, overrideDataSize );

#ifdef EFT_BI_ENDIAN
        if ( endian != EFT_ENDIAN_TYPE )
        {
            // 一時データをフリップ.
            temp.FlipEndian();
        }
#endif

        // フリップしたデータのみをコピる.
        memcpy( pDst, pSrc, overrideDataSize );
    }

    return true;
}

//---------------------------------------------------------------------------
//  モデルプレビューリソースを更新します。
//---------------------------------------------------------------------------
bool ViewerSystem::UpdateResModelPreview( eftcom::Guid   prevGuid,
                                          u32            offset,
                                          void*          overrideData,
                                          u32            overrideDataSize,
                                          u8             endian )
{
    Preview* preview = mPreviewHead.GetPreview( prevGuid );
    if ( !preview ) return false;
    if ( preview->GetPreviewType() != Preview::EFT_VWR_PRV_MODEL ) return false;

    // ResModelPreviewを書き換える
    if ( offset < sizeof( ResModelPreview ) )
    {
        ModelPreview* modelPreview = reinterpret_cast<ModelPreview*>( preview );
        u8* pDst = (u8*)( (u32)modelPreview->GetResModelPreview() + offset );

        // フリップ専用のデータを用意.
        nw::eftvw2::ResModelPreview temp;
        u8* pSrc = (u8*)( &temp ) + offset;

        // 一時データに書き込み
        memcpy( pSrc, overrideData, overrideDataSize );

#ifdef EFT_BI_ENDIAN
        if ( endian != EFT_ENDIAN_TYPE )
        {
            // 一時データをフリップ.
            temp.FlipEndian();
        }
#endif

        // フリップしたデータのみをコピる.
        memcpy( pDst, pSrc, overrideDataSize );
    }
    // ResPreviewを書き換える
    else
    {
        offset -= sizeof( ResModelPreview );
        u8* pDst = (u8*)( (u32)preview->GetResPreview() + offset );

        // フリップ専用のデータを用意.
        nw::eftvw2::ResPreview temp;
        u8* pSrc = (u8*)( &temp ) + offset;

        // 一時データに書き込み
        memcpy( pSrc, overrideData, overrideDataSize );

#ifdef EFT_BI_ENDIAN
        if ( endian != EFT_ENDIAN_TYPE )
        {
            // 一時データをフリップ.
            temp.FlipEndian();
        }
#endif

        // フリップしたデータのみをコピる.
        memcpy( pDst, pSrc, overrideDataSize );
    }

    return true;
}

//------------------------------------------------
// 指定されたemitterの変数を更新します。
//------------------------------------------------
void ViewerSystem::UpdateAssetEmitter( eftcom::Guid resGuid,
                                       const char* emitterName, u32 offset,
                                       void* overrideData,      u32 overrideDataSize,
                                       const bool withReset,    u8 endian )
{
    //再生対象となるリソースを取得
    ResourceManager::EffectResource* res = mResManager.GetResource( resGuid );
    if ( !res ) return;

    nw::eft2::Resource* eftRes = mEftSystem->GetResource( res->resId );
    nw::eft2::EmitterSetResource* emSetRes = eftRes->GetEmitterSetResource( 0 );

    nw::eft2::EmitterResource* emRes = emSetRes->emitterResSet;
    while( emRes )
    {
        // 同名エミッタがあれば上書き
        if ( strcmp( emRes->emitterData->name, emitterName ) == 0 )
        {
            u8* pDst = (u8 *)( (u32)emRes->emitterData + offset );

#ifdef EFT_ENDIAN_LITTLE
            // ResEmitterの編集の場合
            if ( offset < sizeof(nw::eft2::ResEmitter) )
            {
                // フリップ専用データを用意.
                nw::eft2::ResEmitter temp;
                u8* pSrc = (u8*)( &temp ) + offset;

                // 一時データに書き込み.
                memcpy( pSrc, overrideData, overrideDataSize );

#ifdef EFT_BI_ENDIAN
                if ( endian != EFT_ENDIAN_TYPE )
                {
                    // 一時データをフリップ.
                    temp.FlipEndian();
                }
#endif

                // フリップしたデータのみをコピる.
                memcpy( pDst, pSrc, overrideDataSize );
            }
            // サブデータの編集の場合
            // 非常に汚い実装なので要修正!!
            else
            {
#ifdef EFT_BI_ENDIAN
                if ( endian != EFT_ENDIAN_TYPE )
                {
                    // データをフリップ
                    switch ( overrideDataSize )
                    {
                    case 1: break;
                    case 4: nw::eft2::EndianUtil::Flip( (u32*)(overrideData)); break;
                    case 8: nw::eft2::EndianUtil::Flip( (nw::math::VEC2*)(overrideData)); break;
                    case 12: nw::eft2::EndianUtil::Flip( (nw::math::VEC3*)(overrideData)); break;
                    case 16: nw::eft2::EndianUtil::Flip( (nw::math::VEC4*)(overrideData)); break;
                    default:
                        if (overrideDataSize % 16 == 0)
                        {
                            nw::math::VEC4* temp = (nw::math::VEC4*)(overrideData);
                            u32 cnt = overrideDataSize/16;

                            for( u32 i=0; i<cnt; i++ )
                            {
                                nw::eft2::EndianUtil::Flip( (nw::math::VEC4*)(temp) );
                                temp++;
                            }
                            break;
                        }
                        else
                        {
                            nw::eft2::OutputError( "Viewr Param Override Error." );
                        }
                    }
                }
#endif

                // 直書き込み.
                memcpy( pDst, overrideData, overrideDataSize );
            }

            // エミッタリソースの更新を行う
            emRes->Update();
#else
            // staticUbo 部分はLittleになっているので、
            // 一旦、bigに戻してからリソース書き換え。
            GX2EndianSwap( &emRes->emitterData->staticUbo, sizeof(nw::eft2::EmitterStaticUniformBlock) );
            if ( emRes->fieldUniformBlock ) GX2EndianSwap( emRes->fieldUniformBlock, sizeof(nw::eft2::EmitterFieldUniformBlock) );
            memcpy( pDst, overrideData, overrideDataSize );

            // エミッタリソースの更新を行う
            // 下記メソッド内でstaticUboを操作するため、スワップが必要。
            emRes->Update();

            GX2EndianSwap( &emRes->emitterData->staticUbo, sizeof(nw::eft2::EmitterStaticUniformBlock) );
            DCFlushRange(  &emRes->emitterData->staticUbo, sizeof(nw::eft2::EmitterStaticUniformBlock) );

            if ( emRes->fieldUniformBlock )
            {
                GX2EndianSwap( emRes->fieldUniformBlock, sizeof(nw::eft2::EmitterFieldUniformBlock) );
                DCFlushRange(  emRes->fieldUniformBlock, sizeof(nw::eft2::EmitterFieldUniformBlock) );
            }
#endif

            // エミッタを更新
            mEftSystem->UpdateFromResource( emRes, withReset );
        }

        // 子を検索
        for ( u32 i = 0; i < nw::eft2::EFT_EMITTER_INSET_NUM; i++ )
        {
            if ( emRes->childEmitterResSet[i] )
            {
                if ( strcmp(  emRes->childEmitterResSet[i]->emitterData->name, emitterName ) == 0 )
                {
                    u8* pDst = (u8 *)( (u32)emRes->childEmitterResSet[i]->emitterData + offset );
#ifdef EFT_ENDIAN_LITTLE
                    // ResEmitterの編集の場合
                    if ( offset < sizeof(nw::eft2::ResEmitter) )
                    {
                        // フリップ専用データを用意.
                        nw::eft2::ResEmitter temp;
                        u8* pSrc = (u8*)( &temp ) + offset;

                        // 一時データに書き込み.
                        memcpy( pSrc, overrideData, overrideDataSize );

#ifdef EFT_BI_ENDIAN
                        if ( endian != EFT_ENDIAN_TYPE )
                        {
                            // 一時データをフリップ.
                            temp.FlipEndian();
                        }
#endif

                        // フリップしたデータのみをコピる.
                        memcpy( pDst, pSrc, overrideDataSize );
                    }
                    // サブデータの編集の場合
                    // 非常に汚い実装なので要修正!!
                    else
                    {
#ifdef EFT_BI_ENDIAN
                        if ( endian != EFT_ENDIAN_TYPE )
                        {
                            // データをフリップ
                            switch ( overrideDataSize )
                            {
                            case 1: break;
                            case 4: nw::eft2::EndianUtil::Flip( (u32*)(overrideData)); break;
                            case 8: nw::eft2::EndianUtil::Flip( (nw::math::VEC2*)(overrideData)); break;
                            case 12: nw::eft2::EndianUtil::Flip( (nw::math::VEC3*)(overrideData)); break;
                            case 16: nw::eft2::EndianUtil::Flip( (nw::math::VEC4*)(overrideData)); break;
                            default:
                                if (overrideDataSize % 16 == 0)
                                {
                                    nw::math::VEC4* temp = (nw::math::VEC4*)(overrideData);
                                    u32 cnt = overrideDataSize/16;

                                    for( u32 j=0; j<cnt; j++ )
                                    {
                                        nw::eft2::EndianUtil::Flip( (nw::math::VEC4*)(temp) );
                                        temp++;
                                    }
                                    break;
                                }
                                else
                                {
                                    nw::eft2::OutputError( "Viewr Param Override Error." );
                                }
                            }
                        }
#endif

                        // 直書き込み.
                        memcpy( pDst, overrideData, overrideDataSize );
                    }

                    // リソースの更新を行う
                    emRes->childEmitterResSet[i]->Update();
#else
                    // staticUbo 部分はLittleになっている
                    GX2EndianSwap( &emRes->childEmitterResSet[i]->emitterData->staticUbo, sizeof(nw::eft2::EmitterStaticUniformBlock) );

                    memcpy( pDst, overrideData, overrideDataSize );

                    // エミッタリソースの更新を行う
                    emRes->childEmitterResSet[i]->Update();

                    GX2EndianSwap( &emRes->childEmitterResSet[i]->emitterData->staticUbo, sizeof(nw::eft2::EmitterStaticUniformBlock) );
                    DCFlushRange( &emRes->childEmitterResSet[i]->emitterData->staticUbo, sizeof(nw::eft2::EmitterStaticUniformBlock) );
#endif

                    // エミッタを更新
                    mEftSystem->UpdateFromResource( emRes->childEmitterResSet[i], withReset );
                }
            }
        }

        emRes = emRes->next;
    }
}

//------------------------------------------------
//  指定されたエミッタセットのビジビリティを設定します。
//------------------------------------------------
void ViewerSystem::SetEmitterSetVisibility( eftcom::Guid resGuid, bool visible )
{
    //再生対象となるリソースを取得
    ResourceManager::EffectResource* res = mResManager.GetResource( resGuid );
    if ( !res ) return;
    nw::eft2::Resource* eftRes = mEftSystem->GetResource( res->resId );

    nw::eft2::EmitterSetResource* emSetRes = eftRes->GetEmitterSetResource( 0 );
    emSetRes->visibility = visible;

    //Preview* preview = &mPreviewHead;
    //while ( preview )
    //{
    //    Preview* temp = preview;
    //    preview = preview->GetNextPreview();

    //    if ( temp->GetPreviewType() == Preview::EFT_VWR_PRV_EFFECT )
    //    {
    //        EffectPreview* effectPreview = reinterpret_cast<EffectPreview*>( temp );
    //        nw::eft2::Handle* handle = effectPreview->GetEmitterSetHandle();
    //        if ( handle && handle->IsValid() )
    //        {
    //            if ( handle->GetEmitterSet()->GetResourceID() == eftRes->GetResourceID() )
    //            {
    //                //handle->GetEmitterSet()->SetVisible( visible );
    //                effectPreview->SetVisible( visible );
    //            }
    //        }
    //    }
    //}
}

//------------------------------------------------
//  指定されたエミッタのビジビリティを設定します。
//------------------------------------------------
void ViewerSystem::SetEmitterVisibility( eftcom::Guid resGuid, const char* emitterName, bool visible )
{
    //再生対象となるリソースを取得
    ResourceManager::EffectResource* res = mResManager.GetResource( resGuid );
    if ( !res ) return;
    nw::eft2::Resource* eftRes = mEftSystem->GetResource( res->resId );

    nw::eft2::EmitterSetResource* emSetRes = eftRes->GetEmitterSetResource( 0 );

    for( u32 i=0 ; i < emSetRes->emitterAllNum ; i++ )
    {
        if ( strcmp( emitterName, emSetRes->emitterResSet[i].emitterData->name ) == 0 )
        {
            emSetRes->emitterResSet[i].visibility = visible;
        }

        // 子エミッタも検索
        for ( u32 j = 0; j < nw::eft2::EFT_EMITTER_INSET_NUM; j++ )
        {
            if ( emSetRes->emitterResSet[i].childEmitterResSet[j] )
            {
                if ( strcmp( emitterName, emSetRes->emitterResSet[i].childEmitterResSet[j]->emitterData->name ) == 0 )
                {
                    emSetRes->emitterResSet[i].childEmitterResSet[j]->visibility = visible;
                }
            }
        }
    }

    //Preview* preview = &mPreviewHead;
    //while ( preview )
    //{
    //    Preview* temp = preview;
    //    preview = preview->GetNextPreview();

    //    if ( temp->GetPreviewType() == Preview::EFT_VWR_PRV_EFFECT )
    //    {
    //        EffectPreview* effectPreview = reinterpret_cast<EffectPreview*>( temp );
    //        nw::eft2::Handle* handle = effectPreview->GetEmitterSetHandle();

    //        if ( handle && handle->IsValid() )
    //        {
    //            if ( handle->GetEmitterSet()->GetResourceID() == eftRes->GetResourceID() )
    //            {
    //                handle->GetEmitterSet()->SetEmitterVisible( emitterName, visible );
    //            }
    //        }
    //    }
    //}
}

//---------------------------------------------------------------------------
// エミッタセットを作成します。
//---------------------------------------------------------------------------
void ViewerSystem::CreateViewSysEmitterSet( const char* emitterSetName )
{
    EFT_UNUSED_VARIABLE( emitterSetName );

    nw::eft2::Handle handle;
    mEftSystem->CreateEmitterSetID( &handle, 0, 0, 0 );
    if ( handle.IsValid() )
    {
        NW_LOG("Succes New EmitterSet.\n");
    }
}

//------------------------------------------------
//  指定されたプレビューのビジビリティを設定します。
//------------------------------------------------
void ViewerSystem::SetPreviewVisibility( eftcom::Guid prevGuid, bool visible )
{
    Preview* preview = &mPreviewHead;

    while ( preview )
    {
        if ( preview->GetGuID() == prevGuid )
        {
            preview->SetVisible( visible );
        }
        preview = preview->GetNextPreview();
    }
}

//---------------------------------------------------------------------------
//  モデル情報更新
//---------------------------------------------------------------------------
// ボーンバッファのサイズを計算
size_t CalcSize_BoneNameBuff(const char* modelName, IEftModelEnumerator* eftModel)
{
    int boneCount = eftModel->GetNumBone( modelName );
    size_t szDataBuf = 0;
    for( int j = 0; j < boneCount ; ++j )
    {
        // ボーン名
        const char* boneName = eftModel->GetBoneName(modelName, j);
        size_t szName = strlen( boneName );
        // ボーン名サイズ
        szDataBuf += szName;
        szDataBuf += 1; // "," 分＋１
    }
    szDataBuf += 1; // "\0" 分＋１
    szDataBuf += (4 - (szDataBuf & 0x03)) & 0x03;   // ４バイトのパディング

    return szDataBuf;
}

// SendModelInfoV2E パケットを生成
void MakePacket_SendModelInfoV2E(
    nw::eftcom::ModelDataMessage* packetBuff,
    size_t boneNameBufSize,
    IEftModelEnumerator* modelEnumrator,
    const char* modelName,
    u8* modelGuid,
    bool isGame )
{
    int boneCount = modelEnumrator->GetNumBone( modelName );

    // モデル名を設定
    nw::ut::strcpy( packetBuff->modelName, nw::eftcom::max_modelName, modelName);

    // モデルＩＤ(EM4Fから渡されたID)
    memcpy( (void*)&packetBuff->modelGuid, modelGuid, 16 );

    packetBuff->isGame = ( isGame == true ) ? 1: 0;

    // ボーン数
    packetBuff->boneCount = boneCount;

    // ボーン名
    packetBuff->boneDataSize = boneNameBufSize;
    char* pBoneName = &packetBuff->boneData;

    s32 bufferSize = boneNameBufSize;

    for(int i = 0; i < boneCount; ++i)
    {
        // ボーン名を追加します。
        const char* boneName = modelEnumrator->GetBoneName( modelName,  i);
        size_t szName = strlen( boneName );
        nw::ut::MemCpy( pBoneName, boneName, szName );
        // 区切り
        pBoneName += szName;
        *pBoneName++ = ',';
        bufferSize -= szName + 1;
        EFT_ASSERT( bufferSize > 0 );
    }
    *(pBoneName-1) = '\0';
}

//---------------------------------------------------------------------------
//  モデル情報更新
//---------------------------------------------------------------------------
void ViewerSystem::UpdateModelInfo( IEftModelEnumerator* modelEnumrator )
{
    // 未接続時は送信しない。
    if ( !nw::eftvw2::ToolConnecter::IsConnected() ) return;

    mModelEnumrator = modelEnumrator;
    if ( !modelEnumrator ) return;


    // すべてのユーザモデルプレビューを一旦破棄
    {
        Preview* preview = &mPreviewHead;

        while( preview )
        {
            if ( preview->GetPreviewType() == Preview::EFT_VWR_PRV_MODEL )
            {
                Preview* modelPreview = preview;
                preview = preview->GetNextPreview();

                modelPreview->RemovePreview();
                UsrModelPreview::DestroyUsrModelPreview( mViewerHeap, (UsrModelPreview *)modelPreview );
            }
            else
            {
                preview = preview->GetNextPreview();
            }
        }
    }

    // EffectMaker側をリセット、モデル連携UIをリセット
    SendClearPreviewModelV2E();
    nw::eft2::OutputLog( "Clear Model List \n" );

    // モデル数を取得
    u32 modelNum = modelEnumrator->GetNumModel();
    if ( modelNum == 0 ) return;

    // TODO:バッファサイズの算出方法はこのままかな？
    const u32 BufferSize = 1024 * 1024;
    char* packetBuffer = (char*)mViewerHeap->Alloc( BufferSize );
    EFT_ASSERT( packetBuffer );
    if ( !packetBuffer ) return;

    nw::eftcom::ModelDataMessage*  pPacket = NULL;
    u32 modelCnt = 0;

    // モデル情報開始メッセージを送信
    SendBeginPreviewModelV2E();

    // モデル数分、モデル情報パケットを生成して送信する
    for ( s32 i = 0; i <  modelEnumrator->GetNumModel(); ++i )
    {
        // モデル名を取得
        const char* modelName = modelEnumrator->GetModelName( i );
        if ( strlen( modelName ) == 0 ) continue;

        // ボーン数を取得
        u32 boneCount = modelEnumrator->GetNumBone( modelName );
        if ( boneCount == 0 ) continue;

        // ボーン名バッファのサイズを求める
        size_t packetSize       = sizeof( nw::eftcom::ModelDataMessage )  + sizeof(nw::eftcom::ModelHeaderMessage);
        size_t boneNameBufSize  = CalcSize_BoneNameBuff( modelName, modelEnumrator );
        packetSize += boneNameBufSize;

        // バッファオーバーチェック
        NW_ASSERT(packetSize < BufferSize);

        // バッファ確保します。
        void* buf = packetBuffer;
        memset( buf, 0, packetSize );
     //   pPacket = (nw::eftcom::ModelDataMessage*)packetBuffer;
     //   memset( pPacket, 0, packetSize );

        // パケットを生成します。
        nw::eftcom::ModelHeaderMessage* header = (nw::eftcom::ModelHeaderMessage*)packetBuffer;
        header->type = nw::eftcom::MODEL_MESSAGE_TYPE_SEND_MODELINFO;
        header->size = boneNameBufSize + sizeof( nw::eftcom::ModelDataMessage);

        pPacket = (nw::eftcom::ModelDataMessage*)( packetBuffer + sizeof(nw::eftcom::ModelHeaderMessage) );

        u8 modelGuid[16];
        memset( modelGuid, 0, 16 );
        modelGuid[0] = (u8)modelCnt + 16;
        MakePacket_SendModelInfoV2E( pPacket, boneNameBufSize, modelEnumrator, modelName, modelGuid, true );

        // モデル内の情報をEM4Fに送信
        mCmdSender->SendCommand( nw::eftcom::MESSAGE_TYPE_MODEL_INFO,
                                 (void *)packetBuffer, packetSize );
        // モデルプレビューを作成する
        UsrModelPreview* usrModel = UsrModelPreview::CreateUsrModelPreview( mViewerHeap, pPacket->modelGuid, modelName, modelEnumrator );
        if ( usrModel )
        {
            AddPreview( usrModel );
        }

        modelCnt++;
    }

    // モデル情報終了メッセージを送信
    SendEndPreviewModelV2E();

    mViewerHeap->Free( packetBuffer );

}

//-----------------------------------------------------------------------------------------------------
//  エフェクトメーカー操作 共通処理
//-----------------------------------------------------------------------------------------------------
void ViewerSystem::RequestEset( u32 type, const char* emitterSetName, const char* duplicateEmitterSetName )
{
    u32 packetSize = sizeof( nw::eftcom::EsetRequestHeaderMessage ) +
                     sizeof( nw::eftcom::EsetRequestMessage );
    char* packetBuffer = (char*)mViewerHeap->Alloc( packetSize );
    EFT_ASSERT( packetBuffer );
    if ( !packetBuffer ) return;

    memset( packetBuffer, 0, packetSize );
    nw::eftcom::EsetRequestHeaderMessage* header = (nw::eftcom::EsetRequestHeaderMessage*)packetBuffer;
    header->type = type;
    header->size = sizeof( nw::eftcom::EsetRequestMessage );

    nw::eftcom::EsetRequestMessage* eset = (nw::eftcom::EsetRequestMessage*)( (u8*)packetBuffer + sizeof(nw::eftcom::EsetRequestHeaderMessage) );
    memcpy( (void*)eset->emitterSetName, emitterSetName, strlen( emitterSetName) );
    if ( type == nw::eftcom::ESET_MESSAGE_TYPE_REQUEST_DUPLICATE )
    {
        memcpy( (void*)eset->dupEmitterSetName, duplicateEmitterSetName, strlen( duplicateEmitterSetName) );
    }

    mCmdSender->SendCommand( nw::eftcom::MESSAGE_TYPE_ESET_REQUEST,
                             (void *)packetBuffer, packetSize );

    mViewerHeap->Free( packetBuffer );
}

//-----------------------------------------------------------------------------------------------------
//  エフェクトメーカー操作 エミッタセットファイルオープン
//-----------------------------------------------------------------------------------------------------
void ViewerSystem::RequestEsetFileOpen( const char* emitterSetFilePath )
{
    RequestEset( nw::eftcom::ESET_MESSAGE_TYPE_REQUEST_FILE_OPEN, emitterSetFilePath );
}

//-----------------------------------------------------------------------------------------------------
//  エフェクトメーカー操作 エミッタセットファイルクローズ
//-----------------------------------------------------------------------------------------------------
void ViewerSystem::RequestEsetFileClose( const char* emitterSetFilePath )
{
    RequestEset( nw::eftcom::ESET_MESSAGE_TYPE_REQUEST_CLOSE, emitterSetFilePath );
}

//-----------------------------------------------------------------------------------------------------
//  エフェクトメーカー操作 新規エミッタセット作成
//-----------------------------------------------------------------------------------------------------
void ViewerSystem::RequestEsetFileCreate( const char* emitterSetFilePath )
{
    RequestEset( nw::eftcom::ESET_MESSAGE_TYPE_REQUEST_CREATE, emitterSetFilePath );
}

//-----------------------------------------------------------------------------------------------------
//  エフェクトメーカー操作 エミッタセット複製
//-----------------------------------------------------------------------------------------------------
void ViewerSystem::RequestEsetFileDuplicate( const char* srcEmitterSetName, const char* duplicateEmitterSetName )
{
    RequestEset( nw::eftcom::ESET_MESSAGE_TYPE_REQUEST_DUPLICATE, srcEmitterSetName, duplicateEmitterSetName );
}

//-----------------------------------------------------------------------------------------------------
//  リンクエミッタセットメッセージを送信する。
//-----------------------------------------------------------------------------------------------------
void ViewerSystem::SendLinkedEmitterSetV2E(  const char* emitterSetName )
{
    RequestEset( nw::eftcom::ESET_MESSAGE_TYPE_REQUEST_LINKESET, emitterSetName );
}

//-----------------------------------------------------------------------------------------------------
//  アンリンクエミッタセットメッセージを送信する。
//-----------------------------------------------------------------------------------------------------
void ViewerSystem::SendUnLinkedEmitterSetV2E(  const char* emitterSetName )
{
    RequestEset( nw::eftcom::ESET_MESSAGE_TYPE_REQUEST_UNLINKESET, emitterSetName );
}

#if 0 // Test用Send Dummy ModelData
void ViewerSystem::UploadDummyModelDataInfo()
{
    class modelinfo : public IEftModelEnumerator
    {
    public:
        modelinfo( nw::eft2::Heap* viewerHeap )
        {
            heap = viewerHeap;

            for (int i = 0; i < 5; i++ )
            {
                boneName[i] = (char *)heap->Alloc( 64 );
                memset( boneName[i], 0, 64 );
            }

            strcpy( boneName[0], "GameBone1\0" );
            strcpy( boneName[1], "GameBone2\0" );
            strcpy( boneName[2], "GameBone3\0" );
            strcpy( boneName[3], "GameBone4\0" );
            strcpy( boneName[4], "GameBone5\0" );
        }

        ~modelinfo()
        {
            for (int i = 0; i < 5; i++ )
            {
                heap->Free( boneName[i] );
            }
        }

        s32 GetNumModel() { return 1; }
        const char*          GetModelName( s32 ix ) { EFT_UNUSED_VARIABLE( ix ); return "GameModel1\0"; }
        s32                  GetNumBone ( const char* modelName ) { EFT_UNUSED_VARIABLE( modelName ); return  5; }
        const char*          GetBoneName( const char* modelName , s32 ix )
        {
            EFT_UNUSED_VARIABLE(modelName);
            if ( ix > 5 ) return NULL;

            return boneName[ix];
        }
        void GetBoneMtx ( nw::math::Matrix34* boneMatrix, const char* modelName , s32 ix ) { EFT_UNUSED_VARIABLE( boneMatrix ); EFT_UNUSED_VARIABLE( modelName ); EFT_UNUSED_VARIABLE( ix ); }
        void GetModelRootMtx( nw::math::Matrix34* rootMatrix, const char* modelName ) { EFT_UNUSED_VARIABLE(rootMatrix); EFT_UNUSED_VARIABLE(modelName);}

        char* boneName[5];
        nw::eft2::Heap* heap;
    };

    modelinfo modelEnumrator( mViewerHeap );

    // Gameモデル情報の送信
    UpdateModelInfo( reinterpret_cast<IEftModelEnumerator*>(&modelEnumrator) );
}

void ViewerSystem::UploadDummyToolModelDataInfo()
{
    class modelinfo : public IEftModelEnumerator
    {
    public:
        modelinfo( nw::eft2::Heap* viewerHeap )
        {
            heap = viewerHeap;

            for (int i = 0; i < 5; i++ )
            {
                boneName[i] = (char *)heap->Alloc( 64 );
                memset( boneName[i], 0, 64 );
            }

            strcpy( boneName[0], "ToolBone1\0" );
            strcpy( boneName[1], "ToolBone2\0" );
            strcpy( boneName[2], "ToolBone3\0" );
            strcpy( boneName[3], "ToolBone4\0" );
            strcpy( boneName[4], "ToolBone5\0" );
        }

        ~modelinfo()
        {
            for (int i = 0; i < 5; i++ )
            {
                heap->Free( boneName[i] );
            }
        }

        s32 GetNumModel() { return 1; }
        const char*          GetModelName( s32 ix ) { EFT_UNUSED_VARIABLE( ix ); return "ToolModel1\0"; }
        s32                  GetNumBone ( const char* modelName ) { EFT_UNUSED_VARIABLE( modelName ); return  5; }
        const char*          GetBoneName( const char* modelName , s32 ix )
        {
            EFT_UNUSED_VARIABLE(modelName);
            if ( ix > 5 ) return NULL;

            return boneName[ix];
        }
        void GetBoneMtx ( nw::math::Matrix34* boneMatrix, const char* modelName , s32 ix ) { EFT_UNUSED_VARIABLE( boneMatrix ); EFT_UNUSED_VARIABLE( modelName ); EFT_UNUSED_VARIABLE( ix ); }
        void GetModelRootMtx( nw::math::Matrix34* rootMatrix, const char* modelName ) { EFT_UNUSED_VARIABLE(rootMatrix); EFT_UNUSED_VARIABLE(modelName);}

        char* boneName[5];
        nw::eft2::Heap* heap;
    };

    modelinfo modelEnumrator( mViewerHeap );

   // モデル数を取得
    u32 modelNum = modelEnumrator.GetNumModel();
    if ( modelNum == 0 ) return;

    // TODO:バッファサイズの算出方法はこのままかな？
    const u32 BufferSize = 1024 * 1024;
    char* packetBuffer = (char*)mViewerHeap->Alloc( BufferSize );
    EFT_ASSERT( packetBuffer );
    if ( !packetBuffer ) return;

    nw::eftcom::ModelDataMessage*  pPacket = NULL;

     // モデル情報開始メッセージを送信
    SendBeginPreviewModelV2E();

    // モデル情報パケットを生成して送信する
    // モデル名を取得
    const char* modelName = modelEnumrator.GetModelName( 0 );

    // ボーン名バッファのサイズを求める
    size_t packetSize       = sizeof( nw::eftcom::ModelDataMessage ) + sizeof(nw::eftcom::ModelHeaderMessage);
    size_t boneNameBufSize  = CalcSize_BoneNameBuff( modelName, &modelEnumrator );
    packetSize += boneNameBufSize;

    // バッファオーバーチェック
    NW_ASSERT(packetSize < BufferSize);

    // バッファ確保します。
    void* buf = packetBuffer;
    memset( buf, 0, packetSize );

    // パケットを生成します。
    nw::eftcom::ModelHeaderMessage* header = (nw::eftcom::ModelHeaderMessage*)packetBuffer;
    header->type = nw::eftcom::MODEL_MESSAGE_TYPE_SEND_MODELINFO;
    header->size = boneNameBufSize;

    pPacket = (nw::eftcom::ModelDataMessage*)( packetBuffer + sizeof(nw::eftcom::ModelHeaderMessage) );

    u8 modelGuid[16];
    memset( modelGuid, 0, 16 );
    modelGuid[0] = (u8)1;
    MakePacket_SendModelInfoV2E( pPacket, boneNameBufSize, &modelEnumrator, modelName, modelGuid, false );

    // モデル内の情報をEM4Fに送信
    mCmdSender->SendCommand( nw::eftcom::MESSAGE_TYPE_MODEL_INFO,
                             (void *)packetBuffer, packetSize );


    // モデル情報終了メッセージを送信
    SendEndPreviewModelV2E();

    mViewerHeap->Free( packetBuffer );
}
#endif

} // namespace eftvw2
} // namespace nw
