﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eftvw2_CmdSender.h>
#include <nw/eft/eftcom_Message.h>
#include <nw/eft/eftcom_Protocol.h>
#include <nw/eft/eft2_Misc.h>

namespace nw   {
namespace eftvw2 {


CmdSender*      CmdSender::gCmdSender       = NULL;
nw::ut::Mutex   CmdSender::s_Mutex;

//------------------------------------------------------------------------------
//　　　コマンド送信クラスを生成する
//------------------------------------------------------------------------------
CmdSender* CreateCmdSender( nw::eft2::Heap* heap, size_t buffSize )
{
    if ( CmdSender::gCmdSender )
    {
        return CmdSender::gCmdSender;
    }

    void* ptr = heap->Alloc( sizeof(CmdSender) );
    CmdSender::gCmdSender = new (ptr) CmdSender( heap, buffSize );
    return CmdSender::gCmdSender;
}

//------------------------------------------------------------------------------
//　　　コマンド受信クラスを取得する
//------------------------------------------------------------------------------
CmdSender* GetCmdSender()
{
    return CmdSender::gCmdSender;
}

//------------------------------------------------------------------------------
//　　　コマンド受信クラスを破棄する
//------------------------------------------------------------------------------
void DestroyCmdSender( nw::eft2::Heap* heap )
{
    CmdSender::gCmdSender->~CmdSender();
    heap->Free( CmdSender::gCmdSender );
}

//------------------------------------------------------------------------------
//　　　コントラクタ
//------------------------------------------------------------------------------
CmdSender::CmdSender( nw::eft2::Heap* heap, size_t buffSize )
{
    mHeap               = heap;
    mSharedBufferSize   = buffSize;

 //   ClearToolName();

    s_Mutex.Initialize();

    // プロセス内のアドレス空間にファイルのビューをマップ
    mSharedBuffer = (u8*)mHeap->Alloc( mSharedBufferSize );
    EFT_ASSERT( mSharedBuffer );
}

//------------------------------------------------------------------------------
//　　　デストラクタ
//------------------------------------------------------------------------------
CmdSender::~CmdSender()
{
    while( GetCommand() )
    {
        PopCommand();
    }

    mHeap->Free( mSharedBuffer );
}

//------------------------------------------------------------------------------
//　　　EffectMakerにコマンドを送信する
//------------------------------------------------------------------------------
void CmdSender::SendCommand( nw::eftcom::MESSAGE_TYPE type, void* buffer, u32 size )
{
    EFT_UNUSED_VARIABLE( type );
    EFT_UNUSED_VARIABLE( buffer );
    EFT_UNUSED_VARIABLE( size );

    // TODO: 送信データフォーマットを作成すること
    nw::eftcom::Message header;
    header.type = type;
    header.bufferSize = size;

    u32 headerSize = sizeof( nw::eftcom::Message );

    if ( (size+headerSize) > mSharedBufferSize )
    {
        nw::eft2::OutputWarning( "Send Buffer is Over.\n" );
        return;
    }

    // データ
    memcpy( mSharedBuffer, &header, headerSize );
    memcpy( ( (u8*)mSharedBuffer + headerSize ), buffer, size );

    AddCommand( mSharedBuffer, size + headerSize );

    ToolConnecter::SendPacket();
}

//------------------------------------------------------------------------------
//　　　送信コマンドを追加する
//------------------------------------------------------------------------------
void CmdSender::AddCommand( const void* buffer, u32 size )
{
    if ( s_Mutex.TryLock() )
    {
        Cmd* cmd = static_cast<Cmd*>( mHeap->Alloc( sizeof(Cmd) ) );
        EFT_ASSERT( cmd );

        cmd->Initialize();
        cmd->SetCommand( mHeap, (u32)CAFE_CMDID, buffer, size  );
        mCmdQueue.PushBack( cmd );

        s_Mutex.Unlock();
    }
}

//------------------------------------------------------------------------------
//　　　送信コマンドをpopする
//------------------------------------------------------------------------------
void CmdSender::PopCommand()
{
    if ( mCmdQueue.empty() ) return;

    if ( s_Mutex.TryLock() )
    {
        Cmd* cmd = mCmdQueue.back();
        if ( cmd )
        {
            cmd->~Cmd();
            mHeap->Free( cmd );
        }
        mCmdQueue.PopBack();
        s_Mutex.Unlock();
    }
}

//------------------------------------------------------------------------------
//　　　送信コマンドを取得する
//------------------------------------------------------------------------------
Cmd* CmdSender::GetCommand()
{
    if ( mCmdQueue.empty() ) return NULL;
    return mCmdQueue.back();
}



} // namespace eftvw2
} // namespace nw
