﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_TextureSampler.h>
#include <nw/eft/eft2_ShaderRes.h>
#include <nw/eft/eft2_Data.h>

namespace nw   {
namespace eft2 {

//---------------------------------------------------
// サンプラーをバインド
//---------------------------------------------------
bool BindSampler( TextureSamplerInterface sampler, s32 slot, u32 vertLoc, u32 fragLoc )
{
    if ( fragLoc == EFT_INVALID_TEXTURE_LOCATION &&
         vertLoc == EFT_INVALID_TEXTURE_LOCATION )
    {
        return false;
    }

#if EFT_OPENGL
    EFT_UNUSED_VARIABLE( fragLoc );
    EFT_UNUSED_VARIABLE( vertLoc );
    glBindSampler( slot, sampler );
    EFT_GLERR_CHECK();
#elif EFT_GX2
    if ( fragLoc != EFT_INVALID_TEXTURE_SAMPLER ) GX2SetPixelSampler(  sampler, fragLoc );
    if ( vertLoc != EFT_INVALID_TEXTURE_SAMPLER ) GX2SetVertexSampler( sampler, vertLoc );
#else
    EFT_UNUSED_VARIABLE( sampler );
    EFT_UNUSED_VARIABLE( slot );
    EFT_UNUSED_VARIABLE( fragLoc );
    EFT_UNUSED_VARIABLE( vertLoc );
#endif
    return true;
}

//---------------------------------------------------------------------------
//  初期化
//---------------------------------------------------------------------------
bool TextureSmplerLoc::Initialize( ShaderResource* shaderRes, const char* name )
{
    fragSamplerLocation = shaderRes->GetFragmentSamplerLocation( name );
    vertSamplerLocation = shaderRes->GetVertexSamplerLocation( name );

    if ( fragSamplerLocation == EFT_INVALID_TEXTURE_SAMPLER &&
         vertSamplerLocation == EFT_INVALID_TEXTURE_SAMPLER ) return false;
    return true;
}





//---------------------------------------------------------------------------
//  初期化
//---------------------------------------------------------------------------
void TextureSampler::Initialize( ResTextureSampler* resSampler )
{
    textureSamplerRes = resSampler;

#if EFT_OPENGL
    {
        Finalize();
        glGenSamplers( 1, &textureSampler );
        Setup( textureSamplerRes );
        EFT_GLERR_CHECK();
    }
#endif
#if EFT_GX2
    {
        GX2InitSampler( &textureSampler, GX2_TEX_CLAMP_WRAP, GX2_TEX_XY_FILTER_BILINEAR );
        Setup( textureSamplerRes );
    }
#endif
}

//---------------------------------------------------------------------------
//  終了処理
//---------------------------------------------------------------------------
void TextureSampler::Finalize()
{
#if EFT_OPENGL
    if ( textureSampler != EFT_INVALID_TEXTURE_SAMPLER )
    {
        glDeleteSamplers( 1, &textureSampler );
    }
#endif
}

//---------------------------------------------------------------------------
//  無効化
//---------------------------------------------------------------------------
void TextureSampler::Invalidate()
{
    textureSamplerRes = NULL;
#if EFT_OPENGL
    textureSampler    = static_cast<u32>( EFT_INVALID_TEXTURE_SAMPLER );
#endif
}

//---------------------------------------------------------------------------
//  有効かどうか
//---------------------------------------------------------------------------
bool TextureSampler::IsValidate()
{
#if EFT_OPENGL
    return ( textureSampler == EFT_INVALID_TEXTURE_SAMPLER ) ? false : true;
#elif EFT_GX2
    return true;    // TODO : Cafe用に処理を追加すること
#else
    return false;   // サンプラオブジェクトは利用できない
#endif
}

//---------------------------------------------------------------------------
//  ResTextureParamからセットアップ
//---------------------------------------------------------------------------
void TextureSampler::Setup( const ResTextureSampler* sampler )
{
    EFT_NULL_ASSERT( sampler );
    Setup( sampler->filter, sampler->wrapU, sampler->wrapV );
#if EFT_GX2
    SetupLOD( sampler->mipLevel, sampler->mipMapBias );
#endif
}

//---------------------------------------------------------------------------
// サンプラセットアップ
//---------------------------------------------------------------------------
void TextureSampler::Setup( TextureFilterMode filter, TextureWrapMode wrapU, TextureWrapMode wrapV )
{
#if EFT_OPENGL
    {
        if ( textureSampler == EFT_INVALID_TEXTURE_SAMPLER ) return;

        // フィルタモード変更
        if ( filter == EFT_TEXTURE_FILTER_TYPE_LINEAR )
        {
            glSamplerParameteri( textureSampler, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
            glSamplerParameteri( textureSampler, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
        }
        else
        {
            glSamplerParameteri( textureSampler, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
            glSamplerParameteri( textureSampler, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
        }

        // ラッピングモード変更
        switch( wrapU )
        {
        case EFT_TEXTURE_WRAP_TYPE_MIRROR:
            glSamplerParameteri( textureSampler, GL_TEXTURE_WRAP_S, GL_MIRRORED_REPEAT );
            break;

        case EFT_TEXTURE_WRAP_TYPE_REPEAT:
            glSamplerParameteri( textureSampler, GL_TEXTURE_WRAP_S, GL_REPEAT );
            break;

        case EFT_TEXTURE_WRAP_TYPE_CLAMP:
            glSamplerParameteri( textureSampler, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE );
            break;

        case EFT_TEXTURE_WRAP_TYPE_MIROOR_ONCE:
            //        glSamplerParameteri( mTextureSampler, GL_TEXTURE_WRAP_S, GL_ATI_texture_mirror_once );
            glSamplerParameteri( textureSampler, GL_TEXTURE_WRAP_S, GL_MIRRORED_REPEAT );
            break;
        }

        switch( wrapV )
        {
        case EFT_TEXTURE_WRAP_TYPE_MIRROR:
            glSamplerParameteri( textureSampler, GL_TEXTURE_WRAP_T, GL_MIRRORED_REPEAT );
            break;

        case EFT_TEXTURE_WRAP_TYPE_REPEAT:
            glSamplerParameteri( textureSampler, GL_TEXTURE_WRAP_T, GL_REPEAT );
            break;

        case EFT_TEXTURE_WRAP_TYPE_CLAMP:
            glSamplerParameteri( textureSampler, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE );
            break;

        case EFT_TEXTURE_WRAP_TYPE_MIROOR_ONCE:
            // glSamplerParameteri( textureSampler, GL_TEXTURE_WRAP_T, GL_ATI_texture_mirror_once );
            glSamplerParameteri( textureSampler, GL_TEXTURE_WRAP_S, GL_MIRRORED_REPEAT );
            break;
        }
        EFT_GLERR_CHECK();
    }
#endif

#if defined( NW_PLATFORM_SMART_DEVICE )
#if EFT_OPENGLES2
    {
        // フィルタモード変更
        if ( filter == EFT_TEXTURE_FILTER_TYPE_LINEAR )
        {
            glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
            glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
        }
        else
        {
            glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
            glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
        }

        // ラッピングモード変更
        switch( wrapU )
        {
        case EFT_TEXTURE_WRAP_TYPE_MIRROR:
            glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_MIRRORED_REPEAT );
            break;

        case EFT_TEXTURE_WRAP_TYPE_REPEAT:
            glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT );
            break;

        case EFT_TEXTURE_WRAP_TYPE_CLAMP:
            glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE );
            break;

        case EFT_TEXTURE_WRAP_TYPE_MIROOR_ONCE:
            // glSamplerParameteri( mTextureSampler, GL_TEXTURE_WRAP_S, GL_ATI_texture_mirror_once );
            glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_MIRRORED_REPEAT );
            break;
        }

        switch( wrapV )
        {
        case EFT_TEXTURE_WRAP_TYPE_MIRROR:
            glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_MIRRORED_REPEAT );
            break;

        case EFT_TEXTURE_WRAP_TYPE_REPEAT:
            glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT );
            break;

        case EFT_TEXTURE_WRAP_TYPE_CLAMP:
            glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE );
            break;

        case EFT_TEXTURE_WRAP_TYPE_MIROOR_ONCE:
            // glSamplerParameteri( textureSampler, GL_TEXTURE_WRAP_T, GL_ATI_texture_mirror_once );
            glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_MIRRORED_REPEAT );
            break;
        }
        EFT_GLERR_CHECK();
    }
#endif
#endif


#if EFT_GX2
    {
        GX2TexAnisoRatio ratio = GX2_TEX_ANISO_1_TO_1;

        // フィルタモード変更
        if ( filter == EFT_TEXTURE_FILTER_TYPE_LINEAR )
        {
            GX2InitSamplerXYFilter( &textureSampler, GX2_TEX_XY_FILTER_BILINEAR, GX2_TEX_XY_FILTER_BILINEAR, ratio );
        }
        else
        {
            GX2InitSamplerXYFilter( &textureSampler, GX2_TEX_XY_FILTER_POINT, GX2_TEX_XY_FILTER_POINT, ratio );
        }

        // ラッピングモード変更
        GX2TexClamp clampX = GX2_TEX_CLAMP_MIRROR;
        GX2TexClamp clampY = GX2_TEX_CLAMP_MIRROR;
        GX2TexClamp clampZ = GX2_TEX_CLAMP_WRAP;

        switch( wrapU )
        {
        case EFT_TEXTURE_WRAP_TYPE_MIRROR:              clampX = GX2_TEX_CLAMP_MIRROR;      break;
        case EFT_TEXTURE_WRAP_TYPE_REPEAT:              clampX = GX2_TEX_CLAMP_WRAP;        break;
        case EFT_TEXTURE_WRAP_TYPE_CLAMP:               clampX = GX2_TEX_CLAMP_CLAMP;       break;
        case EFT_TEXTURE_WRAP_TYPE_MIROOR_ONCE:         clampX = GX2_TEX_CLAMP_MIRROR_ONCE; break;
        }

        switch( wrapV )
        {
        case EFT_TEXTURE_WRAP_TYPE_MIRROR:              clampY = GX2_TEX_CLAMP_MIRROR;      break;
        case EFT_TEXTURE_WRAP_TYPE_REPEAT:              clampY = GX2_TEX_CLAMP_WRAP;        break;
        case EFT_TEXTURE_WRAP_TYPE_CLAMP:               clampY = GX2_TEX_CLAMP_CLAMP ;      break;
        case EFT_TEXTURE_WRAP_TYPE_MIROOR_ONCE:         clampY = GX2_TEX_CLAMP_MIRROR_ONCE; break;
        }

        GX2InitSamplerClamping( &textureSampler, clampX, clampY, clampZ );
    }
#endif
}

//---------------------------------------------------------------------------
//! @brief  バインド
//---------------------------------------------------------------------------
void TextureSampler::Bind( u32 fragLoc, u32 vertLoc, s32 slot ) const
{
#if EFT_OPENGL
    {
        EFT_UNUSED_VARIABLE( fragLoc );
        EFT_UNUSED_VARIABLE( vertLoc );
        glBindSampler( slot, textureSampler );
        EFT_GLERR_CHECK();
    }
#elif EFT_GX2
    {
        if ( fragLoc != EFT_INVALID_TEXTURE_SAMPLER ) GX2SetPixelSampler(  &textureSampler, fragLoc );
        if ( vertLoc != EFT_INVALID_TEXTURE_SAMPLER ) GX2SetVertexSampler( &textureSampler, vertLoc );
    }
#else
    {
        EFT_UNUSED_VARIABLE( fragLoc );
        EFT_UNUSED_VARIABLE( vertLoc );
        EFT_UNUSED_VARIABLE( slot );
    }
#endif

}

//---------------------------------------------------------------------------
//! @brief  LODバイアスの設定
//---------------------------------------------------------------------------
bool TextureSampler::SetupLOD( f32 maxMip, f32 bais )
{
#ifdef EFT_OGL
    EFT_UNUSED_VARIABLE( maxMip );
    EFT_UNUSED_VARIABLE( bais );
#endif

#if EFT_GX2
    GX2InitSamplerLOD( &textureSampler, 0.0f, maxMip, bais );
#endif
    return true;
}


} // namespace eft2
} // namespace nw

