﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nw/eft/eft2_TemporaryBuffer.h>
#include <nw/eft/eft2_Misc.h>
#include <nw/eft/eft2_MemUtil.h>

namespace nw   {
namespace eft2 {


//---------------------------------------------------------------------------
//  初期化処理を行います。
//---------------------------------------------------------------------------
void TemporaryBuffer::Initialize( u32 size, bool useTriple )
{
    mBufferSide                                     = EFT_TEMPORARY_BUFFER_FIRST;
    mBufferSize                                     = size;
    mAllocedSize   [EFT_TEMPORARY_BUFFER_FIRST]     = 0;
    mFlushedSize   [EFT_TEMPORARY_BUFFER_FIRST]     = 0;
    mPreAllocedSize[EFT_TEMPORARY_BUFFER_FIRST]     = 0;
    mAllocedSize   [EFT_TEMPORARY_BUFFER_SECOND]    = 0;
    mFlushedSize   [EFT_TEMPORARY_BUFFER_SECOND]    = 0;
    mPreAllocedSize[EFT_TEMPORARY_BUFFER_SECOND]    = 0;
    mAllocedSize   [EFT_TEMPORARY_BUFFER_THIRD]     = 0;
    mFlushedSize   [EFT_TEMPORARY_BUFFER_THIRD]     = 0;
    mPreAllocedSize[EFT_TEMPORARY_BUFFER_THIRD]     = 0;
    mBuffer[EFT_TEMPORARY_BUFFER_FIRST]             = AllocFromStaticHeap( mBufferSize, EFT_DOUBLE_BUFFER_ALIGN );
    mBuffer[EFT_TEMPORARY_BUFFER_SECOND]            = AllocFromStaticHeap( mBufferSize, EFT_DOUBLE_BUFFER_ALIGN );
    if ( useTriple )
    {
        mBuffer[EFT_TEMPORARY_BUFFER_THIRD]         = AllocFromStaticHeap( mBufferSize, EFT_DOUBLE_BUFFER_ALIGN );
    }
    else
    {
        mBuffer[EFT_TEMPORARY_BUFFER_THIRD]         = NULL;
    }
    mMaxAllocedSize                                 = 0;
    mInitialized                                    = true;
}

//---------------------------------------------------------------------------
//  無効化処理を行います。
//---------------------------------------------------------------------------
void TemporaryBuffer::Invalidate()
{
    mBufferSide                                     = EFT_TEMPORARY_BUFFER_FIRST;
    mBufferSize                                     = 0;
    mAllocedSize   [EFT_TEMPORARY_BUFFER_FIRST]     = 0;
    mFlushedSize   [EFT_TEMPORARY_BUFFER_FIRST]     = 0;
    mPreAllocedSize[EFT_TEMPORARY_BUFFER_FIRST]     = 0;
    mAllocedSize   [EFT_TEMPORARY_BUFFER_SECOND]    = 0;
    mFlushedSize   [EFT_TEMPORARY_BUFFER_SECOND]    = 0;
    mPreAllocedSize[EFT_TEMPORARY_BUFFER_SECOND]    = 0;
    mAllocedSize   [EFT_TEMPORARY_BUFFER_THIRD]     = 0;
    mFlushedSize   [EFT_TEMPORARY_BUFFER_THIRD]     = 0;
    mPreAllocedSize[EFT_TEMPORARY_BUFFER_THIRD]     = 0;
    mBuffer[EFT_TEMPORARY_BUFFER_FIRST]             = NULL;
    mBuffer[EFT_TEMPORARY_BUFFER_SECOND]            = NULL;
    mBuffer[EFT_TEMPORARY_BUFFER_THIRD]             = NULL;
    mMaxAllocedSize                                 = 0;
    mInitialized                                    = false;
}

//---------------------------------------------------------------------------
//  終了処理を行います。
//---------------------------------------------------------------------------
void TemporaryBuffer::Finalize()
{
    if ( !mInitialized ) return;

    FreeFromStaticHeap( mBuffer[EFT_TEMPORARY_BUFFER_FIRST] );
    FreeFromStaticHeap( mBuffer[EFT_TEMPORARY_BUFFER_SECOND] );
    if ( mBuffer[EFT_TEMPORARY_BUFFER_THIRD] ) FreeFromStaticHeap( mBuffer[EFT_TEMPORARY_BUFFER_THIRD] );
}

//---------------------------------------------------------------------------
//  ダブルバッファをスワップします。
//---------------------------------------------------------------------------
void TemporaryBuffer::Swap()
{
    if ( !mInitialized ) return;

    mAllocedSize[mBufferSide] = 0;
    mFlushedSize[mBufferSide] = 0;
    mPreAllocedSize[mBufferSide] = 0;

    if ( mBuffer[EFT_TEMPORARY_BUFFER_THIRD] )
    {
        if( mBufferSide == EFT_TEMPORARY_BUFFER_THIRD )
        {
            mBufferSide = EFT_TEMPORARY_BUFFER_FIRST;
        }
        else
        {
            mBufferSide++;
        }
    }
    else
    {
        mBufferSide = 1 - mBufferSide;
    }

    mAllocedSize[mBufferSide] = 0;
    mFlushedSize[mBufferSide] = 0;
    mPreAllocedSize[mBufferSide] = 0;

    mAllocedSize[mBufferSide] = mPreAllocedSize[mBufferSide];
}

//---------------------------------------------------------------------------
//  ダブルバッファからメモリを確保する。
//---------------------------------------------------------------------------
void* TemporaryBuffer::Alloc( u32 size )
{
    if ( !mInitialized ) return NULL;
    if ( size == 0 ) return NULL;

    u32 allocSize =  nw::ut::RoundUp( size, EFT_DOUBLE_BUFFER_ALIGN );
    if ( mAllocedSize[mBufferSide] + allocSize > mBufferSize )
    {
        OutputWarning( "DoubleBuffer is Empty.\n" );
        return NULL;
    }

    void* ret = reinterpret_cast<u8*>( mBuffer[mBufferSide] ) + mAllocedSize[mBufferSide];
    mAllocedSize[mBufferSide] += allocSize;

    if ( mMaxAllocedSize < mAllocedSize[mBufferSide] )
    {
        mMaxAllocedSize = mAllocedSize[mBufferSide];
    }

#if EFT_IS_CAFE
    MemUtil::ZeroRange( ret, allocSize );
#endif

    return ret;
}

//---------------------------------------------------------------------------
//  カレントバッファのCPUメモリキャッシュをフラッシュします。
//---------------------------------------------------------------------------
void TemporaryBuffer::FlushCache()
{
    if ( !mInitialized ) return;

    if ( mFlushedSize[mBufferSide] != mAllocedSize[mBufferSide] )
    {
#if EFT_IS_CAFE
        void* startAdress = reinterpret_cast<void*>( reinterpret_cast<u32>( mBuffer[mBufferSide] ) + mFlushedSize[mBufferSide] );
        MemUtil::FlushCache( startAdress, ( mAllocedSize[mBufferSide] - mFlushedSize[mBufferSide] ) );
#endif
        mFlushedSize[mBufferSide] = mAllocedSize[mBufferSide];
    }
}


} // namespace eft2
} // namespace nw

