﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_StreamOutBuffer.h>
#include <nw/eft/eft2_Misc.h>
#include <nw/eft/eft2_MemUtil.h>
#include <nw/eft/eft2_Particle.h>
#include <nw/eft/eft2_EndianUtil.h>

namespace nw   {
namespace eft2 {

//---------------------------------------------------------------------------
//  初期化
//---------------------------------------------------------------------------
void StreamOutAttributeBuffer::Initialize( u32 size )
{
    bufferSize = size;

#if EFT_OPENGL
    glGenBuffers( 1, &uBuffer[EFT_STREAM_OUT_ATTRB_0] );
    if ( uBuffer[EFT_STREAM_OUT_ATTRB_0] != 0xFFFFFFFF )
    {
        glBindBuffer( GL_ARRAY_BUFFER, uBuffer[EFT_STREAM_OUT_ATTRB_0] );
        glBufferData( GL_ARRAY_BUFFER, bufferSize, 0, GL_STREAM_DRAW );
    }
    else
    {
        isError = true;
    }
    glGenBuffers( 1, &uBuffer[EFT_STREAM_OUT_ATTRB_1] );
    if ( uBuffer[EFT_STREAM_OUT_ATTRB_1] != 0xFFFFFFFF )
    {
        glBindBuffer( GL_ARRAY_BUFFER, uBuffer[EFT_STREAM_OUT_ATTRB_1] );
        glBufferData( GL_ARRAY_BUFFER, bufferSize, 0, GL_STREAM_DRAW );
    }
    else
    {
        isError = true;
    }
    glBindBuffer( GL_ARRAY_BUFFER, 0);
#endif
#if EFT_GX2
    u32 memSize = GX2RoundUp( bufferSize + sizeof(GX2StreamOutContext), 32 );

    pBuffer[EFT_STREAM_OUT_ATTRB_0]  = (nw::math::VEC4 *)AllocFromDynamicHeap(
        memSize, GX2_STREAMOUT_BUFFER_ALIGNMENT );
    pBuffer[EFT_STREAM_OUT_ATTRB_1]  = (nw::math::VEC4 *)AllocFromDynamicHeap(
        memSize, GX2_STREAMOUT_BUFFER_ALIGNMENT );

    if ( pBuffer[EFT_STREAM_OUT_ATTRB_0] && pBuffer[EFT_STREAM_OUT_ATTRB_1] )
    {
        MemUtil::ZeroRange( pBuffer[EFT_STREAM_OUT_ATTRB_0], memSize );
        MemUtil::ZeroRange( pBuffer[EFT_STREAM_OUT_ATTRB_1], memSize );

        // TODO:MemUtil::ZeroRange化
        memset( &streamOutBuf[EFT_STREAM_OUT_ATTRB_0],     0, sizeof(GX2StreamOutBuffer) );
        memset( &streamOutContext[EFT_STREAM_OUT_ATTRB_0], 0, sizeof(GX2StreamOutContext) );
        memset( &streamOutBuf[EFT_STREAM_OUT_ATTRB_1],     0, sizeof(GX2StreamOutBuffer) );
        memset( &streamOutContext[EFT_STREAM_OUT_ATTRB_1], 0, sizeof(GX2StreamOutContext) );

        streamOutBuf[EFT_STREAM_OUT_ATTRB_0].size       = bufferSize;
        streamOutBuf[EFT_STREAM_OUT_ATTRB_0].dataPtr    = pBuffer[EFT_STREAM_OUT_ATTRB_0];
        //        streamOutBuf[EFT_STREAM_OUT_ATTRB_0].ctxPtr     = &streamOutContext[EFT_STREAM_OUT_ATTRB_0];
        streamOutBuf[EFT_STREAM_OUT_ATTRB_0].ctxPtr     =
            (GX2StreamOutContext*)((u8*)streamOutBuf[EFT_STREAM_OUT_ATTRB_0].dataPtr +
            streamOutBuf[EFT_STREAM_OUT_ATTRB_0].size);

        streamOutBuf[EFT_STREAM_OUT_ATTRB_1].size       = bufferSize;
        streamOutBuf[EFT_STREAM_OUT_ATTRB_1].dataPtr    = pBuffer[EFT_STREAM_OUT_ATTRB_1];
        //        streamOutBuf[EFT_STREAM_OUT_ATTRB_1].ctxPtr     = &streamOutContext[EFT_STREAM_OUT_ATTRB_1];
        streamOutBuf[EFT_STREAM_OUT_ATTRB_1].ctxPtr     =
            (GX2StreamOutContext*)((u8*)streamOutBuf[EFT_STREAM_OUT_ATTRB_1].dataPtr +
            streamOutBuf[EFT_STREAM_OUT_ATTRB_1].size);

        GX2Invalidate( GX2_INVALIDATE_CPU, streamOutBuf[EFT_STREAM_OUT_ATTRB_0].dataPtr, memSize );
        //        GX2Invalidate( GX2_INVALIDATE_CPU, streamOutBuf[EFT_STREAM_OUT_ATTRB_0].ctxPtr,  sizeof(GX2StreamOutContext) );
        GX2Invalidate( GX2_INVALIDATE_CPU, streamOutBuf[EFT_STREAM_OUT_ATTRB_1].dataPtr, memSize );
        //        GX2Invalidate( GX2_INVALIDATE_CPU, streamOutBuf[EFT_STREAM_OUT_ATTRB_1].ctxPtr,  sizeof(GX2StreamOutContext) );

        streamOutBuf[EFT_STREAM_OUT_ATTRB_0].vertexStride = sizeof(nw::math::VEC4);
        streamOutBuf[EFT_STREAM_OUT_ATTRB_1].vertexStride = sizeof(nw::math::VEC4);
    }
    else
    {
        if ( pBuffer[EFT_STREAM_OUT_ATTRB_0] ) { FreeFromDynamicHeap( pBuffer[EFT_STREAM_OUT_ATTRB_0], false ); pBuffer[EFT_STREAM_OUT_ATTRB_0] = NULL; }
        if ( pBuffer[EFT_STREAM_OUT_ATTRB_1] ) { FreeFromDynamicHeap( pBuffer[EFT_STREAM_OUT_ATTRB_1], false ); pBuffer[EFT_STREAM_OUT_ATTRB_1] = NULL; }
        Invalidate();
        isError = true;
    }
#endif
}

//---------------------------------------------------------------------------
//  終了処理
//---------------------------------------------------------------------------
void StreamOutAttributeBuffer::Finalize()
{
#if EFT_OPENGL
    if ( uBuffer[EFT_STREAM_OUT_ATTRB_0] != 0xFFFFFFFF )
    {
        glDeleteBuffers( 1, &uBuffer[EFT_STREAM_OUT_ATTRB_0] );
        uBuffer[EFT_STREAM_OUT_ATTRB_0] = 0xFFFFFFFF;
    }
    if ( uBuffer[EFT_STREAM_OUT_ATTRB_1] != 0xFFFFFFFF )
    {
        glDeleteBuffers( 1, &uBuffer[EFT_STREAM_OUT_ATTRB_1] );
        uBuffer[EFT_STREAM_OUT_ATTRB_1] = 0xFFFFFFFF;
    }
#endif
#if EFT_GX2
    if ( pBuffer[EFT_STREAM_OUT_ATTRB_0] )
    {
        FreeFromDynamicHeap( pBuffer[EFT_STREAM_OUT_ATTRB_0], false );
        pBuffer[EFT_STREAM_OUT_ATTRB_0] = NULL;
    }
    if ( pBuffer[EFT_STREAM_OUT_ATTRB_1] )
    {
        FreeFromDynamicHeap( pBuffer[EFT_STREAM_OUT_ATTRB_1], false );
        pBuffer[EFT_STREAM_OUT_ATTRB_1] = NULL;
    }
#endif
}

//---------------------------------------------------------------------------
//  無効化
//---------------------------------------------------------------------------
void StreamOutAttributeBuffer::Invalidate()
{
    isError                             = false;

#if EFT_OPENGL
    uBuffer[EFT_STREAM_OUT_ATTRB_0]     = 0xFFFFFFFF;
    uBuffer[EFT_STREAM_OUT_ATTRB_1]     = 0xFFFFFFFF;
#endif
#if EFT_GX2
    pBuffer[EFT_STREAM_OUT_ATTRB_0] = NULL;
    pBuffer[EFT_STREAM_OUT_ATTRB_1] = NULL;
    memset( &streamOutBuf[EFT_STREAM_OUT_ATTRB_0],     0, sizeof(GX2StreamOutBuffer) );
    memset( &streamOutContext[EFT_STREAM_OUT_ATTRB_0], 0, sizeof(GX2StreamOutContext) );
    memset( &streamOutBuf[EFT_STREAM_OUT_ATTRB_1],     0, sizeof(GX2StreamOutBuffer) );
    memset( &streamOutContext[EFT_STREAM_OUT_ATTRB_1], 0, sizeof(GX2StreamOutContext) );
#endif
}

//---------------------------------------------------------------------------
//  入力バッファをバインド
//---------------------------------------------------------------------------
bool StreamOutAttributeBuffer::BindInBuffer( u32 input, u32 flip )
{
    if ( isError ) return false;

#if EFT_OPENGL
    if ( uBuffer[flip] == 0xFFFFFFFF )
    {
        return false;
    }
    glEnableVertexAttribArray( input );
    glBindBuffer( GL_ARRAY_BUFFER, uBuffer[flip] );
    glVertexAttribPointer( input, 4, GL_FLOAT, GL_FALSE, 0, 0 );
    EFT_GLERR_CHECK();
#elif EFT_GX2
    GX2SetAttribBuffer( input, streamOutBuf[flip].size, sizeof(nw::math::VEC4), pBuffer[flip] );
#else
    EFT_UNUSED_VARIABLE( input );
    EFT_UNUSED_VARIABLE( flip );
#endif
    return true;
}

//---------------------------------------------------------------------------
//  入力バッファをバインド
//---------------------------------------------------------------------------
bool StreamOutAttributeBuffer::BindInBuffer( u32 input, u32 flip, u32 start, u32 count )
{
    if ( isError ) return false;

#if EFT_OPENGL
    EFT_UNUSED_VARIABLE( count );
    if ( uBuffer[flip] == 0xFFFFFFFF )
    {
        return false;
    }
    glEnableVertexAttribArray( input );
    glBindBuffer( GL_ARRAY_BUFFER, uBuffer[flip] );
    glVertexAttribPointer( input, 4, GL_FLOAT, GL_FALSE, 0, (void*)( start * sizeof(nw::math::VEC4) ) );
    EFT_GLERR_CHECK();
#elif EFT_GX2
    GX2SetAttribBuffer( input, count * sizeof(nw::math::VEC4), sizeof(nw::math::VEC4), pBuffer[flip][start] );
#else
    EFT_UNUSED_VARIABLE( input );
    EFT_UNUSED_VARIABLE( flip );
    EFT_UNUSED_VARIABLE( start );
    EFT_UNUSED_VARIABLE( count );
#endif
    return true;
}

//---------------------------------------------------------------------------
//  出力バッファをバインド
//---------------------------------------------------------------------------
bool StreamOutAttributeBuffer::BindOutBuffer( u32 index, u32 flip )
{
    if ( isError ) return false;

#if EFT_OPENGL
    glBindBufferBase( GL_TRANSFORM_FEEDBACK_BUFFER, index, uBuffer[1-flip] );
    EFT_GLERR_CHECK();
#elif EFT_GX2
    GX2SetStreamOutBuffer( index,  &streamOutBuf[1-flip] );
    GX2SetStreamOutContext( index, &streamOutBuf[1-flip], GX2_TRUE );
    outputBuffer = 1-flip;
#else
    EFT_UNUSED_VARIABLE( index );
    EFT_UNUSED_VARIABLE( flip );
#endif
    return true;
}

//---------------------------------------------------------------------------
//  バインド解除
//---------------------------------------------------------------------------
void StreamOutAttributeBuffer::UnBind( u32 index )
{
    if ( isError ) return;

#ifdef EFT_OGL
    EFT_UNUSED_VARIABLE( index );
#endif
#if EFT_GX2
    if ( pBuffer[EFT_STREAM_OUT_ATTRB_0] && pBuffer[EFT_STREAM_OUT_ATTRB_1] )
    {
        GX2SaveStreamOutContext( index, &streamOutBuf[outputBuffer] );
        GX2Invalidate(
            (GX2InvalidateType)(GX2_INVALIDATE_ATTRIB_BUFFER | GX2_INVALIDATE_STREAMOUT_BUFFER),
            streamOutBuf[outputBuffer].dataPtr, bufferSize );
    }
#endif
}

#if EFT_GX2
void StreamOutAttributeBuffer::DumpSOBuffer( const char* text ) const
{
    GX2InvalidateType type = (GX2InvalidateType)(
        GX2_INVALIDATE_CPU |
        GX2_INVALIDATE_CPU_ATTRIB_BUFFER |
        GX2_INVALIDATE_ATTRIB_BUFFER |
        GX2_INVALIDATE_STREAMOUT_BUFFER |
        0 );
    GX2Invalidate( type, streamOutBuf[0].dataPtr, bufferSize );
    GX2Invalidate( type, streamOutBuf[1].dataPtr, bufferSize );

    const int count = streamOutBuf[0].size / sizeof(nw::math::VEC4);
    NW_LOG("================================================\n# of element: %d\n================================================\n", count);

    const nw::math::VEC4* pBuf0 = reinterpret_cast<nw::math::VEC4*>(streamOutBuf[0].dataPtr);
    const nw::math::VEC4* pBuf1 = reinterpret_cast<nw::math::VEC4*>(streamOutBuf[1].dataPtr);
    for( int i=0; i<count; ++i )
    {
        nw::math::VEC4 v0 = pBuf0[i];
        nw::math::VEC4 v1 = pBuf1[i];
        EndianUtil::ForceFlip( &v0 );
        EndianUtil::ForceFlip( &v1 );
        NW_LOG( "%s[0](%.2f, %.2f, %.2f, %.2f)/%s[1](%.2f, %.2f, %.2f, %.2f)\n", text, v0.x, v0.y, v0.z, v0.w, text, v1.x, v1.y, v1.z, v1.w  );
    }
}
#endif

} // namespace eft
} // namespace nw

