﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_Enum.h>
#include <nw/eft/eft2_ShaderKey.h>
#include <nw/eft/eft2_Data.h>

namespace nw {
namespace eft2 {

#ifdef EFT_OGL

// シェーダセッティング用バッファ
char ShaderKey::g_CompileSetting[ EFT_SHADER_COMPILE_SETTING_BUFFER_SIZE ];

//---------------------------------------------------------------------------
//! @brief  シェーダキー文字列生成クラス
//---------------------------------------------------------------------------
struct _ShaderKeyString
{
    //---------------------------------------------------------------------------
    //! @brief  コンパイルセッティングを取得します。
    //---------------------------------------------------------------------------
    void Initialize( char* top, u32 size )
    {
        keyTop = top;
        keySize = size;
        current = top;
    }

    //---------------------------------------------------------------------------
    //! @brief  コンパイルセッティングを取得します。
    //---------------------------------------------------------------------------
    void AddKey( const char* format, ... )
    {
        va_list vargs;
        va_start( vargs, format );
#if EFT_IS_WIN
        current += vsprintf_s( current, keySize - ( u32 )( ( uintptr_t )current - ( uintptr_t )keyTop ), format, vargs );
#else
        current += vsprintf( current, format, vargs );
#endif
        va_end( vargs );
    }

    //---------------------------------------------------------------------------
    //! @brief  EOF。
    //---------------------------------------------------------------------------
    void Eof()
    {
        *current = 0;
    }


    char* GetKeyStr() const
    {
        return keyTop;
    }

private:
    char* keyTop;
    u32   keySize;
    char* current;
};

//---------------------------------------------------------------------------
//! @brief  頂点シェーダキー定義1
//---------------------------------------------------------------------------
enum : u64
{
    // 以下は 通常 & 制限 版スペックで利用できる定義
    _PARTICLE_TYPE_BILLBOARD                = ( static_cast< u64 >( 0x01 ) << 0 ),    //!< ビルボード
    _PARTICLE_TYPE_COMPLEX_BILLBOARD        = ( static_cast< u64 >( 0x01 ) << 1 ),    //!< 豪華版ビルボード
    _PARTICLE_TYPE_Y_BILLBOARD              = ( static_cast< u64 >( 0x01 ) << 2 ),    //!< Y軸ビルボード
    _PARTICLE_TYPE_POLYGON_XY               = ( static_cast< u64 >( 0x01 ) << 3 ),    //!< XYポリゴン
    _PARTICLE_TYPE_POLYGON_XZ               = ( static_cast< u64 >( 0x01 ) << 4 ),    //!< XZポリゴン
    _PARTICLE_TYPE_VEL_LOOK                 = ( static_cast< u64 >( 0x01 ) << 5 ),    //!< ディレクショナルY
    _PARTICLE_TYPE_VEL_LOOK_POLYGON         = ( static_cast< u64 >( 0x01 ) << 6 ),    //!< ディレクショナル
    _EMITTER_FOLLOW_TYPE_ALL                = ( static_cast< u64 >( 0x01 ) << 7 ),    //!< 完全追従
    _EMITTER_FOLLOW_TYPE_NONE               = ( static_cast< u64 >( 0x01 ) << 8 ),    //!< 追従なし
    _EMITTER_FOLLOW_TYPE_POS                = ( static_cast< u64 >( 0x01 ) << 9 ),    //!< 位置のみ追従
    _PARTICLE                               = ( static_cast< u64 >( 0x01 ) << 10 ),   //!< パーティクル
    _PRIMITIVE                              = ( static_cast< u64 >( 0x01 ) << 11 ),   //!< プリミティブ

    // 以下は通常版スペックで利用できる定義
    _CALC_TYPE_CPU                          = ( static_cast< u64 >( 0x01 ) << 12 ),    //!< CPUエミッタ
    _CALC_TYPE_GPU_TIME                     = ( static_cast< u64 >( 0x01 ) << 13 ),    //!< GPUタイムエミッタ
    _CALC_TYPE_GPU_SO                       = ( static_cast< u64 >( 0x01 ) << 14 ),    //!< GPU StreamOut エミッタ
    _SHADER_ANIM                            = ( static_cast< u64 >( 0x01 ) << 15 ),    //!< シェーダアニメ
    _WORLD_GRAVITY                          = ( static_cast< u64 >( 0x01 ) << 16 ),    //!< 世界座標系で重力を計算する
    _CAMERA_OFFSET                          = ( static_cast< u64 >( 0x01 ) << 17 ),    //!< カメラオフセット
    _DEPTH_OFFSET                           = ( static_cast< u64 >( 0x01 ) << 18 ),    //!< デプスオフセット
    _USE_ROTATE                             = ( static_cast< u64 >( 0x01 ) << 19 ),    //!< パーティクル回転
    _ROTATE_YZX                             = ( static_cast< u64 >( 0x01 ) << 20 ),    //!< YZX回転
    _ROTATE_XYZ                             = ( static_cast< u64 >( 0x01 ) << 21 ),    //!< XYZ回転
    _ROTATE_ZXY                             = ( static_cast< u64 >( 0x01 ) << 22 ),    //!< ZXY回転
    _COLOR_0_FIXED                          = ( static_cast< u64 >( 0x01 ) << 23 ),    //!< カラー0 固定
    _COLOR_0_RANDOM                         = ( static_cast< u64 >( 0x01 ) << 24 ),    //!< カラー0 ランダム
    _COLOR0_INHERIT                         = ( static_cast< u64 >( 0x01 ) << 25 ),    //!< カラー0 継承
    _COLOR_1_FIXED                          = ( static_cast< u64 >( 0x01 ) << 26 ),    //!< カラー1 固定
    _COLOR_1_RANDOM                         = ( static_cast< u64 >( 0x01 ) << 27 ),    //!< カラー1 ランダム
    _COLOR1_INHERIT                         = ( static_cast< u64 >( 0x01 ) << 28 ),    //!< カラー1 継承
    _ALPHA_0_FIXED                          = ( static_cast< u64 >( 0x01 ) << 29 ),    //!< アルファ0 固定
    _ALPHA0_INHERIT                         = ( static_cast< u64 >( 0x01 ) << 30 ),    //!< アルファ0 継承
    _ALPHA_1_FIXED                          = ( static_cast< u64 >( 0x01 ) << 31 ),    //!< アルファ1 固定
    _ALPHA1_INHERIT                         = ( static_cast< u64 >( 0x01 ) << 32 ),    //!< アルファ1 継承
    _NEAR_DIST_ALPHA                        = ( static_cast< u64 >( 0x01 ) << 33 ),    //!< 近距離アルファ
    _FAR_DIST_ALPHA                         = ( static_cast< u64 >( 0x01 ) << 34 ),    //!< 近距離アルファ
    _MASKING_CHECK                          = ( static_cast< u64 >( 0x01 ) << 35 ),    //!< 遮蔽チェック
    _VERTEX_SOFT                            = ( static_cast< u64 >( 0x01 ) << 36 ),    //!< 頂点ソフト
    _VERTEX_FRESNEL_ALPHA                   = ( static_cast< u64 >( 0x01 ) << 37 ),    //!< 頂点ソフト
    _FLUCTUATION_ALPHA_ENABLE               = ( static_cast< u64 >( 0x01 ) << 38 ),    //!< アルファ揺らぎ 有効/無効
    _FLUCTUATION_SCALE_ENABLE               = ( static_cast< u64 >( 0x01 ) << 39 ),    //!< スケール揺らぎ 有効/無効
    _FLUCTUATION_SCALE_Y_SEPARATE           = ( static_cast< u64 >( 0x01 ) << 40 ),    //!< スケール揺らぎYを別個で計算するか
    _PARTICLE_SCALE_LIMIT_NEAR              = ( static_cast< u64 >( 0x01 ) << 41 ),    //!< カメラ距離スケール制限「近」
    _VELOCITY_SCALE_Y                       = ( static_cast< u64 >( 0x01 ) << 42 ),    //!< 速度スケールY
    _TEXTURE0_USE_UV1                       = ( static_cast< u64 >( 0x01 ) << 43 ),    //!< テクスチャ0 uv1を使用
    _TEXTURE1_USE_UV1                       = ( static_cast< u64 >( 0x01 ) << 44 ),    //!< テクスチャ1 uv1を使用
    _TEXTURE2_USE_UV1                       = ( static_cast< u64 >( 0x01 ) << 45 ),    //!< テクスチャ2 uv1を使用
    _TEX_0_SHIFT_ANIM                       = ( static_cast< u64 >( 0x01 ) << 46 ),    //!< テクスチャ0 座標アニメ
    _TEX_1_SHIFT_ANIM                       = ( static_cast< u64 >( 0x01 ) << 47 ),    //!< テクスチャ1 座標アニメ
    _TEX_2_SHIFT_ANIM                       = ( static_cast< u64 >( 0x01 ) << 48 ),    //!< テクスチャ2 座標アニメ
    _TEX_0_PATTERN_ANIM                     = ( static_cast< u64 >( 0x01 ) << 49 ),    //!< テクスチャ0 パタンアニメ
    _TEX_1_PATTERN_ANIM                     = ( static_cast< u64 >( 0x01 ) << 50 ),    //!< テクスチャ1 パタンアニメ
    _TEX_2_PATTERN_ANIM                     = ( static_cast< u64 >( 0x01 ) << 51 ),    //!< テクスチャ2 パタンアニメ
    _TEX_0_SPHERE_MAP                       = ( static_cast< u64 >( 0x01 ) << 52 ),    //!< テクスチャ0 スフィアマップ
    _TEX_1_SPHERE_MAP                       = ( static_cast< u64 >( 0x01 ) << 53 ),    //!< テクスチャ1 スフィアマップ
    _TEX_2_SPHERE_MAP                       = ( static_cast< u64 >( 0x01 ) << 54 ),    //!< テクスチャ2 スフィアマップ
    _TEX_0_SHIFT_ANIM_ROTATE                = ( static_cast< u64 >( 0x01 ) << 55 ),    //!< テクスチャ1 座標アニメ回転
    _TEX_1_SHIFT_ANIM_ROTATE                = ( static_cast< u64 >( 0x01 ) << 56 ),    //!< テクスチャ2 座標アニメ回転
    _TEX_2_SHIFT_ANIM_ROTATE                = ( static_cast< u64 >( 0x01 ) << 57 ),    //!< テクスチャ3 座標アニメ回転
    _PARTICLE_TYPE_Y_BILLBOARD_PARALLEL     = ( static_cast< u64 >( 0x01 ) << 58 ),    //!< Y軸ビルボード（カメラに平行）
    _PARTICLE_TYPE_WITH_SCALE_Z_DIRECTIONAL = ( static_cast< u64 >( 0x01 ) << 59 ),    //!< 立体風ビルボード（ディレクショナル版）
    _PARTICLE_TYPE_WITH_SCALE_Z             = ( static_cast< u64 >( 0x01 ) << 60 ),    //!< 立体風ビルボード
    _PARTICLE_SCALE_LIMIT_FAR               = ( static_cast< u64 >( 0x01 ) << 61 ),    //!< カメラ距離スケール制限「遠」
};

//---------------------------------------------------------------------------
//! @brief  頂点シェーダキー定義2
//---------------------------------------------------------------------------
enum : u64
{
    _FIELD_RANDOM                           = ( static_cast< u64 >( 0x01 ) << 0  ),    //!< フィールドランダム
    _TEX0_KEEP_TEXTURE_SIZE_U               = ( static_cast< u64 >( 0x01 ) << 1  ),    //!< テクスチャ0 描画サイズをスケール値によって変えない
    _TEX1_KEEP_TEXTURE_SIZE_U               = ( static_cast< u64 >( 0x01 ) << 2  ),    //!< テクスチャ1 描画サイズをスケール値によって変えない
    _TEX2_KEEP_TEXTURE_SIZE_U               = ( static_cast< u64 >( 0x01 ) << 3  ),    //!< テクスチャ2 描画サイズをスケール値によって変えない
    _TEX0_KEEP_TEXTURE_SIZE_V               = ( static_cast< u64 >( 0x01 ) << 4  ),    //!< テクスチャ0 描画サイズをスケール値によって変えない
    _TEX1_KEEP_TEXTURE_SIZE_V               = ( static_cast< u64 >( 0x01 ) << 5  ),    //!< テクスチャ1 描画サイズをスケール値によって変えない
    _TEX2_KEEP_TEXTURE_SIZE_V               = ( static_cast< u64 >( 0x01 ) << 6  ),    //!< テクスチャ2 描画サイズをスケール値によって変えない
    _CONNECT_PTCL_SCALE_TO_Z_OFFSET         = ( static_cast< u64 >( 0x01 ) << 7  ),    //!< ZオフセットとPtclスケールを連動
    _LINK_FOVY_TO_SCALE_LIMIT               = ( static_cast< u64 >( 0x01 ) << 8  ),    //!< カメラ距離スケール制限に画角を連動させる
    _AVOID_Z_FIGHTING                       = ( static_cast< u64 >( 0x01 ) << 9  ),    //!< Z-Fighting 抑制
    _CAMERA_OFFSET_FIXED_SIZE               = ( static_cast< u64 >( 0x01 ) << 10 ),    //!< サイズ固定カメラオフセット
};


//---------------------------------------------------------------------------
//! @brief  フラグメントシェーダキー定義1
//---------------------------------------------------------------------------
enum : u64
{
    _COLOR_PROCESS_COLOR                    = ( static_cast< u64 >( 0x01 ) << 0  ),    //!< Color0
    _COLOR_PROCESS_TEXTURE                  = ( static_cast< u64 >( 0x01 ) << 1  ),    //!< Color0 * Texture
    _COLOR_PROCESS_TEXTURE_INTERPOLATE      = ( static_cast< u64 >( 0x01 ) << 2  ),    //!< Texture * Color0 + ( 1 - Texture ) * Color1
    _COLOR_PROCESS_TEXTURE_ADD              = ( static_cast< u64 >( 0x01 ) << 3  ),    //!< Texture * Color0 + Color1
    _ALPHA_PROCESS_MOD                      = ( static_cast< u64 >( 0x01 ) << 4  ),    //!< Alpha0 * Texture
    _ALPHA_PROCESS_SUB                      = ( static_cast< u64 >( 0x01 ) << 5  ),    //!< ( Texture - ( 1 - Alpha0 ) ) * 2
    _ALPHA_PROCESS_MOD_A0_A1                = ( static_cast< u64 >( 0x01 ) << 6  ),    //!< Texture * Alpha0 * Alpha1
    _ALPHA_PROCESS_SUB_A0_MOD_A1            = ( static_cast< u64 >( 0x01 ) << 7  ),    //!< ( Texture - Alpha0 ) * Alpha1
    _TEX0_COLOR_INPUT_ONE                   = ( static_cast< u64 >( 0x01 ) << 8  ),    //!< テクスチャ0 入力カラーを1.0で置き換え
    _TEX1_COLOR_INPUT_ONE                   = ( static_cast< u64 >( 0x01 ) << 9  ),    //!< テクスチャ1 入力カラーを1.0で置き換え
    _TEX2_COLOR_INPUT_ONE                   = ( static_cast< u64 >( 0x01 ) << 10 ),    //!< テクスチャ2 入力カラーを1.0で置き換え
    _TEX0_ALPHA_INPUT_RED                   = ( static_cast< u64 >( 0x01 ) << 11 ),    //!< テクスチャ0 入力アルファをR値で置き換え
    _TEX0_ALPHA_INPUT_ONE                   = ( static_cast< u64 >( 0x01 ) << 12 ),    //!< テクスチャ0 入力アルファを1.0で置き換え
    _TEX1_ALPHA_INPUT_RED                   = ( static_cast< u64 >( 0x01 ) << 13 ),    //!< テクスチャ1 入力アルファをR値で置き換え
    _TEX1_ALPHA_INPUT_ONE                   = ( static_cast< u64 >( 0x01 ) << 14 ),    //!< テクスチャ1 入力アルファを1.0で置き換え
    _TEX2_ALPHA_INPUT_RED                   = ( static_cast< u64 >( 0x01 ) << 15 ),    //!< テクスチャ2 入力アルファをR値で置き換え
    _TEX2_ALPHA_INPUT_ONE                   = ( static_cast< u64 >( 0x01 ) << 16 ),    //!< テクスチャ2 入力アルファを1.0で置き換え
    _SHADER_TYPE_NORMAL                     = ( static_cast< u64 >( 0x01 ) << 17 ),    //!< 通常シェーダ
    _SHADER_TYPE_REFRACT                    = ( static_cast< u64 >( 0x01 ) << 18 ),    //!< 屈折シェーダ
    _SHADER_TYPE_DISTORTION                 = ( static_cast< u64 >( 0x01 ) << 19 ),    //!< 歪みシェーダ
    _TEXTURE0_ENABLE                        = ( static_cast< u64 >( 0x01 ) << 20 ),    //!< テクスチャ2 を有効にする
    _TEXTURE1_ENABLE                        = ( static_cast< u64 >( 0x01 ) << 21 ),    //!< テクスチャ1 を有効にする
    _TEXTURE2_ENABLE                        = ( static_cast< u64 >( 0x01 ) << 22 ),    //!< テクスチャ2 を有効にする
    _TEXTURE1_COLOR_BLEND_MOD               = ( static_cast< u64 >( 0x01 ) << 23 ),    //!< テクスチャ0RGB * テクスチャ1RGB
    _TEXTURE1_COLOR_BLEND_ADD               = ( static_cast< u64 >( 0x01 ) << 24 ),    //!< テクスチャ0RGB + テクスチャ1RGB
    _TEXTURE1_COLOR_BLEND_SUB               = ( static_cast< u64 >( 0x01 ) << 25 ),    //!< テクスチャ0RGB - テクスチャ1RGB
    _TEXTURE2_COLOR_BLEND_MOD               = ( static_cast< u64 >( 0x01 ) << 26 ),    //!< テクスチャ0RGB * テクスチャ2RGB
    _TEXTURE2_COLOR_BLEND_ADD               = ( static_cast< u64 >( 0x01 ) << 27 ),    //!< テクスチャ0RGB + テクスチャ2RGB
    _TEXTURE2_COLOR_BLEND_SUB               = ( static_cast< u64 >( 0x01 ) << 28 ),    //!< テクスチャ0RGB - テクスチャ2RGB
    _TEXTURE1_ALPHA_BLEND_MOD               = ( static_cast< u64 >( 0x01 ) << 29 ),    //!< テクスチャ0A * テクスチャ1A
    _TEXTURE1_ALPHA_BLEND_ADD               = ( static_cast< u64 >( 0x01 ) << 30 ),    //!< テクスチャ0A + テクスチャ1A
    _TEXTURE1_ALPHA_BLEND_SUB               = ( static_cast< u64 >( 0x01 ) << 31 ),    //!< テクスチャ0A - テクスチャ1A
    _TEXTURE2_ALPHA_BLEND_MOD               = ( static_cast< u64 >( 0x01 ) << 32 ),    //!< テクスチャ0A * テクスチャ2A
    _TEXTURE2_ALPHA_BLEND_ADD               = ( static_cast< u64 >( 0x01 ) << 33 ),    //!< テクスチャ0A + テクスチャ2A
    _TEXTURE2_ALPHA_BLEND_SUB               = ( static_cast< u64 >( 0x01 ) << 34 ),    //!< テクスチャ0A - テクスチャ2A
    _PRIMITIVE_COLOR_BLEND_MOD              = ( static_cast< u64 >( 0x01 ) << 35 ),    //!< プリミティブカラー 乗算
    _PRIMITIVE_COLOR_BLEND_ADD              = ( static_cast< u64 >( 0x01 ) << 36 ),    //!< プリミティブカラー 加算
    _PRIMITIVE_COLOR_BLEND_SUB              = ( static_cast< u64 >( 0x01 ) << 37 ),    //!< プリミティブカラー 減算
    _PRIMITIVE_ALPHA_BLEND_MOD              = ( static_cast< u64 >( 0x01 ) << 38 ),    //!< プリミティブアルファ 乗算
    _PRIMITIVE_ALPHA_BLEND_ADD              = ( static_cast< u64 >( 0x01 ) << 39 ),    //!< プリミティブアルファ 加算
    _PRIMITIVE_ALPHA_BLEND_SUB              = ( static_cast< u64 >( 0x01 ) << 40 ),    //!< プリミティブアルファ 減算
    _PRIMITIVE_COLOR_INPUT_SRC              = ( static_cast< u64 >( 0x01 ) << 41 ),    //!< プリミティブ入力カラーをR値で置き換え
    _PRIMITIVE_COLOR_INPUT_ONE              = ( static_cast< u64 >( 0x01 ) << 42 ),    //!< プリミティブ入力カラーを1.0で置き換え
    _PRIMITIVE_ALPHA_INPUT_A                = ( static_cast< u64 >( 0x01 ) << 43 ),    //!< プリミティブ入力アルファをR値で置き換え
    _PRIMITIVE_ALPHA_INPUT_RED              = ( static_cast< u64 >( 0x01 ) << 44 ),    //!< プリミティブ入力アルファを1.0で置き換え
    _PRIMITIVE_ALPHA_INPUT_ONE              = ( static_cast< u64 >( 0x01 ) << 45 ),    //!< プリミティブ入力アルファを1.0で置き換え
    _FRAGMENT_SOFT                          = ( static_cast< u64 >( 0x01 ) << 46 ),    //!< フラグメントソフト
    _DECAL                                  = ( static_cast< u64 >( 0x01 ) << 47 ),    //!< デカール
    _FRAGMENT_FRESNEL_ALPHA                 = ( static_cast< u64 >( 0x01 ) << 48 ),    //!< デカール
    _REFRACT_APPLY_ALPHA                    = ( static_cast< u64 >( 0x01 ) << 49 ),    //!< 屈折アルファ値を適用する
    _ALPHA_PROCESS_DIST_FILED               = ( static_cast< u64 >( 0x01 ) << 50 ),    //!< clamp( ( Texture - Alpha0 ) *4 )* Alpha1
    _ALPHA_COMPARE_NEVER                    = ( static_cast< u64 >( 0x01 ) << 51 ),    //!< アルファテスト比較モード(常に通過しない)
    _ALPHA_COMPARE_LESS                     = ( static_cast< u64 >( 0x01 ) << 52 ),    //!< アルファテスト比較モード(＜ 基準値)
    _ALPHA_COMPARE_EQUAL                    = ( static_cast< u64 >( 0x01 ) << 53 ),    //!< アルファテスト比較モード(＝ 基準値)
    _ALPHA_COMPARE_LEQUAL                   = ( static_cast< u64 >( 0x01 ) << 54 ),    //!< アルファテスト比較モード(≦ 基準値)
    _ALPHA_COMPARE_GREATER                  = ( static_cast< u64 >( 0x01 ) << 55 ),    //!< アルファテスト比較モード(＞ 基準値)
    _ALPHA_COMPARE_NOTEQUAL                 = ( static_cast< u64 >( 0x01 ) << 56 ),    //!< アルファテスト比較モード(≠ 基準値)
    _ALPHA_COMPARE_GEQUAL                   = ( static_cast< u64 >( 0x01 ) << 57 ),    //!< アルファテスト比較モード(≧ 基準値)
    _ALPHA_COMPARE_ALWAYS                   = ( static_cast< u64 >( 0x01 ) << 58 ),    //!< アルファテスト比較モード(常に通過)
    _TEX0_COLOR_INPUT_SQUARE                = ( static_cast< u64 >( 0x01 ) << 59 ),    //!< テクスチャ0 入力カラーを二乗して取得
    _TEX1_COLOR_INPUT_SQUARE                = ( static_cast< u64 >( 0x01 ) << 60 ),    //!< テクスチャ1 入力カラーを二乗して取得
    _TEX2_COLOR_INPUT_SQUARE                = ( static_cast< u64 >( 0x01 ) << 61 ),    //!< テクスチャ2 入力カラーを二乗して取得
    _REFRACT_BY_CAMERA_DISTANCE             = ( static_cast< u64 >( 0x01 ) << 62 ),    //!< カメラ距離に応じてカラーバッファ屈折の強度を強くする
};

//---------------------------------------------------------------------------
//! @brief  フラグメントシェーダキー定義2
//---------------------------------------------------------------------------
enum : u64
{
    _TEX0_COLOR_INPUT_ONE_MINUS_RGB         = ( static_cast< u64 >( 0x01 ) <<  0 ),    //!< テクスチャ0 入力カラーを (1.0 - RGB) で置き換え
    _TEX1_COLOR_INPUT_ONE_MINUS_RGB         = ( static_cast< u64 >( 0x01 ) <<  1 ),    //!< テクスチャ1 入力カラーを (1.0 - RGB) で置き換え
    _TEX2_COLOR_INPUT_ONE_MINUS_RGB         = ( static_cast< u64 >( 0x01 ) <<  2 ),    //!< テクスチャ2 入力カラーを (1.0 - RGB) で置き換え
    _TEX0_COLOR_INPUT_ALPHA                 = ( static_cast< u64 >( 0x01 ) <<  3 ),    //!< テクスチャ0 入力カラーを Alpha で置き換え
    _TEX1_COLOR_INPUT_ALPHA                 = ( static_cast< u64 >( 0x01 ) <<  4 ),    //!< テクスチャ1 入力カラーを Alpha で置き換え
    _TEX2_COLOR_INPUT_ALPHA                 = ( static_cast< u64 >( 0x01 ) <<  5 ),    //!< テクスチャ2 入力カラーを Alpha で置き換え
    _TEX0_COLOR_INPUT_ONE_MINUS_ALPHA       = ( static_cast< u64 >( 0x01 ) <<  6 ),    //!< テクスチャ0 入力カラーを (1.0 - Alpha) で置き換え
    _TEX1_COLOR_INPUT_ONE_MINUS_ALPHA       = ( static_cast< u64 >( 0x01 ) <<  7 ),    //!< テクスチャ1 入力カラーを (1.0 - Alpha) で置き換え
    _TEX2_COLOR_INPUT_ONE_MINUS_ALPHA       = ( static_cast< u64 >( 0x01 ) <<  8 ),    //!< テクスチャ2 入力カラーを (1.0 - Alpha) で置き換え
    _TEX0_ALPHA_INPUT_ONE_MINUS_RED         = ( static_cast< u64 >( 0x01 ) <<  9 ),    //!< テクスチャ0 入力アルファを (1.0 - Red) で置き換え
    _TEX1_ALPHA_INPUT_ONE_MINUS_RED         = ( static_cast< u64 >( 0x01 ) << 10 ),    //!< テクスチャ1 入力アルファを (1.0 - Red) で置き換え
    _TEX2_ALPHA_INPUT_ONE_MINUS_RED         = ( static_cast< u64 >( 0x01 ) << 11 ),    //!< テクスチャ2 入力アルファを (1.0 - Red) で置き換え
    _TEX0_ALPHA_INPUT_ONE_MINUS_ALPHA       = ( static_cast< u64 >( 0x01 ) << 12 ),    //!< テクスチャ0 入力アルファを (1.0 - Alpha) で置き換え
    _TEX1_ALPHA_INPUT_ONE_MINUS_ALPHA       = ( static_cast< u64 >( 0x01 ) << 13 ),    //!< テクスチャ1 入力アルファを (1.0 - Alpha) で置き換え
    _TEX2_ALPHA_INPUT_ONE_MINUS_ALPHA       = ( static_cast< u64 >( 0x01 ) << 14 ),    //!< テクスチャ2 入力アルファを (1.0 - Alpha) で置き換え
    _PRIMITIVE_COLOR_INPUT_ONE_MINUS_RGB    = ( static_cast< u64 >( 0x01 ) << 15 ),    //!< プリミティブ入力カラーを (1.0 - RGB) で置き換え
    _PRIMITIVE_COLOR_INPUT_ALPHA            = ( static_cast< u64 >( 0x01 ) << 16 ),    //!< プリミティブ入力カラーを Alpha で置き換え
    _PRIMITIVE_COLOR_INPUT_ONE_MINUS_ALPHA  = ( static_cast< u64 >( 0x01 ) << 17 ),    //!< プリミティブ入力カラーを (1.0 - Alpha) で置き換え
    _PRIMITIVE_ALPHA_INPUT_ONE_MINUS_RED    = ( static_cast< u64 >( 0x01 ) << 18 ),    //!< プリミティブ入力アルファを (1.0 - Red) で置き換え
    _PRIMITIVE_ALPHA_INPUT_ONE_MINUS_ALPHA  = ( static_cast< u64 >( 0x01 ) << 19 ),    //!< プリミティブ入力アルファを (1.0 - Alpha) で置き換え
};


//---------------------------------------------------------------------------
//! @brief  EmitterPlugin用 頂点シェーダキー定義
//---------------------------------------------------------------------------
enum : u64
{
    _STRIPE_BILLBOARD                       = ( static_cast< u64 >( 0x01 ) << 0  ),    //!< ストライプ形状：ビルボード
    _STRIPE_EMITTER_MATRIX                  = ( static_cast< u64 >( 0x01 ) << 1  ),    //!< ストライプ形状：エミッタ行列ストライプ
    _STRIPE_EMITTER_UPDOWN                  = ( static_cast< u64 >( 0x01 ) << 2  ),    //!< ストライプ形状：エミッタの上下に伸ばす
    _STRIPE_FOLLOW_EMITTER_MATRIX           = ( static_cast< u64 >( 0x01 ) << 3  ),    //!< ストライプ：エミッタに完全追従
    _STRIPE_RIBBON                          = ( static_cast< u64 >( 0x01 ) << 4  ),    //!< ストライプ形状：リボン
};

//---------------------------------------------------------------------------
//! @brief  EmitterPlugin用 フラグメントシェーダキー定義
//---------------------------------------------------------------------------
enum : u64
{
    _EMITTER_PLUGIN_FRAGMENT_DUMMY          = ( static_cast< u64 >( 0x01 ) << 0 )    //!< Dummy
};

//---------------------------------------------------------------------------
//  StreamOutを利用するか
//---------------------------------------------------------------------------
bool ShaderKey::IsStreamOut() const
{
    return m_VtxFlag.Is( _CALC_TYPE_GPU_SO );
}

//---------------------------------------------------------------------------
//  初期化
//---------------------------------------------------------------------------
void ShaderKey::Initialize( ResEmitter* resEmitter, u32 emitterPluginIndex, void* emitterPluginData, u32 fieldFlag, ResFieldCustom* resCustomField, const char* userDefine )
{
    Invalidate();

#ifdef EFT_DEGRADATION_SPEC
    EFT_UNUSED_VARIABLE( fieldFlag );
    EFT_UNUSED_VARIABLE( emitterPluginIndex );
    EFT_UNUSED_VARIABLE( emitterPluginData );
#endif

    //------------------------------------------------
    // 頂点シェーダ向けキー
    //------------------------------------------------

    // パーティクル or プリミティブ
    SetFigureType( !( resEmitter->ptcl.primitiveIndex == EFT_INVALID_ASSET_GUID ) );

    // パーティクルタイプ
    SetParticleType( resEmitter->ptcl.billboardType );

    // 追従タイプ
    SetEmitterFollowType( resEmitter->emitter.followType );

    // パーティクル回転
    if( resEmitter->ptcl.isRotateX || resEmitter->ptcl.isRotateY || resEmitter->ptcl.isRotateZ )
    {
        m_VtxFlag.Set( _USE_ROTATE );
    }
    SetParticleRotateType( resEmitter->ptcl.rotType );

    // 挙動計算タイプ
    SetEmitterCalcType( resEmitter->emitter.calcType );

#ifndef EFT_DEGRADATION_SPEC

    // 描画パス
    SetDrawPath( resEmitter->emitter.drawPath );

    // パーティクルタイプ
    if( resEmitter->emission.isWorldGravity )
    {
        SetParticleWorldGravity( true );
    }

    // パーティクルカラー計算タイプ
    SetParticleColorCalcType( resEmitter->ptclColor.color0Type, resEmitter->ptclColor.color1Type );

    // パーティクルアルファ計算タイプ
    SetParticleAlphaCalcType( resEmitter->ptclColor.alpha0Type, resEmitter->ptclColor.alpha1Type );

    // テクスチャパタンアニメ 有効/無効
    SetParticleTexturePatternAnim( ( resEmitter->textureAnim0.patternAnimType != EFT_TEXTURE_PTN_ANIM_TYPE_NONE ),
        ( resEmitter->textureAnim1.patternAnimType != EFT_TEXTURE_PTN_ANIM_TYPE_NONE ),
        ( resEmitter->textureAnim2.patternAnimType != EFT_TEXTURE_PTN_ANIM_TYPE_NONE ) );

    // テクスチャ座標アニメ 有効/無効
    SetParticleTextureShiftAnim( ( resEmitter->textureAnim0.isScroll || resEmitter->textureAnim0.isScale ),
        ( resEmitter->textureAnim1.isScroll || resEmitter->textureAnim1.isScale ),
        ( resEmitter->textureAnim2.isScroll || resEmitter->textureAnim2.isScale ) );

    if( resEmitter->textureAnim0.isRotate )
    {
        m_VtxFlag.Set( _TEX_0_SHIFT_ANIM_ROTATE );
    }
    if( resEmitter->textureAnim1.isRotate )
    {
        m_VtxFlag.Set( _TEX_1_SHIFT_ANIM_ROTATE );
    }
    if( resEmitter->textureAnim2.isRotate )
    {
        m_VtxFlag.Set( _TEX_2_SHIFT_ANIM_ROTATE );
    }

    // スフィアマップの 有効/無効
    SetParticleSphereMap( ( resEmitter->textureSampler0.isSphereMap != 0 ),
        ( resEmitter->textureSampler1.isSphereMap != 0 ),
        ( resEmitter->textureSampler2.isSphereMap != 0 ) );

    // スケール キー数を見て変化させる
    //{
    //    SetParticleScaleAnim( resEmitter->staticUbo.scaleAnimKeyNum );
    //}
    m_VtxScaleAnimKey = static_cast< u8 >( resEmitter->staticUbo.scaleAnimKeyNum );
    m_VtxColor0AnimKey = ( resEmitter->ptclColor.color0Type == EFT_PARTICLE_COLOR_CALC_TYPE_8KEY ) ? static_cast< u8 >( resEmitter->staticUbo.color0AnimKeynum ) : 0;
    m_VtxColor1AnimKey = ( resEmitter->ptclColor.color1Type == EFT_PARTICLE_COLOR_CALC_TYPE_8KEY ) ? static_cast< u8 >( resEmitter->staticUbo.color1AnimKeynum ) : 0;
    m_VtxAlpha0AnimKey = ( resEmitter->ptclColor.alpha0Type == EFT_PARTICLE_COLOR_CALC_TYPE_8KEY ) ? static_cast< u8 >( resEmitter->staticUbo.alpha0AnimKeyNum ) : 0;
    m_VtxAlpha1AnimKey = ( resEmitter->ptclColor.alpha1Type == EFT_PARTICLE_COLOR_CALC_TYPE_8KEY ) ? static_cast< u8 >( resEmitter->staticUbo.alpha1AnimKeyNum ) : 0;

    // 揺らぎ 有効/無効
    if( resEmitter->ptclFluctuation.isApplayScale )
    {
        m_VtxFlag.Set( _FLUCTUATION_SCALE_ENABLE );
    }
    if( resEmitter->ptclFluctuation.isApplyAlpha )
    {
        m_VtxFlag.Set( _FLUCTUATION_ALPHA_ENABLE );
    }
    if( resEmitter->ptclFluctuation.isApplayScaleY )
    {
        m_VtxFlag.Set( _FLUCTUATION_SCALE_Y_SEPARATE );
    }

    // 近距離アルファ
    if( resEmitter->ptclColor.isNearDistAlpha )
    {
        m_VtxFlag.Set( _NEAR_DIST_ALPHA );
    }

    // 遠距離アルファ
    if( resEmitter->ptclColor.isFarDistAlpha )
    {
        m_VtxFlag.Set( _FAR_DIST_ALPHA );
    }

    // ソフトパーティクル
    if( resEmitter->ptclColor.isSoftParticle == 1 )
    {
        m_VtxFlag.Set( _VERTEX_SOFT );
    }

    // 遮蔽チェック
    if( resEmitter->ptclColor.isSoftParticle == 3 )
    {
        m_VtxFlag.Set( _MASKING_CHECK );
    }

    // 頂点フレネル
    if( resEmitter->ptclColor.isFresnelAlpha == 1 )
    {
        m_VtxFlag.Set( _VERTEX_FRESNEL_ALPHA );
    }

    // フィールドフラグを立てる
    if( fieldFlag & EFT_SHADER_KEY_FIELD_RANDOM )
    {
        m_VtxFlag2.Set( _FIELD_RANDOM );
    }

    // カメラ前でのスケールを抑制
    if( resEmitter->ptclScale.enableScalingByCameraDistNear == 1 )
    {
        m_VtxFlag.Set( _PARTICLE_SCALE_LIMIT_NEAR );
    }
    if( resEmitter->ptclScale.enableScalingByCameraDistFar == 1 )
    {
        m_VtxFlag.Set( _PARTICLE_SCALE_LIMIT_FAR );
    }
    if( resEmitter->ptclScale.enableLinkFovyToScaleValue == 1 )
    {
        m_VtxFlag2.Set( _LINK_FOVY_TO_SCALE_LIMIT );
    }

    // パーティクルオフセット
    if( resEmitter->ptcl.offsetType == EFT_PARTICLE_OFFSET_TYPE_DEPTH )
    {
        m_VtxFlag.Set( _DEPTH_OFFSET );
    }
    if( resEmitter->ptcl.offsetType == EFT_PARTICLE_OFFSET_TYPE_CAMERA )
    {
        m_VtxFlag.Set( _CAMERA_OFFSET );
    }
    if( resEmitter->ptcl.offsetType == EFT_PARTICLE_OFFSET_TYPE_CAMERA_FIXED_SIZE )
    {
        m_VtxFlag2.Set( _CAMERA_OFFSET_FIXED_SIZE );
    }

    if( resEmitter->ptcl.connectPtclScaleAndZOffset )
    {
        m_VtxFlag2.Set( _CONNECT_PTCL_SCALE_TO_Z_OFFSET );
    }
    if( resEmitter->ptcl.enableAvoidZFighting )
    {
        m_VtxFlag2.Set( _AVOID_Z_FIGHTING );
    }

    // 速度スケールY
    if( resEmitter->ptclScale.enableAddScaleY )
    {
        m_VtxFlag.Set( _VELOCITY_SCALE_Y );
    }

    // テクスチャ描画サイズを固定
    if( resEmitter->textureSampler0.isDensityFixedU )
    {
        m_VtxFlag2.Set( _TEX0_KEEP_TEXTURE_SIZE_U );
    }
    if( resEmitter->textureSampler1.isDensityFixedU )
    {
        m_VtxFlag2.Set( _TEX1_KEEP_TEXTURE_SIZE_U );
    }
    if( resEmitter->textureSampler2.isDensityFixedU )
    {
        m_VtxFlag2.Set( _TEX2_KEEP_TEXTURE_SIZE_U );
    }
    if( resEmitter->textureSampler0.isDensityFixedV )
    {
        m_VtxFlag2.Set( _TEX0_KEEP_TEXTURE_SIZE_V );
    }
    if( resEmitter->textureSampler1.isDensityFixedV )
    {
        m_VtxFlag2.Set( _TEX1_KEEP_TEXTURE_SIZE_V );
    }
    if( resEmitter->textureSampler2.isDensityFixedV )
    {
        m_VtxFlag2.Set( _TEX2_KEEP_TEXTURE_SIZE_V );
    }

    // シェーダ係数アニメ
    if( ( resEmitter->combiner.shaderType != EFT_FRAGMENT_SHADER_TYPE_NORMAL || resEmitter->shader.eftCombinerShaderIndex != -1 ) &&
        resEmitter->staticUbo.shaderAnimKeyNum > 1 )
    {
        m_VtxFlag.Set( _SHADER_ANIM );
    }
    else
    {
        m_VtxFlag.Cancel( _SHADER_ANIM );
    }

    // カラー0 継承
    if( resEmitter->inherit.color0 )
    {
        m_VtxFlag.Set( _COLOR0_INHERIT );
    }

    // カラー1 継承
    if( resEmitter->inherit.color1 )
    {
        m_VtxFlag.Set( _COLOR1_INHERIT );
    }

    // アルファ0 継承
    if( resEmitter->inherit.alpha0 )
    {
        m_VtxFlag.Set( _ALPHA0_INHERIT );
    }

    // アルファ1 継承
    if( resEmitter->inherit.alpha1 )
    {
        m_VtxFlag.Set( _ALPHA1_INHERIT );
    }

    //------------------------------------------------
    // エミッタプラグイン：現在は型を見てチートしてフラグを立てる。
    //------------------------------------------------
    if( emitterPluginIndex > 0 )
    {
        switch( emitterPluginIndex )
        {
        // MEMO: この .cpp を EffectMaker でも使用しているため、エミッタプラグインのヘッダが引けない
        //case ConnectionStripeSystem::PluginId: // 連結式ストライプ
        case EFT_EMITTER_PLUGIN_ID_1:
        {
            ResStripeConnection* data = reinterpret_cast< ResStripeConnection* >( emitterPluginData );
#ifdef NEED_EP_FLIP
            data->FlipEndian();
#endif
            switch( data->calcType )
            {
            case 0: m_VtxFlagEP.Set( _STRIPE_BILLBOARD );         break;
            case 1: m_VtxFlagEP.Set( _STRIPE_EMITTER_MATRIX );    break;
            case 2: m_VtxFlagEP.Set( _STRIPE_EMITTER_UPDOWN );    break;
            }
            if( data->emitterFollow )
            {
                m_VtxFlagEP.Set( _STRIPE_FOLLOW_EMITTER_MATRIX );
            }
#ifdef NEED_EP_FLIP
            data->FlipEndian();
#endif
        }
            break;
        //case StripeSystem::PluginId: // 履歴式ストライプ
        case EFT_EMITTER_PLUGIN_ID_2:
        {
            ResStripeHistory* data = reinterpret_cast< ResStripeHistory* >( emitterPluginData );
#ifdef NEED_EP_FLIP
            data->FlipEndian();
#endif
            switch( data->calcType )
            {
            case 0: m_VtxFlagEP.Set( _STRIPE_BILLBOARD );         break;
            case 1: m_VtxFlagEP.Set( _STRIPE_EMITTER_MATRIX );    break;
            case 2: m_VtxFlagEP.Set( _STRIPE_EMITTER_UPDOWN );    break;
            }
            if( data->emitterFollow )
            {
                m_VtxFlagEP.Set( _STRIPE_FOLLOW_EMITTER_MATRIX );
            }
#ifdef NEED_EP_FLIP
            data->FlipEndian();
#endif
        }
            break;
        //case SuperStripeSystem::PluginId: // 履歴式ストライプ2（スーパーストライプ）
        case EFT_EMITTER_PLUGIN_ID_3:
        {
            ResStripeSuper* data = reinterpret_cast< ResStripeSuper* >( emitterPluginData );
#ifdef NEED_EP_FLIP
            data->FlipEndian();
#endif
            switch( data->calcType )
            {
            case 0: m_VtxFlagEP.Set( _STRIPE_BILLBOARD );         break;
            case 1: m_VtxFlagEP.Set( _STRIPE_EMITTER_MATRIX );    break;
            case 2: m_VtxFlagEP.Set( _STRIPE_EMITTER_UPDOWN );    break;
            case 3: m_VtxFlagEP.Set( _STRIPE_RIBBON );            break;
            }
            if( data->emitterFollow )
            {
                m_VtxFlagEP.Set( _STRIPE_FOLLOW_EMITTER_MATRIX );
            }
#ifdef NEED_EP_FLIP
            data->FlipEndian();
#endif
        }
            break;
        //case AreaLoopSystem::PluginId: // 範囲内ループ
        case EFT_EMITTER_PLUGIN_ID_4:
        {

        }
            break;
        default:
            break;
        }
    }

    // UVチャンネル
    if( resEmitter->textureAnim0.uvChannel == 1 )
    {
        m_VtxFlag.Set( _TEXTURE0_USE_UV1 );
    }
    if( resEmitter->textureAnim1.uvChannel == 1 )
    {
        m_VtxFlag.Set( _TEXTURE1_USE_UV1 );
    }
    if( resEmitter->textureAnim2.uvChannel == 1 )
    {
        m_VtxFlag.Set( _TEXTURE2_USE_UV1 );
    }

#endif  // EFT_DEGRADATION_SPEC

    //------------------------------------------------
    // フラグメントシェーダ向けキー
    //------------------------------------------------

    // シェーダタイプ
    // コンバイナシェーダがセットされてない場合のみ、シェーダタイプのフラグをセットする
    if( resEmitter->shader.eftCombinerShaderIndex == ( u32 )-1 )
    {
        switch( resEmitter->combiner.shaderType )
        {
        case EFT_FRAGMENT_SHADER_TYPE_NORMAL: m_FragFlag.Set( _SHADER_TYPE_NORMAL ); break;
        case EFT_FRAGMENT_SHADER_TYPE_REFRACT:
        {
            m_FragFlag.Set( _SHADER_TYPE_REFRACT );
            if( resEmitter->combiner.applyAlpha )
            {
                m_FragFlag.Set( _REFRACT_APPLY_ALPHA );
            }
            if( resEmitter->combiner.isDistortionByCameraDistance )
            {
                m_FragFlag.Set( _REFRACT_BY_CAMERA_DISTANCE );
            }
        }
            break;
        case EFT_FRAGMENT_SHADER_TYPE_DISTORTION: m_FragFlag.Set( _SHADER_TYPE_DISTORTION ); break;
        }
    }

    // カラー計算式
    switch( resEmitter->combiner.colorCombinerProcess )
    {
    case EFT_COMBINER_COLOR_PROC_COLOR:                 m_FragFlag.Set( _COLOR_PROCESS_COLOR );  break;
    case EFT_COMBINER_COLOR_PROC_TEXTURE:               m_FragFlag.Set( _COLOR_PROCESS_TEXTURE );    break;
    case EFT_COMBINER_COLOR_PROC_TEXTURE_INTERPOLATE:   m_FragFlag.Set( _COLOR_PROCESS_TEXTURE_INTERPOLATE );    break;
    case EFT_COMBINER_COLOR_PROC_TEXTURE_ADD:           m_FragFlag.Set( _COLOR_PROCESS_TEXTURE_ADD );    break;
    }

    // アルファ計算式
    switch( resEmitter->combiner.alphaCombinerProcess )
    {
    case EFT_COMBINER_ALPHA_PROC_MOD:                   m_FragFlag.Set( _ALPHA_PROCESS_MOD );    break;
    case EFT_COMBINER_ALPHA_PROC_MOD_ALPHA0_ALPHA1:     m_FragFlag.Set( _ALPHA_PROCESS_MOD_A0_A1 );    break;
    case EFT_COMBINER_ALPHA_PROC_SUB:                   m_FragFlag.Set( _ALPHA_PROCESS_SUB );  break;
    case EFT_COMBINER_ALPHA_PROC_SUB_ALPHA0_MOD_ALPHA1: m_FragFlag.Set( _ALPHA_PROCESS_SUB_A0_MOD_A1 );  break;
    case EFT_COMBINER_ALPHA_PROC_DIST_FIELD:            m_FragFlag.Set( _ALPHA_PROCESS_DIST_FILED );  break;
    }

    // テクスチャブレンドタイプ
    switch( resEmitter->combiner.texture1ColorBlend )
    {
    case EFT_COMBINER_BLEND_TYPE_MOD:      m_FragFlag.Set( _TEXTURE1_COLOR_BLEND_MOD );  break;
    case EFT_COMBINER_BLEND_TYPE_ADD:      m_FragFlag.Set( _TEXTURE1_COLOR_BLEND_ADD );  break;
    case EFT_COMBINER_BLEND_TYPE_SUB:      m_FragFlag.Set( _TEXTURE1_COLOR_BLEND_SUB );  break;
    }
    switch( resEmitter->combiner.texture2ColorBlend )
    {
    case EFT_COMBINER_BLEND_TYPE_MOD:      m_FragFlag.Set( _TEXTURE2_COLOR_BLEND_MOD );  break;
    case EFT_COMBINER_BLEND_TYPE_ADD:      m_FragFlag.Set( _TEXTURE2_COLOR_BLEND_ADD );  break;
    case EFT_COMBINER_BLEND_TYPE_SUB:      m_FragFlag.Set( _TEXTURE2_COLOR_BLEND_SUB );  break;
    }
    switch( resEmitter->combiner.texture1AlphaBlend )
    {
    case EFT_COMBINER_BLEND_TYPE_MOD:      m_FragFlag.Set( _TEXTURE1_ALPHA_BLEND_MOD );  break;
    case EFT_COMBINER_BLEND_TYPE_ADD:      m_FragFlag.Set( _TEXTURE1_ALPHA_BLEND_ADD );  break;
    case EFT_COMBINER_BLEND_TYPE_SUB:      m_FragFlag.Set( _TEXTURE1_ALPHA_BLEND_SUB );  break;
    }
    switch( resEmitter->combiner.texture2AlphaBlend )
    {
    case EFT_COMBINER_BLEND_TYPE_MOD:      m_FragFlag.Set( _TEXTURE2_ALPHA_BLEND_MOD );  break;
    case EFT_COMBINER_BLEND_TYPE_ADD:      m_FragFlag.Set( _TEXTURE2_ALPHA_BLEND_ADD );  break;
    case EFT_COMBINER_BLEND_TYPE_SUB:      m_FragFlag.Set( _TEXTURE2_ALPHA_BLEND_SUB );  break;
    }

    // テクスチャカラー入力タイプ
    if( resEmitter->combiner.texColor0InputType == EFT_COMBINER_COLOR_INPUT_TYPE_ONE )
    {
        m_FragFlag.Set( _TEX0_COLOR_INPUT_ONE );
    }
    if( resEmitter->combiner.texColor1InputType == EFT_COMBINER_COLOR_INPUT_TYPE_ONE )
    {
        m_FragFlag.Set( _TEX1_COLOR_INPUT_ONE );
    }
    if( resEmitter->combiner.texColor2InputType == EFT_COMBINER_COLOR_INPUT_TYPE_ONE )
    {
        m_FragFlag.Set( _TEX2_COLOR_INPUT_ONE );
    }
    if( resEmitter->combiner.texColor0InputType == EFT_COMBINER_COLOR_INPUT_TYPE_ONE_MINUS_RGB )
    {
        m_FragFlag2.Set( _TEX0_COLOR_INPUT_ONE_MINUS_RGB );
    }
    if( resEmitter->combiner.texColor1InputType == EFT_COMBINER_COLOR_INPUT_TYPE_ONE_MINUS_RGB )
    {
        m_FragFlag2.Set( _TEX1_COLOR_INPUT_ONE_MINUS_RGB );
    }
    if( resEmitter->combiner.texColor2InputType == EFT_COMBINER_COLOR_INPUT_TYPE_ONE_MINUS_RGB )
    {
        m_FragFlag2.Set( _TEX2_COLOR_INPUT_ONE_MINUS_RGB );
    }
    if( resEmitter->combiner.texColor0InputType == EFT_COMBINER_COLOR_INPUT_TYPE_ALPHA )
    {
        m_FragFlag2.Set( _TEX0_COLOR_INPUT_ALPHA );
    }
    if( resEmitter->combiner.texColor1InputType == EFT_COMBINER_COLOR_INPUT_TYPE_ALPHA )
    {
        m_FragFlag2.Set( _TEX1_COLOR_INPUT_ALPHA );
    }
    if( resEmitter->combiner.texColor2InputType == EFT_COMBINER_COLOR_INPUT_TYPE_ALPHA )
    {
        m_FragFlag2.Set( _TEX2_COLOR_INPUT_ALPHA );
    }
    if( resEmitter->combiner.texColor0InputType == EFT_COMBINER_COLOR_INPUT_TYPE_ONE_MINUS_ALPHA )
    {
        m_FragFlag2.Set( _TEX0_COLOR_INPUT_ONE_MINUS_ALPHA );
    }
    if( resEmitter->combiner.texColor1InputType == EFT_COMBINER_COLOR_INPUT_TYPE_ONE_MINUS_ALPHA )
    {
        m_FragFlag2.Set( _TEX1_COLOR_INPUT_ONE_MINUS_ALPHA );
    }
    if( resEmitter->combiner.texColor2InputType == EFT_COMBINER_COLOR_INPUT_TYPE_ONE_MINUS_ALPHA )
    {
        m_FragFlag2.Set( _TEX2_COLOR_INPUT_ONE_MINUS_ALPHA );
    }

    // テクスチャアルファ入力タイプ
    if( resEmitter->combiner.texAlpha0InputType == EFT_COMBINER_ALPHA_INPUT_TYPE_ONE )
    {
        m_FragFlag.Set( _TEX0_ALPHA_INPUT_ONE );
    }
    if( resEmitter->combiner.texAlpha1InputType == EFT_COMBINER_ALPHA_INPUT_TYPE_ONE )
    {
        m_FragFlag.Set( _TEX1_ALPHA_INPUT_ONE );
    }
    if( resEmitter->combiner.texAlpha2InputType == EFT_COMBINER_ALPHA_INPUT_TYPE_ONE )
    {
        m_FragFlag.Set( _TEX2_ALPHA_INPUT_ONE );
    }
    if( resEmitter->combiner.texAlpha0InputType == EFT_COMBINER_ALPHA_INPUT_TYPE_RED )
    {
        m_FragFlag.Set( _TEX0_ALPHA_INPUT_RED );
    }
    if( resEmitter->combiner.texAlpha1InputType == EFT_COMBINER_ALPHA_INPUT_TYPE_RED )
    {
        m_FragFlag.Set( _TEX1_ALPHA_INPUT_RED );
    }
    if( resEmitter->combiner.texAlpha2InputType == EFT_COMBINER_ALPHA_INPUT_TYPE_RED )
    {
        m_FragFlag.Set( _TEX2_ALPHA_INPUT_RED );
    }
    if( resEmitter->combiner.texAlpha0InputType == EFT_COMBINER_ALPHA_INPUT_TYPE_ONE_MINUS_ALPHA )
    {
        m_FragFlag2.Set( _TEX0_ALPHA_INPUT_ONE_MINUS_ALPHA );
    }
    if( resEmitter->combiner.texAlpha1InputType == EFT_COMBINER_ALPHA_INPUT_TYPE_ONE_MINUS_ALPHA )
    {
        m_FragFlag2.Set( _TEX1_ALPHA_INPUT_ONE_MINUS_ALPHA );
    }
    if( resEmitter->combiner.texAlpha2InputType == EFT_COMBINER_ALPHA_INPUT_TYPE_ONE_MINUS_ALPHA )
    {
        m_FragFlag2.Set( _TEX2_ALPHA_INPUT_ONE_MINUS_ALPHA );
    }
    if( resEmitter->combiner.texAlpha0InputType == EFT_COMBINER_ALPHA_INPUT_TYPE_ONE_MINUS_RED )
    {
        m_FragFlag2.Set( _TEX0_ALPHA_INPUT_ONE_MINUS_RED );
    }
    if( resEmitter->combiner.texAlpha1InputType == EFT_COMBINER_ALPHA_INPUT_TYPE_ONE_MINUS_RED )
    {
        m_FragFlag2.Set( _TEX1_ALPHA_INPUT_ONE_MINUS_RED );
    }
    if( resEmitter->combiner.texAlpha2InputType == EFT_COMBINER_ALPHA_INPUT_TYPE_ONE_MINUS_RED )
    {
        m_FragFlag2.Set( _TEX2_ALPHA_INPUT_ONE_MINUS_RED );
    }

    // テクスチャカラー二乗オプション
    if( resEmitter->textureSampler0.isSquareRgb )
    {
        m_FragFlag.Set( _TEX0_COLOR_INPUT_SQUARE );
    }
    if( resEmitter->textureSampler1.isSquareRgb )
    {
        m_FragFlag.Set( _TEX1_COLOR_INPUT_SQUARE );
    }
    if( resEmitter->textureSampler2.isSquareRgb )
    {
        m_FragFlag.Set( _TEX2_COLOR_INPUT_SQUARE );
    }

    // プリミティブカラーブレンドタイプ
    switch( resEmitter->combiner.primitiveColorBlend )
    {
    case EFT_COMBINER_BLEND_TYPE_MOD:      m_FragFlag.Set( _PRIMITIVE_COLOR_BLEND_MOD ); break;
    case EFT_COMBINER_BLEND_TYPE_ADD:      m_FragFlag.Set( _PRIMITIVE_COLOR_BLEND_ADD ); break;
    case EFT_COMBINER_BLEND_TYPE_SUB:      m_FragFlag.Set( _PRIMITIVE_COLOR_BLEND_SUB ); break;
    }
    switch( resEmitter->combiner.primitiveAlphaBlend )
    {
    case EFT_COMBINER_BLEND_TYPE_MOD:      m_FragFlag.Set( _PRIMITIVE_ALPHA_BLEND_MOD ); break;
    case EFT_COMBINER_BLEND_TYPE_ADD:      m_FragFlag.Set( _PRIMITIVE_ALPHA_BLEND_ADD ); break;
    case EFT_COMBINER_BLEND_TYPE_SUB:      m_FragFlag.Set( _PRIMITIVE_ALPHA_BLEND_SUB ); break;
    }

    // プリミティブカラー入力タイプ
    if( resEmitter->combiner.primitiveColorInputType == EFT_COMBINER_COLOR_INPUT_TYPE_ONE )
    {
        m_FragFlag.Set( _PRIMITIVE_COLOR_INPUT_ONE );
    }
    if( resEmitter->combiner.primitiveColorInputType == EFT_COMBINER_COLOR_INPUT_TYPE_SRC )
    {
        m_FragFlag.Set( _PRIMITIVE_COLOR_INPUT_SRC );
    }
    if( resEmitter->combiner.primitiveColorInputType == EFT_COMBINER_COLOR_INPUT_TYPE_ONE_MINUS_RGB )
    {
        m_FragFlag2.Set( _PRIMITIVE_COLOR_INPUT_ONE_MINUS_RGB );
    }
    if( resEmitter->combiner.primitiveColorInputType == EFT_COMBINER_COLOR_INPUT_TYPE_ALPHA )
    {
        m_FragFlag2.Set( _PRIMITIVE_COLOR_INPUT_ALPHA );
    }
    if( resEmitter->combiner.primitiveColorInputType == EFT_COMBINER_COLOR_INPUT_TYPE_ONE_MINUS_ALPHA )
    {
        m_FragFlag2.Set( _PRIMITIVE_COLOR_INPUT_ONE_MINUS_ALPHA );
    }

    // プリミティブアルファ入力タイプ
    if( resEmitter->combiner.primitiveAlphaInputType == EFT_COMBINER_ALPHA_INPUT_TYPE_SRC )
    {
        m_FragFlag.Set( _PRIMITIVE_ALPHA_INPUT_A );
    }
    if( resEmitter->combiner.primitiveAlphaInputType == EFT_COMBINER_ALPHA_INPUT_TYPE_RED )
    {
        m_FragFlag.Set( _PRIMITIVE_ALPHA_INPUT_RED );
    }
    if( resEmitter->combiner.primitiveAlphaInputType == EFT_COMBINER_ALPHA_INPUT_TYPE_ONE )
    {
        m_FragFlag.Set( _PRIMITIVE_ALPHA_INPUT_ONE );
    }
    if( resEmitter->combiner.primitiveAlphaInputType == EFT_COMBINER_ALPHA_INPUT_TYPE_ONE_MINUS_ALPHA )
    {
        m_FragFlag2.Set( _PRIMITIVE_ALPHA_INPUT_ONE_MINUS_ALPHA );
    }
    if( resEmitter->combiner.primitiveAlphaInputType == EFT_COMBINER_ALPHA_INPUT_TYPE_ONE_MINUS_RED )
    {
        m_FragFlag2.Set( _PRIMITIVE_ALPHA_INPUT_ONE_MINUS_RED );
    }

    // テクスチャの有効無効
    if( resEmitter->textureSampler0.guid != EFT_INVALID_ASSET_GUID )
    {
        m_FragFlag.Set( _TEXTURE0_ENABLE );
    }
    if( resEmitter->textureSampler1.guid != EFT_INVALID_ASSET_GUID )
    {
        m_FragFlag.Set( _TEXTURE1_ENABLE );
    }
    if( resEmitter->textureSampler2.guid != EFT_INVALID_ASSET_GUID )
    {
        m_FragFlag.Set( _TEXTURE2_ENABLE );
    }

    if( resEmitter->ptclColor.isSoftParticle == 2 )
    {
        m_FragFlag.Set( _FRAGMENT_SOFT );
    }
    if( resEmitter->ptclColor.isDecal )
    {
        m_FragFlag.Set( _DECAL );
    }

    // 頂点フレネル
    if( resEmitter->ptclColor.isFresnelAlpha == 2 )
    {
        m_FragFlag.Set( _FRAGMENT_FRESNEL_ALPHA );
    }

    // アルファ比較モード
    if ( resEmitter->renderState.isAlphaTest )
    {
        switch (resEmitter->renderState.alphaFunc)
        {
        case EFT_COMPARE_NEVER:
            m_FragFlag.Set( _ALPHA_COMPARE_NEVER );
            break;
        case EFT_COMPARE_LESS:
            m_FragFlag.Set( _ALPHA_COMPARE_LESS );
            break;
        case EFT_COMPARE_EQUAL:
            m_FragFlag.Set( _ALPHA_COMPARE_EQUAL );
            break;
        case EFT_COMPARE_LEQUAL:
            m_FragFlag.Set( _ALPHA_COMPARE_LEQUAL );
            break;
        case EFT_COMPARE_GREATER:
            m_FragFlag.Set( _ALPHA_COMPARE_GREATER );
            break;
        case EFT_COMPARE_NOTEQUAL:
            m_FragFlag.Set( _ALPHA_COMPARE_NOTEQUAL );
            break;
        case EFT_COMPARE_GEQUAL:
            m_FragFlag.Set( _ALPHA_COMPARE_GEQUAL );
            break;
        case EFT_COMPARE_ALWAYS:
            m_FragFlag.Set( _ALPHA_COMPARE_ALWAYS );
            break;
        }
    }

    //------------------------------------------------
    // カスタムシェーダ向けキー
    //------------------------------------------------
    m_CustomShaderIndex = resEmitter->shader.customShaderIndex;
    m_CustomShaderFlag = resEmitter->shader.customShaderFlag;

    //------------------------------------------------
    // カスタムフィールド向けキー
    //------------------------------------------------
    if( resCustomField )
    {
#ifdef NEED_EP_FLIP
        resCustomField->FlipEndian();
#endif
        m_CustomFieldEnabled = 1;
        m_CustomFieldFlag = resCustomField->flags;
#ifdef NEED_EP_FLIP
        resCustomField->FlipEndian();
#endif
    }
    else
    {
        m_CustomFieldEnabled = 0;
        m_CustomFieldFlag = 0;
    }

    m_CustomShaderSwitch = resEmitter->shader.customShaderSwitch;
#if EFT_IS_WIN
    if( userDefine )
    {
        strcpy_s( m_CustomShaderDefine, 16, userDefine );
    }
    else
    {
        memset( m_CustomShaderDefine, 0, 16 );
    }
#else
    if( userDefine )
    {
        strcpy( m_CustomShaderDefine, userDefine );
    }
    else
    {
        memset( m_CustomShaderDefine, 0, 16 );
    }
#endif

    //------------------------------------------------
    // コンバイナシェーダ向けキー
    //------------------------------------------------
    m_CombinerShaderIndex = resEmitter->shader.eftCombinerShaderIndex;

}   // NOLINT (readability/fn_size)


//---------------------------------------------------------------------------
//  コンパイルセッティングを生成します。
//---------------------------------------------------------------------------
void ShaderKey::Generate()
{
    _ShaderKeyString keyString;
    keyString.Initialize( g_CompileSetting, EFT_SHADER_COMPILE_SETTING_BUFFER_SIZE );
    //    OutputLog( mCompileSetting );

#ifdef EFT_USE_INSTANS_DRAW
    keyString.AddKey( "#define EFT_USE_INSTANS_DRAW\n" );
#endif
#ifdef EFT_USE_REGISTER_DRAW
    keyString.AddKey( "#define EFT_USE_REGISTER_DRAW\n" );
#endif

    //----------------------------------
    // 頂点シェーダ向けキー
    //----------------------------------
    if( m_VtxFlag.Is( _CALC_TYPE_CPU ) )
    {
        keyString.AddKey( "#define _CALC_TYPE_CPU\n" );
    }
    if( m_VtxFlag.Is( _CALC_TYPE_GPU_TIME ) )
    {
        keyString.AddKey( "#define _CALC_TYPE_GPU_TIME\n" );
    }
    if( m_VtxFlag.Is( _CALC_TYPE_GPU_SO ) )
    {
        keyString.AddKey( "#define _CALC_TYPE_GPU_SO\n" );
    }
    if( m_VtxFlag.Is( _PARTICLE_TYPE_BILLBOARD ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_BILLBOARD\n" );
    }
    if( m_VtxFlag.Is( _PARTICLE_TYPE_COMPLEX_BILLBOARD ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_COMPLEX_BILLBOARD\n" );
    }
    if( m_VtxFlag.Is( _PARTICLE_TYPE_Y_BILLBOARD ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_Y_BILLBOARD\n" );
    }
    if( m_VtxFlag.Is( _PARTICLE_TYPE_POLYGON_XY ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_POLYGON_XY\n" );
    }
    if( m_VtxFlag.Is( _PARTICLE_TYPE_POLYGON_XZ ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_POLYGON_XZ\n" );
    }
    if( m_VtxFlag.Is( _PARTICLE_TYPE_VEL_LOOK ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_VEL_LOOK\n" );
    }
    if( m_VtxFlag.Is( _PARTICLE_TYPE_VEL_LOOK_POLYGON ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_VEL_LOOK_POLYGON\n" );
    }
    if( m_VtxFlag.Is( _PARTICLE_TYPE_Y_BILLBOARD_PARALLEL ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_Y_BILLBOARD_PARALLEL\n" );
    }
    if( m_VtxFlag.Is( _PARTICLE_TYPE_WITH_SCALE_Z ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_WITH_SCALE_Z\n" );
    }
    if( m_VtxFlag.Is( _PARTICLE_TYPE_WITH_SCALE_Z_DIRECTIONAL ) )
    {
        keyString.AddKey( "#define _PARTICLE_TYPE_WITH_SCALE_Z_DIRECTIONAL\n" );
    }

    if( m_VtxFlag.Is( _EMITTER_FOLLOW_TYPE_ALL ) )
    {
        keyString.AddKey( "#define _EMITTER_FOLLOW_TYPE_ALL\n" );
    }
    if( m_VtxFlag.Is( _EMITTER_FOLLOW_TYPE_NONE ) )
    {
        keyString.AddKey( "#define _EMITTER_FOLLOW_TYPE_NONE\n" );
    }
    if( m_VtxFlag.Is( _EMITTER_FOLLOW_TYPE_POS ) )
    {
        keyString.AddKey( "#define _EMITTER_FOLLOW_TYPE_POS\n" );
    }
    if( m_VtxFlag.Is( _FLUCTUATION_SCALE_ENABLE ) )
    {
        keyString.AddKey( "#define _FLUCTUATION_SCALE_ENABLE\n" );
    }
    if( m_VtxFlag.Is( _FLUCTUATION_ALPHA_ENABLE ) )
    {
        keyString.AddKey( "#define _FLUCTUATION_ALPHA_ENABLE\n" );
    }
    if( m_VtxFlag.Is( _FLUCTUATION_SCALE_Y_SEPARATE ) )
    {
        keyString.AddKey( "#define _FLUCTUATION_SCALE_Y_SEPARATE\n" );
    }
    if( m_VtxFlag.Is( _COLOR_0_FIXED ) )
    {
        keyString.AddKey( "#define _COLOR_0_FIXED\n" );
    }
    if( m_VtxFlag.Is( _COLOR_0_RANDOM ) )
    {
        keyString.AddKey( "#define _COLOR_0_RANDOM\n" );
    }
    if( m_VtxFlag.Is( _COLOR_1_FIXED ) )
    {
        keyString.AddKey( "#define _COLOR_1_FIXED\n" );
    }
    if( m_VtxFlag.Is( _COLOR_1_RANDOM ) )
    {
        keyString.AddKey( "#define _COLOR_1_RANDOM\n" );
    }
    if( m_VtxFlag.Is( _ALPHA_0_FIXED ) )
    {
        keyString.AddKey( "#define _ALPHA_0_FIXED\n" );
    }
    if( m_VtxFlag.Is( _ALPHA_1_FIXED ) )
    {
        keyString.AddKey( "#define _ALPHA_1_FIXED\n" );
    }

    if( m_VtxFlag.Is( _TEX_0_SHIFT_ANIM_ROTATE ) )
    {
        keyString.AddKey( "#define _TEX_0_SHIFT_ANIM_ROTATE\n" );
    }
    else if( m_VtxFlag.Is( _TEX_0_SHIFT_ANIM ) )
    {
        keyString.AddKey( "#define _TEX_0_SHIFT_ANIM\n" );
    }

    if( m_VtxFlag.Is( _TEX_1_SHIFT_ANIM_ROTATE ) )
    {
        keyString.AddKey( "#define _TEX_1_SHIFT_ANIM_ROTATE\n" );
    }
    else if( m_VtxFlag.Is( _TEX_1_SHIFT_ANIM ) )
    {
        keyString.AddKey( "#define _TEX_1_SHIFT_ANIM\n" );
    }

    if( m_VtxFlag.Is( _TEX_2_SHIFT_ANIM_ROTATE ) )
    {
        keyString.AddKey( "#define _TEX_2_SHIFT_ANIM_ROTATE\n" );
    }
    else if( m_VtxFlag.Is( _TEX_2_SHIFT_ANIM ) )
    {
        keyString.AddKey( "#define _TEX_2_SHIFT_ANIM\n" );
    }

    if( m_VtxFlag.Is( _TEX_0_PATTERN_ANIM ) )
    {
        keyString.AddKey( "#define _TEX_0_PATTERN_ANIM\n" );
    }
    if( m_VtxFlag.Is( _TEX_1_PATTERN_ANIM ) )
    {
        keyString.AddKey( "#define _TEX_1_PATTERN_ANIM\n" );
    }
    if( m_VtxFlag.Is( _TEX_2_PATTERN_ANIM ) )
    {
        keyString.AddKey( "#define _TEX_2_PATTERN_ANIM\n" );
    }
    if( m_VtxFlag.Is( _TEX_0_SPHERE_MAP ) )
    {
        keyString.AddKey( "#define _TEX_0_SPHERE_MAP\n" );
    }
    if( m_VtxFlag.Is( _TEX_1_SPHERE_MAP ) )
    {
        keyString.AddKey( "#define _TEX_1_SPHERE_MAP\n" );
    }
    if( m_VtxFlag.Is( _TEX_2_SPHERE_MAP ) )
    {
        keyString.AddKey( "#define _TEX_2_SPHERE_MAP\n" );
    }

    if( m_VtxFlag.Is( _USE_ROTATE ) )
    {
        keyString.AddKey( "#define _USE_ROTATE\n" );
    }
    if( m_VtxFlag.Is( _ROTATE_YZX ) )
    {
        keyString.AddKey( "#define _ROTATE_YZX\n" );
    }
    if( m_VtxFlag.Is( _ROTATE_XYZ ) )
    {
        keyString.AddKey( "#define _ROTATE_XYZ\n" );
    }
    if( m_VtxFlag.Is( _ROTATE_ZXY ) )
    {
        keyString.AddKey( "#define _ROTATE_ZXY\n" );
    }
    if( m_VtxFlag.Is( _NEAR_DIST_ALPHA ) )
    {
        keyString.AddKey( "#define _NEAR_DIST_ALPHA\n" );
    }
    if( m_VtxFlag.Is( _FAR_DIST_ALPHA ) )
    {
        keyString.AddKey( "#define _FAR_DIST_ALPHA\n" );
    }
    if( m_VtxFlag.Is( _MASKING_CHECK ) )
    {
        keyString.AddKey( "#define _MASKING_CHECK\n" );
    }
    if( m_VtxFlag.Is( _VERTEX_SOFT ) )
    {
        keyString.AddKey( "#define _VERTEX_SOFT\n" );
    }
    if( m_VtxFlag.Is( _VERTEX_FRESNEL_ALPHA ) )
    {
        keyString.AddKey( "#define _VERTEX_FRESNEL_ALPHA\n" );
    }
    if( m_VtxFlag.Is( _PARTICLE ) )
    {
        keyString.AddKey( "#define _PARTICLE\n" );
    }
    if( m_VtxFlag.Is( _PRIMITIVE ) )
    {
        keyString.AddKey( "#define _PRIMITIVE\n" );
    }
    if( m_VtxFlag2.Is( _LINK_FOVY_TO_SCALE_LIMIT ) )
    {
        keyString.AddKey( "#define _LINK_FOVY_TO_SCALE_LIMIT\n" );
    }
    if( m_VtxFlag.Is( _PARTICLE_SCALE_LIMIT_NEAR ) )
    {
        keyString.AddKey( "#define _PARTICLE_SCALE_LIMIT_NEAR\n" );
    }
    if( m_VtxFlag.Is( _PARTICLE_SCALE_LIMIT_FAR ) )
    {
        keyString.AddKey( "#define _PARTICLE_SCALE_LIMIT_FAR\n" );
    }
    if( m_VtxFlag2.Is( _AVOID_Z_FIGHTING ) )
    {
        keyString.AddKey( "#define _AVOID_Z_FIGHTING\n" );
    }
    if( m_VtxFlag.Is( _SHADER_ANIM ) )
    {
        keyString.AddKey( "#define _SHADER_ANIM\n" );
    }

    if( m_VtxFlag.Is( _COLOR0_INHERIT ) )
    {
        keyString.AddKey( "#define _COLOR0_INHERIT\n" );
    }
    if( m_VtxFlag.Is( _COLOR1_INHERIT ) )
    {
        keyString.AddKey( "#define _COLOR1_INHERIT\n" );
    }
    if( m_VtxFlag.Is( _ALPHA0_INHERIT ) )
    {
        keyString.AddKey( "#define _ALPHA0_INHERIT\n" );
    }
    if( m_VtxFlag.Is( _ALPHA1_INHERIT ) )
    {
        keyString.AddKey( "#define _ALPHA1_INHERIT\n" );
    }

    if( m_VtxFlag2.Is( _CONNECT_PTCL_SCALE_TO_Z_OFFSET ) )
    {
        keyString.AddKey( "#define _CONNECT_PTCL_SCALE_TO_Z_OFFSET\n" );
    }

    if( m_VtxFlag.Is( _VELOCITY_SCALE_Y ) )
    {
        keyString.AddKey( "#define _VELOCITY_SCALE_Y\n" );
    }
    if( m_VtxFlag.Is( _DEPTH_OFFSET ) )
    {
        keyString.AddKey( "#define _DEPTH_OFFSET\n" );
    }
    if( m_VtxFlag.Is( _CAMERA_OFFSET ) )
    {
        keyString.AddKey( "#define _CAMERA_OFFSET\n" );
    }
    if( m_VtxFlag2.Is( _CAMERA_OFFSET_FIXED_SIZE ) )
    {
        keyString.AddKey( "#define _CAMERA_OFFSET_FIXED_SIZE\n" );
    }
    if( m_VtxFlag.Is( _WORLD_GRAVITY ) )
    {
        keyString.AddKey( "#define _WORLD_GRAVITY\n" );
    }

    if( m_VtxFlag2.Is( _FIELD_RANDOM ) )
    {
        keyString.AddKey( "#define _FIELD_RANDOM\n" );
    }

    if( m_VtxFlag2.Is( _TEX0_KEEP_TEXTURE_SIZE_U ) )
    {
        keyString.AddKey( "#define _TEX0_KEEP_TEXTURE_SIZE_U\n" );
    }
    if( m_VtxFlag2.Is( _TEX1_KEEP_TEXTURE_SIZE_U ) )
    {
        keyString.AddKey( "#define _TEX1_KEEP_TEXTURE_SIZE_U\n" );
    }
    if( m_VtxFlag2.Is( _TEX2_KEEP_TEXTURE_SIZE_U ) )
    {
        keyString.AddKey( "#define _TEX2_KEEP_TEXTURE_SIZE_U\n" );
    }

    if( m_VtxFlag2.Is( _TEX0_KEEP_TEXTURE_SIZE_V ) )
    {
        keyString.AddKey( "#define _TEX0_KEEP_TEXTURE_SIZE_V\n" );
    }
    if( m_VtxFlag2.Is( _TEX1_KEEP_TEXTURE_SIZE_V ) )
    {
        keyString.AddKey( "#define _TEX1_KEEP_TEXTURE_SIZE_V\n" );
    }
    if( m_VtxFlag2.Is( _TEX2_KEEP_TEXTURE_SIZE_V ) )
    {
        keyString.AddKey( "#define _TEX2_KEEP_TEXTURE_SIZE_V\n" );
    }

    // ストライプタイプ
    if( m_VtxFlagEP.Is( _STRIPE_BILLBOARD ) )
    {
        keyString.AddKey( "#define _STRIPE_BILLBOARD\n" );
    }
    if( m_VtxFlagEP.Is( _STRIPE_EMITTER_MATRIX ) )
    {
        keyString.AddKey( "#define _STRIPE_EMITTER_MATRIX\n" );
    }
    if( m_VtxFlagEP.Is( _STRIPE_EMITTER_UPDOWN ) )
    {
        keyString.AddKey( "#define _STRIPE_EMITTER_UPDOWN\n" );
    }
    if( m_VtxFlagEP.Is( _STRIPE_FOLLOW_EMITTER_MATRIX ) )
    {
        keyString.AddKey( "#define _STRIPE_FOLLOW_EMITTER_MATRIX\n" );
    }
    if( m_VtxFlagEP.Is( _STRIPE_RIBBON ) )
    {
        keyString.AddKey( "#define _STRIPE_RIBBON\n" );
    }

    //----------------------------------
    // フラグメントシェーダ向けキー
    //----------------------------------
    if( m_FragFlag.Is( _SHADER_TYPE_NORMAL ) )
    {
        keyString.AddKey( "#define _SHADER_TYPE_NORMAL\n" );
    }
    if( m_FragFlag.Is( _SHADER_TYPE_REFRACT ) )
    {
        keyString.AddKey( "#define _SHADER_TYPE_REFRACT\n" );
    }
    if( m_FragFlag.Is( _SHADER_TYPE_DISTORTION ) )
    {
        keyString.AddKey( "#define _SHADER_TYPE_DISTORTION\n" );
    }
    if( m_FragFlag.Is( _COLOR_PROCESS_COLOR ) )
    {
        keyString.AddKey( "#define _COLOR_PROCESS_COLOR\n" );
    }
    if( m_FragFlag.Is( _COLOR_PROCESS_TEXTURE ) )
    {
        keyString.AddKey( "#define _COLOR_PROCESS_TEXTURE\n" );
    }
    if( m_FragFlag.Is( _COLOR_PROCESS_TEXTURE_INTERPOLATE ) )
    {
        keyString.AddKey( "#define _COLOR_PROCESS_TEXTURE_INTERPOLATE\n" );
    }
    if( m_FragFlag.Is( _COLOR_PROCESS_TEXTURE_ADD ) )
    {
        keyString.AddKey( "#define _COLOR_PROCESS_TEXTURE_ADD\n" );
    }
    if( m_FragFlag.Is( _ALPHA_PROCESS_MOD ) )
    {
        keyString.AddKey( "#define _ALPHA_PROCESS_MOD\n" );
    }
    if( m_FragFlag.Is( _ALPHA_PROCESS_SUB ) )
    {
        keyString.AddKey( "#define _ALPHA_PROCESS_SUB\n" );
    }
    if( m_FragFlag.Is( _ALPHA_PROCESS_MOD_A0_A1 ) )
    {
        keyString.AddKey( "#define _ALPHA_PROCESS_MOD_A0_A1\n" );
    }
    if( m_FragFlag.Is( _ALPHA_PROCESS_SUB_A0_MOD_A1 ) )
    {
        keyString.AddKey( "#define _ALPHA_PROCESS_SUB_A0_MOD_A1\n" );
    }
    if( m_FragFlag.Is( _ALPHA_PROCESS_DIST_FILED ) )
    {
        keyString.AddKey( "#define _ALPHA_PROCESS_DIST_FILED\n" );
    }
    if( m_FragFlag.Is( _TEX0_COLOR_INPUT_ONE ) )
    {
        keyString.AddKey( "#define _TEX0_COLOR_INPUT_ONE\n" );
    }
    if( m_FragFlag.Is( _TEX1_COLOR_INPUT_ONE ) )
    {
        keyString.AddKey( "#define _TEX1_COLOR_INPUT_ONE\n" );
    }
    if( m_FragFlag.Is( _TEX2_COLOR_INPUT_ONE ) )
    {
        keyString.AddKey( "#define _TEX2_COLOR_INPUT_ONE\n" );
    }
    if( m_FragFlag.Is( _TEX0_ALPHA_INPUT_ONE ) )
    {
        keyString.AddKey( "#define _TEX0_ALPHA_INPUT_ONE\n" );
    }
    if( m_FragFlag.Is( _TEX1_ALPHA_INPUT_ONE ) )
    {
        keyString.AddKey( "#define _TEX1_ALPHA_INPUT_ONE\n" );
    }
    if( m_FragFlag.Is( _TEX2_ALPHA_INPUT_ONE ) )
    {
        keyString.AddKey( "#define _TEX2_ALPHA_INPUT_ONE\n" );
    }
    if( m_FragFlag.Is( _TEX0_ALPHA_INPUT_RED ) )
    {
        keyString.AddKey( "#define _TEX0_ALPHA_INPUT_RED\n" );
    }
    if( m_FragFlag.Is( _TEX1_ALPHA_INPUT_RED ) )
    {
        keyString.AddKey( "#define _TEX1_ALPHA_INPUT_RED\n" );
    }
    if( m_FragFlag.Is( _TEX2_ALPHA_INPUT_RED ) )
    {
        keyString.AddKey( "#define _TEX2_ALPHA_INPUT_RED\n" );
    }
    if( m_FragFlag.Is( _TEXTURE0_ENABLE ) )
    {
        keyString.AddKey( "#define _TEXTURE0_ENABLE\n" );
    }
    if( m_FragFlag.Is( _TEXTURE1_ENABLE ) )
    {
        keyString.AddKey( "#define _TEXTURE1_ENABLE\n" );
    }
    if( m_FragFlag.Is( _TEXTURE2_ENABLE ) )
    {
        keyString.AddKey( "#define _TEXTURE2_ENABLE\n" );
    }
    if( m_FragFlag.Is( _TEXTURE1_COLOR_BLEND_MOD ) )
    {
        keyString.AddKey( "#define _TEXTURE1_COLOR_BLEND_MOD\n" );
    }
    if( m_FragFlag.Is( _TEXTURE1_COLOR_BLEND_ADD ) )
    {
        keyString.AddKey( "#define _TEXTURE1_COLOR_BLEND_ADD\n" );
    }
    if( m_FragFlag.Is( _TEXTURE1_COLOR_BLEND_SUB ) )
    {
        keyString.AddKey( "#define _TEXTURE1_COLOR_BLEND_SUB\n" );
    }
    if( m_FragFlag.Is( _TEXTURE2_COLOR_BLEND_MOD ) )
    {
        keyString.AddKey( "#define _TEXTURE2_COLOR_BLEND_MOD\n" );
    }
    if( m_FragFlag.Is( _TEXTURE2_COLOR_BLEND_ADD ) )
    {
        keyString.AddKey( "#define _TEXTURE2_COLOR_BLEND_ADD\n" );
    }
    if( m_FragFlag.Is( _TEXTURE2_COLOR_BLEND_SUB ) )
    {
        keyString.AddKey( "#define _TEXTURE2_COLOR_BLEND_SUB\n" );
    }
    if( m_FragFlag.Is( _TEXTURE1_ALPHA_BLEND_MOD ) )
    {
        keyString.AddKey( "#define _TEXTURE1_ALPHA_BLEND_MOD\n" );
    }
    if( m_FragFlag.Is( _TEXTURE1_ALPHA_BLEND_ADD ) )
    {
        keyString.AddKey( "#define _TEXTURE1_ALPHA_BLEND_ADD\n" );
    }
    if( m_FragFlag.Is( _TEXTURE1_ALPHA_BLEND_SUB ) )
    {
        keyString.AddKey( "#define _TEXTURE1_ALPHA_BLEND_SUB\n" );
    }
    if( m_FragFlag.Is( _TEXTURE2_ALPHA_BLEND_MOD ) )
    {
        keyString.AddKey( "#define _TEXTURE2_ALPHA_BLEND_MOD\n" );
    }
    if( m_FragFlag.Is( _TEXTURE2_ALPHA_BLEND_ADD ) )
    {
        keyString.AddKey( "#define _TEXTURE2_ALPHA_BLEND_ADD\n" );
    }
    if( m_FragFlag.Is( _TEXTURE2_ALPHA_BLEND_SUB ) )
    {
        keyString.AddKey( "#define _TEXTURE2_ALPHA_BLEND_SUB\n" );
    }
    if( m_FragFlag.Is( _PRIMITIVE_COLOR_BLEND_MOD ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_COLOR_BLEND_MOD\n" );
    }
    if( m_FragFlag.Is( _PRIMITIVE_COLOR_BLEND_ADD ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_COLOR_BLEND_ADD\n" );
    }
    if( m_FragFlag.Is( _PRIMITIVE_COLOR_BLEND_SUB ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_COLOR_BLEND_SUB\n" );
    }
    if( m_FragFlag.Is( _PRIMITIVE_ALPHA_BLEND_MOD ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_ALPHA_BLEND_MOD\n" );
    }
    if( m_FragFlag.Is( _PRIMITIVE_ALPHA_BLEND_ADD ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_ALPHA_BLEND_ADD\n" );
    }
    if( m_FragFlag.Is( _PRIMITIVE_ALPHA_BLEND_SUB ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_ALPHA_BLEND_SUB\n" );
    }
    if( m_FragFlag.Is( _PRIMITIVE_COLOR_INPUT_ONE ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_COLOR_INPUT_ONE\n" );
    }
    if( m_FragFlag.Is( _PRIMITIVE_COLOR_INPUT_SRC ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_COLOR_INPUT_SRC\n" );
    }
    if( m_FragFlag.Is( _PRIMITIVE_ALPHA_INPUT_A ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_ALPHA_INPUT_A\n" );
    }
    if( m_FragFlag.Is( _PRIMITIVE_ALPHA_INPUT_RED ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_ALPHA_INPUT_RED\n" );
    }
    if( m_FragFlag.Is( _PRIMITIVE_ALPHA_INPUT_ONE ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_ALPHA_INPUT_ONE\n" );
    }
    if( m_FragFlag.Is( _FRAGMENT_SOFT ) )
    {
        keyString.AddKey( "#define _FRAGMENT_SOFT\n" );
    }
    if( m_FragFlag.Is( _DECAL ) )
    {
        keyString.AddKey( "#define _DECAL\n" );
    }
    if( m_FragFlag.Is( _FRAGMENT_FRESNEL_ALPHA ) )
    {
        keyString.AddKey( "#define _FRAGMENT_FRESNEL_ALPHA\n" );
    }
    if( m_FragFlag.Is( _REFRACT_APPLY_ALPHA ) )
    {
        keyString.AddKey( "#define _REFRACT_APPLY_ALPHA\n" );
    }
    if( m_FragFlag.Is( _ALPHA_COMPARE_NEVER ) )
    {
        keyString.AddKey( "#define _ALPHA_COMPARE_NEVER\n" );
    }
    if( m_FragFlag.Is( _ALPHA_COMPARE_LESS ) )
    {
        keyString.AddKey( "#define _ALPHA_COMPARE_LESS\n" );
    }
    if( m_FragFlag.Is( _ALPHA_COMPARE_EQUAL ) )
    {
        keyString.AddKey( "#define _ALPHA_COMPARE_EQUAL\n" );
    }
    if( m_FragFlag.Is( _ALPHA_COMPARE_LEQUAL ) )
    {
        keyString.AddKey( "#define _ALPHA_COMPARE_LEQUAL\n" );
    }
    if( m_FragFlag.Is( _ALPHA_COMPARE_GREATER ) )
    {
        keyString.AddKey( "#define _ALPHA_COMPARE_GREATER\n" );
    }
    if( m_FragFlag.Is( _ALPHA_COMPARE_NOTEQUAL ) )
    {
        keyString.AddKey( "#define _ALPHA_COMPARE_NOTEQUAL\n" );
    }
    if( m_FragFlag.Is( _ALPHA_COMPARE_GEQUAL ) )
    {
        keyString.AddKey( "#define _ALPHA_COMPARE_GEQUAL\n" );
    }
    if( m_FragFlag.Is( _ALPHA_COMPARE_ALWAYS ) )
    {
        keyString.AddKey( "#define _ALPHA_COMPARE_ALWAYS\n" );
    }
    if( m_FragFlag.Is( _TEX0_COLOR_INPUT_SQUARE ) )
    {
        keyString.AddKey( "#define _TEX0_COLOR_INPUT_SQUARE\n" );
    }
    if( m_FragFlag.Is( _TEX1_COLOR_INPUT_SQUARE ) )
    {
        keyString.AddKey( "#define _TEX1_COLOR_INPUT_SQUARE\n" );
    }
    if( m_FragFlag.Is( _TEX2_COLOR_INPUT_SQUARE ) )
    {
        keyString.AddKey( "#define _TEX2_COLOR_INPUT_SQUARE\n" );
    }

    if( m_FragFlag.Is( _REFRACT_BY_CAMERA_DISTANCE ) )
    {
        keyString.AddKey( "#define _REFRACT_BY_CAMERA_DISTANCE\n" );
    }

    if( m_FragFlag2.Is( _TEX0_COLOR_INPUT_ONE_MINUS_RGB ) )
    {
        keyString.AddKey( "#define _TEX0_COLOR_INPUT_ONE_MINUS_RGB\n" );
    }
    if( m_FragFlag2.Is( _TEX1_COLOR_INPUT_ONE_MINUS_RGB ) )
    {
        keyString.AddKey( "#define _TEX1_COLOR_INPUT_ONE_MINUS_RGB\n" );
    }
    if( m_FragFlag2.Is( _TEX2_COLOR_INPUT_ONE_MINUS_RGB ) )
    {
        keyString.AddKey( "#define _TEX2_COLOR_INPUT_ONE_MINUS_RGB\n" );
    }

    if( m_FragFlag2.Is( _TEX0_COLOR_INPUT_ALPHA ) )
    {
        keyString.AddKey( "#define _TEX0_COLOR_INPUT_ALPHA\n" );
    }
    if( m_FragFlag2.Is( _TEX1_COLOR_INPUT_ALPHA ) )
    {
        keyString.AddKey( "#define _TEX1_COLOR_INPUT_ALPHA\n" );
    }
    if( m_FragFlag2.Is( _TEX2_COLOR_INPUT_ALPHA ) )
    {
        keyString.AddKey( "#define _TEX2_COLOR_INPUT_ALPHA\n" );
    }

    if( m_FragFlag2.Is( _TEX0_COLOR_INPUT_ONE_MINUS_ALPHA ) )
    {
        keyString.AddKey( "#define _TEX0_COLOR_INPUT_ONE_MINUS_ALPHA\n" );
    }
    if( m_FragFlag2.Is( _TEX1_COLOR_INPUT_ONE_MINUS_ALPHA ) )
    {
        keyString.AddKey( "#define _TEX1_COLOR_INPUT_ONE_MINUS_ALPHA\n" );
    }
    if( m_FragFlag2.Is( _TEX2_COLOR_INPUT_ONE_MINUS_ALPHA ) )
    {
        keyString.AddKey( "#define _TEX2_COLOR_INPUT_ONE_MINUS_ALPHA\n" );
    }

    if( m_FragFlag2.Is( _TEX0_ALPHA_INPUT_ONE_MINUS_RED ) )
    {
        keyString.AddKey( "#define _TEX0_ALPHA_INPUT_ONE_MINUS_RED\n" );
    }
    if( m_FragFlag2.Is( _TEX1_ALPHA_INPUT_ONE_MINUS_RED ) )
    {
        keyString.AddKey( "#define _TEX1_ALPHA_INPUT_ONE_MINUS_RED\n" );
    }
    if( m_FragFlag2.Is( _TEX2_ALPHA_INPUT_ONE_MINUS_RED ) )
    {
        keyString.AddKey( "#define _TEX2_ALPHA_INPUT_ONE_MINUS_RED\n" );
    }

    if( m_FragFlag2.Is( _TEX0_ALPHA_INPUT_ONE_MINUS_ALPHA ) )
    {
        keyString.AddKey( "#define _TEX0_ALPHA_INPUT_ONE_MINUS_ALPHA\n" );
    }
    if( m_FragFlag2.Is( _TEX1_ALPHA_INPUT_ONE_MINUS_ALPHA ) )
    {
        keyString.AddKey( "#define _TEX1_ALPHA_INPUT_ONE_MINUS_ALPHA\n" );
    }
    if( m_FragFlag2.Is( _TEX2_ALPHA_INPUT_ONE_MINUS_ALPHA ) )
    {
        keyString.AddKey( "#define _TEX2_ALPHA_INPUT_ONE_MINUS_ALPHA\n" );
    }

    if( m_FragFlag2.Is( _PRIMITIVE_COLOR_INPUT_ONE_MINUS_RGB ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_COLOR_INPUT_ONE_MINUS_RGB\n" );
    }
    if( m_FragFlag2.Is( _PRIMITIVE_COLOR_INPUT_ALPHA ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_COLOR_INPUT_ALPHA\n" );
    }
    if( m_FragFlag2.Is( _PRIMITIVE_COLOR_INPUT_ONE_MINUS_ALPHA ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_COLOR_INPUT_ONE_MINUS_ALPHA\n" );
    }
    if( m_FragFlag2.Is( _PRIMITIVE_ALPHA_INPUT_ONE_MINUS_RED ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_ALPHA_INPUT_ONE_MINUS_RED\n" );
    }
    if( m_FragFlag2.Is( _PRIMITIVE_ALPHA_INPUT_ONE_MINUS_ALPHA ) )
    {
        keyString.AddKey( "#define _PRIMITIVE_ALPHA_INPUT_ONE_MINUS_ALPHA\n" );
    }


    //----------------------------------
    // 8キーアニメ
    //----------------------------------
    if( m_VtxScaleAnimKey == 1 )
    {
        keyString.AddKey( "#define _SCALE_ANIM_1_KEY\n" );
    }
    else if( m_VtxScaleAnimKey == 2 )
    {
        keyString.AddKey( "#define _SCALE_ANIM_2_KEY\n" );
    }
    else if( m_VtxScaleAnimKey == 3 )
    {
        keyString.AddKey( "#define _SCALE_ANIM_3_KEY\n" );
    }
    else if( m_VtxScaleAnimKey == 4 )
    {
        keyString.AddKey( "#define _SCALE_ANIM_4_KEY\n" );
    }
    else if( m_VtxScaleAnimKey == 5 )
    {
        keyString.AddKey( "#define _SCALE_ANIM_5_KEY\n" );
    }
    else if( m_VtxScaleAnimKey == 6 )
    {
        keyString.AddKey( "#define _SCALE_ANIM_6_KEY\n" );
    }
    else if( m_VtxScaleAnimKey == 7 )
    {
        keyString.AddKey( "#define _SCALE_ANIM_7_KEY\n" );
    }
    else if( m_VtxScaleAnimKey == 8 )
    {
        keyString.AddKey( "#define _SCALE_ANIM_8_KEY\n" );
    }

    if( m_VtxColor0AnimKey == 1 )
    {
        keyString.AddKey( "#define _COLOR_0_ANIM_1_KEY\n" );
    }
    else if( m_VtxColor0AnimKey == 2 )
    {
        keyString.AddKey( "#define _COLOR_0_ANIM_2_KEY\n" );
    }
    else if( m_VtxColor0AnimKey == 3 )
    {
        keyString.AddKey( "#define _COLOR_0_ANIM_3_KEY\n" );
    }
    else if( m_VtxColor0AnimKey == 4 )
    {
        keyString.AddKey( "#define _COLOR_0_ANIM_4_KEY\n" );
    }
    else if( m_VtxColor0AnimKey == 5 )
    {
        keyString.AddKey( "#define _COLOR_0_ANIM_5_KEY\n" );
    }
    else if( m_VtxColor0AnimKey == 6 )
    {
        keyString.AddKey( "#define _COLOR_0_ANIM_6_KEY\n" );
    }
    else if( m_VtxColor0AnimKey == 7 )
    {
        keyString.AddKey( "#define _COLOR_0_ANIM_7_KEY\n" );
    }
    else if( m_VtxColor0AnimKey == 8 )
    {
        keyString.AddKey( "#define _COLOR_0_ANIM_8_KEY\n" );
    }

    if( m_VtxColor1AnimKey == 1 )
    {
        keyString.AddKey( "#define _COLOR_1_ANIM_1_KEY\n" );
    }
    else if( m_VtxColor1AnimKey == 2 )
    {
        keyString.AddKey( "#define _COLOR_1_ANIM_2_KEY\n" );
    }
    else if( m_VtxColor1AnimKey == 3 )
    {
        keyString.AddKey( "#define _COLOR_1_ANIM_3_KEY\n" );
    }
    else if( m_VtxColor1AnimKey == 4 )
    {
        keyString.AddKey( "#define _COLOR_1_ANIM_4_KEY\n" );
    }
    else if( m_VtxColor1AnimKey == 5 )
    {
        keyString.AddKey( "#define _COLOR_1_ANIM_5_KEY\n" );
    }
    else if( m_VtxColor1AnimKey == 6 )
    {
        keyString.AddKey( "#define _COLOR_1_ANIM_6_KEY\n" );
    }
    else if( m_VtxColor1AnimKey == 7 )
    {
        keyString.AddKey( "#define _COLOR_1_ANIM_7_KEY\n" );
    }
    else if( m_VtxColor1AnimKey == 8 )
    {
        keyString.AddKey( "#define _COLOR_1_ANIM_8_KEY\n" );
    }

    if( m_VtxAlpha0AnimKey == 1 )
    {
        keyString.AddKey( "#define _ALPHA_0_ANIM_1_KEY\n" );
    }
    else if( m_VtxAlpha0AnimKey == 2 )
    {
        keyString.AddKey( "#define _ALPHA_0_ANIM_2_KEY\n" );
    }
    else if( m_VtxAlpha0AnimKey == 3 )
    {
        keyString.AddKey( "#define _ALPHA_0_ANIM_3_KEY\n" );
    }
    else if( m_VtxAlpha0AnimKey == 4 )
    {
        keyString.AddKey( "#define _ALPHA_0_ANIM_4_KEY\n" );
    }
    else if( m_VtxAlpha0AnimKey == 5 )
    {
        keyString.AddKey( "#define _ALPHA_0_ANIM_5_KEY\n" );
    }
    else if( m_VtxAlpha0AnimKey == 6 )
    {
        keyString.AddKey( "#define _ALPHA_0_ANIM_6_KEY\n" );
    }
    else if( m_VtxAlpha0AnimKey == 7 )
    {
        keyString.AddKey( "#define _ALPHA_0_ANIM_7_KEY\n" );
    }
    else if( m_VtxAlpha0AnimKey == 8 )
    {
        keyString.AddKey( "#define _ALPHA_0_ANIM_8_KEY\n" );
    }

    if( m_VtxAlpha1AnimKey == 1 )
    {
        keyString.AddKey( "#define _ALPHA_1_ANIM_1_KEY\n" );
    }
    else if( m_VtxAlpha1AnimKey == 2 )
    {
        keyString.AddKey( "#define _ALPHA_1_ANIM_2_KEY\n" );
    }
    else if( m_VtxAlpha1AnimKey == 3 )
    {
        keyString.AddKey( "#define _ALPHA_1_ANIM_3_KEY\n" );
    }
    else if( m_VtxAlpha1AnimKey == 4 )
    {
        keyString.AddKey( "#define _ALPHA_1_ANIM_4_KEY\n" );
    }
    else if( m_VtxAlpha1AnimKey == 5 )
    {
        keyString.AddKey( "#define _ALPHA_1_ANIM_5_KEY\n" );
    }
    else if( m_VtxAlpha1AnimKey == 6 )
    {
        keyString.AddKey( "#define _ALPHA_1_ANIM_6_KEY\n" );
    }
    else if( m_VtxAlpha1AnimKey == 7 )
    {
        keyString.AddKey( "#define _ALPHA_1_ANIM_7_KEY\n" );
    }
    else if( m_VtxAlpha1AnimKey == 8 )
    {
        keyString.AddKey( "#define _ALPHA_1_ANIM_8_KEY\n" );
    }

    //----------------------------------
    // UVチャンネル
    //----------------------------------
    if( m_VtxFlag.Is( _TEXTURE0_USE_UV1 ) )
    {
        keyString.AddKey( "#define _TEXTURE0_USE_UV1\n" );
    }
    if( m_VtxFlag.Is( _TEXTURE1_USE_UV1 ) )
    {
        keyString.AddKey( "#define _TEXTURE1_USE_UV1\n" );
    }
    if( m_VtxFlag.Is( _TEXTURE2_USE_UV1 ) )
    {
        keyString.AddKey( "#define _TEXTURE2_USE_UV1\n" );
    }


    //----------------------------------
    // ユーザー向けに定義する文字列
    //----------------------------------

    // 描画パス
    if ( m_DrawPath != -1 )
    {
        keyString.AddKey( "#define DRAW_PATH_%d\n", m_DrawPath );

        // 互換性保持
        keyString.AddKey( "#define DRAW_PATH (%d)\n", m_DrawPath );
    }
    else
    {
        keyString.AddKey( "#define DRAW_PATH_NONE\n" );
    }

    // カスタムシェーダインデックス
    if( m_CustomShaderIndex != 0 )
    {
        keyString.AddKey( "#define CUSTOM_SHADER_INDEX_%d\n", m_CustomShaderIndex );
        // 互換性保持
        keyString.AddKey( "#define USR_SETTING_%d\n", m_CustomShaderIndex );
    }
    else
    {
        keyString.AddKey( "#define CUSTOM_SHADER_NONE\n" );
        // 互換性保持
        keyString.AddKey( "#define USR_SETTING_NONE\n" );
    }

    // カスタムシェーダ チェックボックス
    if( m_CustomShaderFlag != 0 )
    {
        for( u64 i = 0; i < 64; ++i )
        {
            if( m_CustomShaderFlag & ( static_cast<u64>(0x1) << i ) )
            {
                keyString.AddKey( "#define CUSTOM_SHADER_FLAG_%d\n", i );
                // 互換性保持
                keyString.AddKey( "#define USR_FLAG_%d\n", i );
            }
        }
    }
    else
    {
        keyString.AddKey( "#define CUSTOM_SHADER_FLAG_NONE\n" );
    }

    // カスタムシェーダ スイッチフラグ
    if( m_CustomShaderSwitch != 0 )
    {
        for( u64 i = 0; i < 64; ++i )
        {
            if( m_CustomShaderSwitch & ( static_cast<u64>(0x1) << i ) )
            {
                keyString.AddKey( "#define CUSTOM_SHADER_SWITCH_FLAG_%d\n", i );
                keyString.AddKey( "#define USR_SWITCH_FLAG_%d\n", i );
            }
        }
    }
    else
    {
        keyString.AddKey( "#define CUSTOM_SHADER_SWITCH_FLAG_NONE\n" );
    }

    // カスタムフィールド向けキー
    if( m_CustomFieldEnabled != 0 )
    {
        keyString.AddKey( "#define CUSTOM_FIELD_ENABLED\n" );
        for( u32 i = 0; i < 16; ++i )
        {
            if( m_CustomFieldFlag & ( 0x1 << i ) )
            {
                keyString.AddKey( "#define CUSTOM_FIELD_FLAG_%d\n", i );
            }
        }
    }
    else
    {
        keyString.AddKey( "#define CUSTOM_FIELD_NONE\n" );
    }

    // ユーザー定義
    if( strlen( m_CustomShaderDefine ) > 0 )
    {
        char temp[ 32 ];
#if EFT_IS_WIN
        sprintf_s( temp, 32, "#define %s\n", m_CustomShaderDefine );
#else
        sprintf( temp, "#define %s\n", m_CustomShaderDefine );
#endif
        keyString.AddKey( temp );
    }
    keyString.Eof();

}   // NOLINT (readability/fn_size)


//---------------------------------------------------------------------------
//  描画パスを設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetDrawPath( u32 drawPath )
{
    m_DrawPath = drawPath;
}


//---------------------------------------------------------------------------
//  エミッタ挙動タイプを設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetEmitterCalcType( EmitterCalcType type )
{
    if( type == EFT_EMITTER_CALC_TYPE_CPU )
    {
        m_VtxFlag.Set( _CALC_TYPE_CPU );
    }
    if( type == EFT_EMITTER_CALC_TYPE_GPU )
    {
        m_VtxFlag.Set( _CALC_TYPE_GPU_TIME );
    }
    if( type == EFT_EMITTER_CALC_TYPE_GPU_SO )
    {
        m_VtxFlag.Set( _CALC_TYPE_GPU_SO );
    }
}

//---------------------------------------------------------------------------
//  形状タイプを設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetFigureType( ParticleFigureType type )
{
    switch( type )
    {
    case EFT_PARTCILE_FIGURE_TYPE_PARTICLE:  m_VtxFlag.Set( _PARTICLE );  break;
    case EFT_PARTICLE_FIGURE_TYPE_PRIMITIVE: m_VtxFlag.Set( _PRIMITIVE ); break;
    }
}

//---------------------------------------------------------------------------
//! @brief  世界座標系で重力を適用 を設定します。
//---------------------------------------------------------------------------
void  ShaderKey::SetParticleWorldGravity( bool worldGravity )
{
    if( worldGravity )
    {
        m_VtxFlag.Set( _WORLD_GRAVITY );
    }
    else
    {
        m_VtxFlag.Cancel( _WORLD_GRAVITY );
    }
}

//---------------------------------------------------------------------------
//  パーティクルタイプを設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetParticleType( ParticleType type )
{
    switch( type )
    {
    case EFT_PARTICLE_TYPE_BILLBOARD:            m_VtxFlag.Set( _PARTICLE_TYPE_BILLBOARD );            break;
    case EFT_PARTICLE_TYPE_COMPLEX_BILLBOARD:    m_VtxFlag.Set( _PARTICLE_TYPE_COMPLEX_BILLBOARD );    break;
    case EFT_PARTICLE_TYPE_Y_BILLBOARD:          m_VtxFlag.Set( _PARTICLE_TYPE_Y_BILLBOARD );          break;
    case EFT_PARTICLE_TYPE_POLYGON_XY:           m_VtxFlag.Set( _PARTICLE_TYPE_POLYGON_XY );           break;
    case EFT_PARTICLE_TYPE_POLYGON_XZ:           m_VtxFlag.Set( _PARTICLE_TYPE_POLYGON_XZ );           break;
    case EFT_PARTICLE_TYPE_VEL_LOOK:             m_VtxFlag.Set( _PARTICLE_TYPE_VEL_LOOK );             break;
    case EFT_PARTICLE_TYPE_VEL_LOOK_POLYGON:     m_VtxFlag.Set( _PARTICLE_TYPE_VEL_LOOK_POLYGON );     break;
    case EFT_PARTICLE_TYPE_WITH_SCALE_Z:         m_VtxFlag.Set( _PARTICLE_TYPE_WITH_SCALE_Z );         break;
    case EFT_PARTICLE_TYPE_WITH_SCALE_Z_DIRECTIONAL: m_VtxFlag.Set( _PARTICLE_TYPE_WITH_SCALE_Z_DIRECTIONAL ); break;
    case EFT_PARTICLE_TYPE_Y_BILLBOARD_PARALLEL: m_VtxFlag.Set( _PARTICLE_TYPE_Y_BILLBOARD_PARALLEL ); break;
    }
}

//---------------------------------------------------------------------------
//  パーティクル回転を設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetParticleRotateType( ParticleRotType type )
{
    switch( type )
    {
    case EFT_PARTICLE_ROTATE_TYPE_ROTXYZ: m_VtxFlag.Set( _ROTATE_XYZ ); break;
    case EFT_PARTICLE_ROTATE_TYPE_ROTZXY: m_VtxFlag.Set( _ROTATE_ZXY ); break;
    default: m_VtxFlag.Set( _ROTATE_YZX ); break;
    }
}

//---------------------------------------------------------------------------
//! @brief  エミッタ追従タイプを設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetEmitterFollowType( EmitterFollowType type )
{
    switch( type )
    {
    case EFT_EMITTER_FOLLOW_TYPE_ALL:  m_VtxFlag.Set( _EMITTER_FOLLOW_TYPE_ALL ); break;
    case EFT_EMITTER_FOLLOW_TYPE_NONE: m_VtxFlag.Set( _EMITTER_FOLLOW_TYPE_NONE ); break;
    case EFT_EMITTER_FOLLOW_TYPE_POS:  m_VtxFlag.Set( _EMITTER_FOLLOW_TYPE_POS ); break;
    }
}

//---------------------------------------------------------------------------
//  カラー計算タイプを設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetParticleColorCalcType( ParticleColorCalcType color0Type, ParticleColorCalcType color1Type )
{
    switch( color0Type )
    {
    case EFT_PARTICLE_COLOR_CALC_TYPE_FIXED:     m_VtxFlag.Set( _COLOR_0_FIXED );    break;
    case EFT_PARTICLE_COLOR_CALC_TYPE_RANDOM:    m_VtxFlag.Set( _COLOR_0_RANDOM );    break;
    }
    switch( color1Type )
    {
    case EFT_PARTICLE_COLOR_CALC_TYPE_FIXED:     m_VtxFlag.Set( _COLOR_1_FIXED );    break;
    case EFT_PARTICLE_COLOR_CALC_TYPE_RANDOM:    m_VtxFlag.Set( _COLOR_1_RANDOM );    break;
    }
}

//---------------------------------------------------------------------------
//  パーティクルアルファ計算タイプを設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetParticleAlphaCalcType( ParticleColorCalcType alpha0Type, ParticleColorCalcType alpha1Type )
{
    switch( alpha0Type )
    {
    case EFT_PARTICLE_COLOR_CALC_TYPE_FIXED:     m_VtxFlag.Set( _ALPHA_0_FIXED );    break;
    }
    switch( alpha1Type )
    {
    case EFT_PARTICLE_COLOR_CALC_TYPE_FIXED:     m_VtxFlag.Set( _ALPHA_1_FIXED );    break;
    }
}

//---------------------------------------------------------------------------
//  パーティクルテクスチャパターンアニメ有効無効を設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetParticleTexturePatternAnim( bool texture0, bool texture1, bool texture2 )
{
    if( texture0 )
    {
        m_VtxFlag.Set( _TEX_0_PATTERN_ANIM );
    }
    else
    {
        m_VtxFlag.Cancel( _TEX_0_PATTERN_ANIM );
    }
    if( texture1 )
    {
        m_VtxFlag.Set( _TEX_1_PATTERN_ANIM );
    }
    else
    {
        m_VtxFlag.Cancel( _TEX_1_PATTERN_ANIM );
    }
    if( texture2 )
    {
        m_VtxFlag.Set( _TEX_2_PATTERN_ANIM );
    }
    else
    {
        m_VtxFlag.Cancel( _TEX_2_PATTERN_ANIM );
    }
}

//---------------------------------------------------------------------------
//  パーティクルテクスチャ座標アニメ有効無効を設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetParticleTextureShiftAnim( bool texture0, bool texture1, bool texture2 )
{
    if( texture0 )
    {
        m_VtxFlag.Set( _TEX_0_SHIFT_ANIM );
    }
    else
    {
        m_VtxFlag.Cancel( _TEX_0_SHIFT_ANIM );
    }
    if( texture1 )
    {
        m_VtxFlag.Set( _TEX_1_SHIFT_ANIM );
    }
    else
    {
        m_VtxFlag.Cancel( _TEX_1_SHIFT_ANIM );
    }
    if( texture2 )
    {
        m_VtxFlag.Set( _TEX_2_SHIFT_ANIM );
    }
    else
    {
        m_VtxFlag.Cancel( _TEX_2_SHIFT_ANIM );
    }
}

//---------------------------------------------------------------------------
//  パーティクルスフィアマップ有効無効を設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetParticleSphereMap( bool texture0, bool texture1, bool texture2 )
{
    if( texture0 )
    {
        m_VtxFlag.Set( _TEX_0_SPHERE_MAP );
    }
    else
    {
        m_VtxFlag.Cancel( _TEX_0_SPHERE_MAP );
    }
    if( texture1 )
    {
        m_VtxFlag.Set( _TEX_1_SPHERE_MAP );
    }
    else
    {
        m_VtxFlag.Cancel( _TEX_1_SPHERE_MAP );
    }
    if( texture2 )
    {
        m_VtxFlag.Set( _TEX_2_SPHERE_MAP );
    }
    else
    {
        m_VtxFlag.Cancel( _TEX_2_SPHERE_MAP );
    }
}

//---------------------------------------------------------------------------
//  フラグメントシェーダを標準設定にします。
//---------------------------------------------------------------------------
void ShaderKey::SetDefaultFragmentShaderSetting()
{
    m_FragFlag.Set( _SHADER_TYPE_NORMAL );
    m_FragFlag.Set( _COLOR_PROCESS_TEXTURE );
    m_FragFlag.Set( _ALPHA_PROCESS_MOD );
    m_FragFlag.Set( _TEXTURE1_COLOR_BLEND_MOD );
    m_FragFlag.Set( _TEXTURE2_COLOR_BLEND_MOD );
    m_FragFlag.Set( _TEXTURE1_ALPHA_BLEND_MOD );
    m_FragFlag.Set( _TEXTURE2_ALPHA_BLEND_MOD );
    m_FragFlag.Set( _PRIMITIVE_COLOR_BLEND_MOD );
    m_FragFlag.Set( _PRIMITIVE_ALPHA_BLEND_MOD );
    m_FragFlag.Set( _PRIMITIVE_COLOR_INPUT_SRC );
    m_FragFlag.Set( _PRIMITIVE_ALPHA_INPUT_A );
}

//---------------------------------------------------------------------------
//  color/alpha 8keyアニメを設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetColor8KeyAnim( u8 color0Anim, u8 color1Anim )
{
    m_VtxColor0AnimKey = color0Anim;
    m_VtxColor1AnimKey = color1Anim;
}

//---------------------------------------------------------------------------
//  揺らぎを設定します。
//---------------------------------------------------------------------------
void ShaderKey::SetFluctuation( u32 fluctuation )
{
    switch( fluctuation )
    {
    case 1: m_VtxFlag.Set( _FLUCTUATION_SCALE_ENABLE );  break;
    case 2: m_VtxFlag.Set( _FLUCTUATION_ALPHA_ENABLE );  break;
    case 3: m_VtxFlag.Set( _FLUCTUATION_SCALE_Y_SEPARATE );  break;
    }
}
//---------------------------------------------------------------------------
//  全てのパーティクルタイプをクリアする。
//---------------------------------------------------------------------------
void ShaderKey::ClearAllParticleType()
{
    m_VtxFlag.Cancel( _PARTICLE_TYPE_BILLBOARD );
    m_VtxFlag.Cancel( _PARTICLE_TYPE_COMPLEX_BILLBOARD );
    m_VtxFlag.Cancel( _PARTICLE_TYPE_Y_BILLBOARD );
    m_VtxFlag.Cancel( _PARTICLE_TYPE_POLYGON_XY );
    m_VtxFlag.Cancel( _PARTICLE_TYPE_POLYGON_XZ );
    m_VtxFlag.Cancel( _PARTICLE_TYPE_VEL_LOOK );
    m_VtxFlag.Cancel( _PARTICLE_TYPE_VEL_LOOK_POLYGON );
    m_VtxFlag.Cancel( _PARTICLE_TYPE_Y_BILLBOARD_PARALLEL );
    m_VtxFlag.Cancel( _PARTICLE_TYPE_WITH_SCALE_Z );
    m_VtxFlag.Cancel( _PARTICLE_TYPE_WITH_SCALE_Z_DIRECTIONAL );
}

//---------------------------------------------------------------------------
//  全ての追従タイプをクリアする。
//---------------------------------------------------------------------------
void ShaderKey::ClearAllFollowType()
{
    m_VtxFlag.Cancel( _EMITTER_FOLLOW_TYPE_ALL );
    m_VtxFlag.Cancel( _EMITTER_FOLLOW_TYPE_NONE );
    m_VtxFlag.Cancel( _EMITTER_FOLLOW_TYPE_POS );
}

//---------------------------------------------------------------------------
//  全ての挙動計算タイプをクリアする。
//---------------------------------------------------------------------------
void ShaderKey::ClearAllCalcType()
{
    m_VtxFlag.Cancel( _CALC_TYPE_CPU );
    m_VtxFlag.Cancel( _CALC_TYPE_GPU_TIME );
    m_VtxFlag.Cancel( _CALC_TYPE_GPU_SO );
}

//---------------------------------------------------------------------------
//  全ての挙動計算タイプをクリアする。
//---------------------------------------------------------------------------
void ShaderKey::ClearAllFluctuation()
{
    m_VtxFlag.Cancel( _FLUCTUATION_SCALE_ENABLE );
    m_VtxFlag.Cancel( _FLUCTUATION_ALPHA_ENABLE );
    m_VtxFlag.Cancel( _FLUCTUATION_SCALE_Y_SEPARATE );
}

#endif

} // namespace eft2
} // namespace nw

