﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


//------------------------------------------------------------------------------
#include <nw/eft/eft2_Emitter.h>
#include <nw/eft/eft2_EmitterCalc.h>
#include <nw/eft/eft2_System.h>
#include <nw/eft/eft2_KeyFrameAnim.h>

namespace nw   {
namespace eft2 {

//---------------------------------------------------------------------------
//  パーティクル情報を計算する
//---------------------------------------------------------------------------
void EmitterCalc::CalcParticleInfo( Particle* dstPtcl, Emitter* emitter, f32 time, f32 life, u32 index )
{
    EFT_UNUSED_VARIABLE( dstPtcl );
    EFT_UNUSED_VARIABLE( emitter );
    EFT_UNUSED_VARIABLE( time );
    EFT_UNUSED_VARIABLE( life );
    EFT_UNUSED_VARIABLE( index );

    //nw::math::VEC4* random = &emitter->particleAttr[index].random;

    //CalcRotationMatrix( &dstPtcl->localRotate,
    //                    emitter->emitterRes,
    //                    &emitter->particleAttr[index].initRotate,
    //                    &emitter->particleAttr[index].random,
    //                    time );
    //CalcPtclScaleVec(   &dstPtcl->localScale,  emitter, &emitter->particleAttr[index].scale, random, life, time );
    //CalcPtclColor0Vec(  &dstPtcl->localColor0, emitter, random, life, time );
    //CalcPtclColor1Vec(  &dstPtcl->localColor1, emitter, random, life, time );
}

//---------------------------------------------------------------------------
//  親パーティクル情報を計算しチャイルドパーティクルへ継承する
//---------------------------------------------------------------------------
void EmitterCalc::InheritParentParticleInfo( Emitter* emitter, u32 writeIndex )
{
    EFT_UNUSED_VARIABLE( emitter );
    EFT_UNUSED_VARIABLE( writeIndex );

    ParticlePosAttribute* posAttr   = &emitter->particlePosAttr[writeIndex];
    ParticleAttribute*    ptclAttr  = &emitter->particleAttr[writeIndex];

    // 速度継承は、親パーティクルから設定されたベクトルを利用する
    if ( emitter->emitterData->inherit.velocity )
    {
        f32 inheritRate = emitter->emitterData->inherit.velocityRate;
        posAttr->localVec.x += emitter->parentParticleWorldVec.x * inheritRate;
        posAttr->localVec.y += emitter->parentParticleWorldVec.y * inheritRate;
        posAttr->localVec.z += emitter->parentParticleWorldVec.z * inheritRate;
    }

    // 親パーティクルの状態を計算し継承する
    Particle parentParticle;

    // スケールを継承
    if ( emitter->emitterData->inherit.scale )
    {
        CalcPtclScaleVecFromTime( &parentParticle.localScale,
                                  emitter->parentEmitter->emitterRes,
                                  &emitter->parentParticleScale,
                                  &emitter->parentParticleRandom,
                                  emitter->parentParticleLife,
                                  emitter->parentParticleTime );

        const f32 inheritRate = emitter->emitterData->inherit.scaleRate;
        ptclAttr->scale.x = parentParticle.localScale.x * inheritRate;
        ptclAttr->scale.y = parentParticle.localScale.y * inheritRate;
        ptclAttr->scale.z = parentParticle.localScale.z * inheritRate;
        // .w には、パーティクル生成タイムが入力されるため、コピーしない
    }

    // 回転値を継承
    if ( emitter->emitterData->inherit.rotate )
    {
        CalcRotationMatrix( &parentParticle.localRotate,
                            emitter->parentEmitter->emitterRes,
                            &emitter->parentParticleRotate,
                            &emitter->parentParticleRandom,
                            emitter->parentParticleTime );

        emitter->particleAttr[writeIndex].initRotate.x = parentParticle.localRotate.x;
        emitter->particleAttr[writeIndex].initRotate.y = parentParticle.localRotate.y;
        emitter->particleAttr[writeIndex].initRotate.z = parentParticle.localRotate.z;
    }

    // カラー 0 /アルファ 0 継承
    if ( emitter->emitterData->inherit.color0 || emitter->emitterData->inherit.alpha0 )
    {
        CalcPtclColor0Vec(  &parentParticle.localColor0,
                            emitter->parentEmitter->emitterRes,
                            &emitter->parentParticleRandom,
                            &emitter->parentEmitter->color0,
                            static_cast<const nw::math::VEC3 *>( &emitter->parentEmitter->m_EmitterAnim.vcolor0 ),
                            emitter->parentEmitter->m_EmitterAnim.valpha0.x,
                            emitter->parentParticleLife,
                            emitter->parentParticleTime );
    }

    // カラー0を継承
    if ( emitter->emitterData->inherit.color0 )
    {
        ptclAttr->initColor0.x = parentParticle.localColor0.x;
        ptclAttr->initColor0.y = parentParticle.localColor0.y;
        ptclAttr->initColor0.z = parentParticle.localColor0.z;
    }

    // アルファ0を継承
    if ( emitter->emitterData->inherit.alpha0 )
    {
        ptclAttr->initColor0.w = parentParticle.localColor0.w;
    }

    // カラー 1 /アルファ 1 継承
    if ( emitter->emitterData->inherit.color1 || emitter->emitterData->inherit.alpha1 )
    {
        CalcPtclColor1Vec(  &parentParticle.localColor1,
                            emitter->parentEmitter->emitterRes,
                            &emitter->parentParticleRandom,
                            &emitter->parentEmitter->color1,
                            static_cast<const nw::math::VEC3 *>( &emitter->parentEmitter->m_EmitterAnim.vcolor0 ),
                            emitter->parentEmitter->m_EmitterAnim.valpha0.x,
                            emitter->parentParticleLife,
                            emitter->parentParticleTime );
    }

    // カラー1を継承
    if ( emitter->emitterData->inherit.color1 )
    {
        ptclAttr->initColor1.x = parentParticle.localColor1.x;
        ptclAttr->initColor1.y = parentParticle.localColor1.y;
        ptclAttr->initColor1.z = parentParticle.localColor1.z;
    }

    // アルファ1を継承
    if ( emitter->emitterData->inherit.alpha1 )
    {
        ptclAttr->initColor1.w = parentParticle.localColor1.w;
    }

    return;
}

//---------------------------------------------------------------------------
//  検索して死亡枠があればそこに追加、死亡枠が無い場合は後続に追加。
//---------------------------------------------------------------------------
s32 EmitterCalc::_Emit_SearchOrder( Emitter* emitter, u32 emitNum, bool forceSearch )
{
    f32     random          = emitter->random.GetF32();  // 一度のEmit処理の間常に一定値になる乱数を生成。
    bool    addIndex        = true;
    s32     writeLastIndex  = -1;

    // パーティクル生成後コールバック
    ParticleEmitCallback ptclEmtCbEP = NULL;
    ParticleEmitCallback ptclEmtCbCA = NULL;
    ParticleEmitCallback ptclEmtCbCS = NULL;

    if ( emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP] )
    {
        ptclEmtCbEP = emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP]->particleEmit;
    }
    if ( emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA] )
    {
        ptclEmtCbCA = emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA]->particleEmit;
    }
    if ( emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS] )
    {
        ptclEmtCbCS = emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS]->particleEmit;
    }

    if( emitter->emitterRes->emitterData->volume.primEmitType == EFT_EMIT_FROM_PRIMITIVE_UNISON )
    {
        // 等分割系（円・線）の一斉放出の場合は、放出レートを分割数倍にする
        // 「分割数ランダム」の存在があるのでこのタイミングで決めるしかない。
        const ResEmitterVolume* res = &emitter->emitterData->volume;
        if( emitter->emitterRes->emitterData->volume.volumeType == EFT_EMITTER_VOLUME_TYPE_CIRCLE_SAME_DIVIDE )
        {
            emitNum *= res->numDivideCircle - static_cast<u32>( random * res->numDivideCircleRandom * 0.01f * res->numDivideCircle );
        }
        else if( emitter->emitterRes->emitterData->volume.volumeType == EFT_EMITTER_VOLUME_TYPE_LINE_SAME_DIVIDE )
        {
            emitNum *= res->numDivideLine - static_cast<u32>( random * res->numDivideLineRandom * 0.01f * res->numDivideLine );
        }
    }

    // 放出処理
    for( u32 i = 0; i < emitNum; i++ )
    {
        bool isSearch = false;
        if ( forceSearch ) isSearch = true;

        // 書き込み先バッファ
        s32 writeIndex = emitter->ptclAttrFillIndex;
        addIndex       = true;

        // 寿命ランダム＆ワンタイムでない場合は書き込み先を検索する
        // MEMO: ワンタイムの場合は大抵パーティクルの寿命が来る前にエミッタが死ぬパターンが多く、
        //       検索する必要性が薄いので除外する。
        if ( emitter->emitterData->ptcl.lifeRandom != 0 && emitter->emitterData->emission.isOneTime == 0  )
        {
            isSearch = true;
        }

        // これまで生成したバッファ内から空きを検索し、
        // 空きがあればそこを利用する。( なるべく左詰め )
        if ( isSearch )
        {
            s32 _writeIndex = -1;

            if ( emitter->ptclNum == 0 )
            {
                writeIndex  = 0;
            }
            else
            {
                if ( emitter->emitterData->emitter.calcType == EFT_EMITTER_CALC_TYPE_CPU )
                {
                    // CPUの場合は、終了処理が行われるので、createIDで死亡判断
                    for ( u32 j = 0; j < emitter->ptclNum; j++ )
                    {
                        if ( emitter->particleData[j].createID == 0 )
                        {
                            _writeIndex = j;
                            addIndex    = false;
                            break;
                        }
                    }
                }
                else
                {
                    // GPUの場合は、時間と寿命で死亡判断
                    for ( u32 j = 0; j < emitter->ptclNum; j++ )
                    {
                        // GPUパーティクルなのでシングルバッファ。フロントバッファから取ってOK
                        ParticlePosAttribute* posAttr = &emitter->particlePosAttr[j];
                        f32 time = posAttr->GetTime( emitter->time );
                        f32 life = posAttr->GetLife() ;

                        if ( ( time - life ) > 0.0f )
                        {
                            _writeIndex = j;
                            addIndex    = false;
                            break;
                        }
                    }
                }
            }

            if ( _writeIndex != -1 )
            {
                writeIndex = _writeIndex;
            }
        }

        // 不正なデータがアタッチされていないかチェック
        if ( emitter->particleData[writeIndex].emitterPluginData ||
             emitter->particleData[writeIndex].userData )
        {
            Warning( (void *)emitter, EFT_WARNING_PARTICLE_IS_DIRTY );
            return writeLastIndex;
        }

        emitter->isEmit = true;

        ParticleData*           pPtclData           = &emitter->particleData[writeIndex];
        ParentParticleData*     pParentPtclData     = &emitter->parentParticleData[writeIndex];
        ParticleAttribute*      pPtclAttribute      = &emitter->particleAttr[writeIndex];
        ParticlePosAttribute*   pPtclPosAttribute   = &emitter->particlePosAttr[writeIndex];

        // パーティクルの初期状態を計算する
        if( InitializeParticle( emitter,
            writeIndex,
            pPtclData,
            pParentPtclData,
            pPtclAttribute,
            pPtclPosAttribute,
            i, emitNum, random ) )
        {
            writeLastIndex = writeIndex;

            EFT_F32_VEC3_COPY( &emitter->particlePosAttr[writeIndex].localDiff, &emitter->particlePosAttr[writeIndex].localVec );

            // ここまでの処理で、ParticlePosAttrの値は全て初期化済み（という前提）。以降はフロントバッファから値を取得。

            // チャイルドエミッタ継承処理
            if ( emitter->parentEmitter )
            {
                InheritParentParticleInfo( emitter, writeIndex );
            }

            // 裏バッファのポインタを覚えておく
            // 初回でも、この前のCalcEmitterの処理でバッファがスワップされているので、particleAttrOrgへの決め打ちはできない！
            // ダブルバッファの場合、この時点では裏バッファは表と同じ領域を指すようにしておく（コピーのコストを浮かせる）
            // この後の最期のFlushCacheの時点でコピーして同期をとる
            ParticlePosAttribute* particlePosAttrBackOrg = emitter->particlePosAttrBack;
            if( emitter->emitterData->emitter.calcType == EFT_EMITTER_CALC_TYPE_CPU )
            {
                emitter->particlePosAttrBack = emitter->particlePosAttr;
            }
            ParticleAttribute* particleAttrBackOrg = emitter->particleAttrBack;
            if( emitter->IsParticleAttributeDoubleBuffer() )
            {
                emitter->particleAttrBack = emitter->particleAttr;
            }

            // パーティクル生成コールバック
            InvokeParticleEmitCallback( emitter, writeIndex, ptclEmtCbEP, ptclEmtCbCA, ptclEmtCbCS );

            // バッファ書き込みサイズの調整
            if ( addIndex )
            {
                // バッファ書き込み量を加算
                emitter->ptclAttrFillIndex = ( writeIndex + 1 );

                // バッファ折り返しチェック
                if ( emitter->ptclAttrFillIndex == emitter->ptclAttrFillMax )
                {
                    emitter->ptclAttrFillIndex = 0;
                }

                // 書き込んだ最大数
                if ( emitter->ptclNum < emitter->ptclAttrFillMax )
                {
                    emitter->ptclNum++;
                }
            }

            // キャッシュをフラッシュしておく
            MemUtil::FlushCache( &emitter->particleAttr[writeIndex], sizeof( ParticleAttribute ) );
            if( emitter->emitterData->emitter.calcType == EFT_EMITTER_CALC_TYPE_CPU )
            {
                // バックバッファへのポインタを指し直させる
                emitter->particlePosAttrBack = particlePosAttrBackOrg;
            }
            if( emitter->IsParticleAttributeDoubleBuffer() )
            {
                // バックバッファへのポインタを指し直させる
                emitter->particleAttrBack = particleAttrBackOrg;
            }
            // GPUパーティクルの場合は、posAttrもキャッシュフラッシュしておく
            // emitter->delayFlag の場合もキャッシュフラッシュしないと計算処理が回るまでに描画されてしまう。
            if ( emitter->emitterData->emitter.calcType != EFT_EMITTER_CALC_TYPE_CPU )
            {
                MemUtil::FlushCache( &emitter->particlePosAttr[writeIndex], sizeof( ParticlePosAttribute ) );
            }
        }
    }

    return writeLastIndex;
}



//---------------------------------------------------------------------------
//  放出処理を行います。
//---------------------------------------------------------------------------
s32 EmitterCalc::Emit( Emitter* emitter, u32 emitNum, bool forceSearch )
{
    s32 writeLastIndex = _Emit_SearchOrder( emitter, emitNum, forceSearch );
    return writeLastIndex;
}

//------------------------------------------------------------------------------
//  パーティクルを初期化します。
//------------------------------------------------------------------------------
bool EmitterCalc::InitializeParticle
(
    Emitter*                emitter,
    u32                     attrIndex,
    ParticleData*           ptclData,
    ParentParticleData*     parentPtclData,
    ParticleAttribute*      ptclAttr,
    ParticlePosAttribute*   posAttr,
    u32                     emitIndex,
    u32                     emitMax,
    f32                     randomValue )
{
    ResEmitter* res        = emitter->emitterData;
    EmitterSet* emitterSet = emitter->emitterSet;
    f32         life       = 0.0f;

    // localPos/localVec の初期化: （以降、フロントバッファから取得してOK）
    bool result = m_EmitFunctions[res->volume.volumeType]( &posAttr->localPos,
                                                           &posAttr->localVec,
                                                           emitter, emitIndex, emitMax, randomValue, &emitter->m_EmitterAnim);
    if( !result ){ return false; }

    // 生成時のエミッタマトリクス
    EFT_F32_VEC4_COPY( &ptclAttr->emitterMat0, &emitter->matrixSRT.v[0] );
    EFT_F32_VEC4_COPY( &ptclAttr->emitterMat1, &emitter->matrixSRT.v[1] );
    EFT_F32_VEC4_COPY( &ptclAttr->emitterMat2, &emitter->matrixSRT.v[2] );

    EFT_F32_VEC4_COPY( &ptclAttr->emitterRTMat0, &emitter->matrixRT.v[0] );
    EFT_F32_VEC4_COPY( &ptclAttr->emitterRTMat1, &emitter->matrixRT.v[1] );
    EFT_F32_VEC4_COPY( &ptclAttr->emitterRTMat2, &emitter->matrixRT.v[2] );

    // パーティクル生成ID
    emitter->particleCreateID++;
    ptclData->createID = emitter->particleCreateID;

    // Y軸拡散速度を計算する
    if( emitter->emitterData->ptclVel.xzDiffusion != 0.0f )
    {
        nw::math::VEC3 yAxisVel;
        yAxisVel.x = posAttr->localPos.x;
        yAxisVel.y = 0.0f;
        yAxisVel.z = posAttr->localPos.z;
        if( yAxisVel.LengthSquare() <= FLT_MIN )
        {
            yAxisVel.x = emitter->random.GetF32() * 2.0f - 1.0f;
            yAxisVel.z = emitter->random.GetF32() * 2.0f - 1.0f;
        }
        yAxisVel.SafeNormalize(nw::math::VEC3(0.0f, 0.0f, 0.0f));
        yAxisVel *= emitter->emitterData->ptclVel.xzDiffusion;

        posAttr->localVec.x += yAxisVel.x;
        posAttr->localVec.y += yAxisVel.y;
        posAttr->localVec.z += yAxisVel.z;
    }

    f32 velRand = 1.0f - emitter->random.GetF32() * (emitter->emitterData->ptclVel.velRandom/100.f) * emitterSet->m_RandomVel;
    f32 dirVel  = emitter->m_EmitterAnim.designatedDirScale * emitterSet->m_DirectionalVel;
    f32 angle   = emitter->emitterData->ptclVel.diffusionDirAngle;

    // 位置ランダム
    if( emitter->emitterData->emission.posRand != 0.0f )
    {
        nw::math::VEC3 rndVec3 = emitter->random.GetNormalizedVec3();
        posAttr->localPos.x = rndVec3.x * emitter->emitterData->emission.posRand + posAttr->localPos.x;
        posAttr->localPos.y = rndVec3.y * emitter->emitterData->emission.posRand + posAttr->localPos.y;
        posAttr->localPos.z = rndVec3.z * emitter->emitterData->emission.posRand + posAttr->localPos.z;
    }

    if( emitter->manualEmitPoints )
    {
        // マニュアル放出の場合で、各Ptclの位置情報がある場合は加算。
        posAttr->localPos.x += emitter->manualEmitPoints[emitIndex].x;
        posAttr->localPos.y += emitter->manualEmitPoints[emitIndex].y;
        posAttr->localPos.z += emitter->manualEmitPoints[emitIndex].z;
    }
    {
        // マニュアル放出の場合、共通オフセットが入っているので足し込む。
        // 通常時は0ベクトルなので無視できる。
        posAttr->localPos.x += emitter->manualEmitCommonOffset.x;
        posAttr->localPos.y += emitter->manualEmitCommonOffset.y;
        posAttr->localPos.z += emitter->manualEmitCommonOffset.z;
    }

    nw::math::VEC3 designatedDir = emitter->emitterData->ptclVel.designatedDir;
    if( emitter->emitterData->emission.isWorldOrientedVelocity )
    {
        emitter->TransformToLocalVec( &designatedDir, designatedDir, ptclAttr );
    }

    // ローカル速度
    if( angle == 0.0f )
    {
        posAttr->localVec.x = ( posAttr->localVec.x + designatedDir.x * dirVel ) * velRand;
        posAttr->localVec.y = ( posAttr->localVec.y + designatedDir.y * dirVel ) * velRand;
        posAttr->localVec.z = ( posAttr->localVec.z + designatedDir.z * dirVel ) * velRand;
    }
    else
    {
        // TODO:書き直し。

        // 指定方向拡散角度の処理
        // (0,1,0)を基準にランダムを適用した円錐状ベクトルを生成する
        angle = angle / 90.0f;
        angle = 1.0f - angle;

        f32 rot = emitter->random.GetF32() * 2.0f * nw::math::F_PI;
        f32 sinV;
        f32 cosV;
        nw::math::SinCosRad( &sinV, &cosV, rot );
        f32 y = emitter->random.GetF32() * ( 1.0f - angle ) + angle;
        f32 r = SafeSqrt( 1.0f - y * y );

        nw::math::VEC3 velocity;
        velocity.x = r * cosV;
        velocity.z = r * sinV;
        velocity.y = y;

        // (0, 1, 0)を指定方向に回転させる回転行列を生成し適用する
        nw::math::QUAT q;
        nw::math::QUATMakeVectorRotation( &q, nw::math::VEC3( 0.0f, 1.0f, 0.0f ), designatedDir );
        nw::math::MTX34 qmat;
        nw::math::QUATToMTX34( &qmat, q );
        nw::math::VEC3 srcVelocity;
        srcVelocity.Set( velocity.x, velocity.y, velocity.z );
        nw::math::VEC3Transform(&velocity,& qmat, &srcVelocity);

        posAttr->localVec.x = ( posAttr->localVec.x + velocity.x * dirVel ) * velRand;
        posAttr->localVec.y = ( posAttr->localVec.y + velocity.y * dirVel ) * velRand;
        posAttr->localVec.z = ( posAttr->localVec.z + velocity.z * dirVel ) * velRand;
    }

    // ローカル拡散速度
    const nw::math::VEC3 &rndVec3 = emitter->random.GetVec3();
    posAttr->localVec.x += rndVec3.x * emitter->emitterData->ptclVel.diffusion.x;
    posAttr->localVec.y += rndVec3.y * emitter->emitterData->ptclVel.diffusion.y;
    posAttr->localVec.z += rndVec3.z * emitter->emitterData->ptclVel.diffusion.z;

    // エミッタ速度継承
    f32 inheritRate = emitter->emitterData->ptclVel.emVelInherit;
    posAttr->localVec.x += emitter->emitterLocalVec.x * inheritRate;
    posAttr->localVec.y += emitter->emitterLocalVec.y * inheritRate;
    posAttr->localVec.z += emitter->emitterLocalVec.z * inheritRate;

    // ワールド加算ベクトル
    posAttr->localVec.x += emitterSet->m_VelAdd.x * emitterSet->m_MatrixRT.m[0][0] + emitterSet->m_VelAdd.y * emitterSet->m_MatrixRT.m[1][0] + emitterSet->m_VelAdd.z * emitterSet->m_MatrixRT.m[2][0];
    posAttr->localVec.y += emitterSet->m_VelAdd.x * emitterSet->m_MatrixRT.m[0][1] + emitterSet->m_VelAdd.y * emitterSet->m_MatrixRT.m[1][1] + emitterSet->m_VelAdd.z * emitterSet->m_MatrixRT.m[2][1];
    posAttr->localVec.z += emitterSet->m_VelAdd.x * emitterSet->m_MatrixRT.m[0][2] + emitterSet->m_VelAdd.y * emitterSet->m_MatrixRT.m[1][2] + emitterSet->m_VelAdd.z * emitterSet->m_MatrixRT.m[2][2];

    if( emitter->GetCalcType() == EFT_EMITTER_CALC_TYPE_GPU_SO )
    {
        posAttr->localPos.x += posAttr->localVec.x;
        posAttr->localPos.y += posAttr->localVec.y;
        posAttr->localPos.z += posAttr->localVec.z;
    }

    // スケール
    nw::math::VEC4* scale = NULL;
#ifdef EFT_DEGRADATION_SPEC
    scale               = &ptclData->initScale;
#else
    scale               = &ptclAttr->scale;
#endif

    if ( res->ptclScale.baseRandom.x != res->ptclScale.baseRandom.y )
    {
        scale->x    = emitter->m_EmitterAnim.vparticleScale.x *
                      ( 1.f - res->ptclScale.baseRandom.x / 100.0f * emitter->random.GetF32() ) *
                      emitterSet->m_ParticleEmissionScale.x;
        scale->y    = emitter->m_EmitterAnim.vparticleScale.y *
                      ( 1.f - res->ptclScale.baseRandom.y / 100.0f * emitter->random.GetF32() ) *
                      emitterSet->m_ParticleEmissionScale.y;
        scale->z    = emitter->m_EmitterAnim.vparticleScale.z *
                      ( 1.f - res->ptclScale.baseRandom.z / 100.0f * emitter->random.GetF32() ) *
                      emitterSet->m_ParticleEmissionScale.z;
    }
    else
    {
        f32 random  = ( 1.f - res->ptclScale.baseRandom.x / 100.0f * emitter->random.GetF32() );
        scale->x    = emitter->m_EmitterAnim.vparticleScale.x * random * emitterSet->m_ParticleEmissionScale.x;
        scale->y    = emitter->m_EmitterAnim.vparticleScale.y * random * emitterSet->m_ParticleEmissionScale.y;
        scale->z    = emitter->m_EmitterAnim.vparticleScale.z * random * emitterSet->m_ParticleEmissionScale.z;
    }

#ifdef EFT_DEGRADATION_SPEC
    ptclAttr->scale.x   = scale->x;
    ptclAttr->scale.y   = scale->y;
    ptclAttr->scale.z   = scale->z;
#endif

    ptclData->collisionCount = 0;

    // 運動量ランダム
    ptclAttr->scale.w       = 1.0f + res->ptcl.dynamicsRand - res->ptcl.dynamicsRand * emitter->random.GetF32() * 2.0f;

    // 生成時間
    ptclData->createTime = emitter->time;
    posAttr->localVec.w  = ptclData->createTime;

    ptclData->isBothInitialized = false;

    // パーティクル寿命
    if ( !res->ptcl.isLifeInfinity )
    {
        life = static_cast<f32>( ( emitter->m_EmitterAnim.particleLife - emitter->m_EmitterAnim.particleLife * ( emitter->random.GetS32( res->ptcl.lifeRandom ) * 0.01f ) ) );
        life *= emitter->particleLifeScale;
        life *= emitter->emitterSet->GetParticleLifeScale();
    }
    else
    {
        life = static_cast<f32>( EFT_INFINIT_LIFE );
    }
    posAttr->localPos.w = life;
    ptclData->life      = life;

    // ランダム値
    ptclAttr->random.x      = emitter->random.GetF32();
    ptclAttr->random.y      = emitter->random.GetF32();
    ptclAttr->random.z      = emitter->random.GetF32();
    ptclAttr->random.w      = emitter->random.GetF32();

#ifdef EFT_DEGRADATION_SPEC
    // 初期回転値 + 初期回転ランダム
    ptclAttr->initRotate.x  = emitter->emitterRes->rotateInit.x + ( ptclAttr->random.x * emitter->emitterRes->resEmitterStaticUniformBlock->rotateInitRand.x );
    ptclAttr->initRotate.y  = emitter->emitterRes->rotateInit.y + ( ptclAttr->random.y * emitter->emitterRes->resEmitterStaticUniformBlock->rotateInitRand.y );
    ptclAttr->initRotate.z  = emitter->emitterRes->rotateInit.z + ( ptclAttr->random.z * emitter->emitterRes->resEmitterStaticUniformBlock->rotateInitRand.z );
#else
    // 初期回転値
    EFT_F32_VEC3_COPY( &ptclAttr->initRotate, &emitter->emitterRes->rotateInit );
#endif

#ifndef EFT_DEGRADATION_SPEC
    const nw::math::VEC4 One( 1, 1, 1, 1 );
#endif

    // カラー0値
#ifdef EFT_DEGRADATION_SPEC
    EFT_F32_VEC4_COPY( &ptclAttr->initColor0, &emitter->emitterRes->emitterData->ptclColor.color0 );
#else
    EFT_F32_VEC4_COPY( &ptclAttr->initColor0, &One );
#endif

    // カラー1値
#ifdef EFT_DEGRADATION_SPEC
    EFT_F32_VEC4_COPY( &ptclAttr->initColor1, &emitter->emitterRes->emitterData->ptclColor.color1 );
#else
    EFT_F32_VEC4_COPY( &ptclAttr->initColor1, &One );
#endif

    //----------------------------------
    // 子エミッタが存在する場合の処理
    //----------------------------------
    for ( u32 i = 0; i < emitter->emitterRes->childEmitterResCount; i++ )
    {
        if ( emitter->childEmitterResSet[i] )
        {
            Emitter* child = emitterSet->CreateEmitter( emitter->childEmitterResSet[i], 0, emitter, i );
            if ( !child )
            {
                continue;
            }

            // リソースが保持するシェーダを受け取る
            Resource* resource = m_System->GetResource( emitterSet->GetResourceID() );
            EFT_NULL_ASSERT( resource );
            child->shader[EFT_SHADER_TYPE_NORMAL] = emitterSet->m_EmitterSetRes->shaderManager->GetShader( emitter->childEmitterResSet[i]->shaderIndex );
            EFT_NULL_ASSERT( child->shader[EFT_SHADER_TYPE_NORMAL] );

            if ( emitter->childEmitterResSet[i]->drawPathShader1 != EFT_INVALID_SHADER_ID )
            {
                child->shader[EFT_SHADER_TYPE_PATH_DEF1] = emitterSet->m_EmitterSetRes->shaderManager->GetShader( emitter->childEmitterResSet[i]->drawPathShader1 );
            }
            if ( emitter->childEmitterResSet[i]->drawPathShader2 != EFT_INVALID_SHADER_ID )
            {
                child->shader[EFT_SHADER_TYPE_PATH_DEF2] = emitterSet->m_EmitterSetRes->shaderManager->GetShader( emitter->childEmitterResSet[i]->drawPathShader2 );
            }

            // 親パーティクルに子エミッタを登録
            parentPtclData->childEmitter[i] = child;

            // 親エミッタの情報を登録
            child->parentEmitterCreateID    = emitter->emitterCreateID;         // 親エミッタの生成ID
            child->parentParticleCreateID   = emitter->particleCreateID;        // 親パーティクルの生成ID
            child->parentParticleLife       = life;                             // 親パーティクルの情報
            child->parentParticleBirthTime  = emitter->time;                    // 親パーティクルの生成時間

            // 親となるパーティクルの情報を保存する
            child->parentParticleAttrIdx    = attrIndex;

            EFT_F32_VEC3_COPY( &child->parentParticleLocalPos, &posAttr->localPos );
            EFT_F32_VEC3_COPY( &child->parentParticleLocalVec, &posAttr->localVec );

            EFT_F32_VEC4_COPY( &child->parentParticleScale, &ptclAttr->scale );
            EFT_F32_VEC3_COPY( &child->parentParticleRotate, &ptclAttr->initRotate );
            EFT_F32_VEC4_COPY( &child->parentParticleRandom, &ptclAttr->random );

            child->matrixSRT.SetIdentity();
            child->matrixRT.SetIdentity();
        }
        else
        {
            parentPtclData->childEmitter[i] = NULL;
        }
    }

    return true;
}


} // namespace eft2
} // namespace nw
