﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "../capsrvServer_Config.h"
#include <nn/capsrv/capsrv_ThumbnailFormat.h>
#include <nn/capsrv/capsrv_ViewerThumbnailFormat.h>

#define NN_CAPSRV_ALIGNAS_WORKMEMORY NN_ALIGNAS(4096)

namespace nn{ namespace capsrv{ namespace server{

    class ScreenShotWorkMemoryUtility
    {
    public:
        // @brief ワークメモリの単位サイズ。
        // @details
        //   ページサイズに一致させておく
        static const int UnitSize = 4 * 1024;

        // @brief 単位サイズまで切り上げる
        template<int N>
        struct RoundUpToUnitSize{
            enum { Value = ((N + UnitSize - 1) / UnitSize) * UnitSize };
        };

        // @brief 大きい方を選ぶ
        template<int A, int B>
        struct MaxOf{
            enum { Value = (A > B) ? A : B };
        };

    };

    enum ScreenShotWorkMemorySize
    {
        ScreenShotWorkMemorySize_KeepPresentationFullSizeRawImage   = 3801088,

        ScreenShotWorkMemorySize_ScreenShotFullSizeRawImageY        = 1048576,
        ScreenShotWorkMemorySize_ScreenShotFullSizeRawImageU        =  655360,
        ScreenShotWorkMemorySize_ScreenShotFullSizeRawImageV        =  655360,
        ScreenShotWorkMemorySize_ScreenShotOverlayThumbnailRawImage =  131072,
        ScreenShotWorkMemorySize_ScreenShotViewerThumbnailRawImage  =  262144,
        ScreenShotWorkMemorySize_ScreenShotJpegData                 = 512 * 1024, // >= ScreenShotImageSizeLimit
        ScreenShotWorkMemorySize_ScreenShotExifWorkMemory           = ScreenShotExifWorkSize,
        ScreenShotWorkMemorySize_ScreenShotJpegEncoderWorkMemory    = SoftwareJpegEncoderWorkMemorySize,
        ScreenShotWorkMemorySize_ScreenShotOverlayThumbnailRawData  = ScreenShotWorkMemoryUtility::RoundUpToUnitSize<ThumbnailImageDataSize_Default>::Value,
        ScreenShotWorkMemorySize_ScreenShotViewerThumbnailRawData   = ScreenShotWorkMemoryUtility::RoundUpToUnitSize<ViewerThumbnailImageDataSize_Raw>::Value,
        ScreenShotWorkMemorySize_ScreenShotViewerThumbnailJpegData  = ViewerThumbnailImageDataSize_EncodedLimit,

        ScreenShotWorkMemorySize_ScreenShotCaptureBufferDeviceShared =
            ScreenShotWorkMemorySize_KeepPresentationFullSizeRawImage,

        ScreenShotWorkMemorySize_ScreenShotDeviceShared =
            ScreenShotWorkMemorySize_ScreenShotFullSizeRawImageY +
            ScreenShotWorkMemorySize_ScreenShotFullSizeRawImageU +
            ScreenShotWorkMemorySize_ScreenShotFullSizeRawImageV +
            ScreenShotWorkMemorySize_ScreenShotOverlayThumbnailRawImage +
            ScreenShotWorkMemorySize_ScreenShotViewerThumbnailRawImage +
            ScreenShotWorkMemorySize_ScreenShotJpegData +
            ScreenShotWorkMemorySize_ScreenShotOverlayThumbnailRawData +
            ScreenShotWorkMemorySize_ScreenShotViewerThumbnailRawData +
            ScreenShotWorkMemorySize_ScreenShotViewerThumbnailJpegData,

        ScreenShotWorkMemorySize_ScreenShotCpuOnly =
            ScreenShotWorkMemorySize_ScreenShotExifWorkMemory +
            ScreenShotWorkMemorySize_ScreenShotJpegEncoderWorkMemory,

        ScreenShotWorkMemorySize_ScreenShotDeviceSharedReserved =
            ScreenShotWorkMemorySize_ScreenShotCaptureBufferDeviceShared - ScreenShotWorkMemorySize_ScreenShotDeviceShared,

    };

    struct ScreenShotWorkMemory
    {
        // デバイスマップされる領域。
        // デバイス経由で他のプロセスから書き換えられる可能性があるため制御データを含めないこと。
        union DeviceShared
        {
            struct All
            {
                NN_CAPSRV_ALIGNAS_WORKMEMORY char memory[ScreenShotWorkMemorySize_ScreenShotCaptureBufferDeviceShared];
            } all;

            struct ScreenShot
            {
                NN_CAPSRV_ALIGNAS_WORKMEMORY char fullSizeRawImageY       [ScreenShotWorkMemorySize_ScreenShotFullSizeRawImageY];
                NN_CAPSRV_ALIGNAS_WORKMEMORY char fullSizeRawImageU       [ScreenShotWorkMemorySize_ScreenShotFullSizeRawImageU];
                NN_CAPSRV_ALIGNAS_WORKMEMORY char fullSizeRawImageV       [ScreenShotWorkMemorySize_ScreenShotFullSizeRawImageV];
                NN_CAPSRV_ALIGNAS_WORKMEMORY char overlayThumbnailRawImage[ScreenShotWorkMemorySize_ScreenShotOverlayThumbnailRawImage];
                NN_CAPSRV_ALIGNAS_WORKMEMORY char viewerThumbnailRawImage [ScreenShotWorkMemorySize_ScreenShotViewerThumbnailRawImage];
                NN_CAPSRV_ALIGNAS_WORKMEMORY char jpegData                [ScreenShotWorkMemorySize_ScreenShotJpegData];
                NN_CAPSRV_ALIGNAS_WORKMEMORY char overlayThumbnailRawData [ScreenShotWorkMemorySize_ScreenShotOverlayThumbnailRawData];
                NN_CAPSRV_ALIGNAS_WORKMEMORY char viewerThumbnailRawData  [ScreenShotWorkMemorySize_ScreenShotViewerThumbnailRawData];
                NN_CAPSRV_ALIGNAS_WORKMEMORY char viewerThumbnailJpegData [ScreenShotWorkMemorySize_ScreenShotViewerThumbnailJpegData];
                NN_CAPSRV_ALIGNAS_WORKMEMORY char reserved                [ScreenShotWorkMemorySize_ScreenShotDeviceSharedReserved];
            } screenShot;

            struct KeepPresentation
            {
                NN_CAPSRV_ALIGNAS_WORKMEMORY char fullSizeRawImageRgba[ScreenShotWorkMemorySize_KeepPresentationFullSizeRawImage];
            } keepPresentation;
        } deviceShared;

        // デバイスマップされない領域
        // 他のプロセスから書き換えられることがないので安全。
        union CpuOnly
        {
            struct ScreenShot
            {
                NN_CAPSRV_ALIGNAS_WORKMEMORY char exifWorkMemory          [ScreenShotWorkMemorySize_ScreenShotExifWorkMemory];
                NN_CAPSRV_ALIGNAS_WORKMEMORY char jpegEncoderWorkMemory   [ScreenShotWorkMemorySize_ScreenShotJpegEncoderWorkMemory];
            } screenShot;

        } cpuOnly;
    };

    NN_STATIC_ASSERT(sizeof(ScreenShotWorkMemory::DeviceShared) == ScreenShotWorkMemorySize_ScreenShotCaptureBufferDeviceShared);
    NN_STATIC_ASSERT(sizeof(ScreenShotWorkMemory::DeviceShared::ScreenShot) == ScreenShotWorkMemorySize_ScreenShotDeviceShared + ScreenShotWorkMemorySize_ScreenShotDeviceSharedReserved);

}}}
