﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_ScreenShotControlServiceImpl.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/capsrv/capsrv_Result.h>
#include "../capsrvServer_Config.h"
#include "../capsrvServer_ConvertResult.h"
#include "../../capsrv_Macro.h"

#include "visrv_ScreenShotContextMacro.h"
#include "visrv_ScreenShotServerObject.h"
#include "../screenshot/visrv_ScreenShotContext.h"
#include "../screenshot/visrv_TakeScreenShot2.h"
#include "../screenshot/visrv_RawScreenShotReadStream.h"
#include "../screenshot/visrv_SaveApplicationScreenShot.h"
#include "../screenshot/visrv_SetupOverlayMovieThumbnail.h"
#include "../screenshot/visrv_CreateProtoMovieMetaDataNv12.h"
#include "../screenshot/visrv_CreateProtoMovieMetaDataRgba.h"
#include "../detail/capsrvServer_ScreenShotAttributeUtility.h"
#include "../../capture/capsrv_MemoryPool.h"
#include "../../capture/capsrv_ImageBuffer.h"
#include "../capsrvServer_ImplUtility.h"
#include "visrv_ScreenShotServiceImpl.h"

#include "../../../Eris/Sources/Libraries/visrv/local/visrv_LocalApi.h"

#if defined(NN_CAPSRV_USE_HIPC)

#define NN_CAPSRV_SERVER_DO(Function)  \
    NN_RESULT_DO(g_ScreenShotErrorConverter.Convert(Function()));

#define NN_CAPSRV_SERVER_SYNC_DO(Function)  \
    NN_RESULT_DO(g_ScreenShotErrorConverter.Convert(g_ScreenShotRequestQueue.EnqueueSync(::nn::capsrv::server::MakeFsPriorityInheritedFunction(Function))));

#define NN_CAPSRV_SERVER_ASYNC_DO(Function) \
    NN_RESULT_DO(g_ScreenShotErrorConverter.Convert(g_ScreenShotControlRequestQueue.TryEnqueueAsync(Function)));

#elif defined(NN_CAPSRV_USE_DIRECT_FUNCTION_CALL)

#define NN_CAPSRV_SERVER_DO(Function)  \
    NN_RESULT_DO(g_ScreenShotErrorConverter.Convert(Function()));
#define NN_CAPSRV_SERVER_SYNC_DO  NN_CAPSRV_SERVER_DO
#define NN_CAPSRV_SERVER_ASYNC_DO NN_CAPSRV_SERVER_DO

#endif

namespace nn{ namespace capsrv{ namespace server{

    nn::Result ScreenShotControlServiceImpl::CaptureRawImageRgba32IntoArray(const nn::sf::OutBuffer&, int32_t, std::int64_t, std::int64_t, std::int64_t, std::int64_t) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SC(CaptureRawImageRgba32IntoArray);
        NN_RESULT_THROW(nn::capsrv::ResultNotSupported());
    }

    nn::Result ScreenShotControlServiceImpl::CaptureRawImageRgba32IntoArrayWithTimeout(const nn::sf::OutBuffer&, int32_t, std::int64_t, std::int64_t, std::int64_t, std::int64_t, nn::TimeSpan) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SC(CaptureRawImageRgba32IntoArrayWithTimeout);
        NN_RESULT_THROW(nn::capsrv::ResultNotSupported());
    }

    nn::Result ScreenShotControlServiceImpl::AttachSharedBufferToCaptureModule(nn::vi::fbshare::SharedBufferHandle hBuffer) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SC(AttachSharedBufferToCaptureModule);

        // vi から共有バッファを取得
        capture::MemoryPoolImportData poolInfo;
        nn::vi::fbshare::SharedMemoryPoolLayout layout;
        {
            auto function = [&]() -> nn::Result
            {
                NN_RESULT_DO(nn::visrv::local::GetSharedMemoryPoolInfo(
                    &poolInfo.hMem,
                    &poolInfo.memory,
                    &poolInfo.size,
                    &layout,
                    hBuffer
                ));
                NN_RESULT_SUCCESS;
            };

            NN_RESULT_DO(nn::visrv::local::EnqueueLocalRequestSync(function));
        }

        // キャプチャモジュールにアタッチ
        (void)g_TransitionCaptureModule.AttachSharedMemoryPool(poolInfo, layout);
        NN_CAPSRV_LOG_INFO("Shared buffer %llu is attached to capture module\n", hBuffer);

        NN_RESULT_SUCCESS;
    }

    nn::Result ScreenShotControlServiceImpl::CaptureRawImageToAttachedSharedBuffer(std::int32_t dstIndex, std::int32_t layerStack, nn::TimeSpan timeout) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SC(CaptureRawImageToAttachedSharedBuffer);

        NN_RESULT_DO(g_TransitionCaptureModule.CaptureDisplayToSharedTexture(
            static_cast<int>(dstIndex),
            static_cast<nn::vi::LayerStack>(layerStack),
            timeout
        ));

        NN_RESULT_SUCCESS;
    }

    nn::Result ScreenShotControlServiceImpl::SaveScreenShotEx2ViaAm(
        nn::sf::Out<ApplicationAlbumEntry> pOutEntry,
        nn::sf::InBuffer imageBuffer,
        const capsrv::detail::ScreenShotAttributeEx0& attribute,
        const UserIdList& userIdList,
        applet::AppletResourceUserId aruid,
        OverlayNotificationRequestType overlayRequest
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SC(SaveScreenShotEx2ViaAm);
        nn::capsrv::ScreenShotAttribute attr = {};
        NN_RESULT_DO(detail::ScreenShotAttributeUtility::ImportAttribute(
            &attr,
            attribute,
            nn::capsrv::AlbumFileDescription_ScreenShotSaved
        ));
        return ScreenShotServiceImpl::SaveScreenShotImpl(pOutEntry, imageBuffer, attr, userIdList, aruid, overlayRequest);
    }

    //-------------------------------------------------------------------------------------------

    nn::Result ScreenShotControlServiceImpl::RequestTakingScreenShotImpl(
        uint64_t seqNo,
        nn::ncm::ProgramId programId,
        nn::applet::AppletResourceUserId aruid,
        nn::TimeSpan timeout,
        const nn::capsrv::ScreenShotAttribute& attribute,
        const UserIdList& userIdList
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_SERVER_ASYNC_DO((
            [seqNo, programId, attribute, userIdList, timeout]() -> nn::Result {
                NN_CAPSRV_LOG_INFO("Taking ScreenShot ... 0x%016llX\n", programId.value);
                NN_CAPSRV_SERVER_SCREENSHOT_CONTEXT(pContext);
                nn::ncm::ApplicationId applicationId = {programId.value};
                auto layerStack = g_ScreenShotCaptureModule.GetDefaultScreenShotLayerStack();
                g_ScreenShotContextManager.SetupContextForTakingScreenShot(pContext, seqNo, applicationId, layerStack, attribute, userIdList, timeout);
                NN_RESULT_DO(screenshot::TakeScreenShot(*pContext));
                NN_RESULT_SUCCESS;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result ScreenShotControlServiceImpl::RequestTakingScreenShotEx1(
        uint64_t seqNo,
        nn::ncm::ProgramId programId,
        nn::applet::AppletResourceUserId aruid,
        nn::TimeSpan timeout,
        nn::capsrv::detail::ScreenShotAttributeEx0 attribute,
        const UserIdList& userIdList
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SC(RequestTakingScreenShotEx1);
        nn::capsrv::ScreenShotAttribute attr = {};
        NN_RESULT_DO(detail::ScreenShotAttributeUtility::ImportAttribute(
            &attr,
            attribute,
            nn::capsrv::AlbumFileDescription_ScreenShotCaptured
        ));
        return RequestTakingScreenShotImpl(seqNo, programId, aruid, timeout, attr, userIdList);
    }

    nn::Result ScreenShotControlServiceImpl::CancelTakingScreenShot(
        uint64_t seqNo,
        nn::Result reason
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SC(CancelTakingScreenShot);
        g_ScreenShotSequenceManager.RequestCancelSequence(seqNo, reason);
        NN_RESULT_SUCCESS;
    }

    nn::Result ScreenShotControlServiceImpl::SetTakingScreenShotCancelState(
        uint64_t seqNo,
        nn::Result reason
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SC(SetTakingScreenShotCancelState);
        g_ScreenShotSequenceManager.SetCancelState(seqNo, reason);
        NN_RESULT_SUCCESS;
    }

    nn::Result ScreenShotControlServiceImpl::NotifyTakingScreenShotRefused(
        nn::ncm::ProgramId programId
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SC(NotifyTakingScreenShotRefused);
        nn::ovln::format::ScreenShotCapturedData data = {};
        data.result = nn::ovln::format::ScreenShotResult::NotPermitted;
        data.reason = nn::ovln::format::ScreenShotReason::CaptureButtonPressed;
        reinterpret_cast<AlbumEntry*>(&data._entry)->fileId.applicationId.value = programId.value;
        NN_RESULT_DO(g_OverlayNotifier.SendToOverlay(data));
        NN_RESULT_SUCCESS;
    }

    nn::Result ScreenShotControlServiceImpl::NotifyTakingScreenShotFailed(
        nn::ncm::ProgramId programId
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SC(NotifyTakingScreenShotFailed);
        nn::ovln::format::ScreenShotCapturedData data = {};
        //data.result = nn::ovln::format::ScreenShotResult::Failure;
        data.result = nn::ovln::format::ScreenShotResult::ScreenCaptureError;
        data.reason = nn::ovln::format::ScreenShotReason::CaptureButtonPressed;
        reinterpret_cast<AlbumEntry*>(&data._entry)->fileId.applicationId.value = programId.value;
        NN_RESULT_DO(g_OverlayNotifier.SendToOverlay(data));
        NN_RESULT_SUCCESS;
    }

    nn::Result ScreenShotControlServiceImpl::OpenRawScreenShotReadStream(
        nn::sf::Out<std::int64_t> outSize,
        nn::sf::Out<std::int64_t> outWidth,
        nn::sf::Out<std::int64_t> outHeight,
        std::int32_t layerStack,
        nn::TimeSpan timeout
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SC(OpenRawScreenShotReadStream);

        NN_RESULT_THROW_UNLESS(g_ScreenShotEnvironmentInfo.IsRawScreenShotReadStreamEnabled(), ResultControlResourceLimit());

        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                NN_CAPSRV_LOG_INFO("Opening RawScreenShotReadStream (layer %d)\n", static_cast<int>(layerStack));
                NN_CAPSRV_SERVER_SCREENSHOT_CONTEXT(pContext);
                g_ScreenShotContextManager.SetupContextForRawScreenShotReadStream(
                    pContext, static_cast<nn::vi::LayerStack>(layerStack), timeout
                );
                NN_RESULT_DO(screenshot::OpenRawScreenShotReadStream(*pContext));
                NN_RESULT_SUCCESS;
            }
        ));

        outSize.Set(4 * ScreenShotWidth * ScreenShotHeight);
        outWidth.Set(ScreenShotWidth);
        outHeight.Set(ScreenShotHeight);
        NN_RESULT_SUCCESS;

    }

    nn::Result ScreenShotControlServiceImpl::CloseRawScreenShotReadStream() NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SC(CloseRawScreenShotReadStream);

        NN_RESULT_THROW_UNLESS(g_ScreenShotEnvironmentInfo.IsRawScreenShotReadStreamEnabled(), ResultControlResourceLimit());

        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                NN_CAPSRV_LOG_INFO("Closing RawScreenShotReadStream\n");
                NN_CAPSRV_SERVER_RESUME_SCREENSHOT_CONTEXT(pContext, ScreenShotContextSuspendedTask_RawScreenShotReadStream);
                NN_RESULT_DO(screenshot::CloseRawScreenShotReadStream(*pContext));
                NN_RESULT_SUCCESS;
            }
        ));

        NN_RESULT_SUCCESS;
    }

    nn::Result ScreenShotControlServiceImpl::ReadRawScreenShotReadStream(nn::sf::Out<std::int64_t> outReadSize, const nn::sf::OutBuffer& outBuffer, std::int64_t offset) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SC(ReadRawScreenShotReadStream);

        NN_RESULT_THROW_UNLESS(g_ScreenShotEnvironmentInfo.IsRawScreenShotReadStreamEnabled(), ResultControlResourceLimit());

        size_t readSize = 0;
        void* pBuffer = outBuffer.GetPointerUnsafe();
        size_t bufferSize = outBuffer.GetSize();

        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                NN_CAPSRV_SERVER_RESUME_SCREENSHOT_CONTEXT(pContext, ScreenShotContextSuspendedTask_RawScreenShotReadStream);
                NN_RESULT_DO(screenshot::ReadRawScreenShotReadStream(&readSize, pBuffer, bufferSize, static_cast<ptrdiff_t>(offset), *pContext));
                NN_RESULT_SUCCESS;
            }
        ));

        outReadSize.Set(static_cast<int64_t>(readSize));
        NN_RESULT_SUCCESS;
    }


    //-------------------------------------------------------------------------------------------

    struct ImageBufferNv12
    {
    public:
        nn::Result Initialize(
            capture::CaptureModule* pModule,
            void* bufY,
            size_t bufYSize,
            void* bufUv,
            size_t bufUvSize,
            int width,
            int height
        ) NN_NOEXCEPT
        {
            NN_CAPSRV_PROCESS_START();

            NN_RESULT_DO(this->memoryPoolY.Initialize(pModule, bufY, bufYSize));
            NN_CAPSRV_PROCESS_ROLLBACK(this->memoryPoolY.Finalize());

            NN_RESULT_DO(this->memoryPoolUv.Initialize(pModule, bufUv, bufUvSize));
            NN_CAPSRV_PROCESS_ROLLBACK(this->memoryPoolUv.Finalize());

            {
                capture::ImageBufferInfo info = {};
                info.width  = static_cast<uint32_t>(width);
                info.height = static_cast<uint32_t>(height);
                info.format = capture::ImageFormat_Y_NV12;
                size_t size = capture::ImageBuffer::GetRequiredMemorySize(info);
                NN_ABORT_UNLESS_GREATER_EQUAL(bufYSize, size);
                NN_RESULT_DO(this->imageBufferY.Initialize(pModule, info, &memoryPoolY, 0, size));
            }
            NN_CAPSRV_PROCESS_ROLLBACK(imageBufferY.Finalize());

            {
                capture::ImageBufferInfo info = {};
                info.width  = static_cast<uint32_t>(width) / 2;
                info.height = static_cast<uint32_t>(height) / 2;
                info.format = capture::ImageFormat_Uv_NV12;
                size_t size = capture::ImageBuffer::GetRequiredMemorySize(info);
                NN_ABORT_UNLESS_GREATER_EQUAL(bufUvSize, size);
                NN_RESULT_DO(this->imageBufferUv.Initialize(pModule, info, &memoryPoolUv, 0, size));
            }
            NN_CAPSRV_PROCESS_ROLLBACK(imageBufferUv.Finalize());

            NN_CAPSRV_PROCESS_SUCCESS();
            NN_RESULT_SUCCESS;
        }

        void Finalize() NN_NOEXCEPT
        {
            this->imageBufferUv.Finalize();
            this->imageBufferY.Finalize();
            this->memoryPoolUv.Finalize();
            this->memoryPoolY.Finalize();
        }

    public:
        capture::MemoryPool memoryPoolY;
        capture::MemoryPool memoryPoolUv;
        capture::ImageBuffer imageBufferY;
        capture::ImageBuffer imageBufferUv;
    };

    nn::Result ScreenShotControlServiceImpl::SetupOverlayMovieThumbnail(const nn::sf::OutBuffer& imageY, const nn::sf::OutBuffer& imageUv, std::int64_t width, std::int64_t height, const nn::capsrv::AlbumFileId& fileId) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SC(SetupOverlayMovieThumbnail);
        const size_t requiredSizeY  = static_cast<size_t>(width * height);     // 1 byte/px
        const size_t requiredSizeUv = static_cast<size_t>(width * height / 2); // 1 単位 2 バイトだがサイズ 1/4 なので平均 0.5 byte/px

        void* bufY = imageY.GetPointerUnsafe();
        size_t bufYSize = imageY.GetSize();
        void* bufUv = imageUv.GetPointerUnsafe();
        size_t bufUvSize = imageUv.GetSize();

        NN_RESULT_THROW_UNLESS(width == MovieWidth, ResultScreenShotInvalidSize());
        NN_RESULT_THROW_UNLESS(height == MovieHeight, ResultScreenShotInvalidSize());
        NN_RESULT_THROW_UNLESS(bufY != nullptr, ResultScreenShotInvalidData());
        NN_RESULT_THROW_UNLESS(reinterpret_cast<uintptr_t>(bufY) % capture::MemoryPool::GetRequiredAlignment() == 0, ResultScreenShotInvalidData());
        NN_RESULT_THROW_UNLESS(bufYSize >= requiredSizeY, ResultScreenShotInvalidSize());
        NN_RESULT_THROW_UNLESS(bufYSize % capture::MemoryPool::GetRequiredUnitSize() == 0, ResultScreenShotInvalidSize());
        NN_RESULT_THROW_UNLESS(bufUv != nullptr, ResultScreenShotInvalidData());
        NN_RESULT_THROW_UNLESS(reinterpret_cast<uintptr_t>(bufUv) % capture::MemoryPool::GetRequiredAlignment() == 0, ResultScreenShotInvalidData());
        NN_RESULT_THROW_UNLESS(bufUvSize >= requiredSizeUv, ResultScreenShotInvalidSize());
        NN_RESULT_THROW_UNLESS(bufUvSize % capture::MemoryPool::GetRequiredUnitSize() == 0, ResultScreenShotInvalidSize());

        auto pModule = g_ScreenShotCaptureModule.GetModule();

        ImageBufferNv12 imgbuf;
        NN_RESULT_DO(imgbuf.Initialize(pModule, bufY, bufYSize, bufUv, bufUvSize, static_cast<int>(width), static_cast<int>(height)));
        NN_UTIL_SCOPE_EXIT{ imgbuf.Finalize(); };

        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                NN_CAPSRV_SERVER_SCREENSHOT_CONTEXT(pContext);
                g_ScreenShotContextManager.SetupContextForOverlayMovieThumbnail(
                    pContext, &imgbuf.imageBufferY, &imgbuf.imageBufferUv, fileId
                );
                NN_RESULT_DO(screenshot::SetupOverlayMovieThumbnail(*pContext));
                NN_RESULT_SUCCESS;
            }
        ));

        NN_RESULT_SUCCESS;
    }

    //------------------------------------------------------------------------------

    nn::Result ScreenShotControlServiceImpl::CreateProtoMovieMetaDataNv12Impl(
        nn::sf::Out<std::uint64_t> outSize,
        nn::sf::Out<std::int64_t> outMakerNoteOffset,
        nn::sf::Out<std::int64_t> outMakerNoteSize,
        const nn::sf::OutBuffer& outBuffer,
        const nn::sf::OutBuffer& imageY,
        const nn::sf::OutBuffer& imageUv,
        std::int64_t width,
        std::int64_t height,
        const nn::capsrv::AlbumFileId& fileId,
        const nn::capsrv::ScreenShotAttribute& attribute,
        const nn::capsrv::AppletData* pAppletData,
        const nn::capsrv::ApplicationData* pApplicationData
    ) NN_NOEXCEPT
    {
        const size_t requiredSizeY  = static_cast<size_t>(width * height);     // 1 byte/px
        const size_t requiredSizeUv = static_cast<size_t>(width * height / 2); // 1 単位 2 バイトだがサイズ 1/4 なので平均 0.5 byte/px

        void* outBuf = outBuffer.GetPointerUnsafe();
        size_t outBufSize = outBuffer.GetSize();
        void* bufY = imageY.GetPointerUnsafe();
        size_t bufYSize = imageY.GetSize();
        void* bufUv = imageUv.GetPointerUnsafe();
        size_t bufUvSize = imageUv.GetSize();

        NN_RESULT_THROW_UNLESS(outBuf != nullptr, ResultScreenShotError());
        NN_RESULT_THROW_UNLESS(outBufSize >= MovieMetaImageDataSize, ResultScreenShotError());
        NN_RESULT_THROW_UNLESS(width == MovieWidth, ResultScreenShotInvalidSize());
        NN_RESULT_THROW_UNLESS(height == MovieHeight, ResultScreenShotInvalidSize());
        NN_RESULT_THROW_UNLESS(bufY != nullptr, ResultScreenShotInvalidData());
        NN_RESULT_THROW_UNLESS(reinterpret_cast<uintptr_t>(bufY) % capture::MemoryPool::GetRequiredAlignment() == 0, ResultScreenShotInvalidData());
        NN_RESULT_THROW_UNLESS(bufYSize >= requiredSizeY, ResultScreenShotInvalidSize());
        NN_RESULT_THROW_UNLESS(bufYSize % capture::MemoryPool::GetRequiredUnitSize() == 0, ResultScreenShotInvalidSize());
        NN_RESULT_THROW_UNLESS(bufUv != nullptr, ResultScreenShotInvalidData());
        NN_RESULT_THROW_UNLESS(reinterpret_cast<uintptr_t>(bufUv) % capture::MemoryPool::GetRequiredAlignment() == 0, ResultScreenShotInvalidData());
        NN_RESULT_THROW_UNLESS(bufUvSize >= requiredSizeUv, ResultScreenShotInvalidSize());
        NN_RESULT_THROW_UNLESS(bufUvSize % capture::MemoryPool::GetRequiredUnitSize() == 0, ResultScreenShotInvalidSize());

        NN_RESULT_THROW_UNLESS(
            attribute.description == nn::capsrv::AlbumFileDescription_MovieContinuous ||
            attribute.description == nn::capsrv::AlbumFileDescription_MovieTrimmed ||
            attribute.description == nn::capsrv::AlbumFileDescription_MovieMakerSaved,
            ResultScreenShotInvalidData()
        );

        NN_CAPSRV_PROCESS_START();

        // 出力バッファをクリア
        std::memset(outBuf, 0, outBufSize);
        NN_CAPSRV_PROCESS_ROLLBACK(std::memset(outBuf, 0, outBufSize));

        auto pModule = g_ScreenShotCaptureModule.GetModule();

        ImageBufferNv12 imgbuf;
        NN_RESULT_DO(imgbuf.Initialize(pModule, bufY, bufYSize, bufUv, bufUvSize, static_cast<int>(width), static_cast<int>(height)));
        NN_UTIL_SCOPE_EXIT{ imgbuf.Finalize(); };

        int64_t size = 0;
        int64_t makerNoteOffset = 0;
        int64_t makerNoteSize = 0;

        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                NN_CAPSRV_SERVER_SCREENSHOT_CONTEXT(pContext);
                g_ScreenShotContextManager.SetupContextForProtoMovieMetaDataNv12(
                    pContext, outBuf, outBufSize, &imgbuf.imageBufferY, &imgbuf.imageBufferUv, fileId, attribute, pAppletData, pApplicationData
                );
                NN_RESULT_DO(screenshot::CreateProtoMovieMetaDataNv12(*pContext));
                {
                    auto& context = *pContext;
                    NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
                    NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, FileSize);
                    NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, MakerNoteOffset);
                    NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, MakerNoteSize);
                    NN_CAPSRV_SCREENSHOT_CONTEXT_GET(fileSize, context, FileSize);
                    NN_CAPSRV_SCREENSHOT_CONTEXT_GET(mnoteOffset, context, MakerNoteOffset);
                    NN_CAPSRV_SCREENSHOT_CONTEXT_GET(mnoteSize, context, MakerNoteSize);
                    size = fileSize;
                    makerNoteOffset = mnoteOffset;
                    makerNoteSize = mnoteSize;
                }
                NN_RESULT_SUCCESS;
            }
        ));

        NN_CAPSRV_PROCESS_SUCCESS();
        outSize.Set(static_cast<uint64_t>(size));
        outMakerNoteOffset.Set(makerNoteOffset);
        outMakerNoteSize.Set(makerNoteSize);
        NN_RESULT_SUCCESS;
    }

    nn::Result ScreenShotControlServiceImpl::CreateProtoMovieMetaDataNv12Ex2(
        nn::sf::Out<std::uint64_t> outSize,
        nn::sf::Out<std::int64_t> outMakerNoteOffset,
        nn::sf::Out<std::int64_t> outMakerNoteSize,
        const nn::sf::OutBuffer& outBuffer,
        const nn::sf::OutBuffer& imageY,
        const nn::sf::OutBuffer& imageUv,
        std::int64_t width,
        std::int64_t height,
        const nn::capsrv::AlbumFileId& fileId,
        const nn::capsrv::detail::ScreenShotAttributeEx0& attribute,
        const nn::capsrv::AppletData& appletData,
        const nn::capsrv::ApplicationData& applicationData
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SC(CreateProtoMovieMetaDataNv12Ex2);

        nn::capsrv::ScreenShotAttribute attr = {};
        NN_RESULT_DO(detail::ScreenShotAttributeUtility::ImportAttribute(
            &attr,
            attribute,
            attribute.description // description は grc が設定した値で付ける
        ));

        return CreateProtoMovieMetaDataNv12Impl(outSize, outMakerNoteOffset, outMakerNoteSize, outBuffer, imageY, imageUv, width, height, fileId, attr, &appletData, &applicationData);
    }

    nn::Result ScreenShotControlServiceImpl::CreateProtoMovieMetaDataRgbaEx2(
        nn::sf::Out<std::uint64_t> outSize,
        nn::sf::Out<std::int64_t> outMakerNoteOffset,
        nn::sf::Out<std::int64_t> outMakerNoteSize,
        const nn::sf::OutBuffer& outMeta,
        const nn::sf::InBuffer& imageRgba,
        std::int64_t width, std::int64_t height,
        const nn::capsrv::AlbumFileId& fileId,
        const nn::capsrv::detail::ScreenShotAttributeEx0& attribute,
        const nn::capsrv::AppletData& appletData,
        const nn::capsrv::ApplicationData& applicationData
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SC(CreateProtoMovieMetaDataRgbaEx2);

        void* outBuf = outMeta.GetPointerUnsafe();
        size_t outBufSize = outMeta.GetSize();
        const void* pImageData = imageRgba.GetPointerUnsafe();
        size_t imageDataSize = imageRgba.GetSize();


        NN_RESULT_THROW_UNLESS(outBuf != nullptr, ResultScreenShotError());
        NN_RESULT_THROW_UNLESS(outBufSize >= MovieMetaImageDataSize, ResultScreenShotError());
        NN_RESULT_THROW_UNLESS(width == MovieWidth, ResultScreenShotInvalidSize());
        NN_RESULT_THROW_UNLESS(height == MovieHeight, ResultScreenShotInvalidSize());
        NN_RESULT_THROW_UNLESS(pImageData != nullptr, ResultScreenShotInvalidData());
        NN_RESULT_THROW_UNLESS(imageDataSize >= 4 * ScreenShotWidth * ScreenShotHeight, ResultScreenShotInvalidData());

        NN_RESULT_THROW_UNLESS(
            attribute.description == nn::capsrv::AlbumFileDescription_MovieContinuous ||
            attribute.description == nn::capsrv::AlbumFileDescription_MovieTrimmed ||
            attribute.description == nn::capsrv::AlbumFileDescription_MovieMakerSaved,
            ResultScreenShotInvalidData()
        );

        nn::capsrv::ScreenShotAttribute attr = {};
        NN_RESULT_DO(detail::ScreenShotAttributeUtility::ImportAttribute(
            &attr,
            attribute,
            attribute.description // description は grc が設定した値で付ける
        ));

        NN_CAPSRV_PROCESS_START();

        // 出力バッファをクリア
        std::memset(outBuf, 0, outBufSize);
        NN_CAPSRV_PROCESS_ROLLBACK(std::memset(outBuf, 0, outBufSize));

        int64_t size = 0;
        int64_t makerNoteOffset = 0;
        int64_t makerNoteSize = 0;

        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                NN_CAPSRV_SERVER_SCREENSHOT_CONTEXT(pContext);
                g_ScreenShotContextManager.SetupContextForProtoMovieMetaDataRgba(
                    pContext, outBuf, outBufSize, pImageData, imageDataSize, fileId, attr, &appletData, &applicationData
                );
                NN_RESULT_DO(screenshot::CreateProtoMovieMetaDataRgba(*pContext));
                {
                    auto& context = *pContext;
                    NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
                    NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, FileSize);
                    NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, MakerNoteOffset);
                    NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, MakerNoteSize);
                    NN_CAPSRV_SCREENSHOT_CONTEXT_GET(fileSize, context, FileSize);
                    NN_CAPSRV_SCREENSHOT_CONTEXT_GET(mnoteOffset, context, MakerNoteOffset);
                    NN_CAPSRV_SCREENSHOT_CONTEXT_GET(mnoteSize, context, MakerNoteSize);
                    size = fileSize;
                    makerNoteOffset = mnoteOffset;
                    makerNoteSize = mnoteSize;
                }
                NN_RESULT_SUCCESS;
            }
        ));

        NN_CAPSRV_PROCESS_SUCCESS();
        outSize.Set(static_cast<uint64_t>(size));
        outMakerNoteOffset.Set(makerNoteOffset);
        outMakerNoteSize.Set(makerNoteSize);
        NN_RESULT_SUCCESS;
    }

}}}
