﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/capsrv/capsrv_Result.h>
#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>
#include "../visrv_ScreenShotContext.h"
#include "visrv_ActionTimeMeasure.h"

#include <nn/capsrv/capsrv_ViewerThumbnailFormat.h>
#include "../../capsrvServer_Config.h"
#include "../visrv_ScreenShotUtility.h"

namespace nn{ namespace capsrv{ namespace server{ namespace screenshot{ namespace action{

    // 画面写真を JPEG エンコードする
    // @param[in]  context.DisplayBufferYuv
    // @param[in]  context.ExifBuilder
    // @param[out] context.FileDataMemory
    // @param[out] context.FileSize
    // @param      context.EncoderWorkMemory
    static nn::Result EncodeScreenShotJpegYuv(ScreenShotContext& context) NN_NOEXCEPT
    {
        NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, DisplayBufferYuv);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, ExifBuilder);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_MEMORY(context, FileDataMemory, AlbumFileSizeLimit_ScreenShot);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_MEMORY(context, EncoderWorkMemory, SoftwareJpegEncoderWorkMemorySize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OVERWRITEABLE(context, FileSize);

        NN_CAPSRV_SCREENSHOT_ACTION_TIMEMEASURE("encodeYUV ");

        NN_CAPSRV_SCREENSHOT_CONTEXT_GET_MEMORY(pEncodedBuffer, encodedBufferSize, context, FileDataMemory);
        std::memset(pEncodedBuffer, 0, encodedBufferSize);

        NN_CAPSRV_SCREENSHOT_CONTEXT_GET_MEMORY(pWork, workSize, context, EncoderWorkMemory);
        std::memset(pWork, 0, workSize);

        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(pDisplayBuffer, context, DisplayBufferYuv);
        const capture::ImageBuffer* pImageBufferY = nullptr;
        const capture::ImageBuffer* pImageBufferU = nullptr;
        const capture::ImageBuffer* pImageBufferV = nullptr;
        pDisplayBuffer->GetImageBufferYuvPlanar(&pImageBufferY, &pImageBufferU, &pImageBufferV);

        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(pExifBuilder, context, ExifBuilder);

        class ImageSubRegion
        {
        public:
            ImageSubRegion(const capture::ImageBuffer* pY, const capture::ImageBuffer* pU, const capture::ImageBuffer* pV) NN_NOEXCEPT
                : m_pImageBufferY(pY), m_pImageBufferU(pU), m_pImageBufferV(pV)
            {
            }

            static nn::Result Invoke(const jpeg::SoftwareJpegEncoderYuvBuffer& outBuffer, capture::ImageFormat format, uint32_t x, uint32_t y, uint32_t width, uint32_t height, void* userPtr) NN_NOEXCEPT
            {
                auto pArg = reinterpret_cast<ImageSubRegion*>(userPtr);

                const uint32_t horizontalSubsamplingShift = GetHorizontalSubsamplingShift(format);
                const uint32_t verticalSubsamplingShift   = GetVerticalSubsamplingShift(format);
                NN_SDK_REQUIRES_EQUAL(x      % (1 << horizontalSubsamplingShift), 0u);
                NN_SDK_REQUIRES_EQUAL(y      % (1 << verticalSubsamplingShift),   0u);
                NN_SDK_REQUIRES_EQUAL(width  % (1 << horizontalSubsamplingShift), 0u);
                NN_SDK_REQUIRES_EQUAL(height % (1 << verticalSubsamplingShift),   0u);

                const capture::Rectangle  rectY = capture::MakeRectangle(x, y, width, height);
                const capture::Rectangle  rectU = capture::MakeRectangle(x      >> horizontalSubsamplingShift,
                                                                         y      >> verticalSubsamplingShift,
                                                                         width  >> horizontalSubsamplingShift,
                                                                         height >> verticalSubsamplingShift);
                const capture::Rectangle& rectV = rectU;

                size_t sizeY;
                size_t sizeU;
                size_t sizeV;
                NN_RESULT_DO(pArg->m_pImageBufferY->WriteToMemory(&sizeY, outBuffer.pBufferY, outBuffer.bufferSizeY, nn::capsrv::capture::ImageFormat_Y_Bt601, &rectY));
                NN_RESULT_DO(pArg->m_pImageBufferU->WriteToMemory(&sizeU, outBuffer.pBufferU, outBuffer.bufferSizeU, nn::capsrv::capture::ImageFormat_U_Bt601, &rectU));
                NN_RESULT_DO(pArg->m_pImageBufferV->WriteToMemory(&sizeV, outBuffer.pBufferV, outBuffer.bufferSizeV, nn::capsrv::capture::ImageFormat_V_Bt601, &rectV));
                NN_RESULT_SUCCESS;
            }
        private:
            const capture::ImageBuffer* m_pImageBufferY;
            const capture::ImageBuffer* m_pImageBufferU;
            const capture::ImageBuffer* m_pImageBufferV;
        };

        auto subRegion = ImageSubRegion(pImageBufferY, pImageBufferU, pImageBufferV);

        jpeg::SoftwareJpegEncoderStreamInputInfoYuv inputInfo;
        inputInfo.width  = ScreenShotWidth;
        inputInfo.height = ScreenShotHeight;
        inputInfo.format = pDisplayBuffer->GetImageFormat();
        inputInfo.pGetSubRegionFunction = ImageSubRegion::Invoke;
        inputInfo.pGetSubRegionUserPtr  = &subRegion;

        size_t encodedSize = 0;
        jpeg::SoftwareJpegEncoderBufferOutputInfo outputInfo;
        outputInfo.pOutSize   = &encodedSize;
        outputInfo.pBuffer    = pEncodedBuffer;
        outputInfo.bufferSize = (encodedBufferSize < AlbumFileSizeLimit_ScreenShot) ? encodedBufferSize : AlbumFileSizeLimit_ScreenShot;

        NN_RESULT_DO(ScreenShotUtility::EncodeYuvStreamToBuffer(outputInfo, inputInfo, pExifBuilder, pWork, workSize));
        NN_ABORT_UNLESS_GREATER_EQUAL(encodedBufferSize, encodedSize);

        NN_CAPSRV_SCREENSHOT_CONTEXT_OVERWRITE(context, FileSize, static_cast<int64_t>(encodedSize));
        NN_RESULT_SUCCESS;
    }

}}}}}

