﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/capsrv/capsrv_Result.h>
#include <nn/capsrv/capsrv_ViewerThumbnailFormat.h>
#include "../visrv_ScreenShotContext.h"
#include "visrv_ActionTimeMeasure.h"

#include "../../detail/visrv_ConstructExifBuilder.h"
#include "../../detail/capsrvServer_EncryptApplicationId.h"
#include "../../detail/capsrvServer_EncryptedApplicationId.h"
#include "../../detail/capsrvServer_ScreenShotAttributeUtility.h"

#include <cstring>

namespace nn{ namespace capsrv{ namespace server{ namespace screenshot{ namespace action{

    // ExifBuilder を構築する
    // @param[in]  context.Orientation
    // @param[in]  context.AppletData
    // @param[in]  context.MakerNoteVersion
    // @param[in]  context.AlbumFileId
    // @param[in]  context.EncodedViewerThumbnailMemory
    // @param[in]  context.EncodedViewerThumbnailSize
    // @param[in]  context.ExifMemory
    // @param[out] context.ExifBuilder
    // @param[out] context.ExifMemory
    static nn::Result CreateExifBuilder(ScreenShotContext& context) NN_NOEXCEPT
    {
        NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, AlbumFileId);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, MakerNoteVersion);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, ScreenShotAttribute);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, EncodedViewerThumbnailSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN_OPTIONAL(context, AppletData);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN_OPTIONAL(context, ApplicationData);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN_OPTIONAL(context, SystemReservedInfo);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_MEMORY(context, EncodedViewerThumbnailMemory, ViewerThumbnailImageDataSize_EncodedLimit);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_MEMORY(context, ExifMemory, ScreenShotExifWorkSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ExifBuilder);

        NN_CAPSRV_SCREENSHOT_ACTION_TIMEMEASURE("exif      ");

        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(fileId, context, AlbumFileId);
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(version, context, MakerNoteVersion);
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(attribute, context, ScreenShotAttribute);
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET_MEMORY(pThumbData, thumbBufferSize, context, EncodedViewerThumbnailMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(thumbSize, context, EncodedViewerThumbnailSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET_OPTIONAL(appletData, context, AppletData);
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET_OPTIONAL(applicationData, context, ApplicationData);
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET_OPTIONAL(systemReservedInfo, context, SystemReservedInfo);

        NN_CAPSRV_SCREENSHOT_CONTEXT_GET_MEMORY(pExifMemory, exifMemorySize, context, ExifMemory);
        std::memset(pExifMemory, 0, exifMemorySize);

        NN_ABORT_UNLESS_GREATER_EQUAL(thumbBufferSize, thumbSize);

        detail::MakerNoteInfo info = {};
        NN_RESULT_DO(detail::ScreenShotAttributeUtility::ConvertToMakerNoteInfo(
            &info,
            version,
            detail::EncryptApplicationId(fileId.applicationId, IsExtraAlbumFileContents(fileId.contents)),
            fileId.time,
            attribute,
            (appletData ? &*appletData : nullptr),
            (applicationData ? &*applicationData : nullptr),
            (systemReservedInfo ? &*systemReservedInfo : nullptr)
        ));

        bool hasThumb = (pThumbData != nullptr && thumbSize > 0);

        auto pExifBuilder = detail::ConstructExifBuilder(
            pExifMemory,
            exifMemorySize,
            info,
            (hasThumb ? pThumbData : nullptr),
            (hasThumb ? thumbSize : 0)
        );
        NN_ABORT_UNLESS_NOT_NULL(pExifBuilder);

        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ExifBuilder, pExifBuilder);
        NN_RESULT_SUCCESS;
    }

    // ExifBuilder を破棄する
    static void DestroyExifBuilder(ScreenShotContext& context) NN_NOEXCEPT
    {
        NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_INOUT(context, ExifBuilder);

        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(p, context, ExifBuilder);
        if(p)
        {
            detail::DestructExifBuilder(p);
        }
        NN_CAPSRV_SCREENSHOT_CONTEXT_RESET(context, ExifBuilder);
    }

}}}}}
