﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_SoftwareJpegEncoderRgba8.h"

#include <algorithm>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "../capsrvServer_Config.h"
#include "../capsrvServer_ResultPrivate.h"
#include "capsrvServer_JpegErrorHandler.h"

#ifdef NN_CAPSRV_USE_LIBJPEG_TURBO
#include "capsrvServer_SoftwareJpegEncoderType-encoder.libjpegTurbo.h"
#else
#include "capsrvServer_SoftwareJpegEncoderType-encoder.libjpeg.h"
#endif

namespace nn{ namespace capsrv{ namespace server{ namespace jpeg{


    namespace {
        static const int RgbaComponentCount = 4;
        static const int RgbComponentCount  = 3;

        void SetupEncodingParameterRgba8(
            JpegEncoderType::jpeg_compress_struct* pCinfo,
            const SoftwareJpegEncoderBufferInputInfoRgba8& inputInfo,
            int quality,
            bool hasExif
        ) NN_NOEXCEPT
        {
            pCinfo->image_width  = static_cast<JpegEncoderType::JDIMENSION>(inputInfo.width);
            pCinfo->image_height = static_cast<JpegEncoderType::JDIMENSION>(inputInfo.height);
            pCinfo->input_components = RgbComponentCount;
            pCinfo->in_color_space   = JpegEncoderType::J_COLOR_SPACE::JCS_RGB;

            jpeg_set_defaults(pCinfo);
            jpeg_set_colorspace(pCinfo, JpegEncoderType::J_COLOR_SPACE::JCS_YCbCr);

            pCinfo->do_fancy_downsampling = false;
            pCinfo->comp_info[0].h_samp_factor = 2;
            pCinfo->comp_info[0].v_samp_factor = 1;
            pCinfo->comp_info[1].h_samp_factor = pCinfo->comp_info[2].h_samp_factor = 1;
            pCinfo->comp_info[1].v_samp_factor = pCinfo->comp_info[2].v_samp_factor = 1;

            jpeg_set_quality(pCinfo, quality, true);

            pCinfo->dct_method            = JpegEncoderType::J_DCT_METHOD::JDCT_ISLOW;
            pCinfo->optimize_coding       = false;

            pCinfo->write_JFIF_header = !hasExif;
        }

        void ProcessEncodeRgba8(
            JpegEncoderType::jpeg_compress_struct* pCinfo,
            uint8_t* pRgbBuffer,
            int rgbStride,
            int lineCount
        ) NN_NOEXCEPT
        {
            JpegEncoderType::JSAMPROW dataRow[SoftwareJpegEncoderRgba8::ImageSizeVerticalUnit] = {};

            for(int y = 0; y < SoftwareJpegEncoderRgba8::ImageSizeVerticalUnit; y++)
            {
                dataRow[y] = reinterpret_cast<JpegEncoderType::JSAMPROW>(pRgbBuffer + rgbStride * y);
            }

            jpeg_write_scanlines(pCinfo, dataRow, lineCount);
        }
    }

    nn::Result SoftwareJpegEncoderRgba8::EncodeRgba8BufferToBufferImpl(
        SoftwareJpegEncoderBufferOutputInfo& outputInfo,
        const SoftwareJpegEncoderBufferInputInfoRgba8& inputInfo,
        int quality,
        bool hasExif,
        void* pWorkBuffer,
        size_t workBufferSize
    ) NN_NOEXCEPT
    {
        char* pWork = reinterpret_cast<char*>(pWorkBuffer);
        char* pWorkEnd = pWork + workBufferSize;

        int rgbBufferWidth = ((inputInfo.width + ImageSizeHorizontalUnit - 1) / ImageSizeHorizontalUnit) * ImageSizeHorizontalUnit;
        size_t rgbBufferSize = RgbComponentCount * rgbBufferWidth * ImageSizeVerticalUnit;
        NN_RESULT_THROW_UNLESS(
            workBufferSize >= rgbBufferSize,
            ResultInternalJpegWorkMemoryShortage()
        );
        uint8_t* pRgbBuffer = reinterpret_cast<uint8_t*>(pWork);
        pWork += rgbBufferSize;

        JpegEncoderType::jpeg_compress_struct cinfo = {};
        JpegEncoderType::jpeg_workbuf workbuf = {pWork, static_cast<size_t>(pWorkEnd - pWork)};
        cinfo.workbuf = &workbuf;
        unsigned long codedSize = static_cast<unsigned long>(outputInfo.bufferSize);

        JpegErrorHandler errorHandler;
        std::memset(&errorHandler, 0, sizeof(errorHandler));
        cinfo.err = jpeg_std_error(&errorHandler);
        errorHandler.error_exit = JpegErrorHandler::HandleError;
        errorHandler.encoderResult = nn::ResultSuccess();

#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning(push)
#pragma warning(disable: 4611)
#endif
        if (setjmp(errorHandler.jmpContext) == 0)
#if defined(NN_BUILD_CONFIG_COMPILER_SUPPORTS_VC)
#pragma warning(pop)
#endif
        {
            jpeg_create_compress(&cinfo);
            NN_UTIL_SCOPE_EXIT{
                jpeg_destroy_compress(&cinfo);
            };

            jpeg_mem_dest(&cinfo, reinterpret_cast<unsigned char**>(&outputInfo.pBuffer), &codedSize);
            SetupEncodingParameterRgba8(&cinfo, inputInfo, quality, hasExif);

            jpeg_start_compress(&cinfo, true);
            for(int y = 0; y < inputInfo.height; y += ImageSizeVerticalUnit)
            {
                int lineCount = std::min(static_cast<int>(ImageSizeVerticalUnit), inputInfo.height - y);
                int rgbaStride = RgbaComponentCount * inputInfo.width;
                int rgbStride  = RgbComponentCount  * rgbBufferWidth;
                auto pRgbaBuffer = reinterpret_cast<const uint8_t*>(inputInfo.pImageData) + rgbaStride * y;
                for(int i = 0; i < lineCount; i++)
                {
                    auto pSrc = pRgbaBuffer + i * rgbaStride;
                    auto pDst = pRgbBuffer  + i * rgbStride;
                    for(int x = 0; x < inputInfo.width; x++)
                    {
                        // copy rgb
                        for(int c = 0; c < 3; c++)
                        {
                            *(pDst++) = *(pSrc++);
                        }
                        // skip a
                        pSrc++;
                    }
                }
                ProcessEncodeRgba8(&cinfo, pRgbBuffer, rgbStride, lineCount);
            }
            jpeg_finish_compress(&cinfo);
            *outputInfo.pOutSize = static_cast<size_t>(codedSize);
        }
        else
        {
            NN_RESULT_THROW(errorHandler.encoderResult);
        }

        NN_RESULT_SUCCESS;
    }

}}}}
