﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_ConstructExifBuilder.h"

#include <new>
#include <nn/util/util_FormatString.h>
#include <nn/lmem/lmem_FrameHeap.h>
#include <nn/capsrv/capsrv_ConvertScreenShotOrientation.h>
#include "capsrvServer_GenerateMakerNote.h"

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    namespace {
        nn::image::ExifOrientation GetExifOrientation(ScreenShotOrientationType src) NN_NOEXCEPT
        {
            switch(src)
            {
            case ScreenShotOrientation_Default:
                NN_CAPSRV_LOG_DEV("[exif] rotation: Normal\n");
                return nn::image::ExifOrientation_Normal;
            case ScreenShotOrientation_Rotate90:
                NN_CAPSRV_LOG_DEV("[exif] rotation: Rotate90");
                return nn::image::ExifOrientation_Rotate90;
            case ScreenShotOrientation_Rotate180:
                NN_CAPSRV_LOG_DEV("[exif] rotation: Rotate180");
                return nn::image::ExifOrientation_Rotate180;
            case ScreenShotOrientation_Rotate270:
                NN_CAPSRV_LOG_DEV("[exif] rotation: Rotate270\n");
                return nn::image::ExifOrientation_Rotate270;
            default:
                NN_CAPSRV_LOG_DEV("[exif] rotation: Unknown\n");
                return nn::image::ExifOrientation_Normal;
            }
        }
    }

    nn::image::ExifBuilder* ConstructExifBuilder(
        void* memory,
        size_t memorySize,
        const MakerNoteInfo& info,
        const void* pThumbData,
        size_t thumbDataSize
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(memory);
        nn::lmem::HeapCommonHead heapHead;
        nn::lmem::HeapHandle hHeap = nn::lmem::CreateFrameHeap(memory, memorySize, nn::lmem::CreationOption_NoOption, &heapHead);

        // Construct
        nn::image::ExifBuilder* pBuilder = reinterpret_cast<nn::image::ExifBuilder*>(
            nn::lmem::AllocateFromFrameHeap(hHeap, sizeof(nn::image::ExifBuilder), NN_ALIGNOF(nn::image::ExifBuilder))
        );
        NN_SDK_ASSERT_NOT_NULL(pBuilder);
        void* builderWorkMemory = nn::lmem::AllocateFromFrameHeap(hHeap, nn::image::ExifBuilder::GetWorkBufferSize(), NN_ALIGNOF(std::max_align_t));
        NN_SDK_ASSERT_NOT_NULL(builderWorkMemory);
        new(pBuilder) nn::image::ExifBuilder(builderWorkMemory, nn::image::ExifBuilder::GetWorkBufferSize());

        // Exif: DateTime
        {
            const size_t DateTimeStringSize = 20;
            char* str = reinterpret_cast<char*>(nn::lmem::AllocateFromFrameHeap(hHeap, DateTimeStringSize));
            NN_SDK_ASSERT_NOT_NULL(str);
            int length = nn::util::SNPrintf(
                str,
                DateTimeStringSize,
                "%04d:%02d:%02d %02d:%02d:%02d",
                static_cast<int>(info.dateTime.year),
                static_cast<int>(info.dateTime.month),
                static_cast<int>(info.dateTime.day),
                static_cast<int>(info.dateTime.hour),
                static_cast<int>(info.dateTime.minute),
                static_cast<int>(info.dateTime.second)
                );
            NN_SDK_ASSERT_EQUAL(length, static_cast<int>(DateTimeStringSize - 1));
            NN_UNUSED(length);
            NN_CAPSRV_LOG_DEV("[exif] datetime: %s\n", str);
            pBuilder->SetDateTime(str, DateTimeStringSize);
        }

        // Exif: MakerNote
        {
            void* makerNoteMemory = nn::lmem::AllocateFromFrameHeap(hHeap, MakerNoteSizeLimit);
            NN_SDK_ASSERT_NOT_NULL(makerNoteMemory);
            size_t writtenSize = GenerateMakerNote(makerNoteMemory, MakerNoteSizeLimit, info);
            pBuilder->SetMakerNote(makerNoteMemory, writtenSize);
        }

        // Exif: Orientation
        {
            auto ori = GetExifOrientation(info.dataOrientation);
            pBuilder->SetOrientation(ori);
        }

        // Exif: Thumbnail
        if(pThumbData != nullptr && thumbDataSize > 0)
        {
            pBuilder->SetThumbnail(pThumbData, thumbDataSize);
        }

        // Analyze
        auto analyzeResult = pBuilder->Analyze();
        NN_SDK_ASSERT_EQUAL(analyzeResult, nn::image::JpegStatus_Ok);
        NN_UNUSED(analyzeResult);

        nn::lmem::DestroyFrameHeap(hHeap);

        return pBuilder;
    }

    void DestructExifBuilder(nn::image::ExifBuilder* pBuilder) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pBuilder);
        pBuilder->~ExifBuilder();
    }

}}}}
