﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

#include "capsrvServer_MakerNoteEntry.h"

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    template<typename TInfoType, typename Tag>
    class MakerNoteGenerator
    {
    public:
        typedef TInfoType InfoType;

    private:

        //-----------------------------
        // バージョン
        //-----------------------------

        // @brief pBuffer の先頭にバージョンを書き込んで次の位置を返します。
        // pBuffer が nullptr であるか bufferSize が不足した場合 nullptr を返します。
        static char* WriteMakerNoteVersion(char* pBuffer, ptrdiff_t bufferSize, const InfoType& info) NN_NOEXCEPT
        {
            if(pBuffer == nullptr || static_cast<size_t>(bufferSize) < sizeof(MakerNoteVersionType))
            {
                return nullptr;
            }

            NN_UNUSED(bufferSize);
            MakerNoteVersionType value;
            nn::util::StoreLittleEndian(&value, info.version);
            std::memcpy(pBuffer, &value, sizeof(MakerNoteVersionType));
            NN_CAPSRV_LOG_MAKERNOTE("  version %d\n", info.version);
            return pBuffer + sizeof(MakerNoteVersionType);
        }

        //-----------------------------
        // エントリー
        //-----------------------------

        template<typename TVersionPolicy, typename TProperty>
        static size_t CalculateEntrySize(const InfoType& info) NN_NOEXCEPT
        {
            NN_UNUSED(info);
            // サポートされていないエントリーは無視する
            if(TProperty::GetEntryCountMax(TVersionPolicy::Version) == 0)
            {
                return 0;
            }

            return sizeof(MakerNoteEntryTagType) + sizeof(MakerNoteEntrySizeType) + sizeof(typename TProperty::ValueType);
        }

        template<typename TProperty, MakerNoteValueClass TValueClass>
        struct EntryValueWriter;

        template<typename TProperty>
        struct EntryValueWriter<TProperty, MakerNoteValueClass_Bytes>
        {
            typedef typename TProperty::ValueType T;
            static char* Write(char* pBuffer, ptrdiff_t bufferSize, const T& value) NN_NOEXCEPT
            {
                NN_SDK_ASSERT_GREATER_EQUAL(static_cast<size_t>(bufferSize), sizeof(T));
                NN_UNUSED(bufferSize);
                std::memcpy(pBuffer, &value, sizeof(T));
                NN_CAPSRV_LOG_MAKERNOTE("  tag %d (%llu bytes)\n", TProperty::EntryTag, sizeof(T));
                return pBuffer + sizeof(T);
            }
        };

        template<typename TProperty>
        struct EntryValueWriter<TProperty, MakerNoteValueClass_Integer>
        {
            typedef typename TProperty::ValueType T;
            static char* Write(char* pBuffer, ptrdiff_t bufferSize, const T& value) NN_NOEXCEPT
            {
                NN_SDK_ASSERT_GREATER_EQUAL(static_cast<size_t>(bufferSize), sizeof(T));
                NN_UNUSED(bufferSize);
                T tmp = {};
                nn::util::StoreLittleEndian(&tmp, value);
                std::memcpy(pBuffer, &tmp, sizeof(T));
                NN_CAPSRV_LOG_MAKERNOTE("  tag %d (%llu bytes): %lld\n", TProperty::EntryTag, sizeof(T), static_cast<int64_t>(value));
                return pBuffer + sizeof(T);
            }
        };


        // @brief pBuffer の先頭にエントリーを書き込んで次の位置を返します。
        // pBuffer が nullptr であるか bufferSize が不足した場合 nullptr を返します。
        template<typename TVersionPolicy, typename TProperty>
        static char* WriteEntry(char* pBuffer, ptrdiff_t bufferSize, const InfoType& info) NN_NOEXCEPT
        {
            auto sizeToWrite = static_cast<ptrdiff_t>(CalculateEntrySize<TVersionPolicy, TProperty>(info));

            // サポートされていないエントリーは無視する
            if(sizeToWrite == 0)
            {
                NN_CAPSRV_LOG_MAKERNOTE("  tag %d (%llu bytes): ignored\n", TProperty::EntryTag, sizeof(typename TProperty::ValueType));
                return pBuffer;
            }

            // バッファに入りきらない場合無視する
            if(pBuffer == nullptr || bufferSize < sizeToWrite)
            {
                return nullptr;
            }

            MakerNoteEntryTagType tag;
            MakerNoteEntrySizeType size;
            nn::util::StoreLittleEndian(&tag, static_cast<MakerNoteEntryTagType>(TProperty::EntryTag));
            nn::util::StoreLittleEndian(&size, static_cast<MakerNoteEntrySizeType>(sizeof(typename TProperty::ValueType)));
            char* p = pBuffer;
            char*const pEnd = pBuffer + bufferSize;
            std::memcpy(p, &tag, sizeof(MakerNoteEntryTagType));
            p += sizeof(MakerNoteEntryTagType);
            NN_SDK_ASSERT_LESS_EQUAL(p, pEnd);
            std::memcpy(p, &size, sizeof(MakerNoteEntrySizeType));
            p += sizeof(MakerNoteEntrySizeType);
            NN_SDK_ASSERT_LESS_EQUAL(p, pEnd);

            p = EntryValueWriter<TProperty, TProperty::ValueClass>::Write(p, pEnd - p, *TProperty::GetPointer(info));
            NN_SDK_ASSERT_LESS_EQUAL(p, pEnd);
            NN_SDK_ASSERT_EQUAL(p - pBuffer, sizeToWrite);

            return p;
        }

        //-----------------------------
        // パディング
        //-----------------------------

        template<typename TVersionPolicy>
        static char* WritePadding(char* pBuffer, ptrdiff_t bufferSize, size_t currentSize) NN_NOEXCEPT
        {
            if(pBuffer == nullptr)
            {
                return nullptr;
            }

            static const ptrdiff_t Offset   = TVersionPolicy::EncryptionStartOffset;
            static const ptrdiff_t UnitSize = TVersionPolicy::EncryptionPolicy::UnitSize;

            NN_STATIC_ASSERT(UnitSize >= 1);
            if(currentSize <= Offset || currentSize > TVersionPolicy::SizeMax)
            {
                return nullptr;
            }

            ptrdiff_t newSize = ((currentSize - Offset + UnitSize - 1) / UnitSize) * UnitSize + Offset;

            if(newSize > TVersionPolicy::SizeMax)
            {
                return nullptr;
            }

            ptrdiff_t delta = newSize - currentSize;
            NN_SDK_ASSERT_GREATER_EQUAL(delta, 0);

            if(bufferSize < delta)
            {
                return nullptr;
            }

            std::memset(pBuffer, 0, delta);
            return pBuffer + delta;
        }

    public:

        template<typename TVersionPolicy>
        static size_t GenerateVersioned(void* pBuffer, size_t bufferSize, const InfoType& info) NN_NOEXCEPT
        {
            std::memset(pBuffer, 0, bufferSize);
            char* p = reinterpret_cast<char*>(pBuffer);
            char*const pBeg = p;
            char*const pEnd = p + bufferSize;

            NN_CAPSRV_LOG_MAKERNOTE("generating MakerNote(ver%d) ...\n", info.version);

            // バージョンを書き込む
            p = WriteMakerNoteVersion(p, pEnd - p, info);

            // 全プロパティ書き込む
        #define NN_CAPSRV_DETAIL_WRITE_ENTRY(PropertyType)  \
            p = WriteEntry<TVersionPolicy, PropertyType>(p, pEnd - p, info);

            NN_CAPSRV_DETAIL_FOREACH_MAKERNOTEPROPERTY( NN_CAPSRV_DETAIL_WRITE_ENTRY );

        #undef NN_CAPSRV_DETAIL_WRITE_ENTRY

            // パディングを書き込む
            p = WritePadding<TVersionPolicy>(p, pEnd - p, p - pBeg);

            // バッファが不足していないことを確認
            NN_ABORT_UNLESS_NOT_NULL(p);
            size_t makerNoteSize = static_cast<size_t>(p - pBeg);
            NN_ABORT_UNLESS_MINMAX(makerNoteSize, static_cast<size_t>(TVersionPolicy::SizeMin), static_cast<size_t>(TVersionPolicy::SizeMax));

            TVersionPolicy::EncryptionPolicy::EncryptInplace(pBuffer, makerNoteSize, TVersionPolicy::EncryptionStartOffset);

            NN_CAPSRV_LOG_MAKERNOTE("  -> success (ver=%d, size=%llu)\n", info.version, makerNoteSize);
            return makerNoteSize;
        }

        static size_t Generate(void* pBuffer, size_t bufferSize, const InfoType& info) NN_NOEXCEPT
        {
            switch(info.version)
            {
        #define NN_CAPSRV_DETAIL_CASE_VERSION(policy)    \
            case policy::Version: return GenerateVersioned<policy>(pBuffer, bufferSize, info);

            NN_CAPSRV_DETAIL_FOREACH_MAKERNOTEVERSIONPOLICY(NN_CAPSRV_DETAIL_CASE_VERSION)

        #undef NN_CAPSRV_DETAIL_CASE_VERSION
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

    };

}}}}
