﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include "../capsrvServer_Config.h"
#include "capsrvServer_MakerNoteEntry.h"

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    template<typename TVersionPolicy, typename TDecoder, typename TDecoderArgument, typename TEntryParser, typename Tag>
    class VersionedMakerNoteEntryVisitor
    {
    public:
        typedef TVersionPolicy VersionPolicy;

    private:
        static nn::Result CheckSizeImpl(size_t inputSize) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MAKERNOTE("  checking maker note size ...\n");

            NN_RESULT_THROW_UNLESS(inputSize >= VersionPolicy::SizeMin, ResultAlbumInvalidFileData());
            NN_RESULT_THROW_UNLESS(inputSize <= VersionPolicy::SizeMax, ResultAlbumInvalidFileData());

            NN_RESULT_THROW_UNLESS(
                (inputSize - VersionPolicy::EncryptionStartOffset) % VersionPolicy::EncryptionPolicy::UnitSize == 0,
                ResultAlbumInvalidFileData()
            );

            NN_CAPSRV_LOG_MAKERNOTE("  -> ok\n");
            NN_RESULT_SUCCESS;
        }

        static void DecryptInplace(void* data, size_t size) NN_NOEXCEPT
        {
            NN_STATIC_ASSERT(VersionPolicy::SizeMin > VersionPolicy::EncryptionStartOffset);

            NN_CAPSRV_LOG_MAKERNOTE("  decrypting maker note\n");
            VersionPolicy::EncryptionPolicy::DecryptInplace(data, size, VersionPolicy::EncryptionStartOffset);
        }

        static nn::Result ParsePadding(const char** pOutNext, const char* p, ptrdiff_t size) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MAKERNOTE("    checking padding ...\n");
            NN_SDK_REQUIRES_GREATER_EQUAL(size, 0);
            NN_RESULT_THROW_UNLESS(size >= 0, ResultAlbumInvalidFileData());

            // パディングの大きさは UnitSize 未満
            NN_RESULT_THROW_UNLESS(size < VersionPolicy::EncryptionPolicy::UnitSize, ResultAlbumInvalidFileData());

            // パディング部分はゼロ埋めが必要
            for(ptrdiff_t i = 0; i < size; i++)
            {
                NN_RESULT_THROW_UNLESS(p[i] == 0, ResultAlbumInvalidFileData());
            }

            NN_CAPSRV_LOG_MAKERNOTE("    -> ok\n");
            *pOutNext = p + size;
            NN_RESULT_SUCCESS;
        }

    public:

        static nn::Result VisitEntries(
            void* pMakerNoteData,
            size_t makerNoteDataSize,
            TDecoderArgument& decoderArgument
        ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pMakerNoteData);
            NN_SDK_REQUIRES_GREATER(makerNoteDataSize, static_cast<size_t>(0));

            // サイズチェック
            NN_RESULT_DO(CheckSizeImpl(makerNoteDataSize));

            // 難読化解除
            DecryptInplace(pMakerNoteData, makerNoteDataSize);

            const char* p = reinterpret_cast<const char*>(pMakerNoteData);
            const char* pEnd = p + makerNoteDataSize;

            // 既に version は読んでいるので飛ばす
            p += sizeof(MakerNoteVersionType);

            NN_CAPSRV_LOG_MAKERNOTE("  parsing maker note\n");
            int32_t prevTag = -1;
            while(p < pEnd)
            {
                MakerNoteEntry entry = {};

                auto parseEntryResult = TEntryParser::TryParseEntry(&p, &entry, p, pEnd - p);

                if(parseEntryResult.IsSuccess())
                {
                    // NOTE: 今の実装では同一 Tag の重複を許可していない

                    // エントリーは昇順でなければならない
                    int32_t currTag = static_cast<int32_t>(entry.tag);
                    NN_RESULT_THROW_UNLESS(currTag > prevTag, ResultAlbumInvalidFileData());

                    prevTag = currTag;

                    // コールバックの呼び出し
                    NN_RESULT_DO(TDecoder::Invoke(&entry, decoderArgument));
                }
                else
                {
                    break;
                }
            }
            NN_RESULT_DO(ParsePadding(&p, p, pEnd - p));
            NN_SDK_ASSERT_EQUAL(p, pEnd);

            NN_CAPSRV_LOG_MAKERNOTE("  -> success\n");
            NN_RESULT_SUCCESS;
        }
    };

    template<typename Tag>
    class MakerNoteEntryVisitor
    {
    public:

        template<
            template<typename, typename, typename> class TDecoder,
            typename TInfoType,
            typename TDecoderArgument,
            typename TEntryParser
        >
        static nn::Result VisitEntries(
            MakerNoteVersionType* pOutVersion,
            void* pMakerNoteData,
            size_t makerNoteDataSize,
            TDecoderArgument& decoderArgument
            ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pMakerNoteData);

            const char* p = reinterpret_cast<const char*>(pMakerNoteData);
            const char* pEnd = p + makerNoteDataSize;

            MakerNoteVersionType version = {};
            NN_CAPSRV_LOG_MAKERNOTE("  checking maker note version\n");
            NN_RESULT_DO(TEntryParser::TryParseVersion(&p, &version, p, pEnd - p));
            // 未知のバージョンは解析できない
            if(version >= MakerNoteVersion_End)
            {
                NN_RESULT_THROW(ResultAlbumInvalidFileData());
            }
            NN_CAPSRV_LOG_MAKERNOTE("  -> ok: ver %d\n", version);

            // バージョンごとの解析へ
            switch(version)
            {
            #define NN_CAPSRV_CASE_VERSION(policy)  \
            case policy::Version:                   \
                {                                   \
                    typedef policy VersionPolicy;   \
                    typedef TDecoder<TInfoType, VersionPolicy, Tag> Decoder; \
                    auto visitEntries = VersionedMakerNoteEntryVisitor< \
                        VersionPolicy,                                  \
                        Decoder,                                        \
                        TDecoderArgument,                               \
                        TEntryParser,                                   \
                        Tag>::VisitEntries;                             \
                    NN_RESULT_DO(visitEntries(pMakerNoteData, makerNoteDataSize, decoderArgument));   \
                }                               \
                break;

                NN_CAPSRV_DETAIL_FOREACH_MAKERNOTEVERSIONPOLICY(NN_CAPSRV_CASE_VERSION)

            #undef NN_CAPSRV_CASE_VERSION
            default: NN_UNEXPECTED_DEFAULT;
            }

            *pOutVersion = version;
            NN_RESULT_SUCCESS;
        } // NOLINT(style/indent)

    };



}}}}
