﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_AlbumServiceImpl.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/capsrv/capsrv_Result.h>
#include <nn/capsrv/capsrv_ServiceConfig.h>
#include <mutex>

#include "../../capsrv_Macro.h"
#include "../capsrvServer_Config.h"
#include "capsrvServer_AlbumServerObject.h"
#include "../album/capsrvServer_LoadAndDecodeScreenShot.h"
#include "../detail/capsrvServer_GetApplicationAlbumEntry.h"
#include "../detail/capsrvServer_MakerNoteInfo.h"
#include "../album/capsrvServer_AlbumPathUtility.h"
#include "capsrvServer_AlbumAccessorSessionFactory.h"

#if defined(NN_CAPSRV_USE_HIPC)

#define NN_CAPSRV_SERVER_DO(Function)  \
    NN_RESULT_DO(g_AlbumErrorConverter.Convert(Function()));

#define NN_CAPSRV_SERVER_SYNC_DO(Function)  \
    NN_RESULT_DO(g_AlbumErrorConverter.Convert((::std::unique_lock<decltype(g_AlbumGlobalMutex)>(g_AlbumGlobalMutex), Function())));

#elif defined(NN_CAPSRV_USE_DIRECT_FUNCTION_CALL)

#define NN_CAPSRV_SERVER_DO(Function)  \
    NN_RESULT_DO(g_AlbumErrorConverter.Convert(Function()));
#define NN_CAPSRV_SERVER_SYNC_DO  NN_CAPSRV_SERVER_DO

#endif

namespace nn{ namespace capsrv{ namespace server{

    nn::Result AlbumServiceImpl::GetAlbumFileCount(
        nn::sf::Out<std::uint64_t> outCount,
        nn::capsrv::AlbumStorageType storage
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(GetAlbumFileCount);

        AlbumFileContentsFlag contentsMask = {};
        contentsMask.Set(AlbumFileContents_ScreenShot)
                    .Set(AlbumFileContents_Movie);

        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                int count = 0;
                nn::Result result = g_AlbumManager.GetFileCount(&count, storage, contentsMask);
                outCount.Set(static_cast<uint64_t>(count));
                return result;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::GetAlbumFileCountEx0(
        nn::sf::Out<std::uint64_t> outCount,
        nn::capsrv::AlbumStorageType storage,
        nn::capsrv::AlbumFileContentsFlag contentsMask
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(GetAlbumFileCount);
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                int count = 0;
                nn::Result result = g_AlbumManager.GetFileCount(&count, storage, contentsMask);
                outCount.Set(static_cast<uint64_t>(count));
                return result;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::GetAlbumFileList(
        nn::sf::Out<std::uint64_t> outCount,
        const nn::sf::OutArray<nn::capsrv::AlbumEntry>& outList,
        nn::capsrv::AlbumStorageType storage
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(GetAlbumFileList);

        AlbumFileContentsFlag contentsMask = {};
        contentsMask.Set(AlbumFileContents_ScreenShot)
                    .Set(AlbumFileContents_Movie);

        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                int listLength = 0;
                nn::Result result = g_AlbumManager.GetFileList(
                    &listLength,
                    outList.GetData(),
                    static_cast<int>(outList.GetLength()),
                    storage,
                    contentsMask
                    );
                outCount.Set(static_cast<uint64_t>(listLength));
                return result;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::GetAlbumFileListEx0(
        nn::sf::Out<std::uint64_t> outCount,
        const nn::sf::OutArray<nn::capsrv::AlbumEntry>& outList,
        nn::capsrv::AlbumStorageType storage,
        nn::capsrv::AlbumFileContentsFlag contentsMask
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(GetAlbumFileList);
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                int listLength = 0;
                nn::Result result = g_AlbumManager.GetFileList(
                    &listLength,
                    outList.GetData(),
                    static_cast<int>(outList.GetLength()),
                    storage,
                    contentsMask
                    );
                outCount.Set(static_cast<uint64_t>(listLength));
                return result;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::LoadAlbumFile(
        nn::sf::Out<std::uint64_t> outSize,
        const nn::sf::OutBuffer& outBuffer,
        const nn::capsrv::AlbumFileId& fileId
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(LoadAlbumFile);
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                size_t fileSize = 0;
                nn::Result result = g_AlbumManager.LoadFile(
                    &fileSize,
                    outBuffer.GetPointerUnsafe(),
                    outBuffer.GetSize(),
                    &fileId
                    );
                outSize.Set(static_cast<uint64_t>(fileSize));
                return result;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::DeleteAlbumFile(const nn::capsrv::AlbumFileId& fileId) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(DeleteAlbumFile);
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                return g_AlbumManager.DeleteFile(&fileId);
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::StorageCopyAlbumFile(
        const nn::capsrv::AlbumFileId& sourceFileId,
        nn::capsrv::AlbumStorageType destinationStorage
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(StorageCopyAlbumFile);
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                return g_AlbumManager.StorageCopyFile(&sourceFileId, destinationStorage);
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::IsAlbumMounted(
        nn::sf::Out<bool> outValue,
        nn::capsrv::AlbumStorageType storage
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(IsAlbumMounted);
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                NN_RESULT_TRY(g_AlbumManager.EnsureMounted(storage))
                    NN_RESULT_CATCH_ALL
                    {
                        outValue.Set(false);
                        NN_RESULT_SUCCESS;
                    }
                NN_RESULT_END_TRY;
                outValue.Set(true);
                NN_RESULT_SUCCESS;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::GetAlbumMountResult(
        nn::capsrv::AlbumStorageType storage
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(GetAlbumMountResult);
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                return g_AlbumManager.EnsureMounted(storage);
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::GetAlbumUsage(
        nn::sf::Out<nn::capsrv::detail::AlbumUsage2> outValue,
        nn::capsrv::AlbumStorageType storage
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(GetAlbumUsage2);
        auto p = outValue.GetPointer();
        AlbumFileContentsFlag contentsMask = {};
        contentsMask.Set(AlbumFileContents_ScreenShot);
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                return g_AlbumManager.GetUsage(
                    &p->usages[p->Count - 1],   // Unknown
                    &p->usages[0],              // Known
                    p->Count - 1,               // Knwon の数
                    storage,
                    contentsMask
                );
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::GetAlbumUsage3(
        nn::sf::Out<nn::capsrv::detail::AlbumUsage3> outValue,
        nn::capsrv::AlbumStorageType storage
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(GetAlbumUsage3);
        auto p = outValue.GetPointer();
        AlbumFileContentsFlag contentsMask = {};
        contentsMask.Set(AlbumFileContents_ScreenShot).Set(AlbumFileContents_Movie);
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                return g_AlbumManager.GetUsage(
                    &p->usages[p->Count - 1],   // Unknown
                    &p->usages[0],              // Known
                    p->Count - 1,               // Knwon の数
                    storage,
                    contentsMask
                );
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::GetAlbumUsage16(
        nn::sf::Out<nn::capsrv::detail::AlbumUsage16> outValue,
        nn::capsrv::AlbumStorageType storage,
        nn::capsrv::AlbumFileContentsFlag contentsMask
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(GetAlbumUsage16);
        auto p = outValue.GetPointer();
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                return g_AlbumManager.GetUsage(
                    &p->usages[p->Count - 1],   // Unknown
                    &p->usages[0],              // Known
                    p->Count - 1,               // Knwon の数
                    storage,
                    contentsMask
                );
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::GetAlbumFileSize(
        nn::sf::Out<std::uint64_t> outSize,
        const nn::capsrv::AlbumFileId& fileId
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(GetAlbumFileSize);
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                size_t size = 0;
                nn::Result result = g_AlbumManager.GetFileSize(
                    &size,
                    &fileId
                    );
                outSize.Set(static_cast<uint64_t>(size));
                return result;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::LoadAlbumFileThumbnail(
        nn::sf::Out<std::uint64_t> outSize,
        const nn::sf::OutBuffer& outBuffer,
        const nn::capsrv::AlbumFileId& fileId
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(LoadAlbumFileThumbnail);
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                size_t size = 0;
                nn::Result result = g_AlbumManager.LoadFileThumbnail(
                    &size,
                    outBuffer.GetPointerUnsafe(),
                    outBuffer.GetSize(),
                    &fileId
                    );
                outSize.Set(static_cast<uint64_t>(size));
                return result;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::GetAlbumEntryFromApplicationAlbumEntry(
        nn::sf::Out<nn::capsrv::AlbumEntry> outEntry,
        const nn::capsrv::ApplicationAlbumEntry& srcEntry,
        const nn::ncm::ApplicationId& applicationId
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(GetAlbumEntryFromApplicationAlbumEntry);

        AlbumEntry entry = {};
        NN_CAPSRV_SERVER_DO((
            [&]() -> nn::Result
            {
                detail::ApplicationAlbumEntryKey key = {};
                NN_RESULT_TRY(g_ApplicationResourceManager.GetApplicationAlbumEntryKeyFromApplicationId(&key, applicationId))
                    NN_RESULT_CATCH_ALL
                    {
                        NN_RESULT_THROW(ResultAlbumError());
                    }
                NN_RESULT_END_TRY;
                entry = detail::GetAlbumEntryFromApplicationAlbumEntry(srcEntry, key);
                // 値が妥当か検査
                NN_RESULT_TRY(album::AlbumPathUtility::ValidateFileId(&entry.fileId, g_EnvironmentInfo))
                    NN_RESULT_CATCH_ALL
                    {
                        NN_RESULT_THROW(ResultAlbumError());
                    }
                NN_RESULT_END_TRY;
                AlbumEntry emptyEntry = {};
                NN_RESULT_THROW_UNLESS(std::memcmp(&entry.fileId._reserved, emptyEntry.fileId._reserved, sizeof(emptyEntry.fileId._reserved)) == 0, ResultAlbumError());
                NN_RESULT_SUCCESS;
            }
        ));

        outEntry.Set(entry);
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::GetLastOverlayScreenShotThumbnail(
        nn::sf::Out<nn::capsrv::AlbumFileId> outFileId,
        nn::sf::Out<uint64_t> outSize,
        const nn::sf::OutBuffer& outBuffer
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(GetLastOverlayScreenShotThumbnail);
        size_t size = 0;
        NN_CAPSRV_SERVER_DO((
            [&]() -> nn::Result {
                return g_OverlayThumbnailHolder.GetThumbnailImage(
                    outFileId.GetPointer(),
                    &size,
                    outBuffer.GetPointerUnsafe(),
                    outBuffer.GetSize(),
                    nn::capsrv::server::detail::OverlayThumbnailIndex_ScreenShot
                );
            }
        ));
        outSize.Set(static_cast<uint64_t>(size));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::GetLastOverlayMovieThumbnail(
        nn::sf::Out<nn::capsrv::AlbumFileId> outFileId,
        nn::sf::Out<uint64_t> outSize,
        const nn::sf::OutBuffer& outBuffer
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(GetLastOverlayMovieThumbnail);
        size_t size = 0;
        NN_CAPSRV_SERVER_DO((
            [&]() -> nn::Result {
                return g_OverlayThumbnailHolder.GetThumbnailImage(
                    outFileId.GetPointer(),
                    &size,
                    outBuffer.GetPointerUnsafe(),
                    outBuffer.GetSize(),
                    nn::capsrv::server::detail::OverlayThumbnailIndex_Movie
                );
            }
        ));
        outSize.Set(static_cast<uint64_t>(size));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::GetAutoSavingStorage(
        nn::sf::Out<nn::capsrv::AlbumStorageType> outValue
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(GetAutoSavingStorage);
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                return g_AlbumFileIdGenerator.GetAutoSavingStorage(
                    outValue.GetPointer(),
                    false,
                    &g_AlbumManager
                );
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::GetRequiredStorageSpaceSizeToCopyAll(
        nn::sf::Out<std::uint64_t> outSize,
        nn::capsrv::AlbumStorageType dstStorage,
        nn::capsrv::AlbumStorageType srcStorage
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(GetRequiredStorageSpaceSizeToCopyAll);
        size_t size = 0;
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                return g_AlbumManager.GetRequiredStorageSpaceSizeToCopyAll(
                    &size,
                    dstStorage,
                    srcStorage
                );
            }
        ));
        outSize.Set(static_cast<uint64_t>(size));
        NN_RESULT_SUCCESS;
    }

    //--------------------------------------------------------------------------------------------------------------

    namespace {
        nn::Result LoadAlbumScreenShotImageImpl(
            std::uint64_t* pOutWidth,
            std::uint64_t* pOutHeight,
            nn::capsrv::detail::ScreenShotAttributeEx0* pOutAttribute,
            nn::capsrv::AppletData* pOutAppletData,
            const nn::sf::OutBuffer& outBuffer,
            const nn::capsrv::AlbumFileId& fileId,
            const nn::capsrv::ScreenShotDecodeOption& option,
            const nn::sf::OutBuffer& workBuffer
            ) NN_NOEXCEPT
        {
            int width = 0;
            int height = 0;
            ScreenShotAttribute attr = {};

            void* pBuffer = outBuffer.GetPointerUnsafe();
            size_t bufferSize = outBuffer.GetSize();

            void* pFileBuffer = workBuffer.GetPointerUnsafe();
            size_t fileBufferSize = workBuffer.GetSize();
            NN_UTIL_SCOPE_EXIT{ std::memset(pFileBuffer, 0, fileBufferSize); };

            NN_CAPSRV_SERVER_SYNC_DO((
                [&]() -> nn::Result {
                    NN_RESULT_DO(album::LoadAndDecodeScreenShot(
                        &width,
                        &height,
                        &attr,
                        pOutAppletData,
                        nullptr,
                        nullptr,
                        pBuffer,
                        bufferSize,
                        pFileBuffer,
                        fileBufferSize,
                        fileId,
                        option,
                        &g_AlbumManager
                    ));
                    NN_RESULT_SUCCESS;
                }
            ));

            if(pOutWidth)
            {
                *pOutWidth = static_cast<uint64_t>(width);
            }
            if(pOutHeight)
            {
                *pOutHeight = static_cast<uint64_t>(height);
            }
            if(pOutAttribute)
            {
                *pOutAttribute = attr;
            }
            NN_RESULT_SUCCESS;
        }
    }


    nn::Result AlbumServiceImpl::LoadAlbumScreenShotImage(nn::sf::Out<std::uint64_t> outWidth, nn::sf::Out<std::uint64_t> outHeight, const nn::sf::OutBuffer& outBuffer, const nn::capsrv::AlbumFileId& fileId, const nn::sf::OutBuffer& workBuffer) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(LoadAlbumScreenShotImage);
        auto option = nn::capsrv::ScreenShotDecodeOption::GetDefaultValue();
        return LoadAlbumScreenShotImageImpl(outWidth.GetPointer(), outHeight.GetPointer(), nullptr, nullptr, outBuffer, fileId, option, workBuffer);
    }

    nn::Result AlbumServiceImpl::LoadAlbumScreenShotImageEx(nn::sf::Out<std::uint64_t> outWidth, nn::sf::Out<std::uint64_t> outHeight, const nn::sf::OutBuffer& outBuffer, const nn::capsrv::AlbumFileId& fileId, const nn::capsrv::ScreenShotDecodeOption& option, const nn::sf::OutBuffer& workBuffer) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(LoadAlbumScreenShotImageEx);
        return LoadAlbumScreenShotImageImpl(outWidth.GetPointer(), outHeight.GetPointer(), nullptr, nullptr, outBuffer, fileId, option, workBuffer);
    }

    nn::Result AlbumServiceImpl::LoadAlbumScreenShotImageEx0(nn::sf::Out<std::uint64_t> outWidth, nn::sf::Out<std::uint64_t> outHeight, nn::sf::Out<nn::capsrv::detail::ScreenShotAttributeEx0> outAttribute, const nn::sf::OutBuffer& outBuffer, const nn::capsrv::AlbumFileId& fileId, const nn::capsrv::ScreenShotDecodeOption& option, const nn::sf::OutBuffer& workBuffer) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(LoadAlbumScreenShotImageEx0);
        return LoadAlbumScreenShotImageImpl(outWidth.GetPointer(), outHeight.GetPointer(), outAttribute.GetPointer(), nullptr, outBuffer, fileId, option, workBuffer);
    }

    nn::Result AlbumServiceImpl::LoadAlbumScreenShotImageEx1(nn::sf::Out<nn::capsrv::sf::LoadAlbumScreenShotImageOutputEx1> output, const nn::sf::OutBuffer& outBuffer, const nn::capsrv::AlbumFileId& fileId, const nn::capsrv::ScreenShotDecodeOption& option, const nn::sf::OutBuffer& workBuffer) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(LoadAlbumScreenShotImageEx1);
        auto pOut = output.GetPointer();
        return LoadAlbumScreenShotImageImpl(&pOut->width, &pOut->height, &pOut->attribute, &pOut->appletData, outBuffer, fileId, option, workBuffer);
    }

    namespace {
        nn::Result LoadAlbumScreenShotThumbnailImageImpl(
            std::uint64_t* pOutWidth,
            std::uint64_t* pOutHeight,
            nn::capsrv::detail::ScreenShotAttributeEx0* pOutAttribute,
            nn::capsrv::AppletData* pOutAppletData,
            const nn::sf::OutBuffer& outBuffer,
            const nn::capsrv::AlbumFileId& fileId,
            const nn::capsrv::ScreenShotDecodeOption& option,
            const nn::sf::OutBuffer& workBuffer
            ) NN_NOEXCEPT
        {
            int width = 0;
            int height = 0;
            ScreenShotAttribute attr = {};

            void* pBuffer = outBuffer.GetPointerUnsafe();
            size_t bufferSize = outBuffer.GetSize();

            void* pFileBuffer = workBuffer.GetPointerUnsafe();
            size_t fileBufferSize = workBuffer.GetSize();
            NN_UTIL_SCOPE_EXIT{ std::memset(pFileBuffer, 0, fileBufferSize); };

            NN_CAPSRV_SERVER_SYNC_DO((
                [&]() -> nn::Result {
                    NN_RESULT_DO(album::LoadAndDecodeScreenShotThumbnail(
                        &width,
                        &height,
                        &attr,
                        pOutAppletData,
                        nullptr,
                        nullptr,
                        pBuffer,
                        bufferSize,
                        pFileBuffer,
                        fileBufferSize,
                        fileId,
                        option,
                        &g_AlbumManager
                    ));
                    NN_RESULT_SUCCESS;
                }
            ));

            if(pOutWidth)
            {
                *pOutWidth = static_cast<uint64_t>(width);
            }
            if(pOutHeight)
            {
                *pOutHeight = static_cast<uint64_t>(height);
            }
            if(pOutAttribute)
            {
                *pOutAttribute = attr;
            }
            NN_RESULT_SUCCESS;
        }
    }

    nn::Result AlbumServiceImpl::LoadAlbumScreenShotThumbnailImage(nn::sf::Out<std::uint64_t> outWidth, nn::sf::Out<std::uint64_t> outHeight, const nn::sf::OutBuffer& outBuffer, const nn::capsrv::AlbumFileId& fileId, const nn::sf::OutBuffer& workBuffer) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(LoadAlbumScreenShotThumbnailImage);
        auto option = nn::capsrv::ScreenShotDecodeOption::GetDefaultValue();
        return LoadAlbumScreenShotThumbnailImageImpl(outWidth.GetPointer(), outHeight.GetPointer(), nullptr, nullptr, outBuffer, fileId, option, workBuffer);
    }

    nn::Result AlbumServiceImpl::LoadAlbumScreenShotThumbnailImageEx(nn::sf::Out<std::uint64_t> outWidth, nn::sf::Out<std::uint64_t> outHeight, const nn::sf::OutBuffer& outBuffer, const nn::capsrv::AlbumFileId& fileId, const nn::capsrv::ScreenShotDecodeOption& option, const nn::sf::OutBuffer& workBuffer) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(LoadAlbumScreenShotThumbnailImageEx);
        return LoadAlbumScreenShotThumbnailImageImpl(outWidth.GetPointer(), outHeight.GetPointer(), nullptr, nullptr, outBuffer, fileId, option, workBuffer);
    }

    nn::Result AlbumServiceImpl::LoadAlbumScreenShotThumbnailImageEx0(nn::sf::Out<std::uint64_t> outWidth, nn::sf::Out<std::uint64_t> outHeight, nn::sf::Out<nn::capsrv::detail::ScreenShotAttributeEx0> outAttribute, const nn::sf::OutBuffer& outBuffer, const nn::capsrv::AlbumFileId& fileId, const nn::capsrv::ScreenShotDecodeOption& option, const nn::sf::OutBuffer& workBuffer) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(LoadAlbumScreenShotThumbnailImageEx0);
        return LoadAlbumScreenShotThumbnailImageImpl(outWidth.GetPointer(), outHeight.GetPointer(), outAttribute.GetPointer(), nullptr, outBuffer, fileId, option, workBuffer);
    }

    nn::Result AlbumServiceImpl::LoadAlbumScreenShotThumbnailImageEx1(nn::sf::Out<nn::capsrv::sf::LoadAlbumScreenShotImageOutputEx1> output, const nn::sf::OutBuffer& outBuffer, const nn::capsrv::AlbumFileId& fileId, const nn::capsrv::ScreenShotDecodeOption& option, const nn::sf::OutBuffer& workBuffer) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(LoadAlbumScreenShotThumbnailImageEx1);
        auto pOutput = output.GetPointer();
        return LoadAlbumScreenShotThumbnailImageImpl(&pOutput->width, &pOutput->height, &pOutput->attribute, &pOutput->appletData, outBuffer, fileId, option, workBuffer);
    }

    //--------------------------------------------------------------------------------------------------------------

    nn::Result AlbumServiceImpl::SaveEditedScreenShot(
        nn::sf::Out<nn::capsrv::AlbumEntry> /*outFileEntry*/,
        const nn::sf::InBuffer& /*imageBuffer*/,
        const nn::sf::InBuffer& /*thumbnailBuffer*/,
        std::int64_t /*imageWidth*/,
        std::int64_t /*imageHeight*/,
        std::int64_t /*thumbnailWidth*/,
        std::int64_t /*thumbnailHeight*/,
        const nn::capsrv::AlbumFileId& /*originalFileId*/
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(SaveEditedScreenShot);
        NN_RESULT_THROW(ResultAlbumError());
    }


    //---------------------------
    // caps:a (Testing)
    //---------------------------

    nn::Result AlbumServiceImpl::RefreshAlbumCache(
        AlbumStorageType storage
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(RefreshAlbumCache);
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                return g_AlbumManager.RefreshStorageCache(storage);
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::GetAlbumCache(nn::sf::Out<AlbumCacheData>, nn::capsrv::AlbumStorageType) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(GetAlbumCache);
        NN_RESULT_THROW(ResultAlbumDenied());
    }

    nn::Result AlbumServiceImpl::GetAlbumCacheEx(
        nn::sf::Out<AlbumCacheData> outValue,
        nn::capsrv::AlbumStorageType storage,
        nn::capsrv::AlbumFileContentsType contents
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(GetAlbumCacheEx);
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                return g_AlbumManager.GetStorageCache(
                    outValue.GetPointer(),
                    storage,
                    contents
                );
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::ForceAlbumUnmounted(
        nn::capsrv::AlbumStorageType storage
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(ForceAlbumUnmounted);
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                return g_AlbumManager.ForceUnmounted(storage);
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::ResetAlbumMountStatus(
        nn::capsrv::AlbumStorageType storage
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(ResetAlbumMountStatus);
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                return g_AlbumManager.ResetMountStatus(storage);
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::GetAlbumEntryFromApplicationAlbumEntryAruid(
        nn::sf::Out<nn::capsrv::AlbumEntry> outEntry,
        const nn::capsrv::ApplicationAlbumEntry& srcEntry,
        nn::applet::AppletResourceUserId aruid
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(GetAlbumEntryFromApplicationAlbumEntryAruid);

        NN_RESULT_THROW_UNLESS(aruid != nn::applet::AppletResourceUserId::GetInvalidId(), ResultAlbumError());

        AlbumEntry entry = {};
        NN_CAPSRV_SERVER_DO((
            [&]() -> nn::Result
            {
                detail::ApplicationAlbumEntryKey key = {};
                NN_RESULT_TRY(g_ApplicationResourceManager.GetApplicationAlbumEntryKeyFromAruid(&key, aruid))
                    NN_RESULT_CATCH_ALL
                    {
                        NN_RESULT_THROW(ResultAlbumError());
                    }
                NN_RESULT_END_TRY;
                entry = detail::GetAlbumEntryFromApplicationAlbumEntry(srcEntry, key);
                // 値が妥当か検査
                NN_RESULT_TRY(album::AlbumPathUtility::ValidateFileId(&entry.fileId, g_EnvironmentInfo))
                    NN_RESULT_CATCH_ALL
                    {
                        NN_RESULT_THROW(ResultAlbumError());
                    }
                NN_RESULT_END_TRY;
                AlbumEntry emptyEntry = {};
                NN_RESULT_THROW_UNLESS(std::memcmp(&entry.fileId._reserved, emptyEntry.fileId._reserved, sizeof(emptyEntry.fileId._reserved)) == 0, ResultAlbumError());
                NN_RESULT_SUCCESS;
            }
        ));

        outEntry.Set(entry);
        NN_RESULT_SUCCESS;
    }


    nn::Result AlbumServiceImpl::SetInternalErrorConversionEnabled(bool value) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(SetInternalErrorConversionEnabled);
        NN_CAPSRV_SERVER_DO((
            [&]() -> nn::Result {
                g_AlbumErrorConverter.SetConversionEnabled(value);
                NN_RESULT_SUCCESS;
            }
        ));
        NN_RESULT_SUCCESS;
    }

    nn::Result AlbumServiceImpl::OpenAccessorSession(nn::sf::Out<nn::sf::SharedPointer<nn::capsrv::sf::IAlbumAccessorSession>> outValue, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_A(OpenAccessorSession);
        NN_UNUSED(aruid);

        nn::sf::SharedPointer<sf::IAlbumAccessorSession> p;
        NN_RESULT_DO(AlbumAccessorSessionFactory::Create(&p));

        outValue.Set(p);
        NN_RESULT_SUCCESS;
    }


    // デバッグ用（DevMenuCommandSystem 用）
    nn::Result AlbumServiceImpl::LoadMakerNoteInfoForDebug(nn::sf::Out<std::uint64_t> pOutFileSize, nn::sf::OutBuffer outInfoBuffer, const nn::capsrv::AlbumFileId& fileId, nn::sf::OutBuffer workBuffer) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(!IsProdMode(), ResultNotSupported());

        void* pOutInfoBuffer = outInfoBuffer.GetPointerUnsafe();
        size_t infoBufferSize = outInfoBuffer.GetSize();
        NN_RESULT_THROW_UNLESS(infoBufferSize >= sizeof(capsrv::server::detail::MakerNoteInfo), ResultAlbumReadBufferShortage());

        void* pWorkBuffer = workBuffer.GetPointerUnsafe();
        size_t workBufferSize = workBuffer.GetSize();
        NN_RESULT_THROW_UNLESS(workBufferSize >= nn::capsrv::AlbumFileSizeLimit_ScreenShot, ResultAlbumWorkMemoryError());

        std::memset(pOutInfoBuffer, 0, infoBufferSize);
        NN_UTIL_SCOPE_EXIT{ std::memset(pWorkBuffer, 0, workBufferSize); };

        uint64_t fileSize = 0;
        capsrv::server::detail::MakerNoteInfo makerNoteInfo = {};
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                return g_AlbumManager.LoadMakerNoteInfo(&fileSize, &makerNoteInfo, fileId, pWorkBuffer, workBufferSize);
            }
        ));

        *pOutFileSize = fileSize;
        std::memcpy(pOutInfoBuffer, &makerNoteInfo, sizeof(makerNoteInfo));
        NN_RESULT_SUCCESS;
    }


}}}
