﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_AlbumApplicationServiceImpl.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs/fs_FileSystemPrivate.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/capsrv/capsrv_Result.h>
#include <nn/capsrv/capsrv_ServiceConfig.h>
#include <nn/capsrv/capsrv_AlbumStorage.h>
#include <nn/capsrv/capsrv_AlbumFileCountLimit.h>
#include <nn/time/time_PosixTime.h>
#include <nn/time/time_TimeZoneApi.h>
#include <mutex>

#include "capsrvServer_AlbumServerObject.h"

#include "../../capsrv_Macro.h"
#include "../capsrvServer_Config.h"
#include "../album/capsrvServer_LoadAndDecodeScreenShot.h"
#include "../album/capsrvServer_AlbumPathUtility.h"
#include "../detail/capsrvServer_ApplicationResourceManager.h"
#include "../detail/capsrvServer_DecryptApplicationAlbumEntry.h"
#include "capsrvServer_AlbumAccessorSessionFactory.h"

#include <cstring>

#if defined(NN_CAPSRV_USE_HIPC)

#define NN_CAPSRV_SERVER_DO(Function)  \
    NN_RESULT_DO(g_AlbumErrorConverter.Convert(Function()));

#define NN_CAPSRV_SERVER_SYNC_DO(Function)  \
    NN_RESULT_DO(g_AlbumErrorConverter.Convert((::std::unique_lock<decltype(g_AlbumGlobalMutex)>(g_AlbumGlobalMutex), Function())));


#elif defined(NN_CAPSRV_USE_DIRECT_FUNCTION_CALL)

#define NN_CAPSRV_SERVER_DO(Function)  \
    NN_RESULT_DO(g_AlbumErrorConverter.Convert(Function()));
#define NN_CAPSRV_SERVER_SYNC_DO  NN_CAPSRV_SERVER_DO

#endif

namespace nn { namespace capsrv { namespace server {

Result AlbumApplicationServiceImpl::GetAlbumFileListEx0ByAruid(nn::sf::Out<uint64_t> pOutCount, const nn::sf::OutArray<ApplicationAlbumFileEntry>& pOutList, AlbumFileContentsType contents, AlbumFileDateTime beginDateTime, AlbumFileDateTime endDateTime, applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_CAPSRV_IPC_TRACE_U(GetAlbumFileListEx0ByAruid);

    // アプリ向けに許容されている取得可能な contents 種別
    NN_RESULT_THROW_UNLESS(contents == AlbumFileContents_ScreenShot ||
                           contents == AlbumFileContents_Movie ||
                           contents == AlbumFileContents_ExtraMovie,
                           ResultAlbumInvalidFileContents());

    AlbumFileContentsFlag contentsMask = {};
    contentsMask.Set(contents);

    ncm::ApplicationId applicationId = {};
    NN_RESULT_DO(g_ApplicationResourceManager.GetApplicationIdFromAruid(&applicationId, aruid));

    // ファイルリストを生成して返す
    int totalCount = 0;
    int leftCount = pOutList.GetLength();
    ApplicationAlbumFileEntry* pNext = pOutList.GetData();

    for (AlbumStorageType storage : { AlbumStorage_Nand, AlbumStorage_Sd} )
    {
        // マウントされていないストレージは対象ファイル数 0 として扱い、
        // 次の検索対象ストレージへ処理を移す
        [&]() -> nn::Result {
            NN_CAPSRV_SERVER_SYNC_DO((
                [&]() -> nn::Result {
                    int listLength = 0;
                    NN_RESULT_DO(g_AlbumManager.GetFileListForApplication(
                        &listLength,
                        pNext,
                        leftCount,
                        storage,
                        contentsMask,
                        nullptr,
                        beginDateTime,
                        endDateTime,
                        applicationId
                        ));

                    pNext += listLength;
                    leftCount -= listLength;
                    totalCount += listLength;
                    NN_RESULT_SUCCESS;
                }
            ));
            NN_RESULT_SUCCESS;
        }();
    };

    NN_SDK_ASSERT(leftCount >= 0);
    // ファイルリストの残り部分をゼロクリア
    std::memset(pNext, 0, leftCount * sizeof(ApplicationAlbumFileEntry));
    *pOutCount = totalCount;
    NN_RESULT_SUCCESS;
}

Result AlbumApplicationServiceImpl::GetAlbumFileListEx1ByAruid(nn::sf::Out<uint64_t> pOutCount, const nn::sf::OutArray<ApplicationAlbumFileEntry>& pOutList, AlbumFileContentsType contents, const account::Uid& uid, AlbumFileDateTime beginDateTime, AlbumFileDateTime endDateTime, applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_CAPSRV_IPC_TRACE_U(GetAlbumFileListEx1ByAruid);

    // アプリ向けに許容されている取得可能な contents 種別
    NN_RESULT_THROW_UNLESS(contents == AlbumFileContents_ScreenShot ||
                           contents == AlbumFileContents_Movie ||
                           contents == AlbumFileContents_ExtraMovie,
                           ResultAlbumInvalidFileContents());

    AlbumFileContentsFlag contentsMask = {};
    contentsMask.Set(contents);

    ncm::ApplicationId applicationId = {};
    NN_RESULT_DO(g_ApplicationResourceManager.GetApplicationIdFromAruid(&applicationId, aruid));

    // ファイルリストを生成して返す
    int totalCount = 0;
    int leftCount = pOutList.GetLength();
    ApplicationAlbumFileEntry* pNext = pOutList.GetData();

    for (AlbumStorageType storage : { AlbumStorage_Nand, AlbumStorage_Sd} )
    {
        // マウントされていないストレージは対象ファイル数 0 として扱い、
        // 次の検索対象ストレージへ処理を移す
        [&]() -> nn::Result {
            NN_CAPSRV_SERVER_SYNC_DO((
                [&]() -> nn::Result {
                    int listLength = 0;
                    NN_RESULT_DO(g_AlbumManager.GetFileListForApplication(
                        &listLength,
                        pNext,
                        leftCount,
                        storage,
                        contentsMask,
                        &uid,
                        beginDateTime,
                        endDateTime,
                        applicationId
                        ));

                    pNext += listLength;
                    leftCount -= listLength;
                    totalCount += listLength;
                    NN_RESULT_SUCCESS;
                }
            ));
            NN_RESULT_SUCCESS;
        }();
    };

    NN_SDK_ASSERT(leftCount >= 0);
    // ファイルリストの残り部分をゼロクリア
    std::memset(pNext, 0, leftCount * sizeof(ApplicationAlbumFileEntry));
    *pOutCount = totalCount;
    NN_RESULT_SUCCESS;
}

Result AlbumApplicationServiceImpl::GetAlbumFileListByAruid(nn::sf::Out<uint64_t> pOutCount, const nn::sf::OutArray<ApplicationAlbumFileEntry>& pOutList, AlbumFileContentsType contents, int64_t beginPosixTimeValue, int64_t endPosixTimeValue, applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    auto convertToDateTime = [](int64_t posixTimeValue) -> AlbumFileDateTime
    {
        time::PosixTime posixTime = {};
        posixTime.value = posixTimeValue;

        AlbumFileDateTime dateTime;
        dateTime.FromCalendarTime(time::ToCalendarTimeInUtc(posixTime));
        return dateTime;
    };

    auto beginDateTime = convertToDateTime(beginPosixTimeValue);
    auto endDateTime   = convertToDateTime(endPosixTimeValue);

    return this->GetAlbumFileListEx0ByAruid(pOutCount, pOutList, contents, beginDateTime, endDateTime, aruid);
}

Result AlbumApplicationServiceImpl::DeleteAlbumFileByAruid(const ApplicationAlbumFileEntry& srcEntry, AlbumFileContentsType contents, applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_CAPSRV_IPC_TRACE_U(DeleteAlbumFileByAruid);
    // アプリ向けに許容されている削除可能な contents 種別
    NN_RESULT_THROW_UNLESS(contents == AlbumFileContents_ExtraMovie,
                           ResultAlbumInvalidFileContents());

    AlbumFileContentsFlag contentsMask = {};
    contentsMask.Set(contents);

    ncm::ApplicationId applicationId = {};
    NN_RESULT_DO(g_ApplicationResourceManager.GetApplicationIdFromAruid(&applicationId, aruid));

    AlbumEntry entry = {};
    NN_RESULT_DO(detail::DecryptApplicationAlbumEntryAndCheckApplicationId(&entry, srcEntry._encrypted, applicationId));

    // AlbumFileContents が一致しない場合は NotFound を返す
    AlbumFileId& fileId = entry.fileId;
    NN_RESULT_THROW_UNLESS(fileId.contents == contents, ResultAlbumFileNotFound());

    NN_CAPSRV_SERVER_SYNC_DO((
        [&]() -> nn::Result {
            return g_AlbumManager.DeleteFile(&fileId);
        }
    ));
    NN_RESULT_SUCCESS;
}

Result AlbumApplicationServiceImpl::DeleteAlbumFileByAruidForDebug(const ApplicationAlbumFileEntry& srcEntry, applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_CAPSRV_IPC_TRACE_U(DeleteAlbumFileByAruidForDebug);

    // 製品機では使用不可
    NN_RESULT_THROW_UNLESS(!IsProdMode(), ResultNotSupported());

    AlbumFileContentsFlag contentsMask = {};
    contentsMask.Set(AlbumFileContents_ScreenShot)
                .Set(AlbumFileContents_Movie)
                .Set(AlbumFileContents_ExtraScreenShot)
                .Set(AlbumFileContents_ExtraMovie);

    ncm::ApplicationId applicationId = {};
    NN_RESULT_DO(g_ApplicationResourceManager.GetApplicationIdFromAruid(&applicationId, aruid));

    AlbumEntry entry = {};
    NN_RESULT_DO(detail::DecryptApplicationAlbumEntryAndCheckApplicationId(&entry, srcEntry._encrypted, applicationId));

    // AlbumFileContents が一致しない場合は NotFound を返す
    NN_CAPSRV_SERVER_SYNC_DO((
        [&]() -> nn::Result {
            return g_AlbumManager.DeleteFile(&entry.fileId);
        }
    ));
    NN_RESULT_SUCCESS;
}

Result AlbumApplicationServiceImpl::GetAlbumFileSizeByAruid(nn::sf::Out<uint64_t> outSize, const ApplicationAlbumFileEntry& srcEntry, applet::AppletResourceUserId aruid)
{
    NN_CAPSRV_IPC_TRACE_U(GetAlbumFileSizeByAruid);

    ncm::ApplicationId applicationId = {};
    NN_RESULT_DO(g_ApplicationResourceManager.GetApplicationIdFromAruid(&applicationId, aruid));

    AlbumEntry entry = {};
    NN_RESULT_DO(detail::DecryptApplicationAlbumEntryAndCheckApplicationId(&entry, srcEntry._encrypted, applicationId));

    NN_CAPSRV_SERVER_SYNC_DO((
        [&]() -> nn::Result {
            size_t size = 0;
            nn::Result result = g_AlbumManager.GetFileSize(
                &size,
                &entry.fileId
                );
            outSize.Set(static_cast<uint64_t>(size));
            return result;
        }
    ));
    NN_RESULT_SUCCESS;
}

Result AlbumApplicationServiceImpl::LoadAlbumScreenShotImageByAruid(
    nn::sf::Out<sf::LoadAlbumScreenShotImageOutputForApplication> pOut,
    nn::sf::OutBuffer outBuffer,
    const ApplicationAlbumFileEntry& srcEntry,
    const ScreenShotDecodeOption& option,
    nn::sf::OutBuffer workBuffer,
    applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
{
    NN_CAPSRV_IPC_TRACE_U(LoadAlbumScreenShotImageByAruid);

    int width = 0;
    int height = 0;
    ScreenShotAttribute attribute = {};

    ncm::ApplicationId applicationId = {};
    NN_RESULT_DO(g_ApplicationResourceManager.GetApplicationIdFromAruid(&applicationId, aruid));

    AlbumEntry entry = {};
    NN_RESULT_DO(detail::DecryptApplicationAlbumEntryAndCheckApplicationId(&entry, srcEntry._encrypted, applicationId));

    void* pBuffer = outBuffer.GetPointerUnsafe();
    size_t bufferSize = outBuffer.GetSize();

    void* pFileBuffer = workBuffer.GetPointerUnsafe();
    size_t fileBufferSize = workBuffer.GetSize();
    NN_UTIL_SCOPE_EXIT{ std::memset(pFileBuffer, 0, fileBufferSize); };

    NN_CAPSRV_SERVER_SYNC_DO((
        [&]() -> nn::Result {
            NN_RESULT_DO(album::LoadAndDecodeScreenShot(
                &width,
                &height,
                &attribute,
                nullptr,
                &pOut->applicationData,
                nullptr,
                pBuffer,
                bufferSize,
                pFileBuffer,
                fileBufferSize,
                entry.fileId,
                option,
                &g_AlbumManager
            ));
            NN_RESULT_SUCCESS;
        }
    ));

    pOut->width  = static_cast<uint64_t>(width);
    pOut->height = static_cast<uint64_t>(height);
    pOut->attribute.CopyFromScreenShotAttribute(attribute);
    NN_RESULT_SUCCESS;
}

Result AlbumApplicationServiceImpl::LoadAlbumScreenShotThumbnailImageByAruid(
    nn::sf::Out<sf::LoadAlbumScreenShotImageOutputForApplication> pOut,
    nn::sf::OutBuffer outBuffer,
    const ApplicationAlbumFileEntry& srcEntry,
    const ScreenShotDecodeOption& option,
    nn::sf::OutBuffer workBuffer,
    applet::AppletResourceUserId aruid
    ) NN_NOEXCEPT
{
    NN_CAPSRV_IPC_TRACE_U(LoadAlbumScreenShotThumbnailImageByAruid);

    int width = 0;
    int height = 0;
    ScreenShotAttribute attribute = {};

    ncm::ApplicationId applicationId = {};
    NN_RESULT_DO(g_ApplicationResourceManager.GetApplicationIdFromAruid(&applicationId, aruid));

    AlbumEntry entry = {};
    NN_RESULT_DO(detail::DecryptApplicationAlbumEntryAndCheckApplicationId(&entry, srcEntry._encrypted, applicationId));

    void* pBuffer = outBuffer.GetPointerUnsafe();
    size_t bufferSize = outBuffer.GetSize();

    void* pFileBuffer = workBuffer.GetPointerUnsafe();
    size_t fileBufferSize = workBuffer.GetSize();
    NN_UTIL_SCOPE_EXIT{ std::memset(pFileBuffer, 0, fileBufferSize); };

    NN_CAPSRV_SERVER_SYNC_DO((
        [&]() -> nn::Result {
            NN_RESULT_DO(album::LoadAndDecodeScreenShotThumbnail(
                &width,
                &height,
                &attribute,
                nullptr,
                &pOut->applicationData,
                nullptr,
                pBuffer,
                bufferSize,
                pFileBuffer,
                fileBufferSize,
                entry.fileId,
                option,
                &g_AlbumManager
            ));
            NN_RESULT_SUCCESS;
        }
    ));

    pOut->width  = static_cast<uint64_t>(width);
    pOut->height = static_cast<uint64_t>(height);
    pOut->attribute.CopyFromScreenShotAttribute(attribute);
    NN_RESULT_SUCCESS;
}

namespace {

    bool IsAlbumFileContentsSupportedForStorage(AlbumFileContentsType contents, AlbumStorageType storage) NN_NOEXCEPT
    {
        bool isExtra = IsExtraAlbumFileContents(contents);
        switch (storage)
        {
            case AlbumStorage_Nand:
            {
                return false
                    || !isExtra
                    || (isExtra && g_EnvironmentInfo.IsAlbumExtraMovieFileSupportedForNandStorage());
            }
            case AlbumStorage_Sd:
            {
                return true;
            }
            default: NN_UNEXPECTED_DEFAULT;
        }
    }
}

Result AlbumApplicationServiceImpl::PrecheckToCreateContentsByAruid(AlbumFileContentsType contents, uint64_t requiredSize, applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_CAPSRV_IPC_TRACE_U(PrecheckToCreateContentsByAruid);
    NN_UNUSED(aruid);

    // ストレージの空き容量をチェック
    int64_t mostFreeSpace = 0;
    for (AlbumStorageType storage : { AlbumStorage_Nand, AlbumStorage_Sd })
    {
        // コンテンツ種別とストレージの組合せが許容されているか？
        if (!IsAlbumFileContentsSupportedForStorage(contents, storage))
        {
            continue;
        }

        int pathLength = 0;
        char rootpath[album::AlbumPathUtility::ExtraPathSize];
        album::AlbumPathUtility::GetStorageRootPath(&pathLength, rootpath, sizeof(rootpath), storage);

        // SD カードがマウントされていなければ fs::ResultNotMounted() が返る
        int64_t freeSpace = 0;
        auto result = fs::GetFreeSpaceSize(&freeSpace, rootpath);
        NN_CAPSRV_LOG_DEV("PrecheckToCreateContentsByAruid(): path=\x22%s\x22 is %s\n", rootpath, result <= fs::ResultNotMounted() ? "not mounted" : "mounted");
        NN_RESULT_DO(result);

        NN_CAPSRV_LOG_DEV("storage=%d: requiredSize=%lld / freeSpace=%lld\n", storage, requiredSize, freeSpace);
        if (freeSpace > mostFreeSpace)
        {
            mostFreeSpace = freeSpace;
        }
    }

    // 各ストレージごとに対象ファイル数をカウントして総数を求める
    int64_t totalCount = 0;
    int64_t totalCountLimit = 0;

    for (AlbumStorageType storage : { AlbumStorage_Nand, AlbumStorage_Sd })
    {
        // コンテンツ種別とストレージの組合せが許容されているか？
        if (!IsAlbumFileContentsSupportedForStorage(contents, storage))
        {
            continue;
        }

        AlbumCacheData albumCacheData = {};
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                return g_AlbumManager.GetStorageCache(&albumCacheData, storage, contents);
            }
        ));
        auto count = albumCacheData.fileCount;
        totalCount += count;

        // 各ストレージでの最大数を取得
        auto countLimit = g_EnvironmentInfo.GetFileCountLimit(storage, contents);
        totalCountLimit += countLimit;
        NN_CAPSRV_LOG_DEV("storage=%d: fileCount=%lld, fileCountLimit=%lld\n", storage, count, countLimit);
    };
    NN_RESULT_THROW_UNLESS(totalCount < totalCountLimit, ResultAlbumFileCountLimit());

    // 残容量チェック
    NN_RESULT_THROW_UNLESS(mostFreeSpace >= requiredSize, ResultAlbumIsFull());
    NN_RESULT_SUCCESS;
}

Result AlbumApplicationServiceImpl::OpenAccessorSessionForApplication(nn::sf::Out<nn::sf::SharedPointer<nn::capsrv::sf::IAlbumAccessorApplicationSession>> outValue, const ApplicationAlbumFileEntry& srcEntry, applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_CAPSRV_IPC_TRACE_U(OpenAccessorSessionForApplication);

    ncm::ApplicationId applicationId = {};
    NN_RESULT_DO(g_ApplicationResourceManager.GetApplicationIdFromAruid(&applicationId, aruid));

    AlbumEntry entry = {};
    NN_RESULT_DO(detail::DecryptApplicationAlbumEntryAndCheckApplicationId(&entry, srcEntry._encrypted, applicationId));

    nn::sf::SharedPointer<sf::IAlbumAccessorApplicationSession> p;
    NN_RESULT_DO(AlbumAccessorSessionFactory::Create(&p));

    outValue.Set(p);
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::capsrv::server
