﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_LoadAndDecodeScreenShot.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>
#include <nn/capsrv/capsrv_ViewerThumbnailFormat.h>
#include <nn/capsrv/capsrv_Result.h>

#include "../capsrvServer_Config.h"
#include "../../capsrv_Macro.h"
#include "capsrvServer_AlbumManager.h"
#include "../../capsrv_DecoderControlApi.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    nn::Result LoadAndDecodeScreenShot(
        int* pOutImageWidth,
        int* pOutImageHeight,
        ScreenShotAttribute* pOutAttribute,
        AppletData* pOutAppletData,
        ApplicationData* pOutApplicationData,
        SystemReservedInfo* pOutSystemReservedInfo,
        void *pImageBuffer,
        size_t imageBufferSize,
        void* pFileDataBuffer,
        size_t fileDataBufferSize,
        const AlbumFileId& fileId,
        const ScreenShotDecodeOption& option,
        AlbumManager* pAlbumManager
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutImageWidth);
        NN_SDK_REQUIRES_NOT_NULL(pOutImageHeight);

        const int width = ScreenShotWidth;
        const int height = ScreenShotHeight;

        NN_CAPSRV_PROCESS_START();
        NN_CAPSRV_PROCESS_ROLLBACK({
            if(pOutAttribute)
            {
                std::memset(pOutAttribute, 0, sizeof(ScreenShotAttribute));
            }
            if(pOutAppletData)
            {
                std::memset(pOutAppletData, 0, sizeof(AppletData));
            }
            if(pOutApplicationData)
            {
                std::memset(pOutApplicationData, 0, sizeof(ApplicationData));
            }
            if(pOutSystemReservedInfo)
            {
                std::memset(pOutSystemReservedInfo, 0, sizeof(SystemReservedInfo));
            }
        });

        size_t fileSize = 0;
        AlbumFileAttribute attr = {};
        NN_RESULT_DO(pAlbumManager->LoadFileEx(&attr, pOutAppletData, pOutApplicationData, pOutSystemReservedInfo, &fileSize, pFileDataBuffer, fileDataBufferSize, &fileId));

        auto decodeResult = nn::capsrv::DecodeJpeg(
            pImageBuffer,
            imageBufferSize,
            pFileDataBuffer,
            fileSize,
            width,
            height,
            option
        );
        if(decodeResult.IsFailure())
        {
            NN_CAPSRV_LOG_DEV("Decode error(full): %d-%d\n", decodeResult.GetModule(), decodeResult.GetDescription());
            NN_RESULT_THROW(decodeResult);
        }

        NN_CAPSRV_PROCESS_SUCCESS();
        *pOutImageWidth = width;
        *pOutImageHeight = height;
        if(pOutAttribute)
        {
            *pOutAttribute = attr.attribute.screenshot;
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result LoadAndDecodeScreenShotThumbnail(
        int* pOutImageWidth,
        int* pOutImageHeight,
        ScreenShotAttribute* pOutAttribute,
        AppletData* pOutAppletData,
        ApplicationData* pOutApplicationData,
        SystemReservedInfo* pOutSystemReservedInfo,
        void *pImageBuffer,
        size_t imageBufferSize,
        void* pThumbDataBuffer,
        size_t thumbDataBufferSize,
        const AlbumFileId& fileId,
        const ScreenShotDecodeOption& option,
        AlbumManager* pAlbumManager
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutImageWidth);
        NN_SDK_REQUIRES_NOT_NULL(pOutImageHeight);

        const int width = ViewerThumbnailImageSize_Width;
        const int height = ViewerThumbnailImageSize_Height;

        NN_CAPSRV_PROCESS_START();
        NN_CAPSRV_PROCESS_ROLLBACK({
            if(pOutAttribute)
            {
                std::memset(pOutAttribute, 0, sizeof(ScreenShotAttribute));
            }
            if(pOutAppletData)
            {
                std::memset(pOutAppletData, 0, sizeof(AppletData));
            }
            if(pOutApplicationData)
            {
                std::memset(pOutApplicationData, 0, sizeof(ApplicationData));
            }
            if(pOutSystemReservedInfo)
            {
                std::memset(pOutSystemReservedInfo, 0, sizeof(SystemReservedInfo));
            }
        });

        size_t thumbSize = 0;
        AlbumFileAttribute attr = {};
        NN_RESULT_DO(pAlbumManager->LoadFileThumbnailEx(&attr, pOutAppletData, pOutApplicationData, pOutSystemReservedInfo, &thumbSize, pThumbDataBuffer, thumbDataBufferSize, &fileId));


        auto decodeResult = nn::capsrv::DecodeJpeg(
            pImageBuffer,
            imageBufferSize,
            pThumbDataBuffer,
            thumbSize,
            width,
            height,
            option
        );
        if(decodeResult.IsFailure())
        {
            NN_CAPSRV_LOG_DEV("Decode error(thumb): %d-%d\n", decodeResult.GetModule(), decodeResult.GetDescription());
            NN_RESULT_THROW(decodeResult);
        }



        NN_CAPSRV_PROCESS_SUCCESS();
        *pOutImageWidth = width;
        *pOutImageHeight = height;
        if(pOutAttribute)
        {
            *pOutAttribute = attr.attribute.screenshot;
        }
        NN_RESULT_SUCCESS;
    }


}}}}

