﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/capsrv/movie/capsrv_MovieReaderFileSystem.h>
#include <nn/capsrv/movie/capsrv_MovieReaderFileSystemImpl.h>

#include <mutex>
#include <algorithm>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>
#include "../capsrv_Macro.h"

//#define NN_CAPSRV_LOG_MOVIEFS_HEAD()    NN_SDK_LOG("[capsrv]")
//#define NN_CAPSRV_LOG_MOVIEFS_DATA(...) NN_SDK_LOG(__VA_ARGS__)

#ifndef NN_CAPSRV_LOG_MOVIEFS_HEAD
#define NN_CAPSRV_LOG_MOVIEFS_HEAD()
#endif
#ifndef NN_CAPSRV_LOG_MOVIEFS_DATA
#define NN_CAPSRV_LOG_MOVIEFS_DATA(...)
#endif

namespace {
    int g_CallDepth = 0;
}

#define NN_CAPSRV_LOG_MOVIEFS_I(...) \
    do{                                 \
        NN_CAPSRV_LOG_MOVIEFS_HEAD();   \
        for(int i = 0; i < g_CallDepth; i++)    \
        {                                       \
            NN_CAPSRV_LOG_MOVIEFS_DATA("  ");        \
        }                                       \
        NN_CAPSRV_LOG_MOVIEFS_DATA(__VA_ARGS__);     \
    }while(NN_STATIC_CONDITION(false));

#define NN_CAPSRV_LOG_MOVIEFS_II(...)       \
    NN_CAPSRV_LOG_MOVIEFS_I(__VA_ARGS__)    \
    g_CallDepth++;                          \
    NN_UTIL_SCOPE_EXIT{ g_CallDepth--; };

#define NN_CAPSRV_LOG_MOVIEFS_LINE(...)  NN_CAPSRV_LOG_MOVIEFS_I(__VA_ARGS__)

#define NN_CAPSRV_LOG_MOVIEFS_MRFS(...)  NN_CAPSRV_LOG_MOVIEFS_II("MRFS:" __VA_ARGS__)
#define NN_CAPSRV_LOG_MOVIEFS_MRFSI(...) NN_CAPSRV_LOG_MOVIEFS_II("MRFSI:" __VA_ARGS__)
#define NN_CAPSRV_LOG_MOVIEFS_MRFSP(...) NN_CAPSRV_LOG_MOVIEFS_II("MRFSP:" __VA_ARGS__)
#define NN_CAPSRV_LOG_MOVIEFS_MRFI(...)  NN_CAPSRV_LOG_MOVIEFS_II("MRFI:" __VA_ARGS__)
#define NN_CAPSRV_LOG_MOVIEFS_MRFP(...)  NN_CAPSRV_LOG_MOVIEFS_II("MRFP:" __VA_ARGS__)

namespace nn{ namespace capsrv{ namespace movie{

    namespace {

        detail::MovieReaderFileSystemImpl* g_pMovieReaderFileSystemImpl = nullptr;

    }// anon namespace

    namespace detail{

        MovieReaderFileProxy::MovieReaderFileProxy(MovieReaderFileImpl* pImpl) NN_NOEXCEPT
            : m_pImpl(pImpl)
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFP("ctor(p=%llX)\n", pImpl);
            NN_SDK_REQUIRES_NOT_NULL(pImpl);
            NN_SDK_REQUIRES(pImpl->IsInitialized());
        }

        MovieReaderFileProxy::~MovieReaderFileProxy() NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFP("dtor(p=%llX)\n", m_pImpl);
            // Close
            m_pImpl->NotifyClosed();
        }

        void* MovieReaderFileProxy::operator new(size_t size) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(g_pMovieReaderFileSystemImpl);
            return g_pMovieReaderFileSystemImpl->AllocateProxyMemory(size, NN_ALIGNOF(MovieReaderFileProxy));
        }

        void MovieReaderFileProxy::operator delete(void* p) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(g_pMovieReaderFileSystemImpl);
            g_pMovieReaderFileSystemImpl->FreeProxyMemory(p);
        }

        nn::Result MovieReaderFileProxy::DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFP("DoRead()\n");
            return m_pImpl->DoReadImpl(outValue, offset, buffer, size, option);
        }

        nn::Result MovieReaderFileProxy::DoWrite(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFP("DoWrite()\n");
            return m_pImpl->DoWriteImpl(offset, buffer, size, option);
        }

        nn::Result MovieReaderFileProxy::DoFlush() NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFP("DoFlush()\n");
            return m_pImpl->DoFlushImpl();
        }

        nn::Result MovieReaderFileProxy::DoSetSize(int64_t size) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFP("DoSetSize()\n");
            return m_pImpl->DoSetSizeImpl(size);
        }

        nn::Result MovieReaderFileProxy::DoGetSize(int64_t* outValue) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFP("DoGetSize()\n");
            return m_pImpl->DoGetSizeImpl(outValue);
        }

        nn::Result MovieReaderFileProxy::DoOperateRange(
            void* outBuffer,
            size_t outBufferSize,
            fs::OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize
        ) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFP("DoOperateRange()\n");
            return m_pImpl->DoOperateRangeImpl(outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize);
        }

        //-------------------------------------------------------

        MovieReaderFileSystemProxy::MovieReaderFileSystemProxy(MovieReaderFileSystemImpl* pImpl) NN_NOEXCEPT
            : m_pImpl(pImpl)
        {
            NN_SDK_REQUIRES_NOT_NULL(pImpl);
            NN_CAPSRV_LOG_MOVIEFS_MRFSP("ctor(p=%llX)\n", pImpl);
        }

        MovieReaderFileSystemProxy::~MovieReaderFileSystemProxy() NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSP("dtor(p=%llX)\n", m_pImpl);
        }

        void* MovieReaderFileSystemProxy::operator new(size_t size) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(g_pMovieReaderFileSystemImpl);
            return g_pMovieReaderFileSystemImpl->AllocateProxyMemory(size, NN_ALIGNOF(MovieReaderFileSystemProxy));
        }

        void MovieReaderFileSystemProxy::operator delete(void* p) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(g_pMovieReaderFileSystemImpl);
            g_pMovieReaderFileSystemImpl->FreeProxyMemory(p);
        }

        nn::Result MovieReaderFileSystemProxy::DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSP("DoCreateFile(path=%s)\n", path);
            NN_UNUSED(path);
            NN_UNUSED(size);
            NN_UNUSED(option);
            NN_RESULT_THROW(nn::fs::ResultNotImplemented());
        }

        nn::Result MovieReaderFileSystemProxy::DoDeleteFile(const char* path) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSP("DoDeleteFile(path=%s)\n", path);
            NN_UNUSED(path);
            NN_RESULT_THROW(nn::fs::ResultNotImplemented());
        }

        nn::Result MovieReaderFileSystemProxy::DoCreateDirectory(const char* path) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSP("DoCreateDirectory(path=%s)\n", path);
            NN_UNUSED(path);
            NN_RESULT_THROW(nn::fs::ResultNotImplemented());
        }

        nn::Result MovieReaderFileSystemProxy::DoDeleteDirectory(const char* path) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSP("DoDeleteDirectory(path=%s)\n", path);
            NN_UNUSED(path);
            NN_RESULT_THROW(nn::fs::ResultNotImplemented());
        }

        nn::Result MovieReaderFileSystemProxy::DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSP("DoDeleteDirectoryRecursively(path=%s)\n", path);
            NN_UNUSED(path);
            NN_RESULT_THROW(nn::fs::ResultNotImplemented());
        }

        nn::Result MovieReaderFileSystemProxy::DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSP("DoCleanDirectoryRecursively(path=%s)\n", path);
            NN_UNUSED(path);
            NN_RESULT_THROW(nn::fs::ResultNotImplemented());
        }

        nn::Result MovieReaderFileSystemProxy::DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSP("DoRenameFile(curPath=%s,newPath=%s)\n", currentPath, newPath);
            NN_UNUSED(currentPath);
            NN_UNUSED(newPath);
            NN_RESULT_THROW(nn::fs::ResultNotImplemented());
        }

        nn::Result MovieReaderFileSystemProxy::DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSP("DoRenameDirectory(curPath=%s,newPath=%s)\n", currentPath, newPath);
            NN_UNUSED(currentPath);
            NN_UNUSED(newPath);
            NN_RESULT_THROW(nn::fs::ResultNotImplemented());
        }

        nn::Result MovieReaderFileSystemProxy::DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSP("DoGetEntryType(path=%s)\n", path);
            return m_pImpl->DoGetEntryTypeImpl(outValue, path);
        }

        nn::Result MovieReaderFileSystemProxy::DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSP("DoOpenFile(path=%s)\n", path);
            return m_pImpl->DoOpenFileImpl(outValue, path, mode);
        }

        nn::Result MovieReaderFileSystemProxy::DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSP("DoOpenDirectory(path=%s)\n", path);
            NN_UNUSED(outValue);
            NN_UNUSED(path);
            NN_UNUSED(mode);
            NN_RESULT_THROW(nn::fs::ResultNotImplemented());
        }

        nn::Result MovieReaderFileSystemProxy::DoCommit() NN_NOEXCEPT
        {
            NN_RESULT_THROW(nn::fs::ResultNotImplemented());
        }

        //-------------------------------------------------------

        size_t MovieReaderFileImpl::GetRequiredMemorySize(int64_t cacheChunkSize, int64_t cacheChunkCount) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_GREATER(cacheChunkSize, 0);
            NN_SDK_REQUIRES_EQUAL(cacheChunkSize % AlbumMovieDataUnitSize, 0);
            NN_SDK_REQUIRES_GREATER(cacheChunkCount, 0);

            return CachedMovieStream::GetRequiredWorkMemorySize(cacheChunkSize, cacheChunkCount);
        }

        size_t MovieReaderFileImpl::GetRequiredMemoryAlignment() NN_NOEXCEPT
        {
            return CachedMovieStream::GetRequiredWorkMemoryAlignment();
        }

        MovieReaderFileImpl::MovieReaderFileImpl() NN_NOEXCEPT
        {
            m_Accessor.handle = {};
        }

        bool MovieReaderFileImpl::IsInitialized() const NN_NOEXCEPT
        {
            return m_Accessor.handle != AlbumMovieReadStreamHandle();
        }

        void MovieReaderFileImpl::Initialize(
            const FilePath& filename,
            AlbumMovieReadStreamHandle handle,
            int64_t fileSize,
            int64_t cacheChunkSize,
            int64_t cacheChunkCount,
            void* memory,
            size_t memorySize
        ) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFI("Initialize()\n");
            NN_CAPSRV_LOG_MOVIEFS_LINE("  path  =%s\n", filename);
            NN_CAPSRV_LOG_MOVIEFS_LINE("  handle=%llu\n", handle);
            NN_CAPSRV_LOG_MOVIEFS_LINE("  size  =%lld bytes\n", fileSize);
            NN_CAPSRV_LOG_MOVIEFS_LINE("  cache =%lld bytes x %llu\n", cacheChunkSize, cacheChunkCount);
            NN_CAPSRV_LOG_MOVIEFS_LINE("  memory=%llX (%llu bytes)\n", memory, memorySize);
            NN_SDK_REQUIRES_NOT_EQUAL(handle, AlbumMovieReadStreamHandle());
            NN_SDK_REQUIRES_GREATER(cacheChunkSize, 0);
            NN_SDK_REQUIRES_EQUAL(cacheChunkSize % AlbumMovieDataUnitSize, 0);
            NN_SDK_REQUIRES_GREATER(cacheChunkCount, 0);
            NN_SDK_REQUIRES_NOT_NULL(memory);
            NN_SDK_REQUIRES_ALIGNED(memory, GetRequiredMemoryAlignment());
            NN_SDK_REQUIRES_GREATER_EQUAL(memorySize, GetRequiredMemorySize(cacheChunkSize, cacheChunkCount));
            NN_UNUSED(memorySize);

            m_Mutex.Initialize();
            m_Accessor.handle = handle;
            m_Strategy.Initialize(cacheChunkCount);
            m_Stream.Initialize(
                cacheChunkSize,
                cacheChunkCount,
                m_Accessor.GetAccessor(),
                m_Strategy.GetStrategy(),
                memory,
                memorySize,
                (fileSize + AlbumMovieDataUnitSize - 1) / AlbumMovieDataUnitSize
            );
            m_Filename = filename;
            m_FileSize = fileSize;
            m_OpenCount = 0;
        }

        void MovieReaderFileImpl::Finalize() NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFI("Finalize()\n");
            NN_ABORT_UNLESS_EQUAL(m_OpenCount, 0);
            m_Stream.Finalize();
            m_Strategy.Finalize();
            m_Accessor.handle = AlbumMovieReadStreamHandle();

            m_Filename = {};
            m_FileSize = 0;
        }

        bool MovieReaderFileImpl::IsMatchFilename(const FilePath& filename) const NN_NOEXCEPT
        {
            // 初期化時に値が設定されて以降、値が変化しないので Mutex 不要。
            return m_Filename == filename;
        }

        AlbumMovieReadStreamHandle MovieReaderFileImpl::GetHandle() const NN_NOEXCEPT
        {
            // 初期化時に値が設定されて以降、値が変化しないので Mutex 不要。
            return m_Accessor.handle;
        }

        bool MovieReaderFileImpl::IsOpened() const NN_NOEXCEPT
        {
            // atomic_int なので Mutex 不要。
            return m_OpenCount != 0;
        }

        void MovieReaderFileImpl::NotifyOpened() NN_NOEXCEPT
        {
            // atomic_int なので Mutex 不要。
            m_OpenCount++;
        }

        void MovieReaderFileImpl::NotifyClosed() NN_NOEXCEPT
        {
            // atomic_int なので Mutex 不要。
            m_OpenCount--;
            NN_ABORT_UNLESS_GREATER_EQUAL(m_OpenCount, 0);
        }

        nn::Result MovieReaderFileImpl::DoReadImpl(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFI("DoReadImpl()\n");
            NN_UNUSED(option);

            std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

            return m_Stream.Read(outValue, buffer, size, offset);
        }

        nn::Result MovieReaderFileImpl::DoWriteImpl(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFI("DoWriteImpl()\n");
            NN_UNUSED(option);

            std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

            m_FileSize = std::max(m_FileSize, offset + static_cast<int64_t>(size));
            return m_Stream.Write(offset, buffer,size);
        }

        nn::Result MovieReaderFileImpl::DoFlushImpl() NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFI("DoFlushImpl()\n");

            std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

            return m_Stream.Flush();
        }

        nn::Result MovieReaderFileImpl::DoSetSizeImpl(int64_t size) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFI("DoSetSizeImpl()\n");

            std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

            m_FileSize = size;
            return m_Stream.Resize(size);
        }

        nn::Result MovieReaderFileImpl::DoGetSizeImpl(int64_t* outValue) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFI("DoGetSizeImpl()\n");

            std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

            *outValue = m_FileSize;
            NN_RESULT_SUCCESS;
        }

        nn::Result MovieReaderFileImpl::DoOperateRangeImpl(
            void* outBuffer,
            size_t outBufferSize,
            fs::OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize
        ) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFI("DoOperateRangeImpl()\n");
            NN_UNUSED(outBuffer);
            NN_UNUSED(outBufferSize);
            NN_UNUSED(operationId);
            NN_UNUSED(offset);
            NN_UNUSED(size);
            NN_UNUSED(inBuffer);
            NN_UNUSED(inBufferSize);

            //std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

            NN_RESULT_SUCCESS;
        }


        //-------------------------------------------------------

        const int MovieReaderFileSystemImpl::FileCountMax;

        size_t MovieReaderFileSystemImpl::GetRequiredMemorySizeForFile(int64_t cacheChunkSize, int64_t cacheChunkCount) NN_NOEXCEPT
        {
            return MovieReaderFileImpl::GetRequiredMemorySize(cacheChunkSize, cacheChunkCount);
        }

        size_t MovieReaderFileSystemImpl::GetRequiredMemoryAlignmentForFile() NN_NOEXCEPT
        {
            return MovieReaderFileImpl::GetRequiredMemoryAlignment();
        }

        MovieReaderFileSystemImpl::MovieReaderFileSystemImpl() NN_NOEXCEPT
        {
        }

        void MovieReaderFileSystemImpl::Initialize(const char* mountName) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSI("Initialize()\n");

            // operator new の都合で同時に 1 つしか使えない
            NN_SDK_REQUIRES(g_pMovieReaderFileSystemImpl == nullptr, "Multiple MovieReaderFileSystem instances can't be initialized at the same time.\n");
            NN_SDK_REQUIRES_NOT_NULL(mountName);
            NN_CAPSRV_LOG_MOVIEFS_MRFS("Initialize(m=%s)\n", mountName);

            // まずアロケータを使えるようにする
            m_ProxyHeapHandle = nn::lmem::CreateExpHeap(m_ProxyHeapMemory, sizeof(m_ProxyHeapMemory), nn::lmem::CreationOption_ThreadSafe);
            NN_ABORT_UNLESS_NOT_NULL(m_ProxyHeapHandle);
            m_ProxyAllocateCount = 0;
            g_pMovieReaderFileSystemImpl = this;

            m_Mutex.Initialize();

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::fsa::Register(mountName, std::unique_ptr<nn::fs::fsa::IFileSystem>(new detail::MovieReaderFileSystemProxy(this))));
            m_MountName = FilePath::FromString(mountName);
        }

        void MovieReaderFileSystemImpl::Finalize() NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSI("Finalize()\n");
            NN_SDK_REQUIRES_NOT_NULL(g_pMovieReaderFileSystemImpl);
            NN_SDK_REQUIRES_EQUAL(this, g_pMovieReaderFileSystemImpl);
            // 全部のファイルがクローズされていれば残っているのは FileSystemProxy のみ
            int openedFileCount = m_ProxyAllocateCount - 1;
            NN_SDK_REQUIRES_EQUAL(openedFileCount, 0);
            NN_UNUSED(openedFileCount);

            for(int i = 0; i < FileCountMax; i++)
            {
                NN_SDK_REQUIRES(!m_File[i].IsInitialized());
            }

            nn::fs::Unmount(m_MountName.value);
            //nn::fs::fsa::Unregister(m_MountName.value);
            m_MountName = {};

            // nn::fs::Unmount よりも後にアロケータを使用不能にする
            g_pMovieReaderFileSystemImpl = nullptr;
            NN_SDK_ASSERT_EQUAL(m_ProxyAllocateCount, 0);
            nn::lmem::DestroyExpHeap(m_ProxyHeapHandle);
        }

        int MovieReaderFileSystemImpl::FindEmptyFileSlot() const NN_NOEXCEPT
        {
            for(int i = 0; i < FileCountMax; i++)
            {
                if(!m_File[i].IsInitialized())
                {
                    return i;
                }
            }
            return -1;
        }

        int MovieReaderFileSystemImpl::FindFileSlotByStreamHandle(AlbumMovieReadStreamHandle h) const NN_NOEXCEPT
        {
            for(int i = 0; i < FileCountMax; i++)
            {
                if(m_File[i].IsInitialized() && m_File[i].GetHandle() == h)
                {
                    return i;
                }
            }
            return -1;
        }


        int MovieReaderFileSystemImpl::FindFileSlotByFilename(const char* filename) const NN_NOEXCEPT
        {
            auto path = FilePath::FromString(filename);
            for(int i = 0; i < FileCountMax; i++)
            {
                if(m_File[i].IsInitialized() && m_File[i].IsMatchFilename(path))
                {
                    return i;
                }
            }
            return -1;
        }

        nn::Result MovieReaderFileSystemImpl::RegisterMovieReadStreamDataSection(
            const char* filename,
            AlbumMovieReadStreamHandle streamHandle,
            int64_t cacheChunkSize,
            int64_t cacheChunkCount,
            void* memory,
            size_t memorySize
        ) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSI("RegisterMovieReadStreamDataSection(i=%d,name=%s,h=%llu)\n", index, filename, streamHandle.GetInnerValue());
            NN_SDK_REQUIRES_LESS_EQUAL(nn::util::Strnlen(filename, MovieFilenameLengthMax), MovieFilenameLengthMax);
            NN_SDK_REQUIRES_GREATER(cacheChunkSize, 0);
            NN_SDK_REQUIRES_EQUAL(cacheChunkSize % AlbumMovieDataUnitSize, 0);
            NN_SDK_REQUIRES_GREATER_EQUAL(cacheChunkCount, 1);
            NN_SDK_REQUIRES_NOT_NULL(memory);
            NN_SDK_REQUIRES_ALIGNED(memory, GetRequiredMemoryAlignmentForFile());
            NN_SDK_REQUIRES_GREATER_EQUAL(memorySize, GetRequiredMemorySizeForFile(cacheChunkSize, cacheChunkCount));

            std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

            MovieFilePath name = {};
            nn::util::Strlcpy(name.value + 1, filename, sizeof(name) - 1);
            name.value[0] = '/';
            NN_SDK_ASSERT(name.value[sizeof(name) - 1] == '\0');

            // 同じファイル名で複数回登録できない
            NN_RESULT_THROW_UNLESS(FindFileSlotByFilename(name.value) < 0, nn::fs::ResultAlreadyExists());

            int index = FindEmptyFileSlot();
            NN_RESULT_THROW_UNLESS(index >= 0, nn::fs::ResultFileEntryFull());
            NN_SDK_ASSERT_RANGE(index, 0, FileCountMax);
            auto& file = m_File[index];

            int64_t fileSize = 0;
            NN_RESULT_DO(nn::capsrv::GetAlbumMovieReadStreamDataSize(&fileSize, streamHandle));

            // 初期化
            file.Initialize(
                name,
                streamHandle,
                fileSize,
                cacheChunkSize,
                cacheChunkCount,
                memory,
                memorySize
            );
            NN_RESULT_SUCCESS;
        }

        void MovieReaderFileSystemImpl::UnregisterMovieReadStreamDataSection(const char* filename) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSI("UnregisterMovieReadStreamDataSection(name=%s)\n", filename);

            std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

            MovieFilePath name = {};
            nn::util::Strlcpy(name.value + 1, filename, sizeof(name) - 1);
            name.value[0] = '/';
            NN_SDK_ASSERT(name.value[sizeof(name) - 1] == '\0');

            int index = FindFileSlotByFilename(name.value);
            if(index >= 0 && index < FileCountMax)
            {
                auto& file = m_File[index];
                NN_ABORT_UNLESS(file.IsInitialized());
                NN_ABORT_UNLESS(!file.IsOpened());
                file.Finalize();
            }
            else
            {
                // 見つからなければ何もしない
            }
        }

        void MovieReaderFileSystemImpl::UnregisterMovieReadStreamDataSection(AlbumMovieReadStreamHandle streamHandle) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSI("UnregisterMovieReadStreamDataSection(h=%llX)\n", streamHandle);

            std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

            for(;;)
            {
                int index = FindFileSlotByStreamHandle(streamHandle);
                if(index < 0 || index >= FileCountMax)
                {
                    // すべて登録解除できたら終わり
                    break;
                }

                auto& file = m_File[index];
                NN_ABORT_UNLESS(file.IsInitialized());
                NN_ABORT_UNLESS(!file.IsOpened());
                file.Finalize();
            }
        }

        nn::Result MovieReaderFileSystemImpl::DoGetEntryTypeImpl(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSI("DoGetEntryTypeImpl(path=%s)\n", path);

            std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

            auto index = FindFileSlotByFilename(path);
            NN_RESULT_THROW_UNLESS(index >= 0, nn::fs::ResultPathNotFound());
            NN_SDK_ASSERT_RANGE(index, 0, FileCountMax);

            *outValue = nn::fs::DirectoryEntryType_File;
            NN_RESULT_SUCCESS;
        }

        nn::Result MovieReaderFileSystemImpl::DoOpenFileImpl(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_MOVIEFS_MRFSI("DoOpenFileImpl(path=%s)\n", path);
            NN_UNUSED(mode);

            std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

            NN_CAPSRV_PROCESS_START();

            auto index = FindFileSlotByFilename(path);
            NN_RESULT_THROW_UNLESS(index >= 0, nn::fs::ResultPathNotFound());
            NN_SDK_ASSERT_RANGE(index, 0, FileCountMax);

            m_File[index].NotifyOpened();
            NN_CAPSRV_PROCESS_ROLLBACK(m_File[index].NotifyClosed());

            std::unique_ptr<nn::fs::fsa::IFile> pProxy;
            pProxy = std::unique_ptr<nn::fs::fsa::IFile>(new MovieReaderFileProxy(&m_File[index]));
            NN_RESULT_THROW_UNLESS(pProxy != nullptr, nn::fs::ResultAllocationMemoryFailed());

            NN_CAPSRV_PROCESS_SUCCESS();
            *outValue = std::move(pProxy);
            NN_RESULT_SUCCESS;
        }

        void* MovieReaderFileSystemImpl::AllocateProxyMemory(size_t size, size_t alignment) NN_NOEXCEPT
        {
            auto p = nn::lmem::AllocateFromExpHeap(m_ProxyHeapHandle, size, static_cast<int>(alignment));
            if(p != nullptr)
            {
                m_ProxyAllocateCount++;
            }
            return p;
        }

        void MovieReaderFileSystemImpl::FreeProxyMemory(void* ptr) NN_NOEXCEPT
        {
            if(ptr)
            {
                m_ProxyAllocateCount--;
                nn::lmem::FreeToExpHeap(m_ProxyHeapHandle, ptr);
            }
        }


        //-------------------------------------------------------

    }// namespace detail

    const int MovieReaderFileSystem::FileCountMax;

    size_t MovieReaderFileSystem::GetRequiredMemorySizeForFile(int64_t cacheChunkSize, int64_t cacheChunkCount) NN_NOEXCEPT
    {
        return detail::MovieReaderFileSystemImpl::GetRequiredMemorySizeForFile(cacheChunkSize, cacheChunkCount);
    }

    size_t MovieReaderFileSystem::GetRequiredMemoryAlignmentForFile() NN_NOEXCEPT
    {
        return detail::MovieReaderFileSystemImpl::GetRequiredMemoryAlignmentForFile();
    }

    MovieReaderFileSystem::MovieReaderFileSystem() NN_NOEXCEPT
        : m_pImpl(nullptr)
    {
    }

    bool MovieReaderFileSystem::IsInitialized() const NN_NOEXCEPT
    {
        return m_pImpl != nullptr;
    }

    void MovieReaderFileSystem::Initialize(const char* mountName) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIEFS_MRFS("Initialize()\n");
        NN_CAPSRV_LOG_MOVIEFS_LINE("sizeof(Impl)=%llu, alignof(Impl)=%llu\n", sizeof(detail::MovieReaderFileSystemImpl), NN_ALIGNOF(detail::MovieReaderFileImpl));
        NN_SDK_REQUIRES(!IsInitialized());
#if defined(NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2013)
        NN_STATIC_ASSERT(sizeof(m_Storage) >= sizeof(detail::MovieReaderFileSystemImpl));
        //NN_SDK_ASSERT(NN_ALIGNOF(m_Storage) >= NN_ALIGNOF(detail::MovieReaderFileSystemImpl)); // VS2013 だとコンパイラの内部エラーになる
#else
        NN_STATIC_ASSERT(sizeof(m_Storage) >= sizeof(detail::MovieReaderFileSystemImpl));
        NN_STATIC_ASSERT(NN_ALIGNOF(m_Storage) >= NN_ALIGNOF(detail::MovieReaderFileSystemImpl));
#endif
        m_pImpl = new(m_Storage) detail::MovieReaderFileSystemImpl();
        m_pImpl->Initialize(mountName);
    }

    void MovieReaderFileSystem::Finalize() NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_MOVIEFS_MRFS("Finalize()\n");
        NN_SDK_REQUIRES(IsInitialized());
        m_pImpl->Finalize();
        m_pImpl->~MovieReaderFileSystemImpl();
        m_pImpl = nullptr;
    }

    nn::Result MovieReaderFileSystem::RegisterMovieReadStreamDataSection(
        const char* filename,
        AlbumMovieReadStreamHandle streamHandle,
        int64_t cacheChunkSize,
        int64_t cacheChunkCount,
        void* memory,
        size_t memorySize
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return m_pImpl->RegisterMovieReadStreamDataSection(filename, streamHandle, cacheChunkSize, cacheChunkCount, memory, memorySize);
    }

    void MovieReaderFileSystem::UnregisterMovieReadStreamDataSection(const char* filename) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        m_pImpl->UnregisterMovieReadStreamDataSection(filename);
    }

    void MovieReaderFileSystem::UnregisterMovieReadStreamDataSection(AlbumMovieReadStreamHandle streamHandle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        m_pImpl->UnregisterMovieReadStreamDataSection(streamHandle);
    }

}}}
