﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/capsrv/movie/capsrv_MovieReaderCacheStrategy.h>

#include <limits>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

namespace nn{ namespace capsrv{ namespace movie{

    void MovieReaderCacheStrategy::Entry::Reset() NN_NOEXCEPT
    {
        this->chunkIndex = InvalidChunkIndex;
        this->date = 0;
    }

    //---------------

    const int MovieReaderCacheStrategy::EntryCountMax;
    const int64_t MovieReaderCacheStrategy::InvalidChunkIndex;

    //---------------

    void MovieReaderCacheStrategy::Initialize(int64_t cacheChunkCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(cacheChunkCount, 0);
        NN_SDK_REQUIRES_LESS_EQUAL(cacheChunkCount, EntryCountMax);

        for(int64_t i = 0; i < cacheChunkCount; i++)
        {
            m_EntryList[i].Reset();
        }

        m_EntryCount = cacheChunkCount;
        m_CurrentDate = 0;
    }

    void MovieReaderCacheStrategy::Finalize() NN_NOEXCEPT
    {
        m_EntryCount = 0;
        m_CurrentDate = 0;
    }

    //-----------------

    MovieReaderCacheStrategy::Entry* MovieReaderCacheStrategy::FindEntry(int64_t chunkIndex) NN_NOEXCEPT
    {
        for(int64_t i = 0; i < m_EntryCount; i++)
        {
            if(m_EntryList[i].chunkIndex == chunkIndex)
            {
                return &m_EntryList[i];
            }
        }
        return nullptr;
    }

    void MovieReaderCacheStrategy::NotifyFetched(int64_t chunkIndex, void* userData) NN_NOEXCEPT
    {
        auto pSelf = reinterpret_cast<MovieReaderCacheStrategy*>(userData);
        auto pEntry = pSelf->FindEntry(InvalidChunkIndex);
        NN_SDK_ASSERT_NOT_NULL(pEntry);

        pEntry->chunkIndex = chunkIndex;
        pEntry->date = ++pSelf->m_CurrentDate;
    }

    void MovieReaderCacheStrategy::NotifyInvalidated(int64_t chunkIndex, void* userData) NN_NOEXCEPT
    {
        auto pSelf = reinterpret_cast<MovieReaderCacheStrategy*>(userData);
        auto pEntry = pSelf->FindEntry(chunkIndex);
        NN_SDK_ASSERT_NOT_NULL(pEntry);

        pEntry->Reset();
    }

    void MovieReaderCacheStrategy::NotifyRead(int64_t chunkIndex, int64_t offset, int64_t size, void* userData) NN_NOEXCEPT
    {
        NN_UNUSED(offset);
        NN_UNUSED(size);
        auto pSelf = reinterpret_cast<MovieReaderCacheStrategy*>(userData);
        auto pEntry = pSelf->FindEntry(chunkIndex);
        NN_SDK_ASSERT_NOT_NULL(pEntry);
        pEntry->date = ++pSelf->m_CurrentDate;
    }
    void MovieReaderCacheStrategy::NotifyWritten(int64_t chunkIndex, int64_t offset, int64_t size, void* userData) NN_NOEXCEPT
    {
        NN_UNUSED(offset);
        NN_UNUSED(size);
        auto pSelf = reinterpret_cast<MovieReaderCacheStrategy*>(userData);
        auto pEntry = pSelf->FindEntry(chunkIndex);
        NN_SDK_ASSERT_NOT_NULL(pEntry);
        pEntry->date = ++pSelf->m_CurrentDate;
    }

    int64_t MovieReaderCacheStrategy::QueryChunkIndexToInvalidate(void* userData) NN_NOEXCEPT
    {
        auto pSelf = reinterpret_cast<MovieReaderCacheStrategy*>(userData);

        Entry* pEntry = nullptr;
        uint64_t date = std::numeric_limits<uint64_t>::max();
        for(int64_t i = 0; i < pSelf->m_EntryCount; i++)
        {
            auto& e = pSelf->m_EntryList[i];
            if(e.chunkIndex != InvalidChunkIndex && e.date < date)
            {
                pEntry = &e;
                date = e.date;
            }
        }

        if(pEntry == nullptr)
        {
            return InvalidChunkIndex;
        }
        NN_SDK_ASSERT_NOT_EQUAL(pEntry->chunkIndex, InvalidChunkIndex);
        NN_SDK_ASSERT_GREATER(date, 0);

        return pEntry->chunkIndex;
    }

    MovieStreamCacheStrategy MovieReaderCacheStrategy::GetStrategy() NN_NOEXCEPT
    {
        MovieStreamCacheStrategy strategy = {};
        strategy.userData = this;
        strategy.notifyFetched = NotifyFetched;
        strategy.notifyInvalidated = NotifyInvalidated;
        strategy.notifyRead = NotifyRead;
        strategy.notifyWritten = NotifyWritten;
        strategy.queryChunkIndexToInvalidate = QueryChunkIndexToInvalidate;
        return strategy;
    }

}}}
