﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrv_ImageBuffer.h"
#include "capsrv_ImageBuffer-module.nvnflinger.h"

#include <cstring>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include "capsrv_ResultCapture.h"
#include "capsrv_CaptureModule-module.nvnflinger.h"
#include "capsrv_MemoryPool-module.nvnflinger.h"
#include "detail/capsrv_ImageFormatInfo-module.nvnflinger.h"
#include "detail/capsrv_DumpNvRmSurface-module.nvnflinger.h"

#define NN_CAPSRV_STRICT_POINTER_RANGE_CHECK(v, beg, end) NN_SDK_ASSERT(v >= beg && v < end)

namespace nn{ namespace capsrv{ namespace capture{

    namespace {
        void PrepareSurface(NvRmSurface* pOutSurface, const ImageBuffer::InfoType& info) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_GREATER(info.width, 0);
            NN_SDK_REQUIRES_GREATER(info.height, 0);

            NvColorFormat surfaceFormat = detail::ImageFormatInfo::GetNvColorFormat(info.format);
            NvRmSurfaceLayout surfaceLayout = detail::ImageFormatInfo::GetNvRmSurfaceLayout(info.format);

            const NvU32 surfaceAttributes[] = {
                NvRmSurfaceAttribute_Layout, surfaceLayout,
                NvRmSurfaceAttribute_None,
            };

            NvRmSurfaceSetup(
                pOutSurface,
                static_cast<NvU32>(info.width),
                static_cast<NvU32>(info.height),
                surfaceFormat,
                surfaceAttributes
            );

            // 必要ならば上書き
            auto blockHeightLog2 = detail::ImageFormatInfo::GetNvRmBlockHeightLog2(info.format);
            if(blockHeightLog2 >= 0)
            {
                pOutSurface->BlockHeightLog2 = blockHeightLog2;
            }
        }

        size_t CalculateRequiredMemorySize(NvRmSurface* pPreparedSurface) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pPreparedSurface);
            size_t size = static_cast<size_t>(NvRmSurfaceComputeSize(pPreparedSurface));
            size = nn::util::align_up(size, nn::os::MemoryPageSize);
            return size;
        }

        size_t CalculateRequiredMemoryAlignment(CaptureModule* pModule, NvRmSurface* pPreparedSurface) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pModule);
            NN_SDK_REQUIRES_NOT_NULL(pPreparedSurface);
            return static_cast<size_t>(NvRmSurfaceComputeAlignment(pModule->GetVicDevice(), pPreparedSurface));
        }
    }

    size_t ImageBufferData::GetRequiredMemorySize(const InfoType& info) NN_NOEXCEPT
    {
        NvRmSurface surface;
        PrepareSurface(&surface, info);
        return CalculateRequiredMemorySize(&surface);
    }

    size_t ImageBufferData::GetRequiredMemoryAlignment(CaptureModule* pModule, const InfoType& info) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pModule);
        NvRmSurface surface;
        PrepareSurface(&surface, info);
        return CalculateRequiredMemoryAlignment(pModule, &surface);
    }

    nn::Result ImageBufferData::Initialize(CaptureModule* pModule, const InfoType& info, MemoryPool* pMemoryPool, ptrdiff_t offset, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pModule);
        NN_SDK_REQUIRES_NOT_NULL(pMemoryPool);
        NN_SDK_REQUIRES(pMemoryPool->IsInitialized());

        PrepareSurface(&m_Surface, info);
        size_t requiredSize      = CalculateRequiredMemorySize(&m_Surface);
        size_t requiredAlignment = CalculateRequiredMemoryAlignment(pModule, &m_Surface);
        NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
        NN_SDK_REQUIRES_ALIGNED(offset, requiredAlignment);
        NN_SDK_REQUIRES_GREATER_EQUAL(size, requiredSize);
        NN_UNUSED(requiredSize);
        NN_UNUSED(requiredAlignment);

        m_Surface.hMem = pMemoryPool->GetData()->GetHandle();
        m_Surface.Offset = static_cast<NvU32>(offset);

        NN_CAPSRV_CAPTURE_LOG_SURFACE_DUMP("Surface:\n");
        detail::DumpNvRmSurface(m_Surface);

        m_Info = info;
        m_pMemoryPool = pMemoryPool;
        m_Offset = offset;
        m_Size = size;
        NN_RESULT_SUCCESS;
    }

    void ImageBufferData::Finalize() NN_NOEXCEPT
    {
    }

    void ImageBufferData::FlushCache() const NN_NOEXCEPT
    {
        auto head = m_pMemoryPool->GetData()->GetMemoryAddress();
        nn::os::FlushDataCache(reinterpret_cast<const void*>(head + m_Offset), m_Size);
    }

    int ImageBufferData::GetWidth() const NN_NOEXCEPT
    {
        return m_Info.width;
    }

    int ImageBufferData::GetHeight() const NN_NOEXCEPT
    {
        return m_Info.height;
    }

    ImageFormat ImageBufferData::GetImageFormat() const NN_NOEXCEPT
    {
        return m_Info.format;
    }

    const NvRmSurface* ImageBufferData::GetSurface() const NN_NOEXCEPT
    {
        return &m_Surface;
    }

    NvRmSurface* ImageBufferData::GetSurface() NN_NOEXCEPT
    {
        return &m_Surface;
    }

    const MemoryPool* ImageBufferData::GetMemoryPool() const NN_NOEXCEPT
    {
        return m_pMemoryPool;
    }

    ptrdiff_t ImageBufferData::GetMemoryOffset() const NN_NOEXCEPT
    {
        return m_Offset;
    }

    size_t ImageBufferData::GetMemorySize() const NN_NOEXCEPT
    {
        return m_Size;
    }

    //------------------------------------------------------------------------

    ImageBuffer::ImageBuffer() NN_NOEXCEPT
    {
        NN_CAPSRV_CAPTURE_DATASTORAGE_CHECK(DataType, m_DataStorage);
        std::memset(this, 0, sizeof(*this));
    }

    size_t ImageBuffer::GetRequiredMemorySize(const InfoType& info) NN_NOEXCEPT
    {
        return ImageBufferData::GetRequiredMemorySize(info);
    }

    size_t ImageBuffer::GetRequiredMemoryAlignment(CaptureModule* pModule, const InfoType& info) NN_NOEXCEPT
    {
        return ImageBufferData::GetRequiredMemoryAlignment(pModule, info);
    }

    ImageFormat ImageBuffer::GetImageFormat() const NN_NOEXCEPT
    {
        return m_pData->GetImageFormat();
    }

    nn::Result ImageBuffer::Initialize(CaptureModule* pModule, const InfoType& info, MemoryPool* pMemoryPool, ptrdiff_t offset, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        m_pData = new(&m_DataStorage) DataType();
        return m_pData->Initialize(pModule, info, pMemoryPool, offset, size);
    }

    void ImageBuffer::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        m_pData->Finalize();
        m_pData->~DataType();
        m_pData = nullptr;
    }

    void ImageBuffer::FlushCache() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        m_pData->FlushCache();
    }

    namespace {
        nn::Result WriteToMemorySameFormatImpl(
            size_t* pOutSize,
            void* pBuffer,
            size_t bufferSize,
            ImageFormat dstFormat,
            const Rectangle& srcRect,
            const ImageBufferData* pSrcData
        ) NN_NOEXCEPT
        {
            int bpp = detail::ImageFormatInfo::GetPixelSize(pSrcData->GetImageFormat());
            size_t dataSize = static_cast<size_t>(bpp * srcRect.width * srcRect.height);

            NN_SDK_REQUIRES_GREATER_EQUAL(bufferSize, dataSize);
            NN_UNUSED(bufferSize);

            NvRmSurfaceRead(
                const_cast<NvRmSurface*>(pSrcData->GetSurface()),
                static_cast<NvU32>(srcRect.x),
                static_cast<NvU32>(srcRect.y),
                static_cast<NvU32>(srcRect.width),
                static_cast<NvU32>(srcRect.height),
                pBuffer
                );

            *pOutSize = dataSize;
            NN_RESULT_SUCCESS;
        }

        ptrdiff_t CalculateBlockLinearPixelOffset(uint32_t posX, uint32_t posY, uint32_t bytesPerPixel, uint32_t blockHeight, uint32_t pitch) NN_NOEXCEPT
        {
            uint32_t x = posX * bytesPerPixel;
            uint32_t y = posY;
            ptrdiff_t gobOffset =
                (y / (8 * blockHeight)) * pitch * 8 * blockHeight +
                (x / 64) * 512 * blockHeight +
                ((y % (8 * blockHeight)) / 8) * 512;
            ptrdiff_t pixelOffset =
                gobOffset +
                ((x % 64) / 32) * 256 +
                ((y % 8) / 2) * 64 +
                ((x % 32) / 16) * 32 +
                (y % 2) * 16 +
                (x % 16);
            return pixelOffset;
        }

        // NV12 の Y 面から Y8 にコピー
        nn::Result WriteToMemoryY8FromYNv12Impl(
            size_t* pOutSize,
            void* pBuffer,
            size_t bufferSize,
            ImageFormat dstFormat,
            const Rectangle& srcRect,
            const ImageBufferData* pSrcData
        ) NN_NOEXCEPT
        {
            int bpp = 1;

            auto pSrcSurface = pSrcData->GetSurface();
            auto pMemory = pSrcData->GetMemoryPool();
            auto* pSrcBase = reinterpret_cast<const uint8_t*>(pMemory->GetData()->GetMemoryAddress()) + pSrcData->GetMemoryOffset();
            auto* pDstBase = reinterpret_cast<uint8_t*>(pBuffer);

            NN_SDK_ASSERT_EQUAL(pSrcSurface->Kind, NvRmMemKind_Generic_16Bx2);
            NN_SDK_ASSERT_EQUAL(pSrcSurface->Layout, NvRmSurfaceLayout_Blocklinear);
            uint32_t srcBlockHeight = (1 << pSrcSurface->BlockHeightLog2);
            uint32_t srcPitch       = pSrcSurface->Pitch;

            size_t dstStride = bpp * srcRect.width;
            size_t dstDataSize = srcRect.height * dstStride;
            NN_SDK_ASSERT_LESS_EQUAL(dstDataSize, bufferSize);

            for(uint32_t dstY = 0; dstY < srcRect.height; dstY++)
            {
                for(uint32_t dstX = 0; dstX < srcRect.width; dstX++)
                {
                    uint32_t srcY = dstY + srcRect.y;
                    uint32_t srcX = dstX + srcRect.x;

                    // TORIAEZU: やや非効率だが。
                    auto pSrc = pSrcBase + CalculateBlockLinearPixelOffset(srcX, srcY, bpp, srcBlockHeight, srcPitch);
                    auto pDst = pDstBase + dstStride * dstY + bpp * dstX;

                    NN_CAPSRV_STRICT_POINTER_RANGE_CHECK(pSrc, pSrcBase, pSrcBase + pSrcData->GetMemorySize());
                    NN_CAPSRV_STRICT_POINTER_RANGE_CHECK(pDst, pDstBase, pDstBase + bufferSize);

                    // TODO: 色空間変換
                    *pDst = *pSrc;
                }
            }

            *pOutSize = dstDataSize;
            NN_RESULT_SUCCESS;
        }

        // NV12 の UV 面から U8/V8 にコピー
        nn::Result WriteToMemoryX8FromUvNv12(
            size_t* pOutSize,
            void* pBuffer,
            size_t bufferSize,
            ImageFormat dstFormat,
            const Rectangle& srcRect,
            const ImageBufferData* pSrcData,
            int channel
        ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_RANGE(channel, 0, 2);
            int srcBpp = 2;
            int dstBpp = 1;

            auto pSrcSurface = pSrcData->GetSurface();
            auto pMemory = pSrcData->GetMemoryPool();
            auto* pSrcBase = reinterpret_cast<const uint8_t*>(pMemory->GetData()->GetMemoryAddress()) + pSrcData->GetMemoryOffset();
            auto* pDstBase = reinterpret_cast<uint8_t*>(pBuffer);

            NN_SDK_ASSERT_EQUAL(pSrcSurface->Kind, NvRmMemKind_Generic_16Bx2);
            NN_SDK_ASSERT_EQUAL(pSrcSurface->Layout, NvRmSurfaceLayout_Blocklinear);
            uint32_t srcBlockHeight = (1 << pSrcSurface->BlockHeightLog2);
            uint32_t srcPitch       = pSrcSurface->Pitch;

            size_t dstStride = dstBpp * srcRect.width;
            size_t dstDataSize = srcRect.height * dstStride;
            NN_SDK_ASSERT_LESS_EQUAL(dstDataSize, bufferSize);

            for(uint32_t dstY = 0; dstY < srcRect.height; dstY++)
            {
                for(uint32_t dstX = 0; dstX < srcRect.width; dstX++)
                {
                    uint32_t srcY = dstY + srcRect.y;
                    uint32_t srcX = dstX + srcRect.x;

                    // TORIAEZU: やや非効率だが。
                    auto pSrc = pSrcBase + CalculateBlockLinearPixelOffset(srcX, srcY, srcBpp, srcBlockHeight, srcPitch);
                    auto pDst = pDstBase + dstStride * dstY + dstBpp * dstX;

                    NN_CAPSRV_STRICT_POINTER_RANGE_CHECK(pSrc, pSrcBase, pSrcBase + pSrcData->GetMemorySize());
                    NN_CAPSRV_STRICT_POINTER_RANGE_CHECK(pDst, pDstBase, pDstBase + bufferSize);

                    // TODO: 色空間変換
                    *pDst = pSrc[channel];
                }
            }

            *pOutSize = dstDataSize;
            NN_RESULT_SUCCESS;
        }
    }

    nn::Result ImageBuffer::WriteToMemory(
        size_t* pOutSize,
        void* pBuffer,
        size_t bufferSize,
        ImageFormat dstFormat,
        const Rectangle* pSrcRectangle
        ) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);
        if(pSrcRectangle)
        {
            NN_SDK_REQUIRES_MINMAX(pSrcRectangle->x, 0, m_pData->GetWidth());
            NN_SDK_REQUIRES_MINMAX(pSrcRectangle->y, 0, m_pData->GetHeight());
            NN_SDK_REQUIRES_MINMAX(pSrcRectangle->width, 0, m_pData->GetWidth());
            NN_SDK_REQUIRES_MINMAX(pSrcRectangle->height, 0, m_pData->GetHeight());
            NN_SDK_REQUIRES_MINMAX(pSrcRectangle->x + pSrcRectangle->width, 0, m_pData->GetWidth());
            NN_SDK_REQUIRES_MINMAX(pSrcRectangle->y + pSrcRectangle->height, 0, m_pData->GetHeight());
        }

        Rectangle rect;
        rect.x = 0;
        rect.y = 0;
        rect.width = m_pData->GetWidth();
        rect.height = m_pData->GetHeight();
        if(pSrcRectangle)
        {
            rect = *pSrcRectangle;
        }

        auto srcFormat = m_pData->GetImageFormat();
        if(dstFormat == srcFormat)
        {
            return WriteToMemorySameFormatImpl(pOutSize, pBuffer, bufferSize, dstFormat, rect, m_pData);
        }
        else if(dstFormat == ImageFormat_Y_Bt601 && srcFormat == ImageFormat_Y_NV12)
        {
            return WriteToMemoryY8FromYNv12Impl(pOutSize, pBuffer, bufferSize, dstFormat, rect, m_pData);
        }
        else if(dstFormat == ImageFormat_U_Bt601 && srcFormat == ImageFormat_Uv_NV12)
        {
            return WriteToMemoryX8FromUvNv12(pOutSize, pBuffer, bufferSize, dstFormat, rect, m_pData, 0);
        }
        else if(dstFormat == ImageFormat_V_Bt601 && srcFormat == ImageFormat_Uv_NV12)
        {
            return WriteToMemoryX8FromUvNv12(pOutSize, pBuffer, bufferSize, dstFormat, rect, m_pData, 1);
        }
        else
        {
            NN_RESULT_THROW(ResultCaptureNotSupportedImageFormat());
        }

        NN_RESULT_SUCCESS;
    }

}}}
