﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/capsrv/capsrv_ScreenShotForApplication.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/applet/applet_Apis.h>
#include "capsrv_ServiceHolderBase.h"
#include "capsrv_Macro.h"
#include "capsrv_LibraryState.h"

namespace nn{ namespace capsrv{

    namespace {

        class ScreenShotApplicationServiceHolder
            : public ServiceHolderBase<
                ScreenShotApplicationServiceHolder,
                nn::capsrv::sf::IScreenShotApplicationService,
                2
            >
        {
        public:
            static const char* GetServiceName() NN_NOEXCEPT { return sf::ScreenShotUserServiceName; }
        };

        ScreenShotApplicationServiceHolder g_ScreenShotApplicationServiceHolder;
    }

    nn::Result InitializeForApplication() NN_NOEXCEPT
    {
        NN_RESULT_DO(g_ScreenShotApplicationServiceHolder.Initialize(nullptr));
        NN_RESULT_SUCCESS;
    }

    void FinalizeForApplication() NN_NOEXCEPT
    {
        g_ScreenShotApplicationServiceHolder.Finalize();
    }

    namespace {
        nn::Result SaveScreenShotForApplicationImpl(
            ApplicationAlbumEntry* pOutEntry,
            const void* imageData,
            size_t imageDataSize,
            const ScreenShotAttribute& attribute,
            OverlayNotificationRequestType overlayRequest
            ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutEntry);
            NN_SDK_REQUIRES_NOT_NULL(imageData);
            NN_SDK_REQUIRES_GREATER_EQUAL(imageDataSize, static_cast<size_t>(4 * 1280 * 720));
            NN_SDK_REQUIRES_EQUAL(attribute.size, static_cast<ScreenShotSizeType>(ScreenShotSize_1280x720));

            std::memset(pOutEntry, 0, sizeof(ApplicationAlbumEntry));

            NN_CAPSRV_GET_SERVICE_POINTER(pService, g_ScreenShotApplicationServiceHolder);

            ApplicationAlbumEntry entry = {};
            NN_RESULT_DO(pService->SaveScreenShotEx0(
                &entry,
                nn::sf::InBuffer(reinterpret_cast<const char*>(imageData), imageDataSize),
                attribute,
                applet::GetAppletResourceUserId(),
                overlayRequest
            ));

            *pOutEntry = entry;
            NN_RESULT_SUCCESS;
        }
    }

    nn::Result SaveScreenShotForApplication(
        ApplicationAlbumEntry* pOutEntry,
        const void* imageData,
        size_t imageDataSize,
        ScreenShotSizeType size,
        OverlayNotificationRequestType overlayRequest
        ) NN_NOEXCEPT
    {
        ScreenShotAttribute attribute;
        attribute.SetDefault();
        attribute.size = size;

        return SaveScreenShotForApplicationImpl(pOutEntry, imageData, imageDataSize, attribute, overlayRequest);
    }

    nn::Result SaveScreenShotForApplication(
        ApplicationAlbumEntry* pOutEntry,
        const void* imageData,
        size_t imageDataSize,
        const ScreenShotAttribute& attribute,
        OverlayNotificationRequestType overlayRequest
        ) NN_NOEXCEPT
    {
        return SaveScreenShotForApplicationImpl(pOutEntry, imageData, imageDataSize, attribute, overlayRequest);
    }

    nn::Result SaveScreenShotForApplication(
        ApplicationAlbumEntry* pOutEntry,
        const void* imageData,
        size_t imageDataSize,
        const ScreenShotAttribute& attribute,
        const UserIdList& userIdList,
        OverlayNotificationRequestType overlayRequest
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutEntry);
        NN_SDK_REQUIRES_NOT_NULL(imageData);
        NN_SDK_REQUIRES_GREATER_EQUAL(imageDataSize, static_cast<size_t>(4 * 1280 * 720));
        NN_SDK_REQUIRES_EQUAL(attribute.size, static_cast<ScreenShotSizeType>(ScreenShotSize_1280x720));

        std::memset(pOutEntry, 0, sizeof(ApplicationAlbumEntry));

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_ScreenShotApplicationServiceHolder);

        ApplicationAlbumEntry entry = {};
        NN_RESULT_DO(pService->SaveScreenShotEx2(
            &entry,
            nn::sf::InBuffer(reinterpret_cast<const char*>(imageData), imageDataSize),
            attribute,
            userIdList,
            applet::GetAppletResourceUserId(),
            overlayRequest
        ));

        *pOutEntry = entry;
        NN_RESULT_SUCCESS;
    }

}}
