﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/capsrv/capsrv_ScreenShotControl.h>

#include "capsrv_Macro.h"

#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_MemoryResource.h>
#include <nn/sf/sf_Buffers.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/capsrv/sf/capsrv_Services.sfdl.h>
#include <nn/capsrv/sf/capsrv_ServiceName.h>

#if defined(NN_CAPSRV_USE_DIRECT_FUNCTION_CALL)
#include "capsrv_InitializeForLibrary.h"
#endif

namespace nn{ namespace capsrv{

    namespace {

        nn::sf::ProxyObjectAllocator<5> g_ProxyObjectAllocator = NN_SF_PROXY_OBJECT_ALLOCATOR_INITIALIZER;
        nn::sf::ShimLibraryObjectHolder<nn::capsrv::sf::IScreenShotControlService> g_ObjectHolder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;
        nn::sf::ShimLibraryObjectHolder<nn::capsrv::sf::IScreenShotControlService> g_RealtimeObjectHolder = NN_SF_SHIM_LIBRARY_OBJECT_HOLDER_INITIALIZER;

        Result InitializeByHipc(const char* serviceName) NN_NOEXCEPT
        {
            auto success = false;

            g_ProxyObjectAllocator.Initialize();
            NN_UTIL_SCOPE_EXIT
            {
                if (!success)
                {
                    g_ProxyObjectAllocator.Finalize();
                }
            };

            auto pHipcRef = nn::sf::CreateHipcProxyByName<nn::capsrv::sf::IScreenShotControlService>(g_ProxyObjectAllocator.GetMemoryResource(), serviceName);
            NN_RESULT_DO(pHipcRef);

            uint32_t tag = 1;
            nn::sf::HipcRef<nn::capsrv::sf::IScreenShotControlService> realtimeHipcRef;
            NN_RESULT_DO(pHipcRef->CloneSession<nn::sf::MemoryResourceAllocationPolicy>(&realtimeHipcRef, tag, g_ProxyObjectAllocator.GetMemoryResource()));

            success = true;
            g_ObjectHolder.InitializeHolderDirectly(std::move(*pHipcRef));
            g_RealtimeObjectHolder.InitializeHolderDirectly(std::move(realtimeHipcRef));
            NN_RESULT_SUCCESS;
        }

        void FinalizeByHipc() NN_NOEXCEPT
        {
            g_RealtimeObjectHolder.FinalizeHolder();
            g_ObjectHolder.FinalizeHolder();
            g_ProxyObjectAllocator.Finalize();
        }

    }

    nn::Result InitializeScreenShotControl() NN_NOEXCEPT
    {
#if defined(NN_CAPSRV_USE_HIPC)
        NN_RESULT_DO(InitializeByHipc(sf::ScreenShotControlServiceName));
#elif defined(NN_CAPSRV_USE_DIRECT_FUNCTION_CALL)
        InitializeForLibrary();
#endif
        NN_RESULT_SUCCESS;
    }

    void FinalizeScreenShotControl() NN_NOEXCEPT
    {
#if defined(NN_CAPSRV_USE_HIPC)
        FinalizeByHipc();
#elif defined(NN_CAPSRV_USE_DIRECT_FUNCTION_CALL)
        FinalizeForLibrary();
#endif
    }

    nn::Result AttachSharedBufferToCaptureModule(nn::vi::fbshare::SharedBufferHandle hBuffer) NN_NOEXCEPT
    {
        auto&& pService = g_RealtimeObjectHolder.GetObject();
        NN_RESULT_DO(pService->AttachSharedBufferToCaptureModule(hBuffer));
        NN_RESULT_SUCCESS;
    }

    nn::Result RequestTakingScreenShot(
        uint64_t seqNo,
        nn::ncm::ProgramId programId,
        nn::applet::AppletResourceUserId aruid,
        const ScreenShotAttribute& attribute,
        const UserIdList& userIdList,
        nn::TimeSpan timeout
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(seqNo, 1); // 0 は初期状態でキャンセル済なので 1 以上が必要。
        auto&& pService = g_ObjectHolder.GetObject();
        nn::Result result = pService->RequestTakingScreenShotEx1(seqNo, programId, aruid, timeout, attribute, userIdList);
        return result;
    }

    nn::Result CancelTakingScreenShot(
        uint64_t seqNo,
        nn::Result reason
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(reason.IsFailure());
        auto&& pService = g_RealtimeObjectHolder.GetObject();
        nn::Result result = pService->CancelTakingScreenShot(seqNo, reason);
        return result;
    }

    nn::Result SetTakingScreenShotCancelState(
        uint64_t seqNo,
        nn::Result reason
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(reason.IsFailure());
        auto&& pService = g_ObjectHolder.GetObject();
        nn::Result result = pService->SetTakingScreenShotCancelState(seqNo, reason);
        return result;
    }

    nn::Result NotifyTakingScreenShotRefused(nn::ncm::ProgramId programId) NN_NOEXCEPT
    {
        auto&& pService = g_ObjectHolder.GetObject();
        nn::Result result = pService->NotifyTakingScreenShotRefused(programId);
        return result;
    }

    nn::Result NotifyTakingScreenShotFailed(nn::ncm::ProgramId programId) NN_NOEXCEPT
    {
        auto&& pService = g_ObjectHolder.GetObject();
        nn::Result result = pService->NotifyTakingScreenShotFailed(programId);
        return result;
    }

    nn::Result CaptureRawImageToAttachedSharedBuffer(
        int dstIndex,
        vi::LayerStack layerStack,
        nn::TimeSpan timeout
    ) NN_NOEXCEPT
    {
        auto&& pService = g_RealtimeObjectHolder.GetObject();
        return pService->CaptureRawImageToAttachedSharedBuffer(
            static_cast<int32_t>(dstIndex),
            static_cast<int32_t>(layerStack),
            timeout
        );
    }

    nn::Result SaveScreenShotEx2ViaAm(
        nn::sf::Out<ApplicationAlbumEntry> pOutEntry,
        const nn::sf::InBuffer& imageBuffer,
        const capsrv::detail::ScreenShotAttributeEx0& attribute,
        const UserIdList& userIdList,
        applet::AppletResourceUserId aruid,
        OverlayNotificationRequestType overlayRequest
    ) NN_NOEXCEPT
    {
        auto&& pService = g_ObjectHolder.GetObject();
        auto result = pService->SaveScreenShotEx2ViaAm(pOutEntry, imageBuffer, attribute, userIdList, aruid, overlayRequest);
        return result;
    }

    //--------------------------------------------------------

    nn::Result SetupOverlayMovieThumbnail(
        void* pImageBufferY,
        size_t imageBufferYSize,
        void* pImageBufferUv,
        size_t imageBufferUvSize,
        int width,
        int height,
        const nn::capsrv::AlbumFileId& fileId
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pImageBufferY);
        NN_SDK_REQUIRES_ALIGNED(pImageBufferY, OverlayMovieImageBufferMemoryAlignment);
        NN_SDK_REQUIRES_EQUAL(imageBufferYSize % OverlayMovieImageBufferMemoryUnitSize, 0u);
        NN_SDK_REQUIRES_GREATER_EQUAL(imageBufferYSize, static_cast<size_t>(width * height));
        NN_SDK_REQUIRES_NOT_NULL(pImageBufferUv);
        NN_SDK_REQUIRES_ALIGNED(pImageBufferUv, OverlayMovieImageBufferMemoryAlignment);
        NN_SDK_REQUIRES_EQUAL(imageBufferUvSize % OverlayMovieImageBufferMemoryUnitSize, 0u);
        NN_SDK_REQUIRES_GREATER_EQUAL(imageBufferUvSize, static_cast<size_t>(width * height / 2));
        NN_SDK_REQUIRES_EQUAL(width, 1280); // 今の仕様。
        NN_SDK_REQUIRES_EQUAL(height, 720); // 今の仕様。

        auto&& pService = g_ObjectHolder.GetObject();

        return pService->SetupOverlayMovieThumbnail(
            nn::sf::OutBuffer(static_cast<char*>(pImageBufferY), imageBufferYSize),
            nn::sf::OutBuffer(static_cast<char*>(pImageBufferUv), imageBufferUvSize),
            static_cast<int64_t>(width),
            static_cast<int64_t>(height),
            fileId
        );
    }

    namespace {
        bool IsValidMovieFileDescription(AlbumFileDescriptionType description) NN_NOEXCEPT
        {
            switch(description)
            {
            case AlbumFileDescription_MovieContinuous:
            case AlbumFileDescription_MovieTrimmed:
            case AlbumFileDescription_MovieMakerSaved:
                return true;
            default:
                return false;
            }
        }
    }

#define NN_CAPSRV_REQUIRES_VALID_MOVIE_FILE_DESCRIPTION(description)    \
    NN_SDK_REQUIRES(IsValidMovieFileDescription(description));   \
    NN_UNUSED(IsValidMovieFileDescription);

    nn::Result CreateProtoMovieMetaDataRgba(
        size_t* pOutSize,
        int64_t* pOutMakerNoteOffset,
        int64_t* pOutMakerNoteSize,
        void* pOutBuffer,
        size_t outBufferSize,
        const void* pImageBuffer,
        size_t imageBufferSize,
        int width,
        int height,
        const nn::capsrv::AlbumFileId& fileId,
        const ScreenShotAttribute& attribute,
        const AppletData& appletData,
        const ApplicationData& applicationData
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);
        NN_SDK_REQUIRES_NOT_NULL(pOutMakerNoteOffset);
        NN_SDK_REQUIRES_NOT_NULL(pOutMakerNoteSize);
        NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);
        NN_SDK_REQUIRES_GREATER_EQUAL(outBufferSize, movie::MovieMetaImageDataSize);
        NN_SDK_REQUIRES_NOT_NULL(pImageBuffer);
        NN_SDK_REQUIRES_GREATER_EQUAL(imageBufferSize, 4 * width * height);
        NN_SDK_REQUIRES_EQUAL(width, 1280); // 今の仕様。
        NN_SDK_REQUIRES_EQUAL(height, 720); // 今の仕様。
        NN_CAPSRV_REQUIRES_VALID_MOVIE_FILE_DESCRIPTION(attribute.description);

        auto&& pService = g_ObjectHolder.GetObject();

        uint64_t outSize = 0;
        int64_t outMakerNoteOffset = 0;
        int64_t outMakerNoteSize = 0;
        NN_RESULT_DO(pService->CreateProtoMovieMetaDataRgbaEx2(
            &outSize,
            &outMakerNoteOffset,
            &outMakerNoteSize,
            nn::sf::OutBuffer(static_cast<char*>(pOutBuffer), outBufferSize),
            nn::sf::InBuffer(static_cast<const char*>(pImageBuffer), imageBufferSize),
            static_cast<int64_t>(width),
            static_cast<int64_t>(height),
            fileId,
            attribute,
            appletData,
            applicationData
        ));

        *pOutSize = static_cast<size_t>(outSize);
        *pOutMakerNoteOffset = outMakerNoteOffset;
        *pOutMakerNoteSize = outMakerNoteSize;
        NN_RESULT_SUCCESS;
    }

    nn::Result CreateProtoMovieMetaDataNv12(
        size_t* pOutSize,
        int64_t* pOutMakerNoteOffset,
        int64_t* pOutMakerNoteSize,
        void* pOutBuffer,
        size_t outBufferSize,
        void* pImageBufferY,
        size_t imageBufferYSize,
        void* pImageBufferUv,
        size_t imageBufferUvSize,
        int width,
        int height,
        const nn::capsrv::AlbumFileId& fileId,
        const ScreenShotAttribute& attribute,
        const AppletData& appletData,
        const ApplicationData& applicationData
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);
        NN_SDK_REQUIRES_NOT_NULL(pOutMakerNoteOffset);
        NN_SDK_REQUIRES_NOT_NULL(pOutMakerNoteSize);
        NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);
        NN_SDK_REQUIRES_GREATER_EQUAL(outBufferSize, movie::MovieMetaImageDataSize);
        NN_SDK_REQUIRES_NOT_NULL(pImageBufferY);
        NN_SDK_REQUIRES_ALIGNED(pImageBufferY, OverlayMovieImageBufferMemoryAlignment);
        NN_SDK_REQUIRES_EQUAL(imageBufferYSize % OverlayMovieImageBufferMemoryUnitSize, 0u);
        NN_SDK_REQUIRES_GREATER_EQUAL(imageBufferYSize, static_cast<size_t>(width * height));
        NN_SDK_REQUIRES_NOT_NULL(pImageBufferUv);
        NN_SDK_REQUIRES_ALIGNED(pImageBufferUv, OverlayMovieImageBufferMemoryAlignment);
        NN_SDK_REQUIRES_EQUAL(imageBufferUvSize % OverlayMovieImageBufferMemoryUnitSize, 0u);
        NN_SDK_REQUIRES_GREATER_EQUAL(imageBufferUvSize, static_cast<size_t>(width * height / 2));
        NN_SDK_REQUIRES_EQUAL(width, 1280); // 今の仕様。
        NN_SDK_REQUIRES_EQUAL(height, 720); // 今の仕様。
        NN_CAPSRV_REQUIRES_VALID_MOVIE_FILE_DESCRIPTION(attribute.description);

        auto&& pService = g_ObjectHolder.GetObject();

        uint64_t outSize = 0;
        int64_t outMakerNoteOffset = 0;
        int64_t outMakerNoteSize = 0;
        NN_RESULT_DO(pService->CreateProtoMovieMetaDataNv12Ex2(
            &outSize,
            &outMakerNoteOffset,
            &outMakerNoteSize,
            nn::sf::OutBuffer(static_cast<char*>(pOutBuffer), outBufferSize),
            nn::sf::OutBuffer(static_cast<char*>(pImageBufferY), imageBufferYSize),
            nn::sf::OutBuffer(static_cast<char*>(pImageBufferUv), imageBufferUvSize),
            static_cast<int64_t>(width),
            static_cast<int64_t>(height),
            fileId,
            attribute,
            appletData,
            applicationData
        ));

        *pOutSize = static_cast<size_t>(outSize);
        *pOutMakerNoteOffset = outMakerNoteOffset;
        *pOutMakerNoteSize = outMakerNoteSize;
        NN_RESULT_SUCCESS;
    }


    //--------------------------------------------------------

    nn::Result OpenRawScreenShotReadStreamForDevelop(
        size_t* pOutImageDataSize,
        int* pOutImageWidth,
        int* pOutImageHeight,
        vi::LayerStack layerStack,
        nn::TimeSpan timeout
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutImageDataSize);
        NN_SDK_REQUIRES_NOT_NULL(pOutImageWidth);
        NN_SDK_REQUIRES_NOT_NULL(pOutImageHeight);

        auto&& pService = g_ObjectHolder.GetObject();

        int64_t size = 0;
        int64_t width = 0;
        int64_t height = 0;

        NN_RESULT_DO(pService->OpenRawScreenShotReadStream(&size, &width, &height, layerStack, timeout));

        *pOutImageDataSize = static_cast<size_t>(size);
        *pOutImageWidth = static_cast<int>(width);
        *pOutImageHeight = static_cast<int>(height);
        NN_RESULT_SUCCESS;
    }

    void CloseRawScreenShotReadStreamForDevelop() NN_NOEXCEPT
    {
        auto&& pService = g_ObjectHolder.GetObject();
        pService->CloseRawScreenShotReadStream();
    }

    nn::Result ReadRawScreenShotReadStreamForDevelop(
        size_t* pOutReadSize,
        void* pBuffer,
        size_t bufferSize,
        ptrdiff_t startOffset
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutReadSize);
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        NN_SDK_REQUIRES_GREATER_EQUAL(startOffset, 0);
        NN_SDK_REQUIRES_ALIGNED(pBuffer, 4);
        NN_SDK_REQUIRES(bufferSize % 4 == 0);
        NN_SDK_REQUIRES(startOffset % 4 == 0);

        auto&& pService = g_ObjectHolder.GetObject();

        int64_t readSize = 0;

        NN_RESULT_DO(pService->ReadRawScreenShotReadStream(&readSize, nn::sf::OutBuffer(reinterpret_cast<char*>(pBuffer), bufferSize), static_cast<int64_t>(startOffset)));

        *pOutReadSize = static_cast<size_t>(readSize);
        NN_RESULT_SUCCESS;
    }

}}
