﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/capsrv/capsrv_AlbumAccess.h>

#include <nn/time.h>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/capsrv/sf/capsrv_Services.sfdl.h>

#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>

#include "capsrv_ScreenShotServiceHolder.h"
#include "capsrv_Macro.h"
#include "capsrv_LibraryState.h"

namespace nn{ namespace capsrv{

    // 初期化は AlbumAccess と一緒に行われる

    namespace {
        nn::Result SaveScreenShotImpl(
            ApplicationAlbumEntry* pOutEntry,
            const void* imageData,
            size_t imageDataSize,
            const ScreenShotAttribute& attribute,
            OverlayNotificationRequestType overlayRequest
            ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutEntry);
            NN_SDK_REQUIRES_NOT_NULL(imageData);
            NN_SDK_REQUIRES_GREATER_EQUAL(imageDataSize, static_cast<size_t>(4 * 1280 * 720));
            NN_SDK_REQUIRES_EQUAL(attribute.size, static_cast<ScreenShotSizeType>(ScreenShotSize_1280x720));

            std::memset(pOutEntry, 0, sizeof(ApplicationAlbumEntry));

            NN_CAPSRV_GET_SERVICE_POINTER(pService, g_ScreenShotServiceHolder);

            ApplicationAlbumEntry entry = {};
            NN_RESULT_DO(pService->SaveScreenShotEx0(
                &entry,
                nn::sf::InBuffer(reinterpret_cast<const char*>(imageData), imageDataSize),
                attribute,
                LibraryState::GetMyAruid(),
                overlayRequest
            ));

            *pOutEntry = entry;
            NN_RESULT_SUCCESS;
        }
    }

    nn::Result SaveScreenShot(
        ApplicationAlbumEntry* pOutEntry,
        const void* imageData,
        size_t imageDataSize,
        const ScreenShotAttribute& attribute,
        OverlayNotificationRequestType overlayRequest
        ) NN_NOEXCEPT
    {
        return SaveScreenShotImpl(pOutEntry, imageData, imageDataSize, attribute, overlayRequest);
    }

    //----------------------------------------------------------------------------------------------

    namespace {
        nn::Result SaveEditedScreenShotImpl(
            AlbumEntry* pOutEntry,
            const void* imageData,
            size_t imageDataSize,
            int width,
            int height,
            const void* thumbData,
            size_t thumbDataSize,
            int thumbWidth,
            int thumbHeight,
            const ScreenShotAttribute& attribute,
            const AppletData& appletData,
            const AlbumFileId* pOriginalFileId
            ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutEntry);
            NN_SDK_REQUIRES_NOT_NULL(imageData);
            NN_SDK_REQUIRES_NOT_NULL(thumbData);
            NN_SDK_REQUIRES_NOT_NULL(pOriginalFileId);
            NN_CAPSRV_REQUIRES_VALID_CONTENTS(pOriginalFileId->contents);
            NN_CAPSRV_REQUIRES_VALID_STORAGE(pOriginalFileId->storage);
            // TORIAEZU:
            //   現状の条件ではこの値。
            NN_SDK_REQUIRES_EQUAL(imageDataSize, static_cast<size_t>(4 * width * height));
            NN_SDK_REQUIRES_EQUAL(width, 1280);
            NN_SDK_REQUIRES_EQUAL(height, 720);
            NN_SDK_REQUIRES_EQUAL(thumbDataSize, static_cast<size_t>(4 * thumbWidth * thumbHeight));
            NN_SDK_REQUIRES_EQUAL(thumbWidth , static_cast<int>(ViewerThumbnailImageSize_Width ));
            NN_SDK_REQUIRES_EQUAL(thumbHeight, static_cast<int>(ViewerThumbnailImageSize_Height));

            std::memset(pOutEntry, 0, sizeof(AlbumEntry));

            NN_CAPSRV_TRY_GET_SERVICE_POINTER(pService, g_ScreenShotServiceHolder, ResultAlbumError());

            AlbumEntry entry = {};
            NN_RESULT_DO(
                pService->SaveEditedScreenShotEx1(
                    &entry,
                    nn::sf::InBuffer(reinterpret_cast<const char*>(imageData), imageDataSize),
                    nn::sf::InBuffer(reinterpret_cast<const char*>(thumbData), thumbDataSize),
                    static_cast<int64_t>(width),
                    static_cast<int64_t>(height),
                    static_cast<int64_t>(thumbWidth),
                    static_cast<int64_t>(thumbHeight),
                    attribute,
                    appletData,
                    *pOriginalFileId
                )
            );

            *pOutEntry = entry;
            NN_RESULT_SUCCESS;
        }

        nn::Result SaveScreenShotOfMovieImpl(
            AlbumEntry* pOutEntry,
            const void* imageData,
            size_t imageDataSize,
            int width,
            int height,
            const void* thumbData,
            size_t thumbDataSize,
            int thumbWidth,
            int thumbHeight,
            const ScreenShotAttribute& attribute,
            const AppletData& appletData,
            const AlbumFileId* pOriginalFileId
            ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutEntry);
            NN_SDK_REQUIRES_NOT_NULL(imageData);
            NN_SDK_REQUIRES_NOT_NULL(thumbData);
            NN_SDK_REQUIRES_NOT_NULL(pOriginalFileId);
            NN_CAPSRV_REQUIRES_VALID_CONTENTS(pOriginalFileId->contents);
            NN_CAPSRV_REQUIRES_VALID_STORAGE(pOriginalFileId->storage);
            // TORIAEZU:
            //   現状の事前条件ではこの値。
            NN_SDK_REQUIRES_EQUAL(imageDataSize, static_cast<size_t>(4 * width * height));
            NN_SDK_REQUIRES_EQUAL(width, 1280);
            NN_SDK_REQUIRES_EQUAL(height, 720);
            NN_SDK_REQUIRES_EQUAL(thumbDataSize, static_cast<size_t>(4 * thumbWidth * thumbHeight));
            NN_SDK_REQUIRES_EQUAL(thumbWidth , static_cast<int>(ViewerThumbnailImageSize_Width ));
            NN_SDK_REQUIRES_EQUAL(thumbHeight, static_cast<int>(ViewerThumbnailImageSize_Height));

            std::memset(pOutEntry, 0, sizeof(AlbumEntry));

            NN_CAPSRV_TRY_GET_SERVICE_POINTER(pService, g_ScreenShotServiceHolder, ResultAlbumError());

            AlbumEntry entry = {};
            NN_RESULT_DO(
                pService->SaveScreenShotOfMovieEx1(
                    &entry,
                    nn::sf::InBuffer(reinterpret_cast<const char*>(imageData), imageDataSize),
                    nn::sf::InBuffer(reinterpret_cast<const char*>(thumbData), thumbDataSize),
                    static_cast<int64_t>(width),
                    static_cast<int64_t>(height),
                    static_cast<int64_t>(thumbWidth),
                    static_cast<int64_t>(thumbHeight),
                    attribute,
                    appletData,
                    *pOriginalFileId
                )
            );

            *pOutEntry = entry;
            NN_RESULT_SUCCESS;
        }
    }

    nn::Result SaveEditedScreenShot(
        AlbumEntry* pOutEntry,
        const void* imageData,
        size_t imageDataSize,
        int width,
        int height,
        const void* thumbData,
        size_t thumbDataSize,
        int thumbWidth,
        int thumbHeight,
        const ScreenShotAttribute& attribute,
        const AppletData& appletData,
        const AlbumFileId* pOriginalFileId
        ) NN_NOEXCEPT
    {
        return SaveEditedScreenShotImpl(pOutEntry, imageData, imageDataSize, width, height, thumbData, thumbDataSize, thumbWidth, thumbHeight, attribute, appletData, pOriginalFileId);
    }

    nn::Result SaveScreenShotOfMovie(
        AlbumEntry* pOutEntry,
        const void* imageData,
        size_t imageDataSize,
        int width,
        int height,
        const void* thumbData,
        size_t thumbDataSize,
        int thumbWidth,
        int thumbHeight,
        const ScreenShotAttribute& attribute,
        const AppletData& appletData,
        const AlbumFileId* pOriginalFileId
        ) NN_NOEXCEPT
    {
        return SaveScreenShotOfMovieImpl(pOutEntry, imageData, imageDataSize, width, height, thumbData, thumbDataSize, thumbWidth, thumbHeight, attribute, appletData, pOriginalFileId);
    }

}}
