﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/capsrv/capsrv_AlbumAccessForApplication.h>

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time.h>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/applet/applet_Apis.h>
#include <nn/capsrv/sf/capsrv_Services.sfdl.h>
#include <nn/capsrv/sf/capsrv_ServiceType.h>
#include <nn/capsrv/capsrv_AlbumFileContents.h>
#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>
#include <nn/capsrv/capsrv_ScreenShotDecodeOption.h>
#include <nn/capsrv/capsrv_Result.h>

#include "capsrv_Macro.h"
#include "capsrv_ServiceHolderBase.h"

#include <cstring>

#if defined(NN_CAPSRV_USE_DIRECT_FUNCTION_CALL)
#include "capsrv_InitializeForLibrary.h"
#endif

namespace nn { namespace capsrv {

// album::AlbumFileEntry から capsrv::ApplicationAlbumFileEntry への変換チェック
NN_STATIC_ASSERT(sizeof(album::AlbumFileEntry) == sizeof(ApplicationAlbumFileEntry));
NN_STATIC_ASSERT(NN_ALIGNOF(album::AlbumFileEntry) >= NN_ALIGNOF(ApplicationAlbumFileEntry));

namespace {

    class AlbumApplicationServiceHolder
        : public ServiceHolderBase<
            AlbumApplicationServiceHolder,
            nn::capsrv::sf::IAlbumApplicationService,
            4 /* ObjectCountMax */,
            SessionHolder<nn::capsrv::sf::IAlbumAccessorApplicationSession>
          >
    {
    public:
        static const char* GetServiceName() NN_NOEXCEPT { return sf::AlbumApplicationServiceName; }
    };

    AlbumApplicationServiceHolder g_AlbumApplicationServiceHolder;

    typedef nn::sf::SharedPointer<nn::capsrv::sf::IAlbumAccessorApplicationSession> AlbumAccessorApplicationServicePointerType;
    AlbumAccessorApplicationServicePointerType g_pAlbumAccessorApplicationSession;

}   // namespace

// アプリ向けアルバムアクセス初期化
Result InitializeAlbumAccessForApplication() NN_NOEXCEPT
{
    NN_RESULT_DO(g_AlbumApplicationServiceHolder.Initialize(nullptr));
    NN_RESULT_SUCCESS;
}

// アプリ向けアルバムアクセス終了
void FinalizeAlbumAccessForApplication() NN_NOEXCEPT
{
    g_AlbumApplicationServiceHolder.Finalize();
}

// アプリ向けアルバムファイルリスト取得
Result GetAlbumContentsFileListForApplication(int* pOutCount, ApplicationAlbumFileEntry* pBuffer, int bufferLength, AlbumFileContentsType contents, AlbumFileDateTime beginDateTime, AlbumFileDateTime endDateTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_ALIGNED(pBuffer, NN_ALIGNOF(ApplicationAlbumFileEntry));
    NN_SDK_REQUIRES_GREATER(bufferLength, 0);

    uint64_t count = 0;
    NN_UTIL_SCOPE_EXIT
    {
        *pOutCount = static_cast<int>(count);
    };

    NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumApplicationServiceHolder);
    auto aruid = applet::GetAppletResourceUserId();
    NN_ABORT_UNLESS_RESULT_SUCCESS(pService->GetAlbumFileListEx0ByAruid(&count, nn::sf::OutArray<ApplicationAlbumFileEntry>(pBuffer, bufferLength), contents, beginDateTime, endDateTime, aruid));
    NN_RESULT_SUCCESS;
}

// アプリ向けアルバムファイルリスト取得（ユーザ識別子指定）
Result GetAlbumContentsFileListForApplicationEx1(int* pOutCount, ApplicationAlbumFileEntry* pBuffer, int bufferLength, AlbumFileContentsType contents, const account::Uid& uid, AlbumFileDateTime beginDateTime, AlbumFileDateTime endDateTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_ALIGNED(pBuffer, NN_ALIGNOF(ApplicationAlbumFileEntry));
    NN_SDK_REQUIRES_GREATER(bufferLength, 0);
    NN_SDK_REQUIRES(uid != account::InvalidUid);

    uint64_t count = 0;
    NN_UTIL_SCOPE_EXIT
    {
        *pOutCount = static_cast<int>(count);
    };

    NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumApplicationServiceHolder);
    auto aruid = applet::GetAppletResourceUserId();
    NN_ABORT_UNLESS_RESULT_SUCCESS(pService->GetAlbumFileListEx1ByAruid(&count, nn::sf::OutArray<ApplicationAlbumFileEntry>(pBuffer, bufferLength), contents, uid, beginDateTime, endDateTime, aruid));
    NN_RESULT_SUCCESS;
}

// アプリ向けアルバムファイル削除
Result DeleteAlbumContentsFileForApplication(const ApplicationAlbumFileEntry& entry, AlbumFileContentsType contents) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(&entry);

    NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumApplicationServiceHolder);

    auto aruid = applet::GetAppletResourceUserId();
    NN_RESULT_DO(pService->DeleteAlbumFileByAruid(entry, contents, aruid));
    NN_RESULT_SUCCESS;
}

// アプリ向けアルバムファイル削除（製品機では使用不可）
Result DeleteAlbumFileForApplicationDebug(const ApplicationAlbumFileEntry& entry) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(&entry);

    NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumApplicationServiceHolder);

    auto aruid = applet::GetAppletResourceUserId();
    auto result = pService->DeleteAlbumFileByAruidForDebug(entry, aruid);
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(nn::capsrv::ResultNotSupported)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

// アプリ向けメディアファイルのファイルサイズを取得
Result GetAlbumContentsFileSizeForApplication(uint64_t* pOutSize, const ApplicationAlbumFileEntry& entry) NN_NOEXCEPT
{
    NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumApplicationServiceHolder);

    uint64_t size = 0;
    auto aruid = applet::GetAppletResourceUserId();
    NN_ABORT_UNLESS_RESULT_SUCCESS(pService->GetAlbumFileSizeByAruid(&size, entry, aruid));
    *pOutSize = size;
    NN_RESULT_SUCCESS;
}


namespace {

Result LoadAlbumContentsFileScreenShotImageForApplicationImpl(
        int* pOutWidth,
        int* pOutHeight,
        ScreenShotAttributeForApplication* pOutAttribute,
        ApplicationData* pOutApplicationData,
        void* pBuffer,
        size_t bufferSize,
        void* pWorkBuffer,
        size_t workBufferSize,
        const ScreenShotDecodeOption& option,
        const ApplicationAlbumFileEntry& entry
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pWorkBuffer);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0);
    NN_SDK_REQUIRES_GREATER(workBufferSize, 0);

    NN_UTIL_SCOPE_EXIT{ std::memset(pWorkBuffer, 0, workBufferSize); };

    auto cleanupOutput = [&]()
    {
        if (pOutWidth)
        {
            *pOutWidth = 0;
        }
        if (pOutHeight)
        {
            *pOutHeight = 0;
        }
        if (pOutAttribute)
        {
            pOutAttribute->SetDefault();
        }
        if (pOutApplicationData)
        {
            *pOutApplicationData = {};
        }
    };

    NN_CAPSRV_PROCESS_START();

    cleanupOutput();
    NN_CAPSRV_PROCESS_ROLLBACK(cleanupOutput());

    NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumApplicationServiceHolder);

    capsrv::sf::LoadAlbumScreenShotImageOutputForApplication output = {};
    auto aruid = applet::GetAppletResourceUserId();
    auto result = pService->LoadAlbumScreenShotImageByAruid(
        &output,
        nn::sf::OutBuffer(reinterpret_cast<char*>(pBuffer), bufferSize),
        entry,
        option,
        nn::sf::OutBuffer(reinterpret_cast<char*>(pWorkBuffer), workBufferSize),
        aruid
    );
    if (result <= ResultAlbumFileNoThumbnail())
    {
        *pOutWidth  = ViewerThumbnailImageSize_Width;
        *pOutHeight = ViewerThumbnailImageSize_Height;
        std::memset(pBuffer, 0, sizeof(bufferSize));
    }
    NN_RESULT_DO(result);

    NN_CAPSRV_PROCESS_SUCCESS();
    if (pOutWidth)
    {
        *pOutWidth = static_cast<int>(output.width);
    }
    if (pOutHeight)
    {
        *pOutHeight = static_cast<int>(output.height);
    }
    if (pOutAttribute)
    {
        *pOutAttribute = output.attribute;
    }
    if (pOutApplicationData)
    {
        *pOutApplicationData = output.applicationData;
    }
    NN_RESULT_SUCCESS;
}

}   // namespace


// アプリ向けメディアファイルの画像データの読み込み
Result LoadAlbumContentsFileScreenShotImageForApplication(int* pOutWidth, int* pOutHeight, void* pBuffer, size_t bufferSize, void* pWorkBuffer, size_t workBufferSize, const ApplicationAlbumFileEntry& entry) NN_NOEXCEPT
{
    auto option = ScreenShotDecodeOption::GetDefaultValue();
    return LoadAlbumContentsFileScreenShotImageForApplicationImpl(pOutWidth, pOutHeight, nullptr, nullptr, pBuffer, bufferSize, pWorkBuffer, workBufferSize, option, entry);
}

Result LoadAlbumContentsFileScreenShotImageForApplication(int* pOutWidth, int* pOutHeight, ApplicationData* pOutApplicationData, void* pBuffer, size_t bufferSize, void* pWorkBuffer, size_t workBufferSize, const ApplicationAlbumFileEntry& entry) NN_NOEXCEPT
{
    auto option = ScreenShotDecodeOption::GetDefaultValue();
    return LoadAlbumContentsFileScreenShotImageForApplicationImpl(pOutWidth, pOutHeight, nullptr, pOutApplicationData, pBuffer, bufferSize, pWorkBuffer, workBufferSize, option, entry);
}

namespace {

Result LoadAlbumContentsFileThumbnailImageForApplicationImpl(
        int* pOutWidth,
        int* pOutHeight,
        ScreenShotAttributeForApplication* pOutAttribute,
        ApplicationData* pOutApplicationData,
        void* pBuffer,
        size_t bufferSize,
        void* pWorkBuffer,
        size_t workBufferSize,
        const ScreenShotDecodeOption& option,
        const ApplicationAlbumFileEntry& entry
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pWorkBuffer);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0);
    NN_SDK_REQUIRES_GREATER(workBufferSize, 0);

    NN_UTIL_SCOPE_EXIT{ std::memset(pWorkBuffer, 0, workBufferSize); };

    auto cleanupOutput = [&]()
    {
        if (pOutWidth)
        {
            *pOutWidth = 0;
        }
        if (pOutHeight)
        {
            *pOutHeight = 0;
        }
        if (pOutAttribute)
        {
            pOutAttribute->SetDefault();
        }
        if (pOutApplicationData)
        {
            *pOutApplicationData = {};
        }
    };

    NN_CAPSRV_PROCESS_START();

    cleanupOutput();
    NN_CAPSRV_PROCESS_ROLLBACK(cleanupOutput());

    NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumApplicationServiceHolder);

    capsrv::sf::LoadAlbumScreenShotImageOutputForApplication output = {};
    auto aruid = applet::GetAppletResourceUserId();
    NN_RESULT_DO(pService->LoadAlbumScreenShotThumbnailImageByAruid(
        &output,
        nn::sf::OutBuffer(reinterpret_cast<char*>(pBuffer), bufferSize),
        entry,
        option,
        nn::sf::OutBuffer(reinterpret_cast<char*>(pWorkBuffer), workBufferSize),
        aruid
    ));

    NN_CAPSRV_PROCESS_SUCCESS();
    if (pOutWidth)
    {
        *pOutWidth = static_cast<int>(output.width);
    }
    if (pOutHeight)
    {
        *pOutHeight = static_cast<int>(output.height);
    }
    if (pOutAttribute)
    {
        *pOutAttribute = output.attribute;
    }
    if (pOutApplicationData)
    {
        *pOutApplicationData = output.applicationData;
    }
    NN_RESULT_SUCCESS;
}
}

// アプリ向けメディアファイルのサムネイル画像データの読み込み
Result LoadAlbumContentsFileThumbnailImageForApplication(int* pOutWidth, int* pOutHeight, void* pBuffer, size_t bufferSize, void* pWorkBuffer, size_t workBufferSize, const ApplicationAlbumFileEntry& entry) NN_NOEXCEPT
{
    auto option = ScreenShotDecodeOption::GetDefaultValue();
    return LoadAlbumContentsFileThumbnailImageForApplicationImpl(pOutWidth, pOutHeight, nullptr, nullptr, pBuffer, bufferSize, pWorkBuffer, workBufferSize, option, entry);
    NN_RESULT_SUCCESS;
}

Result LoadAlbumContentsFileThumbnailImageForApplication(int* pOutWidth, int* pOutHeight, ApplicationData* pOutApplicationData, void* pBuffer, size_t bufferSize, void* pWorkBuffer, size_t workBufferSize, const ApplicationAlbumFileEntry& entry) NN_NOEXCEPT
{
    auto option = ScreenShotDecodeOption::GetDefaultValue();
    return LoadAlbumContentsFileThumbnailImageForApplicationImpl(pOutWidth, pOutHeight, nullptr, pOutApplicationData, pBuffer, bufferSize, pWorkBuffer, workBufferSize, option, entry);
    NN_RESULT_SUCCESS;
}

//-----------------------------
// セッション構築
//-----------------------------
namespace {

// AlbumAccessorApplicationSession を生成する。既に生成済みなら ResourceLimit を返す。
Result CreateMovieReadStreamSessionForApplication(AlbumAccessorApplicationServicePointerType* pOutValue, const ApplicationAlbumFileEntry entry) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(g_pAlbumAccessorApplicationSession == nullptr, ResultAlbumResourceLimit());

    NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumApplicationServiceHolder);

    AlbumApplicationServiceHolder::SessionPointerType pAccessSession;
    NN_RESULT_DO(pService->OpenAccessorSessionForApplication(&pAccessSession, entry, applet::GetAppletResourceUserId()));
    g_AlbumApplicationServiceHolder.StoreSession(pAccessSession);
    g_pAlbumAccessorApplicationSession = pAccessSession;
    *pOutValue = pAccessSession;
    NN_RESULT_SUCCESS;
}

// AlbumAccessorApplicationSession を破棄する。
void DestroyMovieReadStreamSessionForApplication() NN_NOEXCEPT
{
    g_pAlbumAccessorApplicationSession.Reset();
}

// AlbumAccessorApplicationSession を返す。
Result EnsureMovieReadStreamSessionForApplication(AlbumAccessorApplicationServicePointerType* pOutValue) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pAlbumAccessorApplicationSession != nullptr);
    *pOutValue = g_pAlbumAccessorApplicationSession;
    NN_RESULT_SUCCESS;
}

}   // namespace

//---------------------------------------------------------------------------
// 動画ストリーム関連
//---------------------------------------------------------------------------

// 動画読込ストリームのオープン（ハンドル取得）
Result OpenAlbumMovieReadStreamForApplication(AlbumMovieReadStreamHandle* pOutHandle, const ApplicationAlbumFileEntry& entry) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);

    AlbumAccessorApplicationServicePointerType pSession;
    NN_RESULT_DO(CreateMovieReadStreamSessionForApplication(&pSession, entry));

    detail::AlbumMovieReadStreamHandleType handleValue = {};
    auto aruid = applet::GetAppletResourceUserId();
    NN_RESULT_DO(pSession->OpenAlbumMovieReadStream(&handleValue, entry, aruid));

    *pOutHandle = AlbumMovieReadStreamHandle(handleValue);
    NN_RESULT_SUCCESS;
}

// 動画読込ストリームのクローズ（ハンドル破棄）
void CloseAlbumMovieReadStreamForApplication(AlbumMovieReadStreamHandle handle) NN_NOEXCEPT
{
    auto closeFunction = [&]() -> Result
    {
        AlbumAccessorApplicationServicePointerType pSession;
        NN_RESULT_DO(EnsureMovieReadStreamSessionForApplication(&pSession));
        NN_RESULT_DO(pSession->CloseAlbumMovieReadStream(handle.GetInnerValue()));
        DestroyMovieReadStreamSessionForApplication();
        NN_RESULT_SUCCESS;
    };
    NN_ABORT_UNLESS_RESULT_SUCCESS(closeFunction());
}

// 動画読込ストリームの動画データ部分のサイズ取得
Result GetAlbumMovieReadStreamDataSizeForApplication(uint64_t* pOutSize, AlbumMovieReadStreamHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);

    AlbumAccessorApplicationServicePointerType pSession;
    NN_RESULT_DO(EnsureMovieReadStreamSessionForApplication(&pSession));

    int64_t size = 0;
    NN_RESULT_DO(pSession->GetAlbumMovieReadStreamMovieDataSize(&size, handle.GetInnerValue()));
    *pOutSize = size;
    NN_RESULT_SUCCESS;
}

// 動画読込ストリームの動画データ部分のデータ読込み
Result ReadDataFromAlbumMovieReadStreamForApplication(size_t* pOutReadSize, void* buffer, size_t size, AlbumMovieReadStreamHandle handle, int64_t offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutReadSize);
    NN_SDK_REQUIRES_GREATER_EQUAL(size, 0u);

    AlbumAccessorApplicationServicePointerType pSession;
    NN_RESULT_DO(EnsureMovieReadStreamSessionForApplication(&pSession));

    int64_t readSize = 0;
    NN_RESULT_DO(pSession->ReadMovieDataFromAlbumMovieReadStream(&readSize, nn::sf::OutBuffer(reinterpret_cast<char*>(buffer), size), handle.GetInnerValue(), offset));
    *pOutReadSize = static_cast<size_t>(readSize);
    NN_RESULT_SUCCESS;
}

// 動画読込ストリームの内部エラーの取得
Result GetAlbumMovieReadStreamBrokenReasonForApplication(AlbumMovieReadStreamHandle handle) NN_NOEXCEPT
{
    AlbumAccessorApplicationServicePointerType pSession;
    NN_RESULT_DO(EnsureMovieReadStreamSessionForApplication(&pSession));

    NN_RESULT_DO(pSession->GetAlbumMovieReadStreamBrokenReason(handle.GetInnerValue()));
    NN_RESULT_SUCCESS;
}

// アルバムファイル作成前の事前チェック
Result PrecheckToCreateContentsForApplication(AlbumFileContentsType contents, uint64_t size) NN_NOEXCEPT
{
    NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumApplicationServiceHolder);
    auto aruid = applet::GetAppletResourceUserId();
    NN_RESULT_DO(pService->PrecheckToCreateContentsByAruid(contents, size, aruid));
    NN_RESULT_SUCCESS;
}


}}  // namespace nn::capsrv
