﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <winext/types.h>
#include <winext/cafe/os.h>
#include <winext/cafe/mem.h>
#include "heapCommoni.h"

namespace nw {
namespace internal {
namespace winext {

typedef struct arena_t {
//  void*         lo;          // effective address
//  u32           size;        // size of arena
//  void*         lo_pa;       // physical address
    MEMHeapHandle heap;
} arena_t;


static MEMSLock      arenaLock;
static arena_t       memArena[MEM_ARENA_MAX] = {
    {MEM_HEAP_INVALID_HANDLE},   // arena1
    {MEM_HEAP_INVALID_HANDLE},   // arena2
    {MEM_HEAP_INVALID_HANDLE},   // arena3
    {MEM_HEAP_INVALID_HANDLE},   // arena4
    {MEM_HEAP_INVALID_HANDLE},   // arena5
    {MEM_HEAP_INVALID_HANDLE},   // arena6
    {MEM_HEAP_INVALID_HANDLE},   // arena7
    {MEM_HEAP_INVALID_HANDLE}    // arena8
};


void MEMInitArenaLibrary (void) {
    MEMInitSLock(&arenaLock);
}


// Returns the base heap, if one exists, associated with the arena. An
// associated  heap means that heap is the base heap and owns all of the
// arena.  There may be other heap created on top of the base heap.
MEMHeapHandle MEMGetBaseHeapHandle (MEMArena arena) {
    if (MEM_ARENA_MAX <= arena)
    {
        return MEM_HEAP_INVALID_HANDLE;
    }
    return memArena[arena].heap;
}


// Associates a base heap with the arena. Returns previously associated heap
// handle. MEM_ARENA_NULL if none.
MEMHeapHandle MEMSetBaseHeapHandle (MEMArena arena, MEMHeapHandle heap) {
    MEMHeapHandle previousHeap;

    if (MEM_ARENA_MAX <= arena || heap == MEM_HEAP_INVALID_HANDLE)
    {
        return MEM_HEAP_INVALID_HANDLE;
    }

    MEMLockSLock(&arenaLock);
    previousHeap = memArena[arena].heap;
    if (previousHeap != MEM_HEAP_INVALID_HANDLE)
    {
        //ASSERTMSG1(previousHeap == MEM_HEAP_INVALID_HANDLE,
        //    "Cannot change the baseheap of arena%d. "   \
        //    "It is allowed to set baseheap only once. \n", arena);
        return MEM_HEAP_INVALID_HANDLE;
    }

    if (MEM_ARENA_2 == arena)
    {
        _MEMSetDefaultHeapHandle(heap);
    }
    memArena[arena].heap = heap;

    MEMUnlockSLock(&arenaLock);
    return previousHeap;
}


// Return the arena that is associated with heap. If no arena is found,
// MEM_ARENA_NULL is returned.  Linear search, but we don't expect this
// to be called frequently.  Faster would be to embed the arena id within
// the heap's metadata structure (TODO).
MEMArena MEMGetArena (MEMHeapHandle heap) {
    int i;

    if (heap == MEM_HEAP_INVALID_HANDLE)
    {
        return MEM_ARENA_INVALID;
    }

    for (i=MEM_ARENA_1; i<MEM_ARENA_MAX; i++) {
        if (memArena[i].heap == heap) return MEMArena(i);
    }
    return MEMArena(MEM_ARENA_INVALID);
}

} // namespace winext
} // namespace internal
} // namespace nw
