﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string.h>
#include <math.h>
#include <winext/cafe/os.h>
// #include <private/OSInterruptPrivate.h>
#include <winext/cafe/ax.h>
#include <winext/cafe/axfx.h>
#include "AXFXPrivate.h"

namespace nw {
    namespace internal {
        namespace winext {

/*****************************************************************************
  Standard Reverb  (expanded version)
 *****************************************************************************/

/*---------------------------------------------------------------------------*
  Block Diagram :


    IN--+->[ Early Reclection ]----------->--+
        |                                    |
        |                                    |
        +->[PreDelay]-->[ Fused Reverb ]-->[ + ]---->OUT


    ( Early Reflection )

        (simple delay)

        IN--->[+]-->[ delay line ]-->-+----->[ x gain ]--->OUT
               |                      |
               +-<---[ x gain ]<------+


    ( Fused Reverb )

        IN->-+->[Comb]->[+]->[AllPass]->[LPF]->[AllPass]-[ x gain ]-->OUT
             |           |
             +->[Comb]->-+


    (Comb Filter)

        IN--->[+]--->[ delay line ] ---+---->OUT
               |                       |
               +------[x coef]<--------+


    (All Pass Filter)

                    +---->--[x -coef]---->-----+
                    |                          |
        IN--->--[+]-+-->[  delay line  ]->-+--[+]--->OUT
                 |                         |
                 +-------<--[x coef]--<----+



    (one pole LPF)

        IN--->[x (1 - coef) ]-->[+]------------------+------>OUT
                                 |                   |
                                 +--[x coef]<--[Z]<--+

                        0(high) <= coef < 1 (low)

 *---------------------------------------------------------------------------*/

/*
 * early reflection sizes (sizes must remain prime numbers)
 */
static u32 __EarlySizeTable[AXFX_REVERBSTD_EXP_EARLY_MODE_MAX] =
{
     163,  //  5 msec
     317,  // 10 msec
     479,  // 15 msec
     641,  // 20 msec
     797,  // 25 msec
     967,  // 30 msec
    1123,  // 35 msec
    1283   // 40 msec
};


/*
 * filter delay sizes (sizes must remain prime numbers)
 */
static u32 __FilterSizeTable[AXFX_REVERBSTD_EXP_FUSED_MODE_MAX][4] =
{
    //  C0    C1   A0   A1
    { 1789, 1999, 433, 149 },  // OLD AXFX
    {  149,  293, 251, 103 },  // Metal Tank
    {  947, 1361, 433, 137 },  // Small Room
    { 1279, 1531, 509, 149 },  // Large Room
    { 1531, 1847, 563, 179 },  // Hall
    { 1823, 2357, 571, 137 },  // Cavernous
    { 1823, 2357, 571, 179 }   // Longest (for memory allocation)
};

/*
 * private funcs
 */
static BOOL __AllocDelayLine(AXFX_REVERBSTD_EXP *reverb);
static void __FreeDelayLine(AXFX_REVERBSTD_EXP *reverb);
static void __BzeroDelayLines(AXFX_REVERBSTD_EXP *reverb);
static BOOL __InitParams(AXFX_REVERBSTD_EXP *reverb);


/*---------------------------------------------------------------------------*
  Name:         AXFXReverbStdExpGetMemSize

  Description:  Returns how many memory bytes specified effect needs to alloc.

  Arguments:    reverb  expanded standard reverb structure

  Returns:      memory size (byte)
 *---------------------------------------------------------------------------*/

s32 AXFXReverbStdExpGetMemSize(AXFX_REVERBSTD_EXP *reverb)
{
    s32 i, size = 0;
    u32 samplerate = AXGetInputSamplesPerSec();

    ASSERTMSG(reverb->preDelayTimeMax >= 0.f, AXFX_ERR_PARAMETER);

    size += __EarlySizeTable[AXFX_REVERBSTD_EXP_EARLY_MODE_MAX - 1];
    size += (s32)(reverb->preDelayTimeMax * (f32)samplerate);
    for(i = 0; i < 2; i++)
    {
        size += __FilterSizeTable[AXFX_REVERBSTD_EXP_FUSED_MODE_LONGEST][i];
    }
    for(i = 0; i < 2; i++)
    {
        size += __FilterSizeTable[AXFX_REVERBSTD_EXP_FUSED_MODE_LONGEST][2 + i];
    }

    return (s32)(size * sizeof(f32) * 3);
}


/*---------------------------------------------------------------------------*
  Name:         AXFXReverbStdExpInit

  Description:  initialises starndard reverb effect.

  Arguments:    reverb       standard reverb structure

  Returns:      TRUE=success, FALSE=failure.
 *---------------------------------------------------------------------------*/
BOOL AXFXReverbStdExpInit(AXFX_REVERBSTD_EXP *reverb)
{
    u32 i;
    BOOL result = TRUE;
    u32 samplerate = AXGetInputSamplesPerSec();
    BOOL mask = OSDisableInterrupts();
#if 0
    if (AXGetMode() == AX_MODE_DPL2)
    {
#ifdef _DEBUG
        COSWarn(COSREPORT_MODULE_SOUND,"AXFXReverbStdExpInit(): WARNING: Invalid AX output mode.\n");
#endif
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }
#endif
    reverb->active = AXFX_EFF_INACTIVE;

    //
    // checks parameter
    //
    ASSERTMSG(reverb->preDelayTimeMax >= 0.f, AXFX_ERR_PARAMETER);
    if(reverb->preDelayTimeMax < 0.f)
    {
        AXFXReverbStdExpShutdown(reverb);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    //
    // sets max delay size
    //
    reverb->earlyMaxLength = __EarlySizeTable[AXFX_REVERBSTD_EXP_EARLY_MODE_MAX - 1];

    reverb->preDelayMaxLength = (u32)(reverb->preDelayTimeMax * (f32)samplerate);

    for(i = 0; i < 2; i++)
    {
        reverb->combMaxLength[i] = __FilterSizeTable[AXFX_REVERBSTD_EXP_FUSED_MODE_LONGEST][i];
    }

    for(i = 0; i < 2; i++)
    {
        reverb->allpassMaxLength[i] = __FilterSizeTable[AXFX_REVERBSTD_EXP_FUSED_MODE_LONGEST][2 + i];
    }

    //
    // allocates delay lines
    //
    result = __AllocDelayLine(reverb);
    if(result == FALSE)
    {
        AXFXReverbStdExpShutdown(reverb);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    //
    // bzeros delay lines
    //
    __BzeroDelayLines(reverb);

    //
    // initialises parameters
    //
    result = __InitParams(reverb);
    if(result == FALSE)
    {
        AXFXReverbStdExpShutdown(reverb);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    reverb->active &= ~AXFX_EFF_INACTIVE;
    (void)OSRestoreInterrupts(mask);

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         AXFXReverbStdExpSettings

  Description:  updates standard reverb setting.
                (with free & alloc)

  Arguments:    reverb       standard reverb structure

  Returns:      TRUE=success, FALSE=failure.
 *---------------------------------------------------------------------------*/
BOOL AXFXReverbStdExpSettings(AXFX_REVERBSTD_EXP *reverb)
{
    BOOL result = TRUE;
    BOOL mask = OSDisableInterrupts();

    reverb->active |= AXFX_EFF_INACTIVE;

    AXFXReverbStdExpShutdown(reverb);
    result = AXFXReverbStdExpInit(reverb);
    if(result == FALSE)
    {
        AXFXReverbStdExpShutdown(reverb);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    reverb->active |= AXFX_EFF_UPDATE;
    reverb->active &= ~AXFX_EFF_INACTIVE;
    (void)OSRestoreInterrupts(mask);

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         AXFXReverbStdExpSettingsUpdate

  Description:  updates standard reverb setting.
                (without free & alloc)

  Arguments:    reverb       standard reverb structure

  Returns:      TRUE=success, FALSE=failure.
 *---------------------------------------------------------------------------*/
BOOL AXFXReverbStdExpSettingsUpdate(AXFX_REVERBSTD_EXP *reverb)
{
    BOOL result = TRUE;
    BOOL mask = OSDisableInterrupts();

    reverb->active |= AXFX_EFF_INACTIVE;

    __BzeroDelayLines(reverb);

    result = __InitParams(reverb);
    if(result == FALSE)
    {
        AXFXReverbStdExpShutdown(reverb);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    reverb->active |= AXFX_EFF_UPDATE;
    reverb->active &= ~AXFX_EFF_INACTIVE;
    (void)OSRestoreInterrupts(mask);

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         AXFXReverbStdExpShutdown

  Description:  shuts down the specified instance of the standard reverb effect
                and frees any associated memory.

  Arguments:    reverb       standard reverb structure

  Returns:      None.
 *---------------------------------------------------------------------------*/
void AXFXReverbStdExpShutdown(AXFX_REVERBSTD_EXP *reverb)
{
    BOOL mask = OSDisableInterrupts();

    reverb->active |= AXFX_EFF_INACTIVE;

    __FreeDelayLine(reverb);

    (void)OSRestoreInterrupts(mask);
}


/*---------------------------------------------------------------------------*
  Name:         AXFXReverbStdExpCallback

  Description:  AUX update calback

  Arguments:    bufferUpdate    buffer to update
                reverb          standard reverb structure

  Returns:      None.
 *---------------------------------------------------------------------------*/
void AXFXReverbStdExpCallback(AXFX_BUFFERUPDATE *bufferUpdate, AXFX_REVERBSTD_EXP *reverb)
{
    u32 ch;
    u32 samp;

    u32  earlyPos;
    u32  preDelayPos;
    u32  comb0Pos;
    u32  comb1Pos;
    u32  allpass0Pos;
    u32  allpass1Pos;

    s32 *input[3];
    f32  data;
    f32  output;

    f32 *earlyLine;
    f32  earlyOut;
    f32  earlyCoef;

    f32 *preDelayLine;
    f32  preDelayOut;

    f32 *combLine0;
    f32 *combLine1;
    f32  combOut0;
    f32  combOut1;
    f32  combOut;
    f32  combCoef0;
    f32  combCoef1;

    f32 *allpass;
    f32  outTmp;
    f32  allpassIn;
    f32  allpass0Out;
    f32  allpass1Out;
    f32  allpassCoef;

    f32  lpfOut;

    f32  lpfCoef1;
    f32  lpfCoef2;

    f32  earlyGain;
    f32  fusedGain;

    s32 *inBusData[3];
    s32 *outBusData[3];

    u32 aux_blocksize =  AXGetInputSamplesPerFrame();

    if(reverb->active != 0)
    {
        reverb->active &= ~AXFX_EFF_UPDATE;
        return;
    }

    input[0] = bufferUpdate->left;
    input[1] = bufferUpdate->right;
    input[2] = bufferUpdate->surround;

    if(reverb->busIn != NULL)
    {
        inBusData[0] = reverb->busIn->left;
        inBusData[1] = reverb->busIn->right;
        inBusData[2] = reverb->busIn->surround;
    }

    if(reverb->busOut != NULL)
    {
        outBusData[0] = reverb->busOut->left;
        outBusData[1] = reverb->busOut->right;
        outBusData[2] = reverb->busOut->surround;
    }

    // NOTE : copy from members of structure
    lpfCoef1 = 1.f - reverb->lpfCoef;
    lpfCoef2 = reverb->lpfCoef;

    earlyCoef = reverb->earlyCoef;
    combCoef0 = reverb->combCoef[0];
    combCoef1 = reverb->combCoef[1];
    allpassCoef = reverb->allpassCoef;

    earlyGain = reverb->earlyGain * AXFX_REVERB_OUT_GAIN;
    fusedGain = reverb->fusedGain * AXFX_REVERB_OUT_GAIN;


    for(samp = 0; samp < aux_blocksize; samp++)
    {
        // NOTE : copy from members of structure
        earlyPos    = reverb->earlyPos;
        preDelayPos = reverb->preDelayPos;
        comb0Pos    = reverb->combPos[0];
        comb1Pos    = reverb->combPos[1];
        allpass0Pos = reverb->allpassPos[0];
        allpass1Pos = reverb->allpassPos[1];

        for(ch = 0; ch < 3; ch++)
        {
            // input
            if(reverb->busIn != NULL)
            {
                data = (f32)(*(input[ch]) + *(inBusData[ch]++));
            }
            else
            {
                data = (f32)(*input[ch]);
            }

            //
            // early reflection
            //
            earlyLine = reverb->earlyLine[ch];
            earlyOut = earlyLine[earlyPos];
            earlyLine[earlyPos] = data + (earlyOut * earlyCoef);


            //
            // pre-delay
            //
            if(reverb->preDelayLength != 0)
            {
                preDelayLine = reverb->preDelayLine[ch];
                preDelayOut = preDelayLine[preDelayPos];
                preDelayLine[preDelayPos] = data;
            }
            else
            {
                preDelayOut = data;
            }

            //
            // comb filter
            //
            combLine0 = reverb->combLine[ch][0];
            combOut0 = combLine0[comb0Pos];
            combLine0[comb0Pos] =  preDelayOut
                                + (combOut0 * combCoef0);

            combLine1 = reverb->combLine[ch][1];
            combOut1 = combLine1[comb1Pos];
            combLine1[comb1Pos] =  preDelayOut
                                + (combOut1 * combCoef1);

            combOut =  combOut0 + combOut1;


            //
            // all pass filter
            //

            // 1st step
            allpass = reverb->allpassLine[ch][0];
            outTmp = allpass[allpass0Pos];
            allpassIn =  combOut + outTmp * allpassCoef;
            allpass[allpass0Pos] = allpassIn;
            allpass0Out =  outTmp - allpassIn * allpassCoef;

            //
            // LPF
            //
            lpfOut =  lpfCoef1 * allpass0Out
                    + lpfCoef2 * reverb->lastLpfOut[ch];
            reverb->lastLpfOut[ch] = lpfOut;


            //
            // last all pass filter
            //
            allpass = reverb->allpassLine[ch][1];
            outTmp = allpass[allpass1Pos];
            allpassIn = lpfOut + outTmp * allpassCoef;
            allpass[allpass1Pos] = allpassIn;
            allpass1Out = outTmp - allpassIn * allpassCoef;

            //
            // output
            //
            output = (earlyOut * earlyGain) + (allpass1Out * fusedGain);

            *(input[ch]++) = (s32)(output * reverb->outGain);

            // bus out
            if(reverb->busOut != NULL)
            {
                *(outBusData[ch]++) = (s32)(output * reverb->sendGain);
            }
        }


        //
        // updates delay positions
        //

        // early reflection
        if(++reverb->earlyPos >= reverb->earlyLength)
        {
            reverb->earlyPos = 0;
        }

        // pre-delay
        if(reverb->preDelayLength != 0)
        {
            if(++reverb->preDelayPos >= reverb->preDelayLength)
            {
                reverb->preDelayPos = 0;
            }
        }

        // comb filter
        if(++reverb->combPos[0] >= reverb->combLength[0])
        {
            reverb->combPos[0] = 0;
        }

        if(++reverb->combPos[1] >= reverb->combLength[1])
        {
            reverb->combPos[1] = 0;
        }

        // all pass filter
        if(++reverb->allpassPos[0] >= reverb->allpassLength[0])
        {
            reverb->allpassPos[0] = 0;
        }

        if(++reverb->allpassPos[1] >= reverb->allpassLength[1])
        {
            reverb->allpassPos[1] = 0;
        }
    }
}



/*===========================================================================*
   Private Funcs
 *===========================================================================*/

/*---------------------------------------------------------------------------*
  Name:         __AllocDelayLine

  Description:  allocates delay lines

  Arguments:    reverb       standard reverb structure

  Returns:      TRUE=success, FALSE=failure
 *---------------------------------------------------------------------------*/
static BOOL __AllocDelayLine(AXFX_REVERBSTD_EXP *reverb)
{
    u32 ch, i;

    for(ch = 0; ch < 3; ch++)
    {
        // allocates early reflection delay
        reverb->earlyLine[ch] = (f32*)__AXFXAlloc( sizeof(f32) * reverb->earlyMaxLength );
        ASSERTMSG(reverb->earlyLine[ch], AXFX_ERR_MEMORY);
        if(reverb->earlyLine[ch]  == NULL) return FALSE;

        // allocates pre delay
        if(reverb->preDelayMaxLength != 0)
        {
            reverb->preDelayLine[ch] = (f32*)__AXFXAlloc( sizeof(f32) * reverb->preDelayMaxLength );
            ASSERTMSG(reverb->preDelayLine[ch], AXFX_ERR_MEMORY);
            if(reverb->preDelayLine[ch]  == NULL) return FALSE;
        }
        else
        {
            reverb->preDelayLine[ch] = NULL;
        }

        // allocates comb filter delay line
        for(i = 0; i < 2; i++)
        {
            reverb->combLine[ch][i] = (f32*)__AXFXAlloc( sizeof(f32) * reverb->combMaxLength[i] );
            ASSERTMSG(reverb->combLine[ch][i], AXFX_ERR_MEMORY);
            if(reverb->combLine[ch][i]  == NULL) return FALSE;
        }

        // allocates all pass filter line
        for(i = 0; i < 2; i++)
        {
            reverb->allpassLine[ch][i] = (f32*)__AXFXAlloc( sizeof(f32) * reverb->allpassMaxLength[i] );
            ASSERTMSG(reverb->allpassLine[ch][i], AXFX_ERR_MEMORY);
            if(reverb->allpassLine[ch][i]  == NULL) return FALSE;
        }
    }

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         __BzeroDelayLines

  Description:  bzeros all delay lines

  Arguments:    reverb        standard reverb structure

  Returns:      None
 *---------------------------------------------------------------------------*/
static void __BzeroDelayLines(AXFX_REVERBSTD_EXP* reverb)
{
    u32 ch, i;

    for(ch = 0; ch < 3; ch++)
    {
        // early reflection
        if(reverb->earlyLine[ch] != NULL)
        {
            (void)memset(reverb->earlyLine[ch], 0,
                    sizeof(f32) * reverb->earlyMaxLength);
        }

        // pre delay
        if(reverb->preDelayLine[ch] != NULL)
        {
            (void)memset(reverb->preDelayLine[ch], 0,
                    sizeof(f32) * reverb->preDelayMaxLength);
        }

        // comb filter
        for(i = 0; i < 2; i++)
        {
            if(reverb->combLine[ch][i] != NULL)
            {
                (void)memset(reverb->combLine[ch][i], 0,
                        sizeof(f32) * reverb->combMaxLength[i]);
            }
        }

        // all pass filter
        for(i = 0; i < 2; i++)
        {
            if(reverb->allpassLine[ch][i] != NULL)
            {
                (void)memset(reverb->allpassLine[ch][i], 0,
                        sizeof(f32) * reverb->allpassMaxLength[i]);
            }
        }
    }
}


/*---------------------------------------------------------------------------*
  Name:         __FreeDelayLine

  Description:  frees delay lines

  Arguments:    reverb       standard reverb structure

  Returns:      none
 *---------------------------------------------------------------------------*/
static void __FreeDelayLine(AXFX_REVERBSTD_EXP *reverb)
{
    u32 ch, i;

    for(ch = 0; ch < 3; ch++)
    {
        // early reflection
        if(reverb->earlyLine[ch] != NULL)
        {
            __AXFXFree(reverb->earlyLine[ch]);
            reverb->earlyLine[ch] = NULL;
        }

        // pre delay
        if(reverb->preDelayLine[ch] != NULL)
        {
            __AXFXFree(reverb->preDelayLine[ch]);
            reverb->preDelayLine[ch] = NULL;
        }

        // comb filter
        for(i = 0; i < 2; i++)
        {
            if(reverb->combLine[ch][i] != NULL)
            {
                __AXFXFree(reverb->combLine[ch][i]);
                reverb->combLine[ch][i] = NULL;
            }
        }

        // all pass filter
        for(i = 0; i < 2; i++)
        {
            if(reverb->allpassLine[ch][i] != NULL)
            {
                __AXFXFree(reverb->allpassLine[ch][i]);
                reverb->allpassLine[ch][i] = NULL;
            }
        }
    }
}



/*---------------------------------------------------------------------------*
  Name:         __InitParams

  Description:  initialises delay params

  Arguments:    reverb       standard reverb structure

  Returns:      TRUE=success, FALSE=failure
 *---------------------------------------------------------------------------*/
static BOOL __InitParams(AXFX_REVERBSTD_EXP *reverb)
{
    u32 ch, i;
    u32 samplerate = AXGetInputSamplesPerSec();

    //
    // parameter check
    //

    ASSERTMSG(reverb->earlyMode < AXFX_REVERBSTD_EXP_EARLY_MODE_MAX, AXFX_ERR_PARAMETER);
    if(reverb->earlyMode >= AXFX_REVERBSTD_EXP_EARLY_MODE_MAX) return FALSE;

    ASSERTMSG(reverb->preDelayTime >= 0.f && reverb->preDelayTime <= reverb->preDelayTimeMax, AXFX_ERR_PARAMETER);
    if(reverb->preDelayTime < 0.f || reverb->preDelayTime > reverb->preDelayTimeMax) return FALSE;

    ASSERTMSG(reverb->fusedMode < AXFX_REVERBSTD_EXP_FUSED_MODE_LONGEST, AXFX_ERR_PARAMETER);
    if(reverb->fusedMode >= AXFX_REVERBSTD_EXP_FUSED_MODE_LONGEST) return FALSE;

    ASSERTMSG(reverb->fusedTime >= 0.f, AXFX_ERR_PARAMETER);
    if(reverb->fusedTime < 0.f) return FALSE;

    ASSERTMSG(reverb->coloration >= 0.f && reverb->coloration <= 1.f, AXFX_ERR_PARAMETER);
    if(reverb->coloration < 0.f || reverb->coloration > 1.f) return FALSE;

    ASSERTMSG(reverb->damping >= 0.f && reverb->damping <= 1.f, AXFX_ERR_PARAMETER);
    if(reverb->damping < 0.f || reverb->damping > 1.f) return FALSE;

    ASSERTMSG(reverb->earlyGain >= 0.f && reverb->earlyGain <= 1.f, AXFX_ERR_PARAMETER);
    if(reverb->earlyGain < 0.f || reverb->earlyGain > 1.f) return FALSE;

    ASSERTMSG(reverb->fusedGain >= 0.f && reverb->fusedGain <= 1.f, AXFX_ERR_PARAMETER);
    if(reverb->fusedGain < 0.f || reverb->fusedGain > 1.f) return FALSE;


    ASSERTMSG(reverb->outGain >= 0.f && reverb->outGain <= 1.f, AXFX_ERR_PARAMETER);
    if(reverb->outGain < 0.f || reverb->outGain > 1.f) return FALSE;

    ASSERTMSG(reverb->sendGain >= 0.f && reverb->sendGain <= 1.f, AXFX_ERR_PARAMETER);
    if(reverb->sendGain < 0.f || reverb->sendGain > 1.f) return FALSE;


    //
    // initialise early reflection
    //
    reverb->earlyPos = 0;
    reverb->earlyLength = __EarlySizeTable[reverb->earlyMode];

    if(reverb->earlyMode <= AXFX_REVERBSTD_EXP_EARLY_MODE_20MS)
    {
        // NOTE : inverse phase is for reduction of flanger effect
        reverb->earlyCoef = -0.33f;
    }
    else
    {
        reverb->earlyCoef = 0.33f;
    }

    //
    // initialises pre-delay
    //
    reverb->preDelayPos = 0;
    reverb->preDelayLength = (u32)(reverb->preDelayTime * (f32)samplerate);

    // initialises comb filter
    for(i = 0; i < 2; i++)
    {
        reverb->combPos[i] = 0;
        reverb->combLength[i] = __FilterSizeTable[reverb->fusedMode][i];
        reverb->combCoef[i] =
                        powf(10.f,
                               (-3.f
                                 * (f32)(reverb->combLength[i])
                                 / (f32)(reverb->fusedTime * samplerate)
                               )
                            );
    }


    //
    // intialises allpass filter
    //
    for(i = 0; i < 2; i++)
    {
        reverb->allpassPos[i] = 0;
        reverb->allpassLength[i] = __FilterSizeTable[reverb->fusedMode][2 + i];
    }
    reverb->allpassCoef = reverb->coloration;


    //
    // initialises LPF
    //
    reverb->lpfCoef = 1.f - reverb->damping;
    if(reverb->lpfCoef > 0.95f) reverb->lpfCoef = 0.95f;

    for(ch = 0; ch < 3; ch++)
    {
        reverb->lastLpfOut[ch] = 0.f;
    }

    return TRUE;

}

        } // namespace winext
    } // namespace internal
} // namespace nw
