﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <string.h>
#include <winext/cafe/os.h>
// #include <private/OSInterruptPrivate.h>
#include <winext/cafe/ax.h>
#include <winext/cafe/axfx.h>
#include "AXFXPrivate.h"

namespace nw {
    namespace internal {
        namespace winext {

/*****************************************************************************
  Chorus (expanded version)
 *****************************************************************************/

/*---------------------------------------------------------------------------*
  Block Diagram :

         +-<-----[ x feedback gain]<--+
         |                            |
    IN->[+]--->[  Delay Line ]-->-----+----->OUT
                  ~~~~~~~~~~
                      T(delay time modulation)
                      |
                    [LFO] (sin wave)

 *---------------------------------------------------------------------------*/

static BOOL __AllocDelay(AXFX_CHORUS_EXP *chorus);
static BOOL __InitDelay(AXFX_CHORUS_EXP* chorus);
static void __FreeDelay(AXFX_CHORUS_EXP *chorus);
static BOOL __InitParams(AXFX_CHORUS_EXP* chorus);
static void __CalcLFO(s32* buf, AXFX_CHORUS_EXP_LFO *lfo);


/*---------------------------------------------------------------------------*
  Name:         AXFXChorusExpGetMemSize

  Description:  Returns how many memory bytes specified effect needs to alloc.

  Arguments:    chorus  expanded chorus structure

  Returns:      memory size (byte)
 *---------------------------------------------------------------------------*/

s32 AXFXChorusExpGetMemSize(AXFX_CHORUS_EXP * /*chorus __attribute__((unused))*/ )
{
    u32 sampler_per_100ms = AXGetInputSamplesPerSec() / 10;

    return (sampler_per_100ms * sizeof(f32) * 3);
}


/*---------------------------------------------------------------------------*
  Name:         AXFXChorusExpInit

  Description:  initialises expanded chorus

  Arguments:    chorus        expanded chorus structure

  Returns:      TRUE=success, FALSE=failure
 *---------------------------------------------------------------------------*/
BOOL AXFXChorusExpInit(AXFX_CHORUS_EXP *chorus)
{
    BOOL result;
    BOOL mask = OSDisableInterrupts();
#if 0
    if (AXGetMode() == AX_MODE_DPL2)
    {
#ifdef _DEBUG
        COSWarn(COSREPORT_MODULE_SOUND,"AXFXChorusExpInit(): WARNING: Invalid AX output mode.\n");
#endif
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }
#endif
    chorus->active = AXFX_EFF_INACTIVE;

    // allocates delay line
    result = __AllocDelay(chorus);
    if(result == FALSE)
    {
        AXFXChorusExpShutdown(chorus);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    // initialises delay line
    result = __InitDelay(chorus);
    if(result == FALSE)
    {
        AXFXChorusExpShutdown(chorus);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }


    // checks parameteres
    result = __InitParams(chorus);
    if(result == FALSE)
    {
        AXFXChorusExpShutdown(chorus);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    chorus->active &= ~AXFX_EFF_INACTIVE;

    (void)OSRestoreInterrupts(mask);

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         AXFXChorusExpSettings

  Description:  updates expanded chorus settings.
                (with free & alloc)

  Arguments:    chorus      expanded chorus structure

  Returns:      TRUE=success, FALSE=failure.
 *---------------------------------------------------------------------------*/
BOOL AXFXChorusExpSettings(AXFX_CHORUS_EXP *chorus)
{
    BOOL result;
    BOOL mask = OSDisableInterrupts();

    chorus->active |= AXFX_EFF_INACTIVE;

    AXFXChorusExpShutdown(chorus);

    result = AXFXChorusExpInit(chorus);
    if(result == FALSE)
    {
        AXFXChorusExpShutdown(chorus);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    chorus->active |= AXFX_EFF_UPDATE;
    chorus->active &= ~AXFX_EFF_INACTIVE;

    (void)OSRestoreInterrupts(mask);


    return result;
}



/*---------------------------------------------------------------------------*
  Name:         AXFXChorusExpSettingsUpdate

  Description:  updates expanded chorus settings.
                (without free & alloc)

  Arguments:    chorus      chorus structure

  Returns:      TRUE=success, FALSE=failure.
 *---------------------------------------------------------------------------*/
BOOL AXFXChorusExpSettingsUpdate(AXFX_CHORUS_EXP *chorus)
{
    BOOL result;
    BOOL mask = OSDisableInterrupts();

    chorus->active |= AXFX_EFF_INACTIVE;

    // initialises delay line
    result = __InitDelay(chorus);
    if(result == FALSE)
    {
        AXFXChorusExpShutdown(chorus);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }


    // initialises parameters
    result = __InitParams(chorus);
    if(result == FALSE)
    {
        AXFXChorusExpShutdown(chorus);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    chorus->active |= AXFX_EFF_UPDATE;
    chorus->active &= ~AXFX_EFF_INACTIVE;

    (void)OSRestoreInterrupts(mask);

    return result;
}



/*---------------------------------------------------------------------------*
  Name:         AXFXChorusExpShutdown

  Description:  shuts down expanded chorus

  Arguments:    chorus      expanded chorus structure

  Returns:      none.
 *---------------------------------------------------------------------------*/
void AXFXChorusExpShutdown(AXFX_CHORUS_EXP *chorus)
{
    BOOL mask = OSDisableInterrupts();

    chorus->active |= AXFX_EFF_INACTIVE;

    __FreeDelay(chorus);

    (void)OSRestoreInterrupts(mask);
}


/*---------------------------------------------------------------------------*
  Name:         AXFXChorusCallback

  Description:  AUX update calback

  Arguments:    bufferUpdate    buffer to update
                chorus          expanded chorus structure

  Returns:      None
 *---------------------------------------------------------------------------*/
void AXFXChorusExpCallback(AXFX_BUFFERUPDATE *bufferUpdate, AXFX_CHORUS_EXP *chorus)
{
    u32 ch, samp;
    u32 i;
    AXFX_CHORUS_EXP_DELAY *delay;
    const u32 aux_blocksize =  AXGetInputSamplesPerFrame();

    s32 *indata[3];

    f32 out;

    s32 lfoLevel;   // 0xIIIIDDDD
    s32 readPos;    // 0xIIIIDDDD
    s32 delta;      // 0xIIIIDDDD
    u32 deltaI;
    u32 deltaD;
    u32 curReadPos;
    u32 index;     // 0 - 3

    f32 *coef;
    u32  coefSetNum;

    s32 lfoBuf[AX_IN_SAMPLES_PER_FRAME * 2]; // HACK: VC++ にて、配列の要素数を定数式にしないといけないので、多めの値を設定しておく。

    s32 *inBusData[3];
    s32 *outBusData[3];
    f32 data;

    if(chorus->active != 0)
    {
        chorus->active &= ~AXFX_EFF_UPDATE;
        return;
    }

    indata[0] = bufferUpdate->left;
    indata[1] = bufferUpdate->right;
    indata[2] = bufferUpdate->surround;

    if(chorus->busIn != NULL)
    {
        inBusData[0] = chorus->busIn->left;
        inBusData[1] = chorus->busIn->right;
        inBusData[2] = chorus->busIn->surround;
    }

    if(chorus->busOut != NULL)
    {
        outBusData[0] = chorus->busOut->left;
        outBusData[1] = chorus->busOut->right;
        outBusData[2] = chorus->busOut->surround;
    }


    delay = &(chorus->delay);

    // calculates LFO value
    __CalcLFO( lfoBuf, &(chorus->lfo) );


    for(samp = 0; samp < aux_blocksize; samp++)
    {
        // get LFO value
        lfoLevel = lfoBuf[samp];

        //
        // calculates chorus
        //
        readPos = (s32)(delay->outPos) + lfoLevel;

        if(readPos >= (s32)(delay->sizeFP))
        {
            readPos -= delay->sizeFP;
        }
        else if(readPos < 0)
        {
            readPos += delay->sizeFP;
        }

        delta = readPos - (s32)(delay->lastPos);
        if(delta < 0) delta += delay->sizeFP;
            // (NOTE : delta must not be < 0)

        deltaI = ((u32)delta & 0xffff0000) >> 16;
        deltaD = (u32)(delta) & 0x0000ffff;       // 0x0 - 0xffff

        curReadPos = delay->lastPos >> 16;

        index = chorus->histIndex;

        while(deltaI--)
        {
            chorus->history[0][index]   = delay->line[0][curReadPos];
            chorus->history[1][index]   = delay->line[1][curReadPos];
            chorus->history[2][index++] = delay->line[2][curReadPos++];
            index &= 0x3;

            if(curReadPos >= delay->size)
            {
                curReadPos = 0;
            }
        }

        coefSetNum = (deltaD & 0x0000fe00) >> 9;  // 0 - 127

        delay->lastPos = readPos & 0xffff0000;


        coef = __AXFXGetSrcCoef(coefSetNum);

        for(ch = 0; ch < 3; ch++)
        {
            out = 0.f;
            for(i = 0; i < 4; i++)
            {
                out += chorus->history[ch][index++] * (*(coef + i));
                index &= 0x3;
            }

            // input
            if(chorus->busIn != NULL)
            {
                data = (f32)(*(indata[ch]) + *(inBusData[ch]++));
            }
            else
            {
                data = (f32)(*indata[ch]);
            }

            // delay line input (& feedback)
            delay->line[ch][delay->inPos] = data + out * chorus->feedback;

            // output
            *(indata[ch]++) = (s32)(out * chorus->outGain);

            // bus out
            if(chorus->busOut != NULL)
            {
                *(outBusData[ch]++) = (s32)(out * chorus->sendGain);
            }
        }
        chorus->histIndex = index;

        if(++delay->inPos >= delay->size)
        {
            delay->inPos = 0;
        }

        // updates 'original' delay position
        if((delay->outPos += 0x00010000) >= delay->sizeFP)
        {
            delay->outPos = 0;
        }

    }
}




/*===========================================================================*
   Private Funcs
 *===========================================================================*/

/*---------------------------------------------------------------------------*
  Name:         __AllocDelay

  Description:  allocates delay line

  Arguments:    chorus      expanded chorus structure

  Returns:      TRUE=success, FALSE=failure
 *---------------------------------------------------------------------------*/
static BOOL __AllocDelay(AXFX_CHORUS_EXP *chorus)
{
    u32 ch;
    AXFX_CHORUS_EXP_DELAY *delay;
    u32 sampler_per_100ms = AXGetInputSamplesPerSec() / 10;

    delay = &(chorus->delay);

    delay->size = sampler_per_100ms;

    for(ch = 0; ch < 3; ch++)
    {
        delay->line[ch] = reinterpret_cast<f32*>(__AXFXAlloc(sizeof(f32) * delay->size));

        ASSERTMSG(delay->line[ch], AXFX_ERR_MEMORY);
        if(delay->line[ch] == NULL) return FALSE;
    }

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         __InitDelay

  Description:  initialises delay line

  Arguments:    chorus      expanded chorus structure

  Returns:      TRUE=success, FALSE=failure
 *---------------------------------------------------------------------------*/
static BOOL __InitDelay(AXFX_CHORUS_EXP* chorus)
{
    u32 ch;
    AXFX_CHORUS_EXP_DELAY *delay;
    u32 samples_msec = AXGetInputSamplesPerSec() / 1000;

    delay = &(chorus->delay);

    for(ch = 0; ch < 3; ch++)
    {
        ASSERTMSG(delay->line[ch] != NULL, AXFX_ERR_NO_BUFFER);
        if(delay->line[ch] == NULL) return FALSE;

        (void)memset(delay->line[ch], 0, sizeof(f32) * delay->size);
    }

    delay->inPos = 0;

    delay->outPos = delay->size
                  - (u32)(chorus->delayTime * (f32)samples_msec);

    delay->outPos <<= 16;  // 0xIIII0000
    delay->lastPos = delay->outPos;

    delay->sizeFP = delay->size << 16;  // 0xIIII0000

    return TRUE;
}



/*---------------------------------------------------------------------------*
  Name:         __FreeDelay

  Description:  frees delay line

  Arguments:    chorus      expanded chorus structure

  Returns:      none
 *---------------------------------------------------------------------------*/
static void __FreeDelay(AXFX_CHORUS_EXP *chorus)
{
    u32 ch;
    AXFX_CHORUS_EXP_DELAY *delay;

    delay = &(chorus->delay);

    for(ch = 0; ch < 3; ch++)
    {
        if(delay->line[ch])
        {
            __AXFXFree(delay->line[ch]);
        }
        delay->line[ch] = NULL;
    }

}


/*---------------------------------------------------------------------------*
  Name:         __InitParams

  Description:  initialises parameters

  Arguments:    chorus      expanded chorus structure

  Returns:      TRUE=success, FALSE=failure
 *---------------------------------------------------------------------------*/
static BOOL __InitParams(AXFX_CHORUS_EXP* chorus)
{
    u32 ch, i;
    f32  delay_samp, depth_samp, step_samp, lfo_grad_factor;
    u32 samples_msec = AXGetInputSamplesPerSec() / 1000;

    //
    // checks parameters
    //
    ASSERTMSG(chorus->delayTime >= 0.1f && chorus->delayTime <= 50.f,
              AXFX_ERR_PARAMETER);
    if(chorus->delayTime < 0.1f || chorus->delayTime > 50.f) return FALSE;

    ASSERTMSG(chorus->depth >= 0.f && chorus->depth <= 1.f,
              AXFX_ERR_PARAMETER);
    if(chorus->depth < 0.f || chorus->depth > 1.f) return FALSE;

    ASSERTMSG(chorus->rate >= 0.1f && chorus->rate <= 2.f,
              AXFX_ERR_PARAMETER);
    if(chorus->rate < 0.1f || chorus->rate > 2.f) return FALSE;

    ASSERTMSG(chorus->feedback >= 0.f && chorus->feedback < 1.f,
              AXFX_ERR_PARAMETER);
    if(chorus->feedback < 0.f || chorus->feedback >= 1.f) return FALSE;


    ASSERTMSG(chorus->outGain >= 0.f && chorus->outGain <= 1.f, AXFX_ERR_PARAMETER);
    if(chorus->outGain < 0.f || chorus->outGain > 1.f) return FALSE;

    ASSERTMSG(chorus->sendGain >= 0.f && chorus->sendGain <= 1.f, AXFX_ERR_PARAMETER);
    if(chorus->sendGain < 0.f || chorus->sendGain > 1.f) return FALSE;


    //
    // initialises LFO
    //

    // sets LFO table
    chorus->lfo.table = __AXFXGetLfoSinTable();

    // LFO modulation depth (in samples)
    delay_samp = chorus->delayTime * (f32)samples_msec;
    depth_samp = delay_samp * chorus->depth;
    if(depth_samp >= delay_samp)
    {
        // NOTE : 1 sample of mergin must be needed.
        //        (to omit over-read of delay line)
        depth_samp -= 1.f;
        if(depth_samp < 0.f)
        {
            depth_samp = 0.f;
        }
    }

    chorus->lfo.depthSamp = __F32ToFixedPoint(depth_samp);

    // LFO counter
    chorus->lfo.phaseAdd = __F32ToFixedPoint(
                              chorus->rate * 256.f / (f32)AXFX_SAMPLERATE
                           );

    // samples in 1 counter step of LFO
    step_samp = (f32)AXFX_SAMPLERATE / chorus->rate / 256.f;
    chorus->lfo.stepSamp = __F32ToFixedPoint(step_samp);

    // LFO calcurating factor
    lfo_grad_factor = depth_samp / step_samp;
    chorus->lfo.gradFactor = __F32ToFixedPoint(lfo_grad_factor);

    chorus->lfo.phase     = 0;
    chorus->lfo.sign      = 0;
    chorus->lfo.lastNum   = 0xffffffff;
    chorus->lfo.lastValue = 0;
    chorus->lfo.grad      = 0;


    // initialises histories of channels
    for(ch = 0; ch < 3; ch++)
    {
        for(i = 0; i < 4; i++)
        {
            chorus->history[ch][i] = 0.f;
        }
    }
    chorus->histIndex = 0;


    return TRUE;
}



/*---------------------------------------------------------------------------*
  Name:         __CalcLFO

  Description:  calculates LFO

  Arguments:    buf    buffer for storing LFO values
                          (fixed point : 0xIIII.DDDD)
                lfo    lfo structure

  Returns:      none
 *---------------------------------------------------------------------------*/
static void __CalcLFO(s32* buf, AXFX_CHORUS_EXP_LFO *lfo)
{
    u32 samp;
    u32 num;
    u32 nextNum;
    s32 curValue;
    s32 nextValue;
    s64 level = 0;
    s64 diff;
    u32 aux_blocksize =  AXGetInputSamplesPerFrame();

    for(samp = 0; samp < aux_blocksize; samp++)
    {
        num = lfo->phase & 0xffff0000;

        if(num != lfo->lastNum)
        {
            // linear interpolation

            lfo->lastNum = num;

            num >>= 16;
            nextNum = num + 1;
            nextNum &= 0x7f;  // 0 - 127

            curValue  = lfo->table[num];      // 0xII.DDDDDD
            nextValue = lfo->table[nextNum];

            diff = nextValue - curValue;
            diff *= lfo->gradFactor;
            diff >>= 24;

            lfo->grad = (s32)diff;

            level = (s64)curValue * (s64)lfo->depthSamp;
            level >>= 24;
        }
        else
        {
            level = lfo->lastValue + lfo->grad;
        }
        lfo->lastValue = (s32)level;

        if(lfo->sign >= 1)
        {
            level *= -1;
        }

        lfo->phase += lfo->phaseAdd;   // xxxx.xxxx   phase, phaseAdd
        if(lfo->phase & 0xff800000)
        {
            lfo->phase &= 0x007fffff;
            lfo->sign ^= 1;
        }

        buf[samp] = (s32)level;
    }
}


        } // namespace winext
    } // namespace internal
} // namespace nw
