﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include "AXPrivate.h"
#include "AXRenderCosts.h"
#include "AXHelper.h"
#include <winext/cafe/os/OSInterrupt.h>

#include <math.h>       // sin, cos
#include <string.h>

#if defined(__APPLE__)
#include <TargetConditionals.h>
#endif

namespace nw {
namespace internal {
namespace winext {

namespace {

// DRC の処理を Android は無効化する
#if defined(ANDROID) || TARGET_OS_IPHONE
    const int s_ProcessDrcCount = 0;
#else
    const int s_ProcessDrcCount = AX_MAX_NUM_DRCS;
#endif
}

//#define AX_LOG_OFFSETS
//#define AX_LOG_GET_OFFSETS

#ifndef offsetof
#define offsetof(type, element) ((u32)(&(((type*)0)->element)))
#endif
/*---------------------------------------------------------------------------*
    local vars
 *---------------------------------------------------------------------------*/

// MEMO: WinExt.vcxproj (WinExt.lib) で LITTLE_ENDIAN_PCM16 が定義されています。
//       この WinExt.lib は SoundRuntimeGeneric.vcxproj (SoundRuntimeGeneric.dll)
//       に含まれ SoundMaker のプレビュー再生などで利用されます。
//       SoundMaker.exe では LE の PCM16 が入ってくるようで、エンディアンスワップは不要ですが、
//       seq2wav.dll では BE の PCM16 (～.pcm16.bfwav) が入力されるため、
//       Win32 PF (LE) 上ではエンディアンスワップが必要です。
static bool __AXEnableSwapPcmSample =
#ifdef LITTLE_ENDIAN_PCM16
    false;
#else
    true;
#endif

// data buffers used by the DSP have to be PPC_IO_BUFFER_ALIGN aligned
static AXPB*          __AXPB  = NULL;
ALIGNED_VAR(static AXPB, PPC_IO_BUFFER_ALIGN, __s_AXPB[AX_MAX_VOICES]);
ALIGNED_VAR(static AXPB, PPC_IO_BUFFER_ALIGN, __s_u_AXPB[AX_MAX_VOICES]);

static AXPBITDBUFFER* __AXITD = NULL;
//          static AXPBITDBUFFER      __s_AXITD[AX_MAX_VOICES] ATTRIBUTE_ALIGN(32);
ALIGNED_VAR(static AXPBITDBUFFER, PPC_IO_BUFFER_ALIGN, __s_AXITD[AX_MAX_VOICES]);

// voice or virtual PBs are for the user to read and write
static AXVPB*         __AXVPB = NULL;
static AXVPB          __s_AXVPB[AX_MAX_VOICES];

// For tracking in AXGetVoiceLoopCount()
static u64 __lastOffsets[AX_MAX_VOICES] = { 0 };
static u32 __loopCounts[AX_MAX_VOICES]  = { 0 };

// func of common initializing routine
static void __AXVPBInitCommon(void);

// func of dsp cycles for voice cycle count estimate
static u32 __AXGetSrcCycles(u16 type, AXPBSRC *src);

static void __AXSetVoiceAddr(AXVPB *p, AXPBADDR *addr);

static u32 __AXMaxDspCycles;
static u32 __AXRecDspCosts;
static u32 __AXRecAxCosts;
static u32 __AXNumVoices;
static u32 __AXNumDspVoices;
static u32 __AXDumpedVoices;
       u32 __AXMaxVoices = 0;
static u32 __AXMaxDspVoices = 00;

/*---------------------------------------------------------------------------*
    global vars FOR DEBUG ONLY!!
 *---------------------------------------------------------------------------*/
u32 __axcl_dsp_cycles_fudge_factor = 35;
u32 __axcl_segment_cost = 2;

void AXSetDSPCyclesFudgeFactor(u32 latency)
{
    __axcl_dsp_cycles_fudge_factor = latency;
}
u32 AXGetDSPCyclesFudgeFactor(void)
{
    return __axcl_dsp_cycles_fudge_factor;
}

// set __AXMaxDspVoices to non-zero to use absolute voice cap
// instead of DDR3 latency cycles counting cap

u32 AXGetMaxDspVoices(void)
{
    return __AXMaxDspVoices;
}

void AXSetMaxDspVoices(u32 max_voices)
{
    __AXMaxDspVoices = max_voices;
}

static f32 __AXPpcLoadLimit = 85.0;
static f32 __AXDspLoadLimit = 75.0;
static f32 __AXPpcLoad;
static f32 __AXDspLoad;

void AXSetDspLoadLimit(f32 limit)
{
    __AXDspLoadLimit = limit;
}

f32 AXGetDspLoadLimit(void)
{
    return __AXDspLoadLimit;
}

void AXSetPpcLoadLimit(f32 limit)
{
    __AXPpcLoadLimit = limit;
}

f32 AXGetPpcLoadLimit(void)
{
    return __AXPpcLoadLimit;
}

f32 AXGetPpcLoad(void)
{
    return __AXPpcLoad;
}

f32 AXGetDspLoad(void)
{
    return __AXDspLoad;
}

/*---------------------------------------------------------------------------*
    Return number of voices active.
 *---------------------------------------------------------------------------*/
u32 AXGetNumVoices(void)
{
    return __AXNumVoices;
}
/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
u32 AXGetNumDspVoices(void)
{
    return __AXNumDspVoices;
}
/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
u32 AXGetDroppedVoiceCount(void)
{
    return __AXDumpedVoices;
}

static const s16 __AX4TapsSrcCoef[3][512] =
{
    // fc = 8KHz
    {
        6600,   19426,    6722,       3,
        6479,   19424,    6845,       9,
        6359,   19419,    6968,      15,
        6239,   19412,    7093,      22,
        6121,   19403,    7219,      28,
        6004,   19391,    7345,      34,
        5888,   19377,    7472,      41,
        5773,   19361,    7600,      48,
        5659,   19342,    7728,      55,
        5546,   19321,    7857,      62,
        5434,   19298,    7987,      69,
        5323,   19273,    8118,      77,
        5213,   19245,    8249,      84,
        5104,   19215,    8381,      92,
        4997,   19183,    8513,     101,
        4890,   19148,    8646,     109,
        4785,   19112,    8780,     118,
        4681,   19073,    8914,     127,
        4579,   19031,    9048,     137,
        4477,   18988,    9183,     147,
        4377,   18942,    9318,     157,
        4277,   18895,    9454,     168,
        4179,   18845,    9590,     179,
        4083,   18793,    9726,     190,
        3987,   18738,    9863,     202,
        3893,   18682,   10000,     215,
        3800,   18624,   10137,     228,
        3709,   18563,   10274,     241,
        3618,   18500,   10411,     255,
        3529,   18436,   10549,     270,
        3441,   18369,   10687,     285,
        3355,   18300,   10824,     300,
        3269,   18230,   10962,     317,
        3186,   18157,   11100,     334,
        3103,   18082,   11238,     351,
        3022,   18006,   11375,     369,
        2942,   17927,   11513,     388,
        2863,   17847,   11650,     408,
        2785,   17765,   11788,     428,
        2709,   17681,   11925,     449,
        2635,   17595,   12062,     471,
        2561,   17507,   12198,     494,
        2489,   17418,   12334,     517,
        2418,   17327,   12470,     541,
        2348,   17234,   12606,     566,
        2280,   17140,   12741,     592,
        2213,   17044,   12876,     619,
        2147,   16946,   13010,     647,
        2083,   16846,   13144,     675,
        2020,   16745,   13277,     704,
        1958,   16643,   13409,     735,
        1897,   16539,   13541,     766,
        1838,   16434,   13673,     798,
        1780,   16327,   13803,     832,
        1723,   16218,   13933,     866,
        1667,   16109,   14062,     901,
        1613,   15998,   14191,     937,
        1560,   15885,   14318,     975,
        1508,   15772,   14445,    1013,
        1457,   15657,   14571,    1052,
        1407,   15540,   14695,    1093,
        1359,   15423,   14819,    1134,
        1312,   15304,   14942,    1177,
        1266,   15185,   15064,    1221,
        1221,   15064,   15185,    1266,
        1177,   14942,   15304,    1312,
        1134,   14819,   15423,    1359,
        1093,   14695,   15540,    1407,
        1052,   14571,   15657,    1457,
        1013,   14445,   15772,    1508,
         975,   14318,   15885,    1560,
         937,   14191,   15998,    1613,
         901,   14062,   16109,    1667,
         866,   13933,   16218,    1723,
         832,   13803,   16327,    1780,
         798,   13673,   16434,    1838,
         766,   13541,   16539,    1897,
         735,   13409,   16643,    1958,
         704,   13277,   16745,    2020,
         675,   13144,   16846,    2083,
         647,   13010,   16946,    2147,
         619,   12876,   17044,    2213,
         592,   12741,   17140,    2280,
         566,   12606,   17234,    2348,
         541,   12470,   17327,    2418,
         517,   12334,   17418,    2489,
         494,   12198,   17507,    2561,
         471,   12062,   17595,    2635,
         449,   11925,   17681,    2709,
         428,   11788,   17765,    2785,
         408,   11650,   17847,    2863,
         388,   11513,   17927,    2942,
         369,   11375,   18006,    3022,
         351,   11238,   18082,    3103,
         334,   11100,   18157,    3186,
         317,   10962,   18230,    3269,
         300,   10824,   18300,    3355,
         285,   10687,   18369,    3441,
         270,   10549,   18436,    3529,
         255,   10411,   18500,    3618,
         241,   10274,   18563,    3709,
         228,   10137,   18624,    3800,
         215,   10000,   18682,    3893,
         202,    9863,   18738,    3987,
         190,    9726,   18793,    4083,
         179,    9590,   18845,    4179,
         168,    9454,   18895,    4277,
         157,    9318,   18942,    4377,
         147,    9183,   18988,    4477,
         137,    9048,   19031,    4579,
         127,    8914,   19073,    4681,
         118,    8780,   19112,    4785,
         109,    8646,   19148,    4890,
         101,    8513,   19183,    4997,
          92,    8381,   19215,    5104,
          84,    8249,   19245,    5213,
          77,    8118,   19273,    5323,
          69,    7987,   19298,    5434,
          62,    7857,   19321,    5546,
          55,    7728,   19342,    5659,
          48,    7600,   19361,    5773,
          41,    7472,   19377,    5888,
          34,    7345,   19391,    6004,
          28,    7219,   19403,    6121,
          22,    7093,   19412,    6239,
          15,    6968,   19419,    6359,
           9,    6845,   19424,    6479,
           3,    6722,   19426,    6600
    },

    // fc = 12KHz
    {
        3195,   26287,    3329,     -32,
        3064,   26281,    3467,     -34,
        2936,   26270,    3608,     -38,
        2811,   26253,    3751,     -42,
        2688,   26230,    3897,     -46,
        2568,   26202,    4046,     -50,
        2451,   26169,    4199,     -54,
        2338,   26130,    4354,     -58,
        2227,   26085,    4512,     -63,
        2120,   26035,    4673,     -67,
        2015,   25980,    4837,     -72,
        1912,   25919,    5004,     -76,
        1813,   25852,    5174,     -81,
        1716,   25780,    5347,     -87,
        1622,   25704,    5522,     -92,
        1531,   25621,    5701,     -98,
        1442,   25533,    5882,    -103,
        1357,   25440,    6066,    -109,
        1274,   25342,    6253,    -115,
        1193,   25239,    6442,    -121,
        1115,   25131,    6635,    -127,
        1040,   25018,    6830,    -133,
         967,   24899,    7027,    -140,
         897,   24776,    7227,    -146,
         829,   24648,    7430,    -153,
         764,   24516,    7635,    -159,
         701,   24379,    7842,    -166,
         641,   24237,    8052,    -174,
         583,   24091,    8264,    -181,
         526,   23940,    8478,    -187,
         472,   23785,    8695,    -194,
         420,   23626,    8914,    -202,
         371,   23462,    9135,    -209,
         324,   23295,    9358,    -215,
         279,   23123,    9583,    -222,
         236,   22948,    9809,    -230,
         194,   22769,   10038,    -237,
         154,   22586,   10269,    -243,
         117,   22399,   10501,    -250,
          81,   22208,   10735,    -258,
          47,   22015,   10970,    -265,
          15,   21818,   11206,    -271,
         -16,   21618,   11444,    -277,
         -44,   21415,   11684,    -283,
         -71,   21208,   11924,    -290,
         -97,   20999,   12166,    -296,
        -121,   20786,   12409,    -302,
        -143,   20571,   12653,    -306,
        -163,   20354,   12898,    -311,
        -183,   20134,   13143,    -316,
        -201,   19911,   13389,    -321,
        -218,   19686,   13635,    -325,
        -234,   19459,   13882,    -328,
        -248,   19230,   14130,    -332,
        -261,   18998,   14377,    -335,
        -273,   18765,   14625,    -337,
        -284,   18531,   14873,    -339,
        -294,   18295,   15121,    -341,
        -302,   18057,   15369,    -341,
        -310,   17817,   15617,    -341,
        -317,   17577,   15864,    -340,
        -323,   17335,   16111,    -340,
        -328,   17092,   16357,    -338,
        -332,   16848,   16603,    -336,
        -336,   16603,   16848,    -332,
        -338,   16357,   17092,    -328,
        -340,   16111,   17335,    -323,
        -340,   15864,   17577,    -317,
        -341,   15617,   17817,    -310,
        -341,   15369,   18057,    -302,
        -341,   15121,   18295,    -294,
        -339,   14873,   18531,    -284,
        -337,   14625,   18765,    -273,
        -335,   14377,   18998,    -261,
        -332,   14130,   19230,    -248,
        -328,   13882,   19459,    -234,
        -325,   13635,   19686,    -218,
        -321,   13389,   19911,    -201,
        -316,   13143,   20134,    -183,
        -311,   12898,   20354,    -163,
        -306,   12653,   20571,    -143,
        -302,   12409,   20786,    -121,
        -296,   12166,   20999,     -97,
        -290,   11924,   21208,     -71,
        -283,   11684,   21415,     -44,
        -277,   11444,   21618,     -16,
        -271,   11206,   21818,      15,
        -265,   10970,   22015,      47,
        -258,   10735,   22208,      81,
        -250,   10501,   22399,     117,
        -243,   10269,   22586,     154,
        -237,   10038,   22769,     194,
        -230,    9809,   22948,     236,
        -222,    9583,   23123,     279,
        -215,    9358,   23295,     324,
        -209,    9135,   23462,     371,
        -202,    8914,   23626,     420,
        -194,    8695,   23785,     472,
        -187,    8478,   23940,     526,
        -181,    8264,   24091,     583,
        -174,    8052,   24237,     641,
        -166,    7842,   24379,     701,
        -159,    7635,   24516,     764,
        -153,    7430,   24648,     829,
        -146,    7227,   24776,     897,
        -140,    7027,   24899,     967,
        -133,    6830,   25018,    1040,
        -127,    6635,   25131,    1115,
        -121,    6442,   25239,    1193,
        -115,    6253,   25342,    1274,
        -109,    6066,   25440,    1357,
        -103,    5882,   25533,    1442,
         -98,    5701,   25621,    1531,
         -92,    5522,   25704,    1622,
         -87,    5347,   25780,    1716,
         -81,    5174,   25852,    1813,
         -76,    5004,   25919,    1912,
         -72,    4837,   25980,    2015,
         -67,    4673,   26035,    2120,
         -63,    4512,   26085,    2227,
         -58,    4354,   26130,    2338,
         -54,    4199,   26169,    2451,
         -50,    4046,   26202,    2568,
         -46,    3897,   26230,    2688,
         -42,    3751,   26253,    2811,
         -38,    3608,   26270,    2936,
         -34,    3467,   26281,    3064,
         -32,    3329,   26287,    3195
    },

    // fc = 16KHz
    {
          -68,   32639,      69,      -5,
         -200,   32630,     212,     -15,
         -328,   32613,     359,     -26,
         -450,   32586,     512,     -36,
         -568,   32551,     669,     -47,
         -680,   32507,     832,     -58,
         -788,   32454,    1000,     -69,
         -891,   32393,    1174,     -80,
         -990,   32323,    1352,     -92,
        -1084,   32244,    1536,    -103,
        -1173,   32157,    1724,    -115,
        -1258,   32061,    1919,    -128,
        -1338,   31956,    2118,    -140,
        -1414,   31844,    2322,    -153,
        -1486,   31723,    2532,    -167,
        -1554,   31593,    2747,    -180,
        -1617,   31456,    2967,    -194,
        -1676,   31310,    3192,    -209,
        -1732,   31157,    3422,    -224,
        -1783,   30995,    3657,    -240,
        -1830,   30826,    3897,    -256,
        -1874,   30649,    4143,    -272,
        -1914,   30464,    4393,    -289,
        -1951,   30272,    4648,    -307,
        -1984,   30072,    4908,    -325,
        -2014,   29866,    5172,    -343,
        -2040,   29652,    5442,    -362,
        -2063,   29431,    5716,    -382,
        -2083,   29203,    5994,    -403,
        -2100,   28968,    6277,    -424,
        -2114,   28727,    6565,    -445,
        -2125,   28480,    6857,    -468,
        -2133,   28226,    7153,    -490,
        -2139,   27966,    7453,    -514,
        -2142,   27700,    7758,    -538,
        -2142,   27428,    8066,    -563,
        -2141,   27151,    8378,    -588,
        -2136,   26867,    8694,    -614,
        -2130,   26579,    9013,    -641,
        -2121,   26285,    9336,    -668,
        -2111,   25987,    9663,    -696,
        -2098,   25683,    9993,    -724,
        -2084,   25375,   10326,    -753,
        -2067,   25063,   10662,    -783,
        -2049,   24746,   11000,    -813,
        -2030,   24425,   11342,    -844,
        -2009,   24100,   11686,    -875,
        -1986,   23771,   12033,    -907,
        -1962,   23438,   12382,    -939,
        -1937,   23103,   12733,    -972,
        -1911,   22764,   13086,   -1005,
        -1883,   22422,   13441,   -1039,
        -1855,   22077,   13798,   -1072,
        -1825,   21729,   14156,   -1107,
        -1795,   21380,   14516,   -1141,
        -1764,   21027,   14877,   -1176,
        -1732,   20673,   15239,   -1211,
        -1700,   20317,   15602,   -1246,
        -1667,   19959,   15965,   -1282,
        -1633,   19600,   16329,   -1317,
        -1599,   19239,   16694,   -1353,
        -1564,   18878,   17058,   -1388,
        -1530,   18515,   17423,   -1424,
        -1495,   18151,   17787,   -1459,
        -1459,   17787,   18151,   -1495,
        -1424,   17423,   18515,   -1530,
        -1388,   17058,   18878,   -1564,
        -1353,   16694,   19239,   -1599,
        -1317,   16329,   19600,   -1633,
        -1282,   15965,   19959,   -1667,
        -1246,   15602,   20317,   -1700,
        -1211,   15239,   20673,   -1732,
        -1176,   14877,   21027,   -1764,
        -1141,   14516,   21380,   -1795,
        -1107,   14156,   21729,   -1825,
        -1072,   13798,   22077,   -1855,
        -1039,   13441,   22422,   -1883,
        -1005,   13086,   22764,   -1911,
         -972,   12733,   23103,   -1937,
         -939,   12382,   23438,   -1962,
         -907,   12033,   23771,   -1986,
         -875,   11686,   24100,   -2009,
         -844,   11342,   24425,   -2030,
         -813,   11000,   24746,   -2049,
         -783,   10662,   25063,   -2067,
         -753,   10326,   25375,   -2084,
         -724,    9993,   25683,   -2098,
         -696,    9663,   25987,   -2111,
         -668,    9336,   26285,   -2121,
         -641,    9013,   26579,   -2130,
         -614,    8694,   26867,   -2136,
         -588,    8378,   27151,   -2141,
         -563,    8066,   27428,   -2142,
         -538,    7758,   27700,   -2142,
         -514,    7453,   27966,   -2139,
         -490,    7153,   28226,   -2133,
         -468,    6857,   28480,   -2125,
         -445,    6565,   28727,   -2114,
         -424,    6277,   28968,   -2100,
         -403,    5994,   29203,   -2083,
         -382,    5716,   29431,   -2063,
         -362,    5442,   29652,   -2040,
         -343,    5172,   29866,   -2014,
         -325,    4908,   30072,   -1984,
         -307,    4648,   30272,   -1951,
         -289,    4393,   30464,   -1914,
         -272,    4143,   30649,   -1874,
         -256,    3897,   30826,   -1830,
         -240,    3657,   30995,   -1783,
         -224,    3422,   31157,   -1732,
         -209,    3192,   31310,   -1676,
         -194,    2967,   31456,   -1617,
         -180,    2747,   31593,   -1554,
         -167,    2532,   31723,   -1486,
         -153,    2322,   31844,   -1414,
         -140,    2118,   31956,   -1338,
         -128,    1919,   32061,   -1258,
         -115,    1724,   32157,   -1173,
         -103,    1536,   32244,   -1084,
          -92,    1352,   32323,    -990,
          -80,    1174,   32393,    -891,
          -69,    1000,   32454,    -788,
          -58,     832,   32507,    -680,
          -47,     669,   32551,    -568,
          -36,     512,   32586,    -450,
          -26,     359,   32613,    -328,
          -15,     212,   32630,    -200,
           -5,      69,   32639,     -68
}};

enum AddrMode {
    ADDR_MODE_CURRENT,
    ADDR_MODE_LOOP_START,
    ADDR_MODE_WAVE_END
};

u64 GetU64Address( const AXPB* vpb, AddrMode mode )
{
    u32 hi, lo;
    const AXPBADDR& addr = vpb->addr;

    switch ( mode )
    {
        case ADDR_MODE_CURRENT:
            hi = addr.currentAddressHi;
            lo = addr.currentAddressLo;
            break;
        case ADDR_MODE_LOOP_START:
            hi = addr.loopAddressHi;
            lo = addr.loopAddressLo;
            break;
        case ADDR_MODE_WAVE_END:
            hi = addr.endAddressHi;
            lo = addr.endAddressLo;
            break;
    }
    u64 ret = hi;
    ret <<= 32;
    ret |= lo;
    return ret;
}

inline void SetCurrentAddress( AXPB* vpb, u64 addr )
{
    vpb->addr.currentAddressHi = ( addr >> 32 ) & 0xffffffff;
    vpb->addr.currentAddressLo = ( addr >>  0 ) & 0xffffffff;
}

u32 GetDspAddressBySampleForDspAdpcm( const void* baseAddress, u32 samples )
{
#ifdef NW_PLATFORM_RVL
    // CTRWIN ではおそらく不要
    if ( baseAddress != NULL )
    {
        baseAddress = reinterpret_cast<const void*>(
                OS_CachedToPhysical( baseAddress) );
    }
#endif /* NW_PLATFORM_RVL */
    u32 addr =
        ( reinterpret_cast<u32>( baseAddress ) << 1 ) +
        ( samples / 14 ) * 16 + ( samples % 14 ) + 2
        ;
    return addr;
}

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
s16 AXDecodeDspAdpcm( AXPBADPCM* adpcm, u8 code )
{
    s32 pred  = (s32)(adpcm->pred_scale >> 4);
    s32 scale = (s32)(adpcm->pred_scale & 0xf);

    s16 nibble = code;
    nibble <<= 12;
    nibble >>= 1;

    s16 a1   = (s16)adpcm->a[pred][0];
    s16 a2   = (s16)adpcm->a[pred][1];
    s16 gain = (s16)(1 << scale);

#ifdef DEBUG_DECODE_PRINT
    s16 yn1 = adpcm->yn1;
    s16 yn2 = adpcm->yn2;
#endif /* DEBUG_DECODE_PRINT */

    s32 val = a1 * (s16)adpcm->yn1;
    val  += a2 * (s16)adpcm->yn2;
    val  += gain * nibble;
    val >>= 10;
    val  += 1;
    val >>= 1;

    if (val > 32767) val = 32767;
    else if (val < -32768) val = -32768;

    adpcm->yn2 = adpcm->yn1;
    adpcm->yn1 = (u16)val;

#ifdef DEBUG_DECODE_PRINT
    if ( s_DebugFlag )
    {
        OSReport("[DSP ADPCM デコード] pred(%d) yn1(%d=>%d) yn2(%d=>%d)\n",
                adpcm->predScale, yn1, adpcm->yn1, yn2, adpcm->yn2 );
    }
#endif /* DEBUG_DECODE_PRINT */

    return (s16)val;
}

// 単極 LPF フィルタ
//  * Arguments   : s16* input         フィルタ入出力
//  *               u32  frameCount    サンプルの長さ
//  *               s16* hist          履歴データ。アプリケーションが 0 で初期化。
//  *               u16  a0            フィルタ係数
//  *               u16  b0            フィルタ係数
void OnePoleFilter( s16* input, u32 frameCount, s16* hist, u16 a0, u16 b0 )
{
    s16  xn;
    s16  yn1;
    s64  stmp64;

    yn1 = *hist;

    for (u32 ii = 0; ii < frameCount; ii++)
    {
        xn = *input;

        stmp64   = (xn * a0 + yn1 * b0);
        stmp64 >>= 15;
        if (stmp64 > 32767)
        {
            stmp64 = 32767;
        }
        else if (stmp64 < -32768)
        {
            stmp64 = -32768;
        }

        yn1 = (s16)stmp64;

        *input++ = yn1;
    }

    *hist = yn1;
}

// Description : Biquad フィルタ
//
// Arguments   : s16*   input       フィルタ入出力
//               u32    frameCount  サンプルの長さ
//               BHIST* hist        履歴データ。アプリケーションが 0 で初期化。
//               s16    b0          フィルタ係数
//               s16    b1          フィルタ係数
//               s16    b2          フィルタ係数
//               s16    a0          フィルタ係数
//               s16    a1          フィルタ係数
struct BHIST
{
    s16 xn1;
    s16 xn2;
    s16 yn1;
    s16 yn2;
};
void BiquadFilter(
    s16* input,
    u32 frameCount,
    BHIST* hist,
    s16 b0, s16 b1, s16 b2, s16 a1, s16 a2 )
{
    s16  xn,  xn1, xn2;
    s16  yn1, yn2;
    s64  stmp64;
    u16  utmp16;

    xn1 = hist->xn1;
    xn2 = hist->xn2;
    yn1 = hist->yn1;
    yn2 = hist->yn2;

    for (u32 ii = 0; ii < frameCount; ii++)
    {
        xn = *input;

        stmp64   = xn * b0 + xn1 * b1 + xn2 * b2 + yn1 * a1 + yn2 * a2;
        stmp64 <<= 2;
        utmp16   = (u16)(stmp64 & 0xffff);
        stmp64 >>= 16;
        if ((utmp16 > 0x8000) || ((utmp16 == 0x8000) && (stmp64 & 0x1)))
        {
            stmp64++;
        }
        if (stmp64 > 32767)
        {
            stmp64 = 32767;
        }
        else if (stmp64 < -32768)
        {
            stmp64 = -32768;
        }

        xn2 = xn1;
        xn1 = xn;
        yn2 = yn1;
        yn1 = (s16)stmp64;

        *input++ = yn1;
    }

    hist->xn1 = xn1;
    hist->xn2 = xn2;
    hist->yn1 = yn1;
    hist->yn2 = yn2;
}

void AXSetEnableSwapPcmSample(bool enable)
{
    __AXEnableSwapPcmSample = enable;
}
bool AXGetEnableSwapPcmSample()
{
    return __AXEnableSwapPcmSample;
}
/*---------------------------------------------------------------------------*
    Service a PB, this is called by the __AXSyncPBs() function below for each
    PB that is running. This function synchronizes the user PB parameters
    with the DSP PB parameters.
 *---------------------------------------------------------------------------*/
s16 __AXDecode1Sample_PCM16(AXPB* pupb)
{
    // カレントサンプルを取得
    const s16* currentAddress = reinterpret_cast<const s16*>(
            GetU64Address( pupb, ADDR_MODE_CURRENT ) << 1 );
    u16 sample = *currentAddress;

    if (__AXEnableSwapPcmSample)
    {
        sample = (sample<<8) | (sample>>8);
        // s16 値をバイトスワップすると符号？がおかしくなるので、
        // u16 状態でバイトスワップする
    }

    // サンプルデータの末尾と currentAddress を比較、
    // 末尾をはみ出していたら、currentAddress とステートを調整
    const s16* waveEndAddress = reinterpret_cast<const s16*>(
        GetU64Address( pupb, ADDR_MODE_WAVE_END ) << 1 );
    if ( currentAddress == waveEndAddress )
    {
        const s16* loopStartAddress = reinterpret_cast<const s16*>(
            GetU64Address( pupb, ADDR_MODE_LOOP_START ) << 1 );
        currentAddress = loopStartAddress;
        if ( pupb->addr.loopFlag == 0 )
        {
            pupb->state = AX_PB_STATE_STOP;
        }
    }
    else
    {
        // アドレスを次のサンプルに進める
        currentAddress++;
    }

    // カレントアドレスを移動させる
    u32 addr = reinterpret_cast<u32>( currentAddress ) >> 1;
    SetCurrentAddress( pupb, addr );

    return static_cast<s16>(sample);
}

s16 __AXDecode1Sample_PCM8(AXPB* pupb)
{
    // カレントサンプルを取得
    const s8* currentAddress = reinterpret_cast<const s8*>(
        GetU64Address( pupb, ADDR_MODE_CURRENT ) );
    s16 sample = *currentAddress << 8;

    const s8* waveEndAddress = reinterpret_cast<const s8*>(
        GetU64Address( pupb, ADDR_MODE_WAVE_END ) );
    if ( currentAddress == waveEndAddress )
    {
        const s8* loopStartAddress = reinterpret_cast<const s8*>(
            GetU64Address( pupb, ADDR_MODE_LOOP_START ) );
        currentAddress = loopStartAddress;
        if ( pupb->addr.loopFlag == 0 )
        {
            pupb->state = AX_PB_STATE_STOP;
        }
    }
    else
    {
        // アドレスを次のサンプルに進める
        currentAddress++;
    }

    // カレントアドレスを移動させる
    u32 addr = reinterpret_cast<u32>( currentAddress );
    SetCurrentAddress( pupb, addr );

    // デバッグ用にサンプルをファイル書き込み
    // WriteDump( sample );

    return sample;
}

s16 __AXDecode1Sample_DSP_ADPCM(AXPB* pupb)
{
    // ADPCM パラメータを取得
    AXPBADPCM* adpcm = &pupb->adpcm;

    // カレントアドレス
    u64 currentAddress = GetU64Address( pupb, ADDR_MODE_CURRENT );

    if ( ! (currentAddress & 0xf) )
    {
        // 8 バイトごとに pred/scale を更新する
        u8 byte = *reinterpret_cast<u8*>( currentAddress >> 1 );
        adpcm->pred_scale = (u16)byte;
        currentAddress += 2;
    }

    // ニブル取得
    u8 byte = *reinterpret_cast<u8*>( currentAddress >> 1 );
    u8 nibble;
    if ( currentAddress & 0x1 )
    {
        nibble = static_cast<u8>( byte & 0xf ); // 下位 4 ビット
    }
    else
    {
        nibble = static_cast<u8>( byte >> 4 );  // 上位 4 ビット
    }

#ifdef DEBUG_DECODE_PRINT
    // デバッグ
    if ( currentAddress == GetU32Address( pupb, ADDR_MODE_WAVE_END ) )
    {
        s_DebugFlag = true;
    }
#endif /* DEBUG_DECODE_PRINT */

    // ADPCM デコード
    s16 sample = AXDecodeDspAdpcm( adpcm, nibble );

    // サンプルデータの末尾と currentAddress を比較、
    // 末尾をはみ出していたら、currentAddress とステートを調整
    u64 waveEndAddress = GetU64Address( pupb, ADDR_MODE_WAVE_END );
    if ( currentAddress == waveEndAddress )
    {
        if ( pupb->addr.loopFlag == 0 )
        {
            pupb->state = AX_PB_STATE_STOP;
        }

        u64 loopStartAddress = GetU64Address( pupb, ADDR_MODE_LOOP_START );
        currentAddress = loopStartAddress;

        const AXPBADPCMLOOP* adpcmLoop = &pupb->adpcmLoop;
        adpcm->pred_scale = adpcmLoop->loop_pred_scale;
        if ( pupb->type == AX_PB_TYPE_NORMAL )
        {
            adpcm->yn1 = adpcmLoop->loop_yn1;
            adpcm->yn2 = adpcmLoop->loop_yn2;
        }
#ifdef DEBUG_DECODE_PRINT
        // デバッグ
        s_DebugFlag = false;
#endif /* DEBUG_DECODE_PRINT */
    }
    else
    {
        // アドレスを次のサンプルに進める
        currentAddress++;
    }

    // カレントアドレスを移動させる
    SetCurrentAddress( pupb, currentAddress );

    return sample;
}

s16 __AXDecode1Sample(AXPB* pupb)
{
    if ( pupb->state == AX_PB_STATE_STOP ) return 0;

    switch ( pupb->addr.format )
    {
        case AX_PB_FORMAT_PCM16:
            return __AXDecode1Sample_PCM16( pupb );
            break;
        case AX_PB_FORMAT_PCM8:
            return __AXDecode1Sample_PCM8( pupb );
            break;
        case AX_PB_FORMAT_ADPCM:
            return __AXDecode1Sample_DSP_ADPCM( pupb );
            break;
        // case AX_PB_FORMAT_IMA_ADPCM:
        //     return __AXDecode1Sample_IMA_ADPCM( pupb );
        //     break;
        default:
            //ASSERTMSG( 0, "no support PCM8, ADPCM, etc...\n");
            return 0;
            break;
    }
}

#if defined(ANDROID) || TARGET_OS_IPHONE
// #define PROFILE_ENABLED

#ifdef PROFILE_ENABLED
#include <android/log.h>
#include <time.h>

namespace {
double g_StartTime = 0;
const s32 g_HowManyCount = 10000;

double currentTime(void)
{
    struct timespec ts;
    clock_gettime(CLOCK_THREAD_CPUTIME_ID, &ts);
    return 1000000.0 * ts.tv_sec + (double) ts.tv_nsec / 1e3; // [us]
}

void checkStart()
{
    g_StartTime = currentTime();
}

void checkEnd()
{
    static double cnt = 0;
    static double sum = 0;
    static const u32 count = g_HowManyCount;
    static double overall = 0;

    double tmp = (currentTime() - g_StartTime);
    sum += tmp;

    static double max = 0;

    if ( max < tmp)
    {
        max = tmp;
    }

    if ( cnt++ > count )
    {
        if (overall)
        {
            overall = (overall + sum/count) /2;
        }
        else
        {
            overall = sum/count;
        }
#if defined(ANDROID)
        __android_log_print(ANDROID_LOG_INFO, "[siglo]", "max: %.2f[us] all %.2f[us] overall %2.f[us]\n", max, sum/count, overall);
#elif TARGET_OS_IPHONE
        printf("max: %.2f[us] all %.2f[us] overall %2.f[us]\n", max, sum/count, overall);
#endif
        cnt = 0;
        sum = 0;
        max = 0;
    }
}

void checkAt(int n)
{
    // 5箇所まで対応
    static double cnt[5] = {0};
    static double sum[5] = {0};
    static const u32 count = g_HowManyCount;

    sum[n] += (currentTime() - g_StartTime);
    if ( cnt[n]++ > count )
    {
#if defined(ANDROID)
        __android_log_print(ANDROID_LOG_INFO, "[siglo]", "pos:%d mean %.2f[us]\n", n, sum[n]/count);
#elif TARGET_OS_IPHONE
        printf("pos:%d mean %.2f[us]\n", n, sum[n]/count);
#endif
        cnt[n] = 0;
        sum[n] = 0;
    }
}

} // namespace
#endif
#endif


void SynthVPB(
    AXVPB *pvpb,
    int /* channels */,
    BusBuffer* busBuffer,
    unsigned long samples,
    int sampleRate
)
{
    //    AXPB* pupb    = &__s_u_AXPB[pvpb->index];
    AXPB* pupb    = &__AXPB[pvpb->index];
    const u32 sync = pvpb->sync;
    s16 waveBuffer[AX_IN_SAMPLES_PER_SEC];

#ifdef PROFILE_ENABLED
    checkStart();
#endif

    if ( pupb->state == AX_PB_STATE_RUN )
    {
        AXPBSRC* src = &pupb->src;

        const s16* coef = __AX4TapsSrcCoef[pupb->coefSelect];
        u32 acc = src->currentAddressFrac;
        u32 pitch = (src->ratioHi << 16 | src->ratioLo);

        s16 history[4];
        for (int i = 0; i < 4; i++)
        {
            history[i] = (s16)src->last_samples[i];
        }

        if ( pupb->srcSelect == AX_PB_SRCSEL_POLYPHASE)
        {
            for(unsigned long ii = 0; ii < samples; ii++)
            {
                acc += pitch;
                u32 news = acc >> 16;
                acc &= 0xffff;

                for (u32 jj = 0; jj < news; jj++)
                {
                    s16 decoded = __AXDecode1Sample(pupb);

                    history[3] = history[2];
                    history[2] = history[1];
                    history[1] = history[0];
                    history[0] = decoded;
                }

                int smp;
                u32 phase = acc;
                phase >>= 9;
                phase *= 4;

                smp  = coef[phase + 0] * history[3];
                smp += coef[phase + 1] * history[2];
                smp += coef[phase + 2] * history[1];
                smp += coef[phase + 3] * history[0];

                smp >>= 15;

                if ( smp > 0x7fff ) smp = 0x7fff;
                else if ( smp < -0x8000 ) smp = -0x8000;

                waveBuffer[ii] = smp;
            }
        }
        else if ( pupb->srcSelect == AX_PB_SRCSEL_LINEAR )
        {
            for(unsigned long ii = 0; ii < samples; ii++)
            {
                acc += pitch;
                u32 news = acc >> 16;
                acc &= 0xffff;

                for (u32 jj = 0; jj < news; jj++)
                {
                    s16 decoded = __AXDecode1Sample(pupb);

                    history[3] = history[2];
                    history[2] = history[1];
                    history[1] = history[0];
                    history[0] = decoded;
                }

                int smp;
                smp  = (int)(65536 - acc) * history[3];
                smp += acc * history[2];
                smp >>= 16;

                if ( smp > 0x7fff ) smp = 0x7fff;
                else if ( smp < -0x8000 ) smp = -0x8000;

                waveBuffer[ii] = smp;
            }
        }
        else // if ( pupb->srcSelect == AX_PB_SRCSEL_NONE)
        {
            for(unsigned long ii = 0; ii < samples; ii++)
            {
                acc += pitch;
                u32 news = acc >> 16;
                acc &= 0xffff;

                news = 1;
                acc = 0;

                for (u32 jj = 0; jj < news; jj++)
                {
                    s16 decoded = __AXDecode1Sample(pupb);

                    history[3] = history[2];
                    history[2] = history[1];
                    history[1] = history[0];
                    history[0] = decoded;
                }

                int smp;
                smp = history[0];

                if ( smp > 0x7fff ) smp = 0x7fff;
                else if ( smp < -0x8000 ) smp = -0x8000;

                waveBuffer[ii] = smp;
            }
        }

        AXPBLPF* lpf = &pupb->lpf;
        if ( lpf->on )
        {
            OnePoleFilter( waveBuffer, samples, (s16*)&(lpf->yn1), lpf->a0, lpf->b0 );
        }

        AXPBBIQUAD* biquad = &pupb->biquad;
        if ( biquad->on )
        {
            BHIST hist = {
                biquad->xn1,
                biquad->xn2,
                biquad->yn1,
                biquad->yn2 };
                BiquadFilter( waveBuffer, samples, &hist,
                    biquad->b0,
                    biquad->b1,
                    biquad->b2,
                    biquad->a1,
                    biquad->a2 );

                // ヒストリーデータ更新
                biquad->xn1 = hist.xn1;
                biquad->xn2 = hist.xn2;
                biquad->yn1 = hist.yn1;
                biquad->yn2 = hist.yn2;
        }

#if defined(ANDROID) || TARGET_OS_IPHONE
        signed int** dest = busBuffer->buffers;
        u16 startVol = pupb->ve.currentVolume;
        u16 vol = startVol;

#ifdef PROFILE_ENABLED
    checkAt(0);
#endif
        // TV
        {
            // TV
            for (u32 ch = 0; ch < AX_MAX_NUM_TV_CHS; ch++)
            {
                auto mixCh = pupb->tvMix[AX_TV_ID0][ch];

                for (u32 busId = 0; busId < AX_MAX_NUM_BUSES; busId++)
                {
                    vol = startVol;
                    // Register サイズに合わせて 32bit 幅に伸張
                    auto curVol   = static_cast<u32>(mixCh.bus[busId].vol);
                    auto busDelta = static_cast<s32>(mixCh.bus[busId].volDelta);
                    auto veDelta  = static_cast<s32>(pupb->ve.currentDelta);

                    auto globalBusId = ch + AX_MAX_NUM_TV_CHS * busId;

                    auto* d = dest[globalBusId];

                    for(u32 ii = 0; ii < samples; ii++)
                    {
                        auto smp = static_cast<s32>(waveBuffer[ii]);  // waveBuffer は上で clamp されている
                        smp *= vol;
                        smp >>= 15;
                        smp *= curVol;
                        smp >>= 15;
                        *d += (smp << 8) ;
                        d++;
                        vol += veDelta;
                        curVol += busDelta;
                    }
                    mixCh.bus[busId].vol = curVol;
                }
            }
        }

        // DRC
        vol = startVol;
        for(unsigned long ii = 0; ii < samples; ii++)
        {
            float smp = static_cast<float>(waveBuffer[ii] * vol);

            for (u32 id = 0 ; id < s_ProcessDrcCount; id++)
            {
                u32 ofs = id * MIX_BUFFER_COUNT_DRC + MIX_BUFFER_COUNT_TV * AX_MAX_NUM_TVS;

                for (u32 ch = 0; ch < AX_MAX_NUM_DRC_CHS; ch++)
                {
                    for (u32 busId = 0; busId < AX_MAX_NUM_BUSES; busId++)
                    {
                        s32 s = (int)(smp * pupb->drcMix[id][ch].bus[busId].vol / (32768 * 32768));
                        pupb->drcMix[id][ch].bus[busId].vol += pupb->drcMix[id][ch].bus[busId].volDelta;
                        u32 globalBusId = ofs + ch + AX_MAX_NUM_DRC_CHS * busId;
                        dest[globalBusId][ii] += (s << 8);
                    }
                }
            }

#if 0 // TODO
            // RMT
            for (u32 id = 0 ; id < AX_MAX_NUM_RMTS; id++)
            {
                int smps[AX_MAX_NUM_RMT_CHS] = {0};
                u32 ofs = id * MIX_BUFFER_COUNT_RMT + MIX_BUFFER_COUNT_TV * AX_MAX_NUM_TVS + MIX_BUFFER_COUNT_DRC * AX_MAX_NUM_DRCS;
                for (u32 ch = 0; ch < AX_MAX_NUM_RMT_CHS; ch++)
                {
                    smps[ch] = (int)(smp * pupb->rmtMix[id][ch].bus[AX_MAIN_BUS].vol / (32768 * 32768));
                    pupb->rmtMix[id][ch].bus[AX_MAIN_BUS].vol += pupb->rmtMix[id][ch].bus[AX_MAIN_BUS].volDelta;
                    dest[ofs + ch][ii] += (smps[ch] << 8);
                }
            }
#endif
            vol += pupb->ve.currentDelta;
        }

#ifdef PROFILE_ENABLED
    checkAt(1);
#endif

        pupb->ve.currentVolume += pupb->ve.currentDelta * samples;
#else // defined(ANDROID) || TARGET_OS_IPHONE
        signed int** dest = busBuffer->buffers;

        for(unsigned long ii = 0; ii < samples; ii++)
        {
            float r = ii / (float)samples;
            float smp = static_cast<float>(waveBuffer[ii] * pupb->ve.currentVolume);

            // TV
            {
                for (u32 busId = 0; busId < AX_MAX_NUM_BUSES; busId++)
                {
                    for (u32 ch = 0; ch < AX_MAX_NUM_TV_CHS; ch++)
                    {
                        s32 s = (int)(smp * pupb->tvMix[AX_TV_ID0][ch].bus[busId].vol / (32768 * 32768));
                        pupb->tvMix[AX_TV_ID0][ch].bus[busId].vol += pupb->tvMix[AX_TV_ID0][ch].bus[busId].volDelta;

                        u32 globalBusId = ch + AX_MAX_NUM_TV_CHS * busId;
                        dest[globalBusId][ii] += (s << 8);
                    }
                }
            }

            // DRC
            for (u32 id = 0 ; id < s_ProcessDrcCount; id++)
            {
                u32 ofs = id * MIX_BUFFER_COUNT_DRC + MIX_BUFFER_COUNT_TV * AX_MAX_NUM_TVS;

                for (u32 busId = 0; busId < AX_MAX_NUM_BUSES; busId++)
                {
                    for (u32 ch = 0; ch < AX_MAX_NUM_DRC_CHS; ch++)
                    {
                        s32 s = (int)(smp * pupb->drcMix[id][ch].bus[busId].vol / (32768 * 32768));
                        pupb->drcMix[id][ch].bus[busId].vol += pupb->drcMix[id][ch].bus[busId].volDelta;
                        u32 globalBusId = ofs + ch + AX_MAX_NUM_DRC_CHS * busId;
                        dest[globalBusId][ii] += (s << 8);
                    }
                }
            }

#if 0 // TODO
            // RMT
            for (u32 id = 0 ; id < AX_MAX_NUM_RMTS; id++)
            {
                int smps[AX_MAX_NUM_RMT_CHS] = {0};
                u32 ofs = id * MIX_BUFFER_COUNT_RMT + MIX_BUFFER_COUNT_TV * AX_MAX_NUM_TVS + MIX_BUFFER_COUNT_DRC * AX_MAX_NUM_DRCS;
                for (u32 ch = 0; ch < AX_MAX_NUM_RMT_CHS; ch++)
                {
                    smps[ch] = (int)(smp * pupb->rmtMix[id][ch].bus[AX_MAIN_BUS].vol / (32768 * 32768));
                    pupb->rmtMix[id][ch].bus[AX_MAIN_BUS].vol += pupb->rmtMix[id][ch].bus[AX_MAIN_BUS].volDelta;
                    dest[ofs + ch][ii] += (smps[ch] << 8);
                }
            }
#endif
            pupb->ve.currentVolume += pupb->ve.currentDelta;
        }
#endif // defined(ANDROID) || TARGET_OS_IPHONE

        src->currentAddressFrac = (u16)acc;
        for (int ii = 0; ii < 4; ii++)
        {
            src->last_samples[ii] = (u16)history[ii];
        }
    }

#ifdef PROFILE_ENABLED
    checkEnd();
#endif

}

#define MAKEAXLONG(h,l) (((h)<<16)|(((u32)(l))&0x0000ffff))
#define MAKEAXLONG64(h,l) (((u64)(h)<<32)|(((u64)(l))&0x0000ffffffff))
/*---------------------------------------------------------------------------*
    Return number of voices active.
 *---------------------------------------------------------------------------*/
u32 __AXGetNumVoices(void)
{
    return __AXNumVoices;
}

u32 __AXGetNumDspVoices(void)
{
    return __AXNumDspVoices;
}

u32 AXGetBumpedVoiceCount(void)
{
    return __AXDumpedVoices;
}

BOOL AXIsVoiceRunning(AXVPB *pvpb)
{
    AXPB  *ppbDsp;

    ppbDsp  = &__s_u_AXPB[pvpb->index];

    return (ppbDsp->state == AX_PB_STATE_RUN);
}


/*---------------------------------------------------------------------------*
    Service a PB, this is called by the __AXSyncPBs() function below for each
    PB that is running. This function synchronizes the user PB parameters
    with the DSP PB parameters.
 *---------------------------------------------------------------------------*/
#ifdef TRACE_HISTORIES
    char hist_name[16] = "AXPBSRC_0_0";
#endif
void __AXServiceVPB(AXVPB *pvpb)
{
    AXPB  *ppbDsp;
    AXPB  *pupb;
    u32    sync;
    u32   csync;    // pipelined copy of sync

    ASSERT((pvpb->index < __AXMaxVoices));

    ppbDsp = &__AXPB[pvpb->index];
    pupb   = &__s_u_AXPB[pvpb->index];
    sync   = pvpb->sync;
    csync  = ppbDsp->sync | sync;

    /*---------------------------------------------------------------------------*
        Optimized Linkage Changes

        1: the few words always passed to the DSP,
           never modified by the DSP, and required
           whether or not a lock is active, have been
           moved into the first 32 bytes of the AXPB

           Along with this, no locations potentially
           modified by the DSP are now in the first
           32 bytes (state was one such).

        2: Introduced the use of DCTouchRange for fields
           about to be changed by pushing to DSP to avoid
           losing adjacent settings modified by DSP because
           of invalidating the range
     *---------------------------------------------------------------------------*/
#if (0)
    OSReport("%s %s: d state = %d,  u state = %d, d sync = 0x%8.8x, sync = 0x%8.8x\n",
            __FILE__, __FUNCTION__,
            ppbDsp->state, pupb->state, csync, sync);
    OSReport("%s %s: d fmt = %d,  u fmt = %d\n",
            __FILE__, __FUNCTION__,
            ppbDsp->addr.format, pupb->addr.format);
#endif
    // 1: before we start:
    //    If the voice was processed by the DSP previous frame
    //       touch the linkages block and invalidate the rest
//    DCTouchRange(ppbDsp, 32);
    if (AX_PB_RENDERER_SELECT_DSP == ppbDsp->mixerSelected)
    {
//        DCInvalidateRange(&((u8*)ppbDsp)[32], offsetof(AXPB, index)-32);
    }
    // 2: update 'must be current' fields
    //    in --> DSP range
    ppbDsp->nextHi   = pupb->nextHi;
    ppbDsp->nextLo   = pupb->nextLo;
    //    in PPC Only range
    ppbDsp->PPC_next = pupb->PPC_next ?  &__AXPB[pupb->PPC_next->index] : NULL;

    //   in Renderer --> PPC range
    ppbDsp->mixerSelect   = pupb->mixerSelect;
    ppbDsp->mixerSelected = pupb->mixerSelected;
    if (!(sync&AX_SYNC_USER_STATE))
    {
        pvpb->state = pupb->state = ppbDsp->state;
    }
    if (!(sync&(AX_SYNC_USER_CURRADDR|AX_SYNC_USER_ADDR)))
    {
        pupb->addr.currentAddressHi  = ppbDsp->addr.currentAddressHi;
        pupb->addr.currentAddressLo  = ppbDsp->addr.currentAddressLo;
    }
    if (!(sync & AX_SYNC_USER_VE))
    {
        pvpb->currentVolume      =
        pupb->ve.currentVolume   = ppbDsp->ve.currentVolume;
    }

    if (__AXVoiceIsProtected(pvpb))
    {
        ppbDsp->sync = csync;
        if (AX_PB_RENDERER_SELECT_DSP == ppbDsp->mixerSelected)
        {
            // have to maintain the process linkages anyway!
//            DCFlushRange(ppbDsp, offsetof(AXPB, index));
        }
        return;
    }
    ppbDsp->sync = sync;

#ifdef TRACE_HISTORIES
    hist_name[8]  = (pvpb->index/10)+'0';
    hist_name[10] = (pvpb->index%10)+'0';
    ISRSendMessage(ISRR_COMMAND_PRINT_SHORTS, hist_name, 7, (u8 *)&ppbDsp->src.ratioHi);
#endif

#if (0) // optimizations should make this nxt section unnecessary
    if (sync == AX_SYNC_USER_ALLPARAMS)
    {
        u32 *src, *dst;

        // copy the whole PB
        pupb->mixerSelect = pvpb->mixerSelect;
        src = (u32*) pupb;
        dst = (u32*) ppbDsp;
        memcpy(dst, src, offsetof(AXPB, PB_PAD));
        ppbDsp->PPC_next = pupb->PPC_next;
        DCFlushRange(ppbDsp, sizeof(AXPB));
    }
    // lets selectively update user params per sync flag
    else if (!sync)
    {
#ifdef CATCH_GLITCHES
AXGlitch_trace(sync, 0, 0, 0xf0f0ffff);
#endif
        // optimization - saves all those if ()'s below
        pvpb->state                  = pupb->state
                                     = ppbDsp->state;
        ppbDsp->mixerSelect          = pupb->mixerSelect
                                     = pvpb->mixerSelect;
        pvpb->currentVolume =
        pupb->ve.currentVolume       = ppbDsp->ve.currentVolume ;
        pupb->addr.currentAddressHi  = ppbDsp->addr.currentAddressHi;
        pupb->addr.currentAddressLo  = ppbDsp->addr.currentAddressLo;

// Debug DSP Renderer
#ifdef _DEBUG
        pupb->PB_PAD[0]              = ppbDsp->PB_PAD[0];
        pupb->PB_PAD[1]              = ppbDsp->PB_PAD[1];
        pupb->PB_PAD[2]              = ppbDsp->PB_PAD[2];
        pupb->PB_PAD[3]              = ppbDsp->PB_PAD[3];
        pupb->PB_PAD[4]              = ppbDsp->PB_PAD[4];
#endif

        // this is really only to pass ppbDsp->mixerSelect to the DSP!
//        DCFlushRange(ppbDsp, sizeof(AXPB));
        DCFlushRange(&ppbDsp->mixerSelect, sizeof(ppbDsp->mixerSelect));
    }
    else
#endif // optimizations
    {
#ifdef CATCH_GLITCHES
AXGlitch_trace(sync, 0, 0, 0xf0f0f0f0);
#endif

// Bug 982:         u32 old_mixer = pupb->mixerSelect;

        // src select
        if (sync & AX_SYNC_USER_SRCSELECT)
        {
            ppbDsp->srcSelect   = pupb->srcSelect;
            ppbDsp->coefSelect  = pupb->coefSelect;
        }

        // mixctrl
        if (sync & AX_SYNC_USER_MIXCTRL)
        {
            memcpy(ppbDsp->tvMixCtrl, pupb->tvMixCtrl, sizeof(pupb->tvMixCtrl));
            memcpy(ppbDsp->drcMixCtrl, pupb->drcMixCtrl, sizeof(pupb->drcMixCtrl));
            memcpy(ppbDsp->rmtMixCtrl, pupb->rmtMixCtrl, sizeof(pupb->rmtMixCtrl));
        }

        // state
        if (sync & AX_SYNC_USER_STATE)
            ppbDsp->state       = pupb->state;

        // type
        if (sync & AX_SYNC_USER_TYPE)
            ppbDsp->type        = pupb->type;

        // mix
        if (sync & AX_SYNC_USER_MIX)
        {
            u16 *src, *dst;

            src = (u16*)&pupb->tvMix;
            dst = (u16*)&ppbDsp->tvMix;
            memcpy(dst, src, sizeof(pupb->tvMix));

            src = (u16*)&pupb->drcMix;
            dst = (u16*)&ppbDsp->drcMix;
            memcpy(dst, src, sizeof(pupb->drcMix));

            src = (u16*)&pupb->rmtMix;
            dst = (u16*)&ppbDsp->rmtMix;
            memcpy(dst, src, sizeof(pupb->rmtMix));
        }

        // itd
        if (sync & AX_SYNC_USER_ITDTARGET)
        {
            ppbDsp->itd.targetShiftL = pupb->itd.targetShiftL;
            ppbDsp->itd.targetShiftR = pupb->itd.targetShiftR;
        }
        else if (sync & AX_SYNC_USER_ITD)
        {
            u16 *src, *dst;
            u32 *dst_;

            src = (u16*)&pupb->itd;
            dst = (u16*)&ppbDsp->itd;

            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src;

            dst_ = (u32*)(pvpb->itdBuffer);

            *dst_ = 0; dst_++;
            *dst_ = 0; dst_++;
            *dst_ = 0; dst_++;
            *dst_ = 0; dst_++;
            *dst_ = 0; dst_++;
            *dst_ = 0; dst_++;
            *dst_ = 0; dst_++;
            *dst_ = 0; dst_++;
            *dst_ = 0; dst_++;
            *dst_ = 0; dst_++;
            *dst_ = 0; dst_++;
            *dst_ = 0; dst_++;
            *dst_ = 0; dst_++;
            *dst_ = 0; dst_++;
            *dst_ = 0; dst_++;
            *dst_ = 0;
        }

        // ve
        if (sync & AX_SYNC_USER_VEDELTA)
        {
            ppbDsp->ve.currentDelta  = pupb->ve.currentDelta;
        }
        else if (sync & AX_SYNC_USER_VE)
        {
            ppbDsp->ve.currentVolume = pupb->ve.currentVolume;
            ppbDsp->ve.currentDelta  = pupb->ve.currentDelta;
        }

        // addr
        if (sync & AX_SYNC_USER_ADDR)
        {
            memcpy(&ppbDsp->addr, &pupb->addr, sizeof(AXPBADDR));
        }
        else
        if (sync & (AX_SYNC_USER_LOOPADDR | AX_SYNC_USER_ENDADDR |
                   AX_SYNC_USER_CURRADDR | AX_SYNC_USER_LOOP ))
        {
            if (sync & AX_SYNC_USER_LOOP)
            {
                ppbDsp->addr.loopFlag = pupb->addr.loopFlag;
            }

            if (sync & AX_SYNC_USER_LOOPADDR)
            {
                *(u64*)&ppbDsp->addr.loopAddressHi = *(u64*)&pupb->addr.loopAddressHi;
            }

            if (sync & AX_SYNC_USER_ENDADDR)
            {
                *(u64*)&ppbDsp->addr.endAddressHi = *(u64*)&pupb->addr.endAddressHi;
            }

            if (sync & AX_SYNC_USER_CURRADDR)
            {
                *(u64*)&ppbDsp->addr.currentAddressHi = *(u64*)&pupb->addr.currentAddressHi;
            }
        }

        // adpcm
        if (sync & AX_SYNC_USER_ADPCM)
        {

            u32 *src, *dst;

            dst = (u32*)&ppbDsp->adpcm;
            src = (u32*)&pupb->adpcm;

            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src;
        }

        // src
        if (sync & AX_SYNC_USER_SRC)
        {
            u16 *src, *dst;

            dst = (u16*)&ppbDsp->src;
            src = (u16*)&pupb->src;

            *dst = *src;
            dst++;
            src++;
            *dst = *src;
            dst++;
            src++;
            *dst = *src;
            dst++;
            src++;
            *dst = *src;
            dst++;
            src++;
            *dst = *src;
            dst++;
            src++;
            *dst = *src;
            dst++;
            src++;
            *dst = *src;
        }
        else if (sync & AX_SYNC_USER_SRCRATIO)
        {
            ppbDsp->src.ratioHi = pupb->src.ratioHi;
            ppbDsp->src.ratioLo = pupb->src.ratioLo;
        }
        // adpcm loop
        if (sync & AX_SYNC_USER_ADPCMLOOP)
        {
            u16 *src, *dst;

            dst = (u16*)&ppbDsp->adpcmLoop;
            src = (u16*)&pupb->adpcmLoop;

            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src;
        }

        // low pass filter
        if (sync & AX_SYNC_USER_LPF_COEF)
        {
            ppbDsp->lpf.a0 = pupb->lpf.a0;
            ppbDsp->lpf.b0 = pupb->lpf.b0;
        }
        else if (sync & AX_SYNC_USER_LPF)
        {
            u16 *src, *dst;

            dst = (u16*)&ppbDsp->lpf;
            src = (u16*)&pupb->lpf;

            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src;
        }

        // high pass filter
        if (sync & AX_SYNC_USER_BIQUAD_COEF)
        {
            ppbDsp->biquad.b0 = pupb->biquad.b0;
            ppbDsp->biquad.b1 = pupb->biquad.b1;
            ppbDsp->biquad.b2 = pupb->biquad.b2;
            ppbDsp->biquad.a1 = pupb->biquad.a1;
            ppbDsp->biquad.a2 = pupb->biquad.a2;
        }
        else if (sync & AX_SYNC_USER_BIQUAD)
        {
            u16 *src, *dst;

            dst = (u16*)&ppbDsp->biquad;
            src = (u16*)&pupb->biquad;

            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src;
        }

        if (sync & AX_SYNC_USER_REMOTE)
        {
            ppbDsp->remote = pupb->remote;
        }

        if (sync & AX_SYNC_USER_RMTDPOP)
        {
            u16 *src, *dst;

            dst = (u16*)&ppbDsp->rmtDpop;
            src = (u16*)&pupb->rmtDpop;

            memcpy(dst, src, sizeof(AXPBRMTDPOP));
        }

        if (sync & AX_SYNC_USER_RMTSRC)
        {
            u16 *src, *dst;

            dst = (u16*)&ppbDsp->rmtSrc;
            src = (u16*)&pupb->rmtSrc;

            memcpy(dst, src, sizeof(AXPBRMTSRC));
        }

        if (sync & AX_SYNC_USER_RMTIIR_COEF1)
        {
            ppbDsp->rmtIIR.lpf.a0 = pupb->rmtIIR.lpf.a0;
            ppbDsp->rmtIIR.lpf.b0 = pupb->rmtIIR.lpf.b0;
        }
        else if (sync & AX_SYNC_USER_RMTIIR_COEF2)
        {
            ppbDsp->rmtIIR.biquad.b0 = pupb->rmtIIR.biquad.b0;
            ppbDsp->rmtIIR.biquad.b1 = pupb->rmtIIR.biquad.b1;
            ppbDsp->rmtIIR.biquad.b2 = pupb->rmtIIR.biquad.b2;
            ppbDsp->rmtIIR.biquad.a1 = pupb->rmtIIR.biquad.a1;
            ppbDsp->rmtIIR.biquad.a2 = pupb->rmtIIR.biquad.a2;
        }
        else if (sync & AX_SYNC_USER_RMTIIR)
        {
            u16 *src, *dst;

            dst = (u16*)&ppbDsp->rmtIIR;
            src = (u16*)&pupb->rmtIIR;

            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src; dst++; src++;
            *dst = *src;
        }
    }
#if (0)
    OSReport("%s %s: d state = %d,  u state = %d, d sync = 0x%8.8x\n",
            __FILE__, __FUNCTION__,
            ppbDsp->state, pupb->state, ppbDsp->sync);
    OSReport("%s %s: d fmt = %d,  u fmt = %d\n",
            __FILE__, __FUNCTION__,
            ppbDsp->addr.format, pupb->addr.format);
#endif
    if (ppbDsp->mixerSelected == AX_PB_RENDERER_SELECT_DSP)
    {
        //DCFlushRange(&((u8*)ppbDsp)[32], offsetof(AXPB, index)-32);
//        DCFlushRange(((u8*)ppbDsp), offsetof(AXPB, index));
    }
}

/*---------------------------------------------------------------------------*
    Get rid of this voice, this is called by __AXSyncPBs() to get rid of voices
    that the DSP doesn't have cycles for. Makse sure is no longer makes noise.
 *---------------------------------------------------------------------------*/
void __AXDumpVPB(AXVPB *pvpb)
{
    AXPB  *ppbDsp = &__AXPB[pvpb->index];
    AXPB  *pupb  = &__s_u_AXPB[pvpb->index];

    // if it's running depop the voice
    // DJW invert sense now run is DSP or PPC
    // if (ppbDsp->state == AX_PB_STATE_RUN)
    if (ppbDsp->state != AX_PB_STATE_STOP)
        __AXDepopVoice(ppbDsp);

    // stop the PB.
    pvpb->state = AX_PB_STATE_STOP;
    ppbDsp->state  = AX_PB_STATE_STOP;
    pupb->state = AX_PB_STATE_STOP;

    // put it in the callback stack to callback later
    __AXPushCallbackStack(pvpb);
}

/*---------------------------------------------------------------------------*
 Go through all the PBs and synchronize user parameters with DSP PB
 parameters. Also estimates DSP cycles for the PBs and dumps the remaining
 PBs if the total allowed cycles are exceeded. PBs are services from the
 top of the highest priority list to the bottom of the lowest priority. So
 when we run out of cycles, the oldest voices of lowest priority are dumped.
 *---------------------------------------------------------------------------*/
#define __AXGetDspFormatCost(f) \
    ((f == AX_PB_FORMAT_ADPCM) ? AX_COST_DSP_ADPCM_VOICE : \
     (f == AX_PB_FORMAT_PCM8)  ? AX_COST_DSP_PCM8_VOICE  : AX_COST_DSP_PCM16_VOICE)
#define __AXGetPpcFormatCost(f) \
    ((f == AX_PB_FORMAT_ADPCM) ? AX_COST_PPC_ADPCM_VOICE : \
     (f == AX_PB_FORMAT_PCM8)  ? AX_COST_PPC_PCM8_VOICE  : AX_COST_PPC_PCM16_VOICE)

static f32 __AXGetMixCost(u16 *ctrls, u32 num, f32 cost)
{
    f32 sum = 0.0;
    while (num--)
    {
        if (*ctrls&0x0003) sum += cost;
        if (*ctrls&0x000c) sum += cost;
        if (*ctrls&0x0030) sum += cost;
        if (*ctrls&0x00c0) sum += cost;
        if (*ctrls&0x0300) sum += cost;
        if (*ctrls&0x0c00) sum += cost;
        ctrls++;
    }
    return sum;
}

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
static f32 __AXGetDspSrcCost(u16 type, AXPBSRC *src)
{
    f32 pitch;
    u32 index;

    pitch = ((f32)src->ratioHi) + (((f32)src->ratioLo)/((f32)0x10000));

    for (index = 0; index < MAX_SRC_RATIO_ENTRIES; index++)
    {
        if (pitch < __AXCostSrcRatio[index])
        {
            return __AXCostPpcSrc[type][index];
        }
    }
    return 0.0;
}

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
static f32 __AXGetPpcSrcCost(u16 type, AXPBSRC *src)
{
    f32 pitch;
    u32 index;

    pitch = ((f32)src->ratioHi) + (((f32)src->ratioLo)/((f32)0x10000));

    for (index = 0; index < MAX_SRC_RATIO_ENTRIES; index++)
    {
        if (pitch < __AXCostSrcRatio[index])
        {
            return __AXCostPpcSrc[type][index];
        }
    }
    return 0.0;
}

int __AXSyncPBs(
    int channels,
    BusBuffer* busBuffer,
    unsigned long samples,
    int sampleRate
)
#if 0
void __AXSyncPBs(u32 lessDspCycles, // TBD redefine to Cafe method of estimating load
        AXPB **root_dsp, // receives first DSP-side DSP assigned AXPB for next frame
        AXPB **root_ppc // receives first DSP-side PPC assigned AXPB for next frame
        )
#endif
{
//static char dbg[256];
//u32 dbg_offset = 0;

    f32 ppc_load;
    f32 dsp_load;
    f32 last_ppc_load;
    f32 last_dsp_load;
    u32 i;
    u32 b_voice_assigned;
    u32 b_dsp_loaded;
    u32 b_ppc_loaded;
    AXVPB *pvpb; // current AXVPB
    AXPB *puvpb; // current AXPB
    AXPB *ppbDsp; // and its Renderer partner
    AXPB *pred_dsp_puvpb; // previous one to run on the DSP
    AXPB *pred_ppc_puvpb; // previous one to run on the PPC

    __AXNumVoices    = 0;
    __AXNumDspVoices = 0;
    __AXDumpedVoices = 0;
    b_voice_assigned = 0;
    b_dsp_loaded     = 0;
    b_ppc_loaded     = 0;
    pred_dsp_puvpb    = NULL;
    pred_ppc_puvpb    = NULL;
    dsp_load          = 0.0;
    ppc_load          = 0.0;
    __AXDspLoad       = 0.0;
    __AXPpcLoad       = 0.0;

#pragma warning(push)
#pragma warning(disable:4305) // possible loss of data
    // initialize cost accumulators
    __AXGetCommandListCosts(&dsp_load, &ppc_load);
// we'll factor this properly TBD
//    dsp_load += lessDspCycles;
    dsp_load  += AX_COST_DSP_MIX_IDLE;
    ppc_load  += AX_COST_PPC_MIX_IDLE;
    ppc_load  += AX_COST_PPC_POST_MIX_IDLE;
    ppc_load  += AX_COST_OUTPUT_STAGE_IDLE;

    last_dsp_load = dsp_load;
    last_ppc_load = ppc_load;

//    dbg[dbg_offset++] = '|';
//    dbg_offset += sprintf(&dbg[dbg_offset], "%3.3f %3.3f:", dsp_load, ppc_load);


    // service PB from the newest of highest priority
    // start by threading them onto the DSP list
    // if the DSP cycles budget gets exceeded
    // switch to threading the voices on the PPC list
    // dump the rest by stopping the PBs and putting them in the
    // callback stack

    // to handle unpinned voices, need to do an initial scan
    // to identify all the DSP pinned voices first - TBD
    //    -  this is tricky, defer for first cut
    //       priorities are intermingled, who gets dumped???
    for (i = AX_PRIORITY_NODROP; i > 0; i--)
    {
        pvpb = __AXGetStackHead(i);

        while (pvpb)
        {
            puvpb  = &__s_u_AXPB[pvpb->index];
            ppbDsp = &__AXPB[pvpb->index];

            b_voice_assigned   = 0;
            puvpb->mixerSelect = pvpb->mixerSelect;

            // see if we should depop
            if (pvpb->depop)
            {
                __AXDepopVoice(ppbDsp);
                pvpb->depop = FALSE;
            }

            // sum and store cycles estimate for this voice
            if (   !b_dsp_loaded
                && ((puvpb->state == AX_PB_STATE_RUN) || (pvpb->sync))
                &&  (  (puvpb->mixerSelect == AX_PB_RENDERER_SELECT_DSP_OR_PPC)
                    || (puvpb->mixerSelect == AX_PB_RENDERER_SELECT_DSP)))
            {
                last_dsp_load = dsp_load;

                dsp_load += static_cast<f32>(__AXGetDspFormatCost(puvpb->addr.format));
                dsp_load += __AXGetDspSrcCost(puvpb->srcSelect, &puvpb->src);

                if (puvpb->lpf.on)
                    dsp_load += AX_COST_DSP_LPF;

                if (puvpb->biquad.on)
                    dsp_load += AX_COST_DSP_BIQUAD;

                dsp_load += __AXGetMixCost((u16 *)&puvpb->tvMixCtrl[0][0], AX_MAX_NUM_TVS * AX_MAX_NUM_BUSES, AX_COST_DSP_MIX_PER_CH);
                dsp_load += __AXGetMixCost((u16 *)&puvpb->drcMixCtrl[0][0], AX_MAX_NUM_DRCS * AX_MAX_NUM_BUSES, AX_COST_DSP_MIX_PER_CH);
#if 0
                if (puvpb->remote == AX_PB_REMOTE_ON)
                {
                    dsp_load += AX_COST_PPC_RMT_MIX_IDLE;
                    dsp_load += __AXGetMixCost((u16 *)&puvpb->rmtMixCtrl[0][0], AX_MAX_NUM_RMTS * AX_MAX_NUM_BUSES, AX_COST_DSP_MIX_PER_CH);

                    if (puvpb->rmtIIR.lpf.on == AX_PB_LPF_ON)
                        dsp_load += AX_COST_DSP_RMT_LPF;

                    else if (puvpb->rmtIIR.lpf.on == AX_PB_BIQUAD_ON)
                        dsp_load += AX_COST_DSP_RMT_BIQUAD;
                }
#endif //
                // Now see if we have enough HP to handle this voice
                if (dsp_load > __AXDspLoadLimit)
                {
                    // No - bump it to the PPC Renderer
//                    dbg[dbg_offset++] = 'X';
                    b_dsp_loaded = 1;
                    dsp_load     = last_dsp_load;
                }
                else
                {
                    __AXNumDspVoices++;
                    __AXNumVoices++;


                    // thread onto the DSP processing list
                    puvpb->mixerSelected = AX_PB_RENDERER_SELECT_DSP;
                    if (pred_dsp_puvpb)
                    {
                        puvpb->nextHi             = pred_dsp_puvpb->currHi;
                        puvpb->nextLo             = pred_dsp_puvpb->currLo;
                        pred_dsp_puvpb->PPC_next  = puvpb;
                    }
                    else
                    {
                        puvpb->nextHi = 0;
                        puvpb->nextLo = 0;
                    }
                    puvpb->PPC_next = 0;
                    pred_dsp_puvpb  = puvpb;

                    if ((0 != __AXMaxDspVoices)
                            && (__AXMaxDspVoices == __AXNumDspVoices))
                    {
                        // keep this voice but move all the rest onto the PPC
                        b_dsp_loaded = 1;
                    }
                    last_dsp_load  = dsp_load;
                    b_voice_assigned = 1;
//                    dbg[dbg_offset++] = 'D';

                    SynthVPB(
                        pvpb,
                        channels,
                        busBuffer,
                        samples,
                        sampleRate
                    );
                }
            }
            if (   !b_voice_assigned && !b_ppc_loaded
                && ((puvpb->state == AX_PB_STATE_RUN) || (pvpb->sync))
               )
            {
                // same sums for PPC loading since we can saturate
                // but use different stats
                // may set b_ppc_loaded
                last_ppc_load = ppc_load;

                ppc_load += static_cast<f32>(__AXGetPpcFormatCost(puvpb->addr.format));
                ppc_load += __AXGetPpcSrcCost(puvpb->srcSelect, &puvpb->src);

                if (puvpb->lpf.on)
                    ppc_load += AX_COST_PPC_LPF;

                if (puvpb->biquad.on)
                    ppc_load += AX_COST_PPC_BIQUAD;

                ppc_load += __AXGetMixCost((u16 *)&puvpb->tvMixCtrl[0][0], AX_MAX_NUM_TVS * AX_MAX_NUM_BUSES, AX_COST_PPC_MIX_PER_CH);
                ppc_load += __AXGetMixCost((u16 *)&puvpb->drcMixCtrl[0][0], AX_MAX_NUM_DRCS * AX_MAX_NUM_BUSES, AX_COST_PPC_MIX_PER_CH);
#pragma warning(pop) // C4305
#if 0
                if (puvpb->remote == AX_PB_REMOTE_ON)
                {
                    ppc_load += AX_COST_PPC_RMT_MIX_IDLE;
                    ppc_load += __AXGetMixCost((u16 *)&puvpb->rmtMixCtrl[0][0], AX_MAX_NUM_RMTS * AX_MAX_NUM_BUSES, AX_COST_PPC_MIX_PER_CH);

                    if (puvpb->rmtIIR.lpf.on == AX_PB_LPF_ON)
                        ppc_load += AX_COST_PPC_RMT_LPF;

                    else if (puvpb->rmtIIR.lpf.on == AX_PB_BIQUAD_ON)
                        ppc_load += AX_COST_PPC_RMT_BIQUAD;
                }
#endif //
                // see if we should service or dump this voice
                // Now see if we have enough HP to handle this voice
                if (ppc_load > __AXPpcLoadLimit)
                {
                    // No - dump it
                    b_ppc_loaded = 1;
                    ppc_load     = last_ppc_load;
//                    dbg[dbg_offset++] = 'x';
                }
                if (!b_ppc_loaded) // OK
                {
                    __AXNumVoices++;
                    puvpb->mixerSelected = AX_PB_RENDERER_SELECT_PPC;

                    // thread onto the PPC processing list
                    if (pred_ppc_puvpb)
                    {
                        puvpb->nextHi   = pred_ppc_puvpb->currHi;
                        puvpb->nextLo   = pred_ppc_puvpb->currLo;
                        puvpb->PPC_next = pred_ppc_puvpb;
                    }
                    else
                    {
                        puvpb->nextHi   = 0;
                        puvpb->nextLo   = 0;
                        puvpb->PPC_next = 0;
                    }
                    pred_ppc_puvpb = puvpb;

#if (0) // for when we also do a over-all max voices limit
                    __AXNumPpcVoices++;
                    if ((0 != __AXMaxPpcVoices) && (__AXMaxPpcVoices == __AXNumPpcVoices))
                    {
                        // eep this voice but move all the rest onto the PPC
                        b_ppc_loaded = 1;
                    }
#endif
                    last_ppc_load   = ppc_load;
                    b_voice_assigned = 1;
//                    dbg[dbg_offset++] = 'P';
                }
                else
                {
                    last_ppc_load = ppc_load;
                    // dump this voice
                    __AXDumpVPB(pvpb);
                    __AXDumpedVoices++;
//                    dbg[dbg_offset++] = 'm';
                }
            }

            if (b_voice_assigned)
            {
                __AXServiceVPB(pvpb);
            }
            else
            {
                ppbDsp->mixerSelected =
                puvpb->mixerSelected  = AX_PB_RENDERER_SELECT_DSP_OR_PPC;
                puvpb->nextHi = 0;
                puvpb->nextLo = 0;
            }

            // reset the vpb for next frame
            if (!__AXVoiceIsProtected(pvpb))
            {
                pvpb->sync = AX_SYNC_NONEWPARAMS;
                pvpb->depop = FALSE;
            }

            pvpb = reinterpret_cast<AXVPB*>(pvpb->next);
        } // end of while loop for each stack
    } // end of for loop
//    dbg_offset += sprintf(&dbg[dbg_offset], "%3.3f %3.3f\n", dsp_load, ppc_load);
//OSReport(dbg);
    // record the DSP cycles that we would have used incase the user needs
    // this number to do profiling
    __AXDspLoad = last_dsp_load;
    __AXPpcLoad = last_ppc_load;

    // finally make sure all the free voices are stopped
    pvpb = __AXGetStackHead(AX_PRIORITY_FREE);

    while (pvpb)
    {
        // see if we should depop
        if (pvpb->depop)
        {
            __AXDepopVoice(&__AXPB[pvpb->index]);
            pvpb->depop = FALSE;
        }

        // make sure the state is stopped.

        __s_u_AXPB[pvpb->index].mixerSelected = AX_PB_RENDERER_SELECT_DSP_OR_PPC;
        __AXPB    [pvpb->index].mixerSelected = AX_PB_RENDERER_SELECT_DSP_OR_PPC;

        __s_u_AXPB[pvpb->index].state = 0;
        __AXPB    [pvpb->index].state = 0;
                   pvpb->state        = 0;

        // get the next one
        pvpb = reinterpret_cast<AXVPB*>(pvpb->next);
    }
 //   if (root_dsp) *root_dsp = pred_dsp_puvpb ? &__AXPB[pred_dsp_puvpb->index] : NULL;
 //   if (root_ppc) *root_ppc = pred_ppc_puvpb ? &__AXPB[pred_ppc_puvpb->index] : NULL;

    return __AXNumVoices;
}

/*---------------------------------------------------------------------------*
 Returns pointer to PBs, used to generate the CL.
 *---------------------------------------------------------------------------*/
AXPB* __AXGetPBs(void)
{
    return &__AXPB[0];
}
/*---------------------------------------------------------------------------*
 Returns pointer to PBs, for debug test access
 TBD **** remove from SK releases!!!!.
 *---------------------------------------------------------------------------*/
//#ifdef _DEBUG
AXPB* __AXGetUPBs(void)
{
    return &__s_u_AXPB[0];
}
AXVPB* __AXGetVPBs(void)
{
    return &__AXVPB[0];
}
//#endif

/*---------------------------------------------------------------------------*
    Sets a PB's params to default, just enough to not get any noise.
 *---------------------------------------------------------------------------*/
void __AXSetPBDefault(AXVPB *p)
{
    AXPB  *pupb  = &__s_u_AXPB[p->index];

    pupb->state  = p->state = AX_PB_STATE_STOP;
    pupb->itd.flag          = AX_PB_ITD_OFF;
//    p->sync                 = AX_SYNC_USER_STATE | AX_SYNC_USER_ITD | AX_SYNC_USER_LPF | AX_SYNC_USER_BIQUAD | AX_SYNC_USER_REMOTE | AX_SYNC_USER_RMTSRC | AX_SYNC_USER_RMTIIR;
    p->sync = AX_SYNC_USER_STATE | AX_SYNC_USER_ITD | AX_SYNC_USER_LPF
            | AX_SYNC_USER_BIQUAD | AX_SYNC_USER_REMOTE | AX_SYNC_USER_SRC
            | AX_SYNC_USER_RMTSRC | AX_SYNC_USER_RMTIIR;
    pupb->lpf.on = 0;
    pupb->biquad.on = 0;
    pupb->remote = AX_PB_REMOTE_OFF;
    pupb->rmtIIR.lpf.on = 0;

    pupb->mixerSelect = p->mixerSelect = AXGetDefaultMixerSelect();

    // clear AXPBSRC - Bug 1275 !!!
    pupb->src.currentAddressFrac = 0;
    pupb->src.last_samples[0]    = 0;
    pupb->src.last_samples[1]    = 0;
    pupb->src.last_samples[2]    = 0;
    pupb->src.last_samples[3]    = 0;

    // clear AXPBRMTSRC
    pupb->rmtSrc.currentAddressFrac = 0;
    pupb->rmtSrc.last_samples[0] = 0;
    pupb->rmtSrc.last_samples[1] = 0;
    pupb->rmtSrc.last_samples[2] = 0;
    pupb->rmtSrc.last_samples[3] = 0;

    __lastOffsets[p->index] = 0;
    __loopCounts[p->index] = 0;

    memset(pupb->tvMixCtrl, 0, sizeof(pupb->tvMixCtrl));
    memset(pupb->drcMixCtrl, 0, sizeof(pupb->drcMixCtrl));
    memset(pupb->rmtMixCtrl, 0, sizeof(pupb->rmtMixCtrl));
    memset(pupb->tvDpop, 0, sizeof(pupb->tvDpop));
    memset(pupb->drcDpop, 0, sizeof(pupb->drcDpop));
    memset(pupb->rmtDpop, 0, sizeof(pupb->rmtDpop));

}

/*---------------------------------------------------------------------------*
 Initialize this code module.
 *---------------------------------------------------------------------------*/
void __AXVPBInit(void)
{

#ifdef _DEBUG
    OSReport("Initializing AXVPB code module\n");
#endif

    // set max num of voices
    __AXMaxVoices = AX_MAX_VOICES;

    // set memory
    __AXPB  = __s_AXPB;
    __AXITD = __s_AXITD;
    __AXVPB = __s_AXVPB;

    __AXVPBInitCommon();
}

void __AXVPBInitSpecifyMem(u32 num, void* mem)
{
    u8              *ptr;

#ifdef _DEBUG
    OSReport("Initializing AXVPB code module\n");
#endif

    ASSERT(num <= AX_MAX_VOICES);
    ASSERT(mem);
    ASSERT(!((u32)mem & 0x1f));

    if ((num > AX_MAX_VOICES) || !mem || ((u32)mem & 0x1f))
    {
        return;
    }

    // set max num of voices
    __AXMaxVoices = num;

    // set memory
    ptr = (u8*)mem;

#if (1)
    __AXPB  = __s_AXPB;
#else
    __AXPB = (AXPB*)ptr;
    ptr += __AXMaxVoices * sizeof(AXPB);
#endif
    __AXITD = (AXPBITDBUFFER*)ptr;
    ptr += __AXMaxVoices * sizeof(AXPBITDBUFFER);

    __AXVPB = (AXVPB*)ptr;
    ptr += __AXMaxVoices * sizeof(AXVPB);

    __AXVPBInitCommon();
}

static void __AXVPBInitCommon(void)
{
    u32             i;
    AXPB            *ppb;
    AXPB            *pupb;
    AXPBITDBUFFER   *ppbi;
    AXVPB           *pvpb;
    u32             *p;
    u32              ppb_p;
    u32              ppbi_p;

    // initialize dsp cycles
    __AXMaxDspCycles = AX_DSP_CYCLES;
    __AXRecDspCosts = 0;
    __AXRecAxCosts = 0;

    // 0 everything
    p = (u32*)&__AXPB[0];

#if (0) // This section helps when making AXPB changes so the DSP stays clam
    {
        u32              IO_pad_needs;
        IO_pad_needs = (((offsetof(AXPB, IO_pad[0]) + (PPC_IO_BUFFER_ALIGN - 1))
            & ~(PPC_IO_BUFFER_ALIGN - 1)) - offsetof(AXPB, IO_pad[0]));
        if (0) // (IO_pad_needs != sizeof(__s_AXPB[0].IO_pad))
        {
            OSReport("sizeof(AXPB)              = %d (0x%8.8x)\n", sizeof(AXPB),
                    sizeof(AXPB));
            OSReport("offsetof(AXPB, state)     =  0x%4.4x\n", offsetof(AXPB, state));
            OSReport("offsetof(AXPB, index)     =  0x%4.4x\n", offsetof(AXPB, index));
            OSReport("offsetof(AXPB, IO_pad[0]) =  0x%4.4x\n", offsetof(AXPB, IO_pad[0]));
            OSReport("IO_pad size need:            %d x u16's\n", IO_pad_needs / 2);
            OSReport(
                    "PB_DMA_LEN, PAD         = %d, %d\n",
                    offsetof(AXPB, index),
                    (((offsetof(AXPB, PB_PAD) + 31) & 0xffffffe0)
                            - offsetof(AXPB, PB_PAD)) / 2);
            OSReport(".bss rmtMix      (u16) = %d\n", sizeof(__s_AXPB[0].rmtMix) / 2);
            OSReport(".bss rmtDpop     (u16) = %d\n",
                    sizeof(__s_AXPB[0].rmtDpop) / 2);
            OSReport(".bss rmtMixCtrl) (u16) = %d\n",
                    sizeof(__s_AXPB[0].rmtMixCtrl) / 2);
    // get this from core.lst, addr of PB_nextHi
    #define paramblk 0x2b20
            OSReport("DSP word addresses\n");
            OSReport("paramblk: 0x%4.4x\n", paramblk);
            OSReport("state:    0x%4.4x\n", (offsetof(AXPB, state) / 2) + paramblk);
            OSReport("addr:     0x%4.4x\n", (offsetof(AXPB, addr) / 2) + paramblk);
            OSReport("tvMixCtrl:0x%4.4x\n", (offsetof(AXPB, tvMixCtrl) / 2) + paramblk);
            OSReport("tvMix:    0x%4.4x\n", (offsetof(AXPB, tvMix) / 2) + paramblk);
            OSReport("tvDpop:   0x%4.4x\n", (offsetof(AXPB, tvDpop) / 2) + paramblk);
            OSReport("drcDpop:  0x%4.4x\n",
                    (offsetof(AXPB, drcDpop) / 2) + paramblk);
            OSReport("lpf:      0x%4.4x\n", (offsetof(AXPB, lpf) / 2) + paramblk);
            OSReport("biquad:   0x%4.4x\n", (offsetof(AXPB, biquad) / 2) + paramblk);
            OSReport("rmtDpop:  0x%4.4x\n",
                    (offsetof(AXPB, rmtDpop) / 2) + paramblk);
            OSReport("remote:   0x%4.4x\n", (offsetof(AXPB, remote) / 2) + paramblk);
            OSReport("rmtSrc:   0x%4.4x\n", (offsetof(AXPB, rmtSrc) / 2) + paramblk);
            OSReport("PB_PAD:   0x%4.4x\n", (offsetof(AXPB, PB_PAD) / 2) + paramblk);
            OSReport("index:    0x%4.4x\n", (offsetof(AXPB, index) / 2) + paramblk);
        }
    }
#endif
    i = (sizeof(AXPB) / 4) * __AXMaxVoices;

    while (i)
    {
        *p = 0;
        p++;
        i--;
    }

    // 0 everything
    p = (u32*) &__s_u_AXPB[0];

    i = (sizeof(AXPB) / 4) * __AXMaxVoices;

    while (i)
    {
        *p = 0;
        p++;
        i--;
    }

    // 0 everything
    p = (u32*)&__AXITD[0];

    i = (sizeof(AXPBITDBUFFER) / 4) * __AXMaxVoices;

    while (i)
    {
        *p = 0;
        p++;
        i--;
    }

    p = (u32*)&__AXVPB[0];

    i = (sizeof(AXVPB) / 4) * __AXMaxVoices;

    while (i)
    {
        *p = 0;
        p++;
        i--;
    }

    // initialize VPBs
    for (i = 0; i < __AXMaxVoices; i++)
    {
        ppb     = &__AXPB[i];
        pupb    = &__s_u_AXPB[i];
        ppbi    = &__AXITD[i];
        pvpb    = &__AXVPB[i];

        ppb_p   = reinterpret_cast<u32>(ppb);
        ppbi_p  = reinterpret_cast<u32>(ppbi);

        ASSERT(!((u32)ppb  & (PPC_IO_BUFFER_ALIGN - 1) ));
        ASSERT(!((u32)ppbi  & (PPC_IO_BUFFER_ALIGN - 1) ));

        // initailize necessary data members
        pvpb->index         = i;
        pvpb->itdBuffer     = ppbi;

        __AXSetPBDefault(pvpb);

        // initialize the PB next pointer, the DSP program uses this
        // to load the next PB, terminate the last one
        if (i == __AXMaxVoices - 1)
        {
            pupb->nextHi =
            pupb->nextLo =
            ppb->nextHi     =
            ppb->nextLo     = 0;
        }
        else
        {
            pupb->nextHi = (u16)(((u32)ppb_p + sizeof(AXPB)) >> 16);
            pupb->nextLo = (u16)(((u32)ppb_p + sizeof(AXPB)) & 0xFFFF);
            ppb->nextHi     = (u16)(((u32)ppb_p + sizeof(AXPB)) >> 16);
            ppb->nextLo     = (u16)(((u32)ppb_p + sizeof(AXPB)) & 0xFFFF);
        }

        ppb->index          = i;
        pupb->index         = i;
        // initialize the PB current pointer, the DSP program uses
        // this to put the PB back in system RAM after processing
        pupb->currHi = (u16)((u32)ppb_p >> 16);
        pupb->currLo = (u16)((u32)ppb_p & 0xFFFF);
        ppb->currHi     = (u16)((u32)ppb_p >> 16);
        ppb->currLo     = (u16)((u32)ppb_p & 0xFFFF);

        // initialize the PB itdbuffer
        pupb->itd.bufferHi   = (u16)((u32)ppbi_p >> 16);
        pupb->itd.bufferLo   = (u16)((u32)ppbi_p & 0xFFFF);
        ppb->itd.bufferHi       = (u16)((u32)ppbi_p >> 16);
        ppb->itd.bufferLo       = (u16)((u32)ppbi_p & 0xFFFF);

        // put it in the free stack
        pvpb->priority = 1;
        __AXPushFreeStack(pvpb);
    }

    // flush PBs to mem
//    DCFlushRange(&__AXPB[0],    sizeof(AXPB) * __AXMaxVoices);
}

/*---------------------------------------------------------------------------*
    Shutdoen this code module.
 *---------------------------------------------------------------------------*/
void __AXVPBQuit(void)
{
#ifdef _DEBUG
    OSReport("Shutting down AXVPB code module\n");
#endif

    __AXPB  = NULL;
    __AXITD = NULL;
    __AXVPB = NULL;

    __AXMaxVoices = 0;
}

/*---------------------------------------------------------------------------*
    Exposed API functions
 *---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*
    Flush sample data from cache into real RAM
 *---------------------------------------------------------------------------*/
void AXPrepareEfxData(const void *samples, u32 size)
{
    ASSERTMSG(samples&&size, "AXPrepareEfxData: passed NULL or empty data");

//    DCFlushRange(samples, size);
}

/*---------------------------------------------------------------------------*
    User API to ser SRC type for PB.
 *---------------------------------------------------------------------------*/
void AXSetVoiceSrcType(AXVPB *p, u32 type)
{
    int old;
    AXPB  *pupb;

    ASSERT(p);
    ASSERT(type <= AX_SRC_TYPE_4TAP_16K);

    old = OSDisableInterrupts();

    pupb = &__s_u_AXPB[p->index];

    switch (type)
    {
    case AX_SRC_TYPE_NONE:

        pupb->srcSelect  = AX_PB_SRCSEL_NONE;

        break;

    case AX_SRC_TYPE_LINEAR:

        pupb->srcSelect  = AX_PB_SRCSEL_LINEAR;

        break;

    case AX_SRC_TYPE_4TAP_8K:

        pupb->srcSelect  = AX_PB_SRCSEL_POLYPHASE;
        pupb->coefSelect = AX_PB_COEFSEL_8KHZ;

        break;

    case AX_SRC_TYPE_4TAP_12K:

        pupb->srcSelect  = AX_PB_SRCSEL_POLYPHASE;
        pupb->coefSelect = AX_PB_COEFSEL_12KHZ;

        break;

    case AX_SRC_TYPE_4TAP_16K:

        pupb->srcSelect  = AX_PB_SRCSEL_POLYPHASE;
        pupb->coefSelect = AX_PB_COEFSEL_16KHZ;

        break;
    }

    p->sync |= AX_SYNC_USER_SRCSELECT;
    __AXVoiceProtect(p);

    OSRestoreInterrupts(old);
}


/*---------------------------------------------------------------------------*
    User API to set/get default mixer for PB.
 *---------------------------------------------------------------------------*/
u32 AXSetVoiceMixerSelect(AXVPB *p, u32 mixerSelect)
{
    int old = OSDisableInterrupts();
    u32 oldSelect = p->mixerSelect;

    if ((__AXMixingMode == AXMIX_DSP_ENABLE) && (mixerSelect == AX_PB_MIXER_SELECT_PPC))
        mixerSelect = AX_PB_MIXER_SELECT_DSP;
    else
    if ((__AXMixingMode == AXMIX_PPC_ENABLE) && (mixerSelect == AX_PB_MIXER_SELECT_DSP))
        mixerSelect = AX_PB_MIXER_SELECT_PPC;

    p->mixerSelect = mixerSelect;

    OSRestoreInterrupts(old);

    return oldSelect;
}

u32 AXGetVoiceMixerSelect(AXVPB *p)
{
    return p->mixerSelect;
}

/*---------------------------------------------------------------------------*
    User API to set state for PB.
 *---------------------------------------------------------------------------*/
void AXSetVoiceState(AXVPB *p, u16 state)
{
    int old = OSDisableInterrupts();

    if (p->state == state)
    {
        OSRestoreInterrupts(old);
        return;
    }

    p->state = state;
    __s_u_AXPB[p->index].state = state;
    p->sync |= AX_SYNC_USER_STATE;
    __AXVoiceProtect(p);

    if (state == AX_PB_STATE_STOP)
    {
        p->depop = TRUE;
    }

    OSRestoreInterrupts(old);
}

/*---------------------------------------------------------------------------*
    User API to set voice type for PB.
 *---------------------------------------------------------------------------*/
void AXSetVoiceType(AXVPB *p, u16 type)
{
    int old = OSDisableInterrupts();

    __s_u_AXPB[p->index].type = type;
    p->sync |= AX_SYNC_USER_TYPE;
    __AXVoiceProtect(p);

    OSRestoreInterrupts(old);
}

/*---------------------------------------------------------------------------*
    User APIs to set mix params for PB.

    DSP rewrite: deprecated but kept for now and remapped
 *---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*
 Map legacy API calls to V2.0 Renderer structs

 There are 2 layers of legacy:

 1: AXPBMIX, the original, includes S channels which are now ignored.
    AXPBDRCMIX is a recent vriant.
 2: AXPB2CHMIX, v0 of new structs, temporarily in place for initial code testing
 *---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
void AXSetVoiceMix(AXVPB *p, const AXPBMIX *mix)
{

    // declare 6ch mix,
    // copy the mix values over to 6 ch
    // call new API
    AXPBCHMIX tmpMix[AX_MAX_NUM_TV_CHS];

    memset(tmpMix, 0, sizeof(tmpMix));

    tmpMix[AX_CH_LEFT].bus[AX_MAIN_BUS].vol = mix->vL;
    tmpMix[AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta = mix->vDeltaL;
    tmpMix[AX_CH_LEFT].bus[AX_AUXA_BUS].vol = mix->vAuxAL;
    tmpMix[AX_CH_LEFT].bus[AX_AUXA_BUS].volDelta = mix->vDeltaAuxAL;
    tmpMix[AX_CH_LEFT].bus[AX_AUXB_BUS].vol = mix->vAuxBL;
    tmpMix[AX_CH_LEFT].bus[AX_AUXB_BUS].volDelta = mix->vDeltaAuxBL;
    tmpMix[AX_CH_LEFT].bus[AX_AUXC_BUS].vol = mix->vAuxCL;
    tmpMix[AX_CH_LEFT].bus[AX_AUXC_BUS].volDelta = mix->vDeltaAuxCL;

    tmpMix[AX_CH_RIGHT].bus[AX_MAIN_BUS].vol = mix->vR;
    tmpMix[AX_CH_RIGHT].bus[AX_MAIN_BUS].volDelta = mix->vDeltaR;
    tmpMix[AX_CH_RIGHT].bus[AX_AUXA_BUS].vol = mix->vAuxAR;
    tmpMix[AX_CH_RIGHT].bus[AX_AUXA_BUS].volDelta = mix->vDeltaAuxAR;
    tmpMix[AX_CH_RIGHT].bus[AX_AUXB_BUS].vol = mix->vAuxBR;
    tmpMix[AX_CH_RIGHT].bus[AX_AUXB_BUS].volDelta = mix->vDeltaAuxBR;
    tmpMix[AX_CH_RIGHT].bus[AX_AUXC_BUS].vol = mix->vAuxCR;
    tmpMix[AX_CH_RIGHT].bus[AX_AUXC_BUS].volDelta = mix->vDeltaAuxCR;

    u32 mode;
    if(AXPB_ERROR_NONE == AXGetDeviceMode(AX_DEVICE_TV, &mode))
    {
        // in surround mode, we remap to channels

        if(AX_MODE_STEREO != mode)
        {
            // mapping is as follows
            // MainL_DPL2      .equ    MainBusLeft
            // MainR_DPL2      .equ    MainBusRight
            // MainSL_DPL2     .equ    MainBusSurr
            // MainSR_DPL2     .equ    AuxCBusLeft

            // AuxAL_DPL2      .equ    AuxABusLeft
            // AuxAR_DPL2      .equ    AuxABusRight
            // AuxASL_DPL2     .equ    AuxABusSurr
            // AuxASR_DPL2     .equ    AuxCBusRight

            // AuxBL_DPL2      .equ    AuxBBusLeft
            // AuxBR_DPL2      .equ    AuxBBusRight
            // AuxBSL_DPL2     .equ    AuxBBusSurr
            // AuxBSR_DPL2     .equ    AuxCBusSurr
            //
            tmpMix[AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].vol = mix->vS;
            tmpMix[AX_CH_SUR_LEFT].bus[AX_MAIN_BUS].volDelta = mix->vDeltaS;

            tmpMix[AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].vol = mix->vAuxCL;
            tmpMix[AX_CH_SUR_RIGHT].bus[AX_MAIN_BUS].volDelta = mix->vDeltaAuxCL;
            // zero out previously assigned
            tmpMix[AX_CH_LEFT].bus[AX_AUXC_BUS].vol = 0;
            tmpMix[AX_CH_LEFT].bus[AX_AUXC_BUS].volDelta = 0;

            tmpMix[AX_CH_SUR_LEFT].bus[AX_AUXA_BUS].vol = mix->vAuxAS;
            tmpMix[AX_CH_SUR_LEFT].bus[AX_AUXA_BUS].volDelta = mix->vDeltaAuxAS;
            tmpMix[AX_CH_SUR_LEFT].bus[AX_AUXB_BUS].vol = mix->vAuxBS;
            tmpMix[AX_CH_SUR_LEFT].bus[AX_AUXB_BUS].volDelta = mix->vDeltaAuxBS;

            tmpMix[AX_CH_SUR_RIGHT].bus[AX_AUXA_BUS].vol = mix->vAuxCR;
            tmpMix[AX_CH_SUR_RIGHT].bus[AX_AUXA_BUS].volDelta = mix->vDeltaAuxCR;
            // zero out previously assigned
            tmpMix[AX_CH_RIGHT].bus[AX_AUXC_BUS].vol = 0;
            tmpMix[AX_CH_RIGHT].bus[AX_AUXC_BUS].volDelta = 0;

            tmpMix[AX_CH_SUR_RIGHT].bus[AX_AUXB_BUS].vol = mix->vAuxCS;
            tmpMix[AX_CH_SUR_RIGHT].bus[AX_AUXB_BUS].volDelta = mix->vDeltaAuxCS;
        }
    }

    AXSetVoiceDeviceMix(p, AX_DEVICE_TV, AX_TV_ID0, tmpMix);

    return;


}
/*---------------------------------------------------------------------------*
    User API to set DRC mix params for PB. Note this fuction will not ASSERT
    AX_PB_MIXCTRL_DPL2 bit. The user has to OR this bit after calling this API
    if DPL2 is used.

    DSP rewrite: deprecated but kept for now and remapped
 *---------------------------------------------------------------------------*/
void AXSetVoiceDRCMix(AXVPB *p, const AXPBDRCMIX *mix)
{

    // declare 6ch mix,
    // copy the mix values over to 6 ch
    // call new API
    AXPBCHMIX tmpMix[AX_MAX_NUM_DRC_CHS];

    memset(tmpMix, 0, sizeof(tmpMix));

    tmpMix[AX_CH_LEFT].bus[AX_MAIN_BUS].vol = mix->vL;
    tmpMix[AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta = mix->vDeltaL;
    tmpMix[AX_CH_LEFT].bus[AX_AUXA_BUS].vol = mix->vAuxL;
    tmpMix[AX_CH_LEFT].bus[AX_AUXA_BUS].volDelta = mix->vDeltaAuxL;

    tmpMix[AX_CH_RIGHT].bus[AX_MAIN_BUS].vol = mix->vR;
    tmpMix[AX_CH_RIGHT].bus[AX_MAIN_BUS].volDelta = mix->vDeltaR;
    tmpMix[AX_CH_RIGHT].bus[AX_AUXA_BUS].vol = mix->vAuxR;
    tmpMix[AX_CH_RIGHT].bus[AX_AUXA_BUS].volDelta = mix->vDeltaAuxR;

    AXSetVoiceDeviceMix(p, AX_DEVICE_DRC, AX_DRC_ID0, tmpMix);

    return ;
}
/*---------------------------------------------------------------------------*
    Renderer 2.0 User API to set voice mix to an output device.
 *---------------------------------------------------------------------------*/

static u32 __s_AX_TV_mode_ch_count[] = {
    2,  // AX_MODE_STEREO_32K
    3,  // AX_MODE_SURROUND_32K
    4,  // AX_MODE_4CHAN_32K
    6,  // AX_MODE_6CHAN_32K
    2,  // AX_MODE_STEREO_48K
    3,  // AX_MODE_SURROUND_48K
    4,  // AX_MODE_4CHAN_48K
    6   // AX_MODE_6CHAN_48K
};

static u32 __s_AX_DRC_mode_ch_count[] = {
    1,  // AX_DRC_MODE_MONO
    2,  // AX_DRC_MODE_STEREO
    4,  // AX_DRC_MODE_SURROUND
    1,  // AX_DRC_MODE_DUAL_MONO
    2,  // AX_DRC_MODE_DUAL_STEREO
    4   // AX_DRC_MODE_DUAL_SURROUND
};

/*---------------------------------------------------------------------------*
    User API to set ITD for PB.
 *---------------------------------------------------------------------------*/
void AXSetVoiceItdOn(AXVPB *p)
{
    int old = OSDisableInterrupts();
    AXPB      *pupb;

    pupb = &__s_u_AXPB[p->index];

    // don't copy the buffer param
    pupb->itd.flag          = AX_PB_ITD_ON;
    pupb->itd.shiftL        =
    pupb->itd.shiftR        =
    pupb->itd.targetShiftL  =
    pupb->itd.targetShiftR  = 0;

    // clear the itd shift bit as it takes presidence in updates
    p->sync &= ~AX_SYNC_USER_ITDTARGET;
    p->sync |= AX_SYNC_USER_ITD;
    __AXVoiceProtect(p);

    OSRestoreInterrupts(old);
}


/*---------------------------------------------------------------------------*
    User API to set target for ITD.
 *---------------------------------------------------------------------------*/
void AXSetVoiceItdTarget(AXVPB *p, u16 lShift, u16 rShift)
{
    AXPB      *pupb;
    int old = OSDisableInterrupts();

    pupb = &__s_u_AXPB[p->index];

    pupb->itd.targetShiftL = lShift;
    pupb->itd.targetShiftR = rShift;
    p->sync |= AX_SYNC_USER_ITDTARGET;
    __AXVoiceProtect(p);

    OSRestoreInterrupts(old);
}

/*---------------------------------------------------------------------------*
    User API to set volume envelope for PB.
 *---------------------------------------------------------------------------*/
void AXSetVoiceVe(AXVPB *p, const AXPBVE *ve)
{
    AXPB      *pupb;
    int old = OSDisableInterrupts();

    pupb = &__s_u_AXPB[p->index];
    /*
    if (   (pupb->ve.vol  != ve->vol)
        || (pupb->ve.volDelta  != ve->volDelta)
       )
    */
    {
        pupb->ve.currentVolume  = ve->currentVolume;
        pupb->ve.currentDelta   = ve->currentDelta;

        p->sync |= AX_SYNC_USER_VE;
        __AXVoiceProtect(p);
    }

    OSRestoreInterrupts(old);
}

/*---------------------------------------------------------------------------*
    User API to set volume envelope delta for PB.
 *---------------------------------------------------------------------------*/
void AXSetVoiceVeDelta(AXVPB *p, s16 delta)
{
    AXPB      *pupb;
    int old = OSDisableInterrupts();

    pupb = &__s_u_AXPB[p->index];

    if (pupb->ve.currentDelta != delta)
    {
        pupb->ve.currentDelta = delta;
        p->sync |= AX_SYNC_USER_VEDELTA;
        __AXVoiceProtect(p);
    }

    OSRestoreInterrupts(old);
}


/*---------------------------------------------------------------------------*
    User API to set CAFE addressing for PB.

    maddr is assumed to contain sample offsets into maddr->samples
    get the physical address of maddr->samples
    extract its segment number into segnum
    convert all the pointers into real addresses
    according to the format.

    Check that voice does not wrap over a segment boundary.
    If it does we should probably copy it and re-do this locating
    Currently just ASSERT on it.

    Then call __AXSetVoiceAddr()
 *---------------------------------------------------------------------------*/
#define Bytes2Nibbles(n) (n << 1)
void AXSetVoiceOffsets(AXVPB *p, const AXPBOFFSET *maddr)
{
    u64 MRAM_addr, MRAM_addr4, MRAM_addr4_nibbles, MRAM_addr8, MRAM_addr16, MRAM_addr16_shorts;
    u64 sample_base;
    AXPBADDR addr;
#ifdef _DEBUG
    u32 sample_size_bytes, sample_count;
#endif
    ASSERTMSG(p&&maddr&&maddr->samples, "AXSetVoiceOffsets: NULL argument\n");

    p->offsets     = *maddr;
    addr.format   = maddr->format;
    addr.loopFlag = maddr->loopFlag;

    MRAM_addr      = reinterpret_cast<u32>((void*)maddr->samples);
    addr.segoff   = static_cast<u16>(MRAM_addr>>29);
#ifdef _DEBUG
    sample_count   = maddr->endOffset - maddr->currentOffset;
#endif

    switch (addr.format)
    {
    case AX_PB_FORMAT_ADPCM:
        MRAM_addr4_nibbles       = Bytes2Nibbles(MRAM_addr);
        MRAM_addr4               = static_cast<s32>(maddr->loopOffset) + MRAM_addr4_nibbles;
        addr.loopAddressHi      = (u32)((MRAM_addr4 >> 32)&0xffffffff);
        addr.loopAddressLo      = (u32)(MRAM_addr4 & 0xFFFFFFFF);

        MRAM_addr4               = (maddr->endOffset + MRAM_addr4_nibbles);
        addr.endAddressHi       = (u32)((MRAM_addr4 >> 32)&0xffffffff);
        addr.endAddressLo       = (u32)(MRAM_addr4 & 0xFFFFFFFF);

        MRAM_addr4               = (maddr->currentOffset + MRAM_addr4_nibbles);
        addr.currentAddressHi   = (u32)((MRAM_addr4 >> 32)&0xffffffff);
        addr.currentAddressLo   = (u32)(MRAM_addr4 & 0xFFFFFFFF);

        sample_base               = maddr->currentOffset;
#ifdef _DEBUG
        sample_size_bytes         = sample_count>>1;
#endif
        sample_base               = (sample_base>>1)|(addr.segoff<<29);
        break;

    case AX_PB_FORMAT_PCM16:
        MRAM_addr16_shorts       = MRAM_addr >> 1;
        MRAM_addr16              = (static_cast<s32>(maddr->loopOffset) + MRAM_addr16_shorts);
        addr.loopAddressHi      = (u32)((MRAM_addr16 >> 32)&0xffffffff);
        addr.loopAddressLo      = (u32)(MRAM_addr16 & 0xFFFFFFFF);

        MRAM_addr16              = (maddr->endOffset + MRAM_addr16_shorts);
        addr.endAddressHi       = (u32)((MRAM_addr16 >> 32)&0xffffffff);
        addr.endAddressLo       = (u32)(MRAM_addr16 & 0xFFFFFFFF);

        MRAM_addr16              = (maddr->currentOffset + MRAM_addr16_shorts);
        addr.currentAddressHi   = (u32)((MRAM_addr16 >> 32)&0xffffffff);
        addr.currentAddressLo   = (u32)(MRAM_addr16 & 0xFFFFFFFF);

        sample_base              = maddr->currentOffset;
#ifdef _DEBUG
        sample_size_bytes        = sample_count<<1;
#endif
        sample_base              = (sample_base<<1)|(addr.segoff<<29);
        break;

    case AX_PB_FORMAT_PCM8:
        MRAM_addr8               = MRAM_addr;
        MRAM_addr8               = (static_cast<s32>(maddr->loopOffset) + MRAM_addr);
        addr.loopAddressHi      = (u32)((MRAM_addr8 >> 32)&0xffffffff);
        addr.loopAddressLo      = (u32)(MRAM_addr8 & 0xFFFFFFFF);

        MRAM_addr8               = (maddr->endOffset + MRAM_addr);
        addr.endAddressHi       = (u32)((MRAM_addr8 >> 32)&0xffffffff);
        addr.endAddressLo       = (u32)(MRAM_addr8 & 0xFFFFFFFF);

        MRAM_addr8               = (maddr->currentOffset + MRAM_addr);
        addr.currentAddressHi   = (u32)((MRAM_addr8 >> 32)&0xffffffff);
        addr.currentAddressLo   = (u32)(MRAM_addr8 & 0xFFFFFFFF);

        sample_base              = maddr->currentOffset;
#ifdef _DEBUG
        sample_size_bytes        = sample_count;
#endif
        sample_base              = sample_base|(addr.segoff<<29);
        break;

    default:
        ASSERTMSG(0, "AXSetVoiceOffsets: unknown format in AXPBOFFSET\n");
    }
    /// \TBD move sample into clean MRAM to avoid this message in future
    /***TBD***/
//    ASSERTMSG(((sample_base+sample_size_bytes)>>29)==addr.segoff, "AXSetVoiceOffsets: Sample wraps over segment boundaries\n");
    __AXSetVoiceAddr(p, &addr);

    // clean stack!!!!!
    sample_base = MRAM_addr = MRAM_addr4 = MRAM_addr4_nibbles
                = MRAM_addr8 = MRAM_addr16 = MRAM_addr16_shorts = 0;
    memset(&addr, 0, sizeof(AXPBADDR));
#ifdef AX_LOG_OFFSETS
    OSReport("AXSetVoiceOffsets       l  = 0x%8.8x e  = 0x%8.8x c  = 0x%8.8x, se =  0x%8.8x\n", p->offsets.loopOffset, p->offsets.endOffset, p->offsets.currentOffset, p->offsets.samples);
#endif
}
/*---------------------------------------------------------------------------*
    User API to set addressing for PB.

    NB: removed legacy *dst++ = *src++ copying - too brittle, possibly incorrect,
    and does not follow data structure changes very well.
    memcpy and memset are optimized these days.
 *---------------------------------------------------------------------------*/
static void __AXSetVoiceAddr(AXVPB *p, AXPBADDR *addr)
{
    int old;
    AXPB *pb = &__s_u_AXPB[p->index];

    old = OSDisableInterrupts();

    memcpy(&pb->addr, addr, sizeof(AXPBADDR));

    // non-ADPCM format needs to have ADPCM decoder gain set to 0x0800 and
    // rest of ADPCM params set to 0 load a pre made AXPBADPCM block for the
    // PCM formats.

    // also check to make sure the APDCM addressing does not land on a
    // frame header
    switch (addr->format)
    {
    case AX_PB_FORMAT_ADPCM:

        ASSERTMSG(
            (addr->loopAddressLo & 0x000f) > 1,
            "*** loop address on ADPCM frame header! ***\n"
            );

        ASSERTMSG( (addr->endAddressLo & 0x000f) > 1,
                "*** end address on ADPCM frame header! ***\n");

        ASSERTMSG(
            (addr->currentAddressLo & 0x000f) > 1,
            "*** current address on ADPCM frame header! ***\n"
            );

        break;

    case AX_PB_FORMAT_PCM16:

        memset(&pb->adpcm, 0, sizeof(AXPBADPCM));
        pb->adpcm.gain = 0x0800;

        p->sync |= AX_SYNC_USER_ADPCM;

        break;

    case AX_PB_FORMAT_PCM8:

        memset(&pb->adpcm, 0, sizeof(AXPBADPCM));
        pb->adpcm.gain = 0x0100;

        p->sync |= AX_SYNC_USER_ADPCM;

        break;

    default:

        ASSERTMSG(0, "unknown addr->format in PB\n");
    }

    // clear the other addr sync bits as they take precedence
    p->sync &= ~(AX_SYNC_USER_LOOPADDR | AX_SYNC_USER_ENDADDR |
                 AX_SYNC_USER_CURRADDR | AX_SYNC_USER_LOOP);

    p->sync |= AX_SYNC_USER_ADDR;;
    __AXVoiceProtect(p);

    OSRestoreInterrupts(old);
}

/*---------------------------------------------------------------------------*
    User API to set loop for PB.
 *---------------------------------------------------------------------------*/
void AXSetVoiceLoop(AXVPB *p, u16 loop)
{
    AXPB      *pupb;
    int old = OSDisableInterrupts();

    pupb = &__s_u_AXPB[p->index];

    p->offsets.loopFlag = loop;
    pupb->addr.loopFlag = loop;
    p->sync |= AX_SYNC_USER_LOOP;
    __AXVoiceProtect(p);

    OSRestoreInterrupts(old);
}

/*---------------------------------------------------------------------------*
    User API to set loop address for PB.
 *---------------------------------------------------------------------------*/
void AXSetVoiceLoopOffset(AXVPB *p, u32 offset)
{
    AXPB      *pupb;
    u64       addr;
    int old = OSDisableInterrupts();

    p->offsets.loopOffset = offset;
    pupb = &__s_u_AXPB[p->index];
    addr = reinterpret_cast<u32>((void*)p->offsets.samples);
    //addr &= 0x1fffffff; // remove segment offset portion

    switch (pupb->addr.format)
    {
    case AX_PB_FORMAT_ADPCM:
        addr = addr << 1;
        break;

    case AX_PB_FORMAT_PCM16:
        addr = addr >> 1;
        break;

    case AX_PB_FORMAT_PCM8:
        break;

    default:
        ASSERTMSG(0, "AXSetVoiceLoopAddrEx: unknown addr->format in PB\n");
        break;
    }

    addr += offset;
    pupb->addr.loopAddressHi = (u32)(addr >> 32);
    pupb->addr.loopAddressLo = (u32)(addr & 0xFFFFFFFF);

    p->sync |= AX_SYNC_USER_LOOPADDR;
    __AXVoiceProtect(p);

    OSRestoreInterrupts(old);
#ifdef AX_LOG_OFFSETS
    OSReport("AXSetVoiceLoopOffset(0x%8.8x) s = 0x%8.8x l = 0x%8.8x e = 0x%8.8x c = 0x%8.8x\n", offset, p->offsets.samples, p->offsets.loopOffset, p->offsets.endOffset, p->offsets.currentOffset);
#endif

    // clean stack!!!
    addr = 0;
}

/*---------------------------------------------------------------------------*
    User API to set end address for PB.
 *---------------------------------------------------------------------------*/
void AXSetVoiceEndOffset(AXVPB *p, u32 offset)
{
    AXPB      *pupb;
    u64       addr;
    int old = OSDisableInterrupts();

    p->offsets.endOffset = offset;
    pupb = &__s_u_AXPB[p->index];
    addr = reinterpret_cast<u32>((void*)p->offsets.samples);
    //addr &= 0x1fffffff; // remove segment offset portion

    switch (pupb->addr.format)
    {
    case AX_PB_FORMAT_ADPCM:
        addr = addr << 1;
        break;

    case AX_PB_FORMAT_PCM16:
        addr = addr >> 1;
        break;

    case AX_PB_FORMAT_PCM8:
        break;

    default:
        ASSERTMSG(0, "AXSetVoiceLoopAddrEx: unknown addr->format in PB\n");
        break;
    }

    addr += offset;

    pupb->addr.endAddressHi = (u32)(addr >> 32);
    pupb->addr.endAddressLo = (u32)(addr & 0xFFFFFFFF);
    p->sync |= AX_SYNC_USER_ENDADDR;
    __AXVoiceProtect(p);

    OSRestoreInterrupts(old);

#ifdef AX_LOG_OFFSETS
    OSReport("AXSetVoiceEndOffset (0x%8.8x) s = 0x%8.8x l = 0x%8.8x e = 0x%8.8x c = 0x%8.8x\n", offset, p->offsets.samples, p->offsets.loopOffset, p->offsets.endOffset, p->offsets.currentOffset);
#endif
    // clean stack!!!
    addr = 0;
    pupb = NULL;
}

/*---------------------------------------------------------------------------*
    User API to set current address for PB.
 *---------------------------------------------------------------------------*/
void AXSetVoiceCurrentOffset(AXVPB *p, u32 offset)
{
    AXPB      *pupb;
    u64       addr;
    int old = OSDisableInterrupts();

    pupb = &__s_u_AXPB[p->index];
    addr = reinterpret_cast<u32>((void*)p->offsets.samples);
    //addr &= 0x1fffffff; // remove segment offset portion

    switch (pupb->addr.format)
    {
    case AX_PB_FORMAT_ADPCM:
        addr = addr << 1;
        break;

    case AX_PB_FORMAT_PCM16:
        addr = addr >> 1;
        break;

    case AX_PB_FORMAT_PCM8:
        break;

    default:
        ASSERTMSG(0, "AXSetVoiceLoopAddrEx: unknown addr->format in PB\n");
    }

    addr += offset;

    pupb->addr.currentAddressHi = (u32)(addr >> 32);
    pupb->addr.currentAddressLo = (u32)(addr & 0xFFFFFFFF);
    p->sync |= AX_SYNC_USER_CURRADDR;
    __AXVoiceProtect(p);

    OSRestoreInterrupts(old);

#ifdef AX_LOG_OFFSETS
    OSReport("AXSetVoiceCurrentOffset(0x%8.8x) s = 0x%8.8x l = 0x%8.8x e = 0x%8.8x c = 0x%8.8x\n", offset, p->offsets.samples, p->offsets.loopOffset, p->offsets.endOffset, p->offsets.currentOffset);
#endif
    // clean stack!!!
    addr = 0;
}

/*---------------------------------------------------------------------------*
    User API to get current offsets for PB.
 *---------------------------------------------------------------------------*/
void AXGetVoiceOffsets(AXVPB *p, AXPBOFFSET *addr)
{
    AXPB      *pupb;
    u64       base;
    u64       loop;
    u64       end;
    u64       current;
    int old = OSDisableInterrupts();

    ASSERT(p&&addr);

    *addr = p->offsets;
    pupb = &__s_u_AXPB[p->index];
    base = reinterpret_cast<u64>((void*)p->offsets.samples);
    //base &= 0x1fffffff; // remove segment offset portion

    switch (pupb->addr.format)
    {
    case AX_PB_FORMAT_ADPCM:
        base = base << 1;
        break;

    case AX_PB_FORMAT_PCM16:
        base = base >> 1;
        break;

    case AX_PB_FORMAT_PCM8:
        break;

    default:
        ASSERTMSG(0, "AXGetVoiceOffsets: unknown addr->format in PB\n");
    }
    loop = (((u64)pupb->addr.loopAddressHi)<<32)|((u64)pupb->addr.loopAddressLo);
    addr->loopOffset = static_cast<u32>(loop - base);
    end = (((u64)pupb->addr.endAddressHi)<<32)|((u64)pupb->addr.endAddressLo);
    addr->endOffset = static_cast<u32>(end - base);
    current = (((u64)pupb->addr.currentAddressHi)<<32)|((u64)pupb->addr.currentAddressLo);
    addr->currentOffset = static_cast<u32>(current - base);
/*
    p->offsets.loopOffset    = addr->loopOffset;
    p->offsets.endOffset     = addr->endOffset;
    p->offsets.currentOffset = addr->currentOffset;
*/
    OSRestoreInterrupts(old);

#ifdef AX_LOG_GET_OFFSETS
    OSReport("AXGetVoiceOffsets       le = 0x%8.8x ee = 0x%8.8x ce = 0x%8.8x, se =  0x%8.8x\n", addr->loopOffset, addr->endOffset, addr->currentOffset, addr->samples);
    OSReport("                        lp = 0x%8.8x ep = 0x%8.8x cp = 0x%8.8x\n", loop, end, current);
#endif
    // clean stack!!!
    loop = end = current = base = 0;
    pupb = NULL;
}
/*---------------------------------------------------------------------------*
 SDK 2.0.3 simplification
 *---------------------------------------------------------------------------*/
void AXSetVoiceSamplesAddr(AXVPB *p, const void *samples)
{
    int old;

    ASSERT(p&&samples);

    old = OSDisableInterrupts();

    p->offsets.samples = samples;
    AXGetVoiceOffsets(p, &p->offsets);

    OSRestoreInterrupts(old);
}
/*---------------------------------------------------------------------------*
 Bug 607 Request
 *---------------------------------------------------------------------------*/
void AXGetVoiceOffsetsEx(AXVPB *p, AXPBOFFSET *addr, const void *samples)
{
    int old = OSDisableInterrupts();

    ASSERT(p&&addr&&samples);

    p->offsets.samples = samples;
#ifdef AX_LOG_GET_OFFSETS
{
    OSReport("AXGetVoiceOffsetsEx\n");
}
#endif
    AXGetVoiceOffsets(p, addr);
    p->offsets = *addr;

    OSRestoreInterrupts(old);
}
/*---------------------------------------------------------------------------*
 Bug 607 Request
 *---------------------------------------------------------------------------*/
void AXSetVoiceOffsetsEx(AXVPB *p, const AXPBOFFSET *addr, const void *samples)
{
    int old = OSDisableInterrupts();
    AXPBOFFSET addr_local;
    ASSERT(p&&addr&&samples);

    // need to make a local copy because of the const nature of addr
    addr_local = *addr;
    addr_local.samples = samples;
    AXSetVoiceOffsets(p, &addr_local);

#ifdef AX_LOG_GET_OFFSETS
{
    AXPBOFFSET laddr;
    OSReport("AXSetVoiceOffsetsEx\n");
    AXGetVoiceOffsets(p, &laddr);
}
#endif
    OSRestoreInterrupts(old);
}
/*---------------------------------------------------------------------------*
 Bug 607 Request
 *---------------------------------------------------------------------------*/
void AXSetVoiceLoopOffsetEx(AXVPB *p, u32 offset, const void *samples)
{
    AXPBOFFSET laddr;
    int old = OSDisableInterrupts();

    ASSERT(p&&samples);

    p->offsets.samples = samples;
#ifdef AX_LOG_GET_OFFSETS
    OSReport("AXSetVoiceLoopOffsetEx\n");
#endif
    AXGetVoiceOffsets(p, &laddr);
    AXSetVoiceLoopOffset(p, offset);

    OSRestoreInterrupts(old);
}
/*---------------------------------------------------------------------------*
 Bug 607 Request
 *---------------------------------------------------------------------------*/
void AXSetVoiceEndOffsetEx(AXVPB *p, u32 offset, const void *samples)
{
    AXPBOFFSET laddr;
    int old = OSDisableInterrupts();

    ASSERT(p&&samples);

    p->offsets.samples = samples;
#ifdef AX_LOG_GET_OFFSETS
    OSReport("AXSetVoiceEndOffsetEx\n");
#endif
    AXGetVoiceOffsets(p, &laddr);
    AXSetVoiceEndOffset(p, offset);

    OSRestoreInterrupts(old);
}
/*---------------------------------------------------------------------------*
 Bug 607 Request
 *---------------------------------------------------------------------------*/
u32 AXGetVoiceCurrentOffsetEx(AXVPB *p, const void *samples)
{
    AXPBOFFSET addr;
    int old = OSDisableInterrupts();

    ASSERT(p&&samples);

#ifdef AX_LOG_GET_OFFSETS
{
    OSReport("AXGetVoiceCurrentOffsetEx\n");
}
#endif
    AXGetVoiceOffsetsEx(p, &addr, samples);

    OSRestoreInterrupts(old);

    return addr.currentOffset;
}
/*---------------------------------------------------------------------------*
 Bug 607 Request
 *---------------------------------------------------------------------------*/
void AXSetVoiceCurrentOffsetEx(AXVPB *p, u32 offset, const void *samples)
{
    AXPBOFFSET laddr;
    int old = OSDisableInterrupts();

    ASSERT(p&&samples);

    p->offsets.samples = samples;
#ifdef AX_LOG_GET_OFFSETS
    OSReport("AXSetVoiceCurrentOffsetEx\n");
#endif
    AXGetVoiceOffsets(p, &laddr);
    AXSetVoiceCurrentOffset(p, offset);

    OSRestoreInterrupts(old);
}
/*---------------------------------------------------------------------------*
 Bug 607 Request
 *---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*
 There are three scenarios to consider:

 1: loop size, taking into account SRC Ratio, is less than one frame (96 samples)
    end > loop && (end - loop) / SRCRatio <= 96
    in which case compute how many loops occurred during the previous frame

 2: Normal or loop moved down (streaming)
    end >= current(n-1) > current(n) >= loop

 3: loop moved beyond end (streaming)
    current(n) >= loop > end >= current(n-1)
 *---------------------------------------------------------------------------*/
u32 AXGetVoiceLoopCount(AXVPB *p)
{
    AXPB      *pupb;
    u64        loop;
    u64        end;
    u64        current;
    u32        srcRatio;
    u32        loopLength;
    int        old;

    ASSERT(p);
    if (!p) return 0;

    old = OSDisableInterrupts();

    pupb = &__s_u_AXPB[p->index];

    loop            = MAKEAXLONG64(pupb->addr.loopAddressHi,    pupb->addr.loopAddressLo);
    end             = MAKEAXLONG64(pupb->addr.endAddressHi,     pupb->addr.endAddressLo);
    current         = MAKEAXLONG64(pupb->addr.currentAddressHi, pupb->addr.currentAddressLo);
    if (pupb->srcSelect != AX_PB_SRCSEL_NONE)
        srcRatio        = MAKEAXLONG(pupb->src.ratioHi,           pupb->src.ratioLo);
    else
        srcRatio        = 0x00010000;
    // bug 6095:
    loopLength      = srcRatio ? static_cast<u32>((((end - loop)<<16)/srcRatio)) : 0;

    if ((end > loop) && srcRatio && (loopLength <= AX_IN_SAMPLES_PER_FRAME))
    {
        __loopCounts[p->index] += static_cast<u32>(((__lastOffsets[p->index]-loop)+AX_IN_SAMPLES_PER_FRAME)/loopLength);
    }
    else
    if (   end                      >= __lastOffsets[p->index]
        && __lastOffsets[p->index]  >  current
        && current                  >= loop)
    {
        __loopCounts[p->index]++;
    }
    else
    if (   current                  >= loop
        && loop                     >  end
        && end                      >= __lastOffsets[p->index])
    {
        __loopCounts[p->index]++;
    }
    __lastOffsets[p->index] = current;
    loop = __loopCounts[p->index];

    OSRestoreInterrupts(old);

    // clean stack!!!
    end = current = 0;
    pupb = NULL;

    return static_cast<u32>(loop);
}
/*---------------------------------------------------------------------------*
    User API to set ADPCM params for PB.
 *---------------------------------------------------------------------------*/
void AXSetVoiceAdpcm(AXVPB *p, const AXPBADPCM *adpcm)
{
    int old;
    u32 *dst, *src;

    ASSERT(p);
    if (!p) return;

    dst = (u32*)&__s_u_AXPB[p->index].adpcm;
    src = (u32*)adpcm;

    old = OSDisableInterrupts();

    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src;

    p->sync |= AX_SYNC_USER_ADPCM;
    __AXVoiceProtect(p);

    OSRestoreInterrupts(old);
}

/*---------------------------------------------------------------------------*
    User API to set SRC for PB.
 *---------------------------------------------------------------------------*/
void AXSetVoiceSrc(AXVPB *p, const AXPBSRC *src_)
{
    int old;
    u16 *dst, *src;

    dst = (u16*)&__s_u_AXPB[p->index].src;
    src = (u16*)src_;

    old = OSDisableInterrupts();

    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src;

    p->sync &= ~AX_SYNC_USER_SRCRATIO;
    p->sync |= AX_SYNC_USER_SRC;
    __AXVoiceProtect(p);

    OSRestoreInterrupts(old);
}

/*---------------------------------------------------------------------------*
    User API to set SRC ratio for PB.

 *---------------------------------------------------------------------------*/
s32 AXSetVoiceSrcRatio(AXVPB *p, f32 ratio)
{
    s32 ret = AX_SRC_RATIO_OK;
    u32 r;
    AXPB      *pupb;
    int old = OSDisableInterrupts();

    pupb = &__s_u_AXPB[p->index];

    r = (u32)(ratio * 0x10000);

// Bug 901: re-instate clamping
// bug 1205: iRD request: 8x SRC Ratio
#if (1)
    // the src cannot sample over 8.0
    if (r > 0x00080000)
    {
        ret = AX_SRC_RATIO_TOO_LARGE;
        r = 0x00080000;
    }
#else
    // the src cannot sample over 4.0
    if (r > 0x00040000)
    {
        ret = AX_SRC_RATIO_TOO_LARGE;
        r = 0x00040000;
    }
#endif

#if (1) // bug 1262 -  decrease lower SRC limit to 0
    // the src cannot sample under 0.0
    if (ratio < 0.0)
    {
        ret = AX_SRC_RATIO_TOO_SMALL;
        r = 0x00000000;
    }
#else
    // the src cannot sample under 0.25
    if (r < 0x00004000)
    {
        ret = AX_SRC_RATIO_TOO_SMALL;
        r = 0x00004000;
    }
#endif
// end bug 901

    if (   (pupb->src.ratioHi != (u16)(r >> 16))
        || (pupb->src.ratioLo != (u16)(r & 0xFFFF))
       )
    {
        pupb->src.ratioHi = (u16)(r >> 16);
        pupb->src.ratioLo = (u16)(r & 0xFFFF);
        p->sync |= AX_SYNC_USER_SRCRATIO;
        __AXVoiceProtect(p);
    }

    OSRestoreInterrupts(old);

    return ret;
}

/*---------------------------------------------------------------------------*
    User API to set ADPCM loop context for PB.
 *---------------------------------------------------------------------------*/
void AXSetVoiceAdpcmLoop(AXVPB *p, const AXPBADPCMLOOP *adpcmloop)
{
    int old;
    AXPB *pb;
    AXPBADPCMLOOP *loop;

    pb   = &__s_u_AXPB[p->index];
    loop = &pb->adpcmLoop;

    old = OSDisableInterrupts();

    loop->loop_pred_scale = adpcmloop->loop_pred_scale;
    if (AX_PB_TYPE_NORMAL == pb->type)
    {
        loop->loop_yn1 = adpcmloop->loop_yn1;
        loop->loop_yn2 = adpcmloop->loop_yn2;
    }

    p->sync |= AX_SYNC_USER_ADPCMLOOP;
    __AXVoiceProtect(p);

    OSRestoreInterrupts(old);
}

/*---------------------------------------------------------------------------*
    User API to set low pass filter params for PB.
 *---------------------------------------------------------------------------*/
void AXSetVoiceLpf(AXVPB *p, const AXPBLPF *lpf)
{
    int old;
    u16 *dst, *src;

    dst = (u16*)&__s_u_AXPB[p->index].lpf;
    src = (u16*)lpf;

    old = OSDisableInterrupts();

    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src;

    p->sync |= AX_SYNC_USER_LPF;
    __AXVoiceProtect(p);

    OSRestoreInterrupts(old);
}

void AXSetVoiceLpfCoefs(AXVPB *p, u16 a0, u16 b0)
{
    int old;

    old = OSDisableInterrupts();

    __s_u_AXPB[p->index].lpf.a0 = a0;
    __s_u_AXPB[p->index].lpf.b0 = b0;

    p->sync |= AX_SYNC_USER_LPF_COEF;
    __AXVoiceProtect(p);

    OSRestoreInterrupts(old);
}

#ifndef PI
#define PI 3.14159265358979323846f
#endif
#define AX_SAMPLE_RATE  AX_IN_SAMPLES_PER_SEC      // 32000
#define AX_HALF_RATE   (AX_IN_SAMPLES_PER_SEC / 2) // 16000

void AXComputeLpfCoefs(u16 freq, u16 *a0, u16 *b0)
{
    f32 bb, cc;

    ASSERTMSG((freq <= AX_HALF_RATE), "freq is out of range\n");

    cc = 2.0f - (f32)cos(2.0f * PI * freq / AX_SAMPLE_RATE);
    bb = (f32)sqrt(cc * cc - 1.0f) - cc;

    *b0 = (u16)(-bb * 32768.0f);
    *a0 = (u16)(32767 - *b0);
}

/*---------------------------------------------------------------------------*
    User API to set high pass filter params for PB.
 *---------------------------------------------------------------------------*/
void AXSetVoiceBiquad(AXVPB *p, const AXPBBIQUAD *biquad)
{
    int old;
    u16 *dst, *src;

    dst = (u16*)&__s_u_AXPB[p->index].biquad;
    src = (u16*)biquad;

    old = OSDisableInterrupts();

    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src;

    p->sync |= AX_SYNC_USER_BIQUAD;
    __AXVoiceProtect(p);

    OSRestoreInterrupts(old);
}

void AXSetVoiceBiquadCoefs(AXVPB *p, u16 b0, u16 b1, u16 b2, u16 a1, u16 a2)
{
    AXPB      *pupb;
    int old = OSDisableInterrupts();

    pupb = &__s_u_AXPB[p->index];

    pupb->biquad.b0 = b0;
    pupb->biquad.b1 = b1;
    pupb->biquad.b2 = b2;
    pupb->biquad.a1 = a1;
    pupb->biquad.a2 = a2;

    p->sync |= AX_SYNC_USER_BIQUAD_COEF;
    __AXVoiceProtect(p);

    OSRestoreInterrupts(old);
}

/*---------------------------------------------------------------------------*
 This function is called by the user to set the maximum aloowed DSP cycles
 to spend on this audio app per audio frame. During PB sync, the number of
 cycles will be estimated for the next processing frame, if the count
 exceeds maximum, voices will be dropped until we are within the allowed
 amount of cycles.
 *---------------------------------------------------------------------------*/
void AXSetMaxDspCycles(u32 cycles)
{
    __AXMaxDspCycles = cycles;
}

/*---------------------------------------------------------------------------*
    This function is called by the user to get the maximum aloowed DSP cycles
    to spend on this audio app per audio frame.
 *---------------------------------------------------------------------------*/
u32 AXGetMaxDspCycles(void)
{
    return __AXMaxDspCycles;
}

/*---------------------------------------------------------------------------*
    This function returns the count of extimated DSP cycles used to process
    the last frame.
 *---------------------------------------------------------------------------*/
u32 AXGetDspCycles(void)
{
    return __AXRecDspCosts;
}
/*---------------------------------------------------------------------------*
 This function returns the DSP and PPC processing load
 for the the last frame.
 *---------------------------------------------------------------------------*/
void AXGetRenderCosts(u32 *dsp_cost, u32 *ppc_cost)
{
    *dsp_cost = __AXRecDspCosts;
    *ppc_cost = __AXRecAxCosts;
}

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
void AXSetVoiceRmtOn(AXVPB *p, u16 on)
{
    BOOL old = OSDisableInterrupts();

    __s_u_AXPB[p->index].remote = on;

    p->sync |= AX_SYNC_USER_REMOTE;
        __AXVoiceProtect(p);

    OSRestoreInterrupts(old);
}

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
void AXSetVoiceRmtMix(AXVPB *p, const AXPBRMTMIX  *mix)
{
    AXPBCHMIX tmpRmtMix[AX_MAX_NUM_RMTS][AX_MAX_NUM_RMT_CHS];

    memset(tmpRmtMix, 0, sizeof(tmpRmtMix));

    tmpRmtMix[AX_RMT_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].vol = mix->vMain0;
    tmpRmtMix[AX_RMT_ID0][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta = mix->vDeltaMain0;
    tmpRmtMix[AX_RMT_ID1][AX_CH_LEFT].bus[AX_MAIN_BUS].vol = mix->vMain1;
    tmpRmtMix[AX_RMT_ID1][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta = mix->vDeltaMain1;
    tmpRmtMix[AX_RMT_ID2][AX_CH_LEFT].bus[AX_MAIN_BUS].vol = mix->vMain2;
    tmpRmtMix[AX_RMT_ID2][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta = mix->vDeltaMain2;
    tmpRmtMix[AX_RMT_ID3][AX_CH_LEFT].bus[AX_MAIN_BUS].vol = mix->vMain3;
    tmpRmtMix[AX_RMT_ID3][AX_CH_LEFT].bus[AX_MAIN_BUS].volDelta = mix->vDeltaMain3;

    tmpRmtMix[AX_RMT_ID0][AX_CH_LEFT].bus[AX_AUXA_BUS].vol = mix->vAux0;
    tmpRmtMix[AX_RMT_ID0][AX_CH_LEFT].bus[AX_AUXA_BUS].volDelta = mix->vDeltaAux0;
    tmpRmtMix[AX_RMT_ID1][AX_CH_LEFT].bus[AX_AUXA_BUS].vol = mix->vAux1;
    tmpRmtMix[AX_RMT_ID1][AX_CH_LEFT].bus[AX_AUXA_BUS].volDelta = mix->vDeltaAux1;
    tmpRmtMix[AX_RMT_ID2][AX_CH_LEFT].bus[AX_AUXA_BUS].vol = mix->vAux2;
    tmpRmtMix[AX_RMT_ID2][AX_CH_LEFT].bus[AX_AUXA_BUS].volDelta = mix->vDeltaAux2;
    tmpRmtMix[AX_RMT_ID3][AX_CH_LEFT].bus[AX_AUXA_BUS].vol = mix->vAux3;
    tmpRmtMix[AX_RMT_ID3][AX_CH_LEFT].bus[AX_AUXA_BUS].volDelta = mix->vDeltaAux3;

    // call the new one
    u16 deviceCnt;
    for(deviceCnt = 0; deviceCnt < AX_MAX_NUM_RMTS ; deviceCnt++)
    {
        AXSetVoiceDeviceMix( p, AX_DEVICE_RMT, deviceCnt, tmpRmtMix[deviceCnt]);
    }

}

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
void AXSetVoiceRmtSrc(AXVPB *p, const AXPBRMTSRC *src_)
{
    u16* dst;
    u16* src;
    BOOL old;

    dst = (u16*)&__s_u_AXPB[p->index].rmtSrc;
    src = (u16*)src_;

    old = OSDisableInterrupts();

    memcpy(dst, src, sizeof(AXPBRMTSRC));

    p->sync |= AX_SYNC_USER_RMTSRC;
    __AXVoiceProtect(p);

    OSRestoreInterrupts(old);
}

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
u32 AXGetMaxVoices(void)
{
    return __AXMaxVoices;
}

/*---------------------------------------------------------------------------*
    User API to set low pass filter params for remote speaker
 *---------------------------------------------------------------------------*/
void AXSetVoiceRmtIIR(AXVPB *p, const AXPBRMTIIR *iir)
{
    int old;
    u16 *dst, *src;

    dst = (u16*)&__s_u_AXPB[p->index].rmtIIR;
    src = (u16*)iir;

    old = OSDisableInterrupts();

    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src; dst++; src++;
    *dst = *src;

    p->sync |= AX_SYNC_USER_RMTIIR;
    __AXVoiceProtect(p);

    OSRestoreInterrupts(old);
}

void AXSetVoiceRmtIIRCoefs(AXVPB *p, u16 type, ...)
{
    va_list argp;
    s32     ii, num;
    u16     coefs[5];
    BOOL    old;
    AXPB    *pupb;

    ASSERT((type == AX_PB_LPF_ON) || (type == AX_PB_BIQUAD_ON));

    if (type == AX_PB_LPF_ON)
    {
        num = 2;
    }
    else if (type == AX_PB_BIQUAD_ON)
    {
        num = 5;
    }
    else
    {
        return;
    }
    pupb = &__s_u_AXPB[p->index];

    va_start(argp, type);

    for (ii = 0; ii < num; ii++)
    {
        coefs[ii] = (u16)va_arg(argp, int);
    }

    va_end(argp);

    old = OSDisableInterrupts();

    if (type == AX_PB_LPF_ON)
    {
        pupb->rmtIIR.lpf.a0 = coefs[0];
        pupb->rmtIIR.lpf.b0 = coefs[1];

        p->sync |= AX_SYNC_USER_RMTIIR_COEF1;
        __AXVoiceProtect(p);
    }
    else
    {
        pupb->rmtIIR.biquad.b0 = coefs[0];
        pupb->rmtIIR.biquad.b1 = coefs[1];
        pupb->rmtIIR.biquad.b2 = coefs[2];
        pupb->rmtIIR.biquad.a1 = coefs[3];
        pupb->rmtIIR.biquad.a2 = coefs[4];

        p->sync |= AX_SYNC_USER_RMTIIR_COEF2;
        __AXVoiceProtect(p);
    }

    OSRestoreInterrupts(old);
}

/*******************************************************************************
 * Function: __AXSetVoiceChMix
 * Arguments:
 *  dst: destination where to copy the values
 *  src: source from where to copy the mix values from
 *  returns the mix control for this particular channel
*******************************************************************************/
void __AXSetVoiceChMix(AXPBCHMIX *dst, AXPBCHMIX const *src, u16 *curMixCtrl)
{
    u8 mixCntr=0;
    u32 busCnt;
    u8 busCntr;
    for(busCnt=0 ; busCnt < AX_MAX_NUM_BUSES; busCnt++)
    {
        busCntr = 0;

        // for first time readers the following statement might seem odd. here is how it works
        // if ( a=b) assigns b to a and checks the value of a. if it not zero, then does the
        // block of code. so busCntr is set to 1 or 3 depending if vol/volDelta is not zero
        if( (dst->bus[busCnt].vol = src->bus[busCnt].vol) )
        {
            busCntr = 1;
        }

        if( (dst->bus[busCnt].volDelta = src->bus[busCnt].volDelta) )
        {
            busCntr = 3;
        }

        *curMixCtrl++ = busCntr;

    } // end of all buses

    return ;
} // __AXSetVoiceChMix
/*******************************************************************************
 * Function: AXSetVoiceDeviceMix
 * Arguments:
 *      AXVPB *voice :          (in) pointer to the voice for which to set the mix
 *      AXPBDeviceType device:  (in) TV/DRC/RMT?
 *      u32 Id:                 (in) which TV/DRC/RMT?
 *      AXPBCHMIX *mix:         (in) pointer to the array of mixes
 *  Sets the mix for the individual devices.
 *  For TV, the mix will point to an array of AX_MAX_NUM_TV_CHS AXPBCHMIX array
 *  For DRC, the mix will point to an array of AX_MAX_NUM_DRC_CHS AXPBCHMIX array
 *  For RMT, the mix will point to an array of AX_MAX_NUM_RMT_CHS AXPBCHMIX array
*******************************************************************************/
AXPB_ERROR_CODE AXSetVoiceDeviceMix(AXVPB *voice, AXPBDeviceType device, u32 Id, AXPBCHMIX *mix)
{
    AXPB_ERROR_CODE retErrCode = AXPB_ERROR_NONE;

    BOOL old;
    AXPB    *pupb;
    AXPBCHMIX *voiceMixPtr;
    u16 *mixCtrlPtr;
    u32 maxChns;
    u16 curMixCtrl[AX_MAX_NUM_BUSES];
    u16 newMixCtrl[AX_MAX_NUM_BUSES];
    u16 busCnt, chCnt;

    // Steps:
    // - Check for validity of the inputs
    // - in the switch code, set up max channels, pointers appropriately
    if(voice == NULL)
    {
        //ASSERT(0);
        return AXPB_ERROR_NULL_VOICE_PTR;
    }
    if(mix == NULL)
    {
        //ASSERT(0);
        return AXPB_ERROR_NULL_MIX_PTR;
    }

    // if device + Id combination is wrong, then retErrCode will not
    // be AXPB_ERROR_NONE
    retErrCode = __AXCheckDeviceArgs(device, Id);


    if(AXPB_ERROR_NONE == retErrCode)
    {

        pupb = &__s_u_AXPB[voice->index];

        switch(device)
        {
            case AX_DEVICE_TV:
                // case of TV

                maxChns = AX_MAX_NUM_TV_CHS;
                voiceMixPtr = &pupb->tvMix[Id][0];
                mixCtrlPtr = &pupb->tvMixCtrl[Id][0];

                break;  // end of case TV
            case AX_DEVICE_DRC:
                // now do the mix updates as required
                maxChns = AX_MAX_NUM_DRC_CHS;
                voiceMixPtr = &pupb->drcMix[Id][0];
                mixCtrlPtr = &pupb->drcMixCtrl[Id][0];
                break;  // end of case DRC
            case AX_DEVICE_RMT:
                // now do the mix updates as required
                maxChns = AX_MAX_NUM_RMT_CHS;
                voiceMixPtr = &pupb->rmtMix[Id][0];
                mixCtrlPtr = &pupb->rmtMixCtrl[Id][0];
                break;  // end of case RMT
        } // end of switch(device)


       // maxChns, mixCtrlPtr, and VoiceMixPtr are set appropriately
        // if the mixCntrl changed, then set the sync bits
        // and protect the voice

        for(busCnt = 0 ; busCnt < AX_MAX_NUM_BUSES; busCnt++)
        {
            newMixCtrl[busCnt]=0;
        }

        // disabling interrupts from now on....
        old = OSDisableInterrupts();

        for (chCnt = 0; chCnt < maxChns; chCnt++)
        {
            __AXSetVoiceChMix(voiceMixPtr, mix, curMixCtrl);

            for(busCnt = 0 ; busCnt < AX_MAX_NUM_BUSES ; busCnt++)
            {
                newMixCtrl[busCnt] |= (curMixCtrl[busCnt] << (2*chCnt));
            }

            // increment mix pointer to next AXPBCHMIX
            mix++;
            voiceMixPtr++;

        } // end of all channels

        // newMixCtrl has the mixcontrol to be updated to
        for(busCnt = 0 ; busCnt < AX_MAX_NUM_BUSES ; busCnt++)
        {
           *mixCtrlPtr =  newMixCtrl[busCnt];
                //OSReport("*mixCtrlPtr = 0x%4.4x\n", *mixCtrlPtr);
           mixCtrlPtr++;
        }
        voice->sync |= (AX_SYNC_USER_MIX | AX_SYNC_USER_MIXCTRL);
        __AXVoiceProtect(voice);

        voiceMixPtr = &pupb->tvMix[Id][0];
        mixCtrlPtr = &pupb->tvMixCtrl[Id][0];

        // restoring interrupts...
        OSRestoreInterrupts(old);

#if (0)
    OSReport("rmtMixCtrl[0]: %8.8x - %4.4x %4.4x %4.4x %4.4x\n", &pupb->rmtMixCtrl[0], pupb->rmtMixCtrl[0][0], pupb->rmtMixCtrl[0][1], pupb->rmtMixCtrl[0][2], pupb->rmtMixCtrl[0][3]);
    OSReport("rmtMixCtrl[1]: %8.8x - %4.4x %4.4x %4.4x %4.4x\n", &pupb->rmtMixCtrl[1], pupb->rmtMixCtrl[1][0], pupb->rmtMixCtrl[1][1], pupb->rmtMixCtrl[1][2], pupb->rmtMixCtrl[1][3]);
    OSReport("rmtMixCtrl[2]: %8.8x - %4.4x %4.4x %4.4x %4.4x\n", &pupb->rmtMixCtrl[2], pupb->rmtMixCtrl[2][0], pupb->rmtMixCtrl[2][1], pupb->rmtMixCtrl[2][2], pupb->rmtMixCtrl[2][3]);
    OSReport("rmtMixCtrl[3]:%8.8x -  %4.4x %4.4x %4.4x %4.4x\n", &pupb->rmtMixCtrl[3], pupb->rmtMixCtrl[3][0], pupb->rmtMixCtrl[3][1], pupb->rmtMixCtrl[3][2], pupb->rmtMixCtrl[3][3]);
#endif
    }// end of processing

    return retErrCode;

} // end of AXSetVoiceDeviceMix

/*******************************************************************************
 * Function AXSetMaxVoices
 * Arguments
 *          u32 num:    (in) max number of voices to be used
 *
 * Sets the maximum number of voices to be allocated during rendering
*******************************************************************************/
void AXSetMaxVoices(u32 num)
{
    //__AXMaxVoices = num;
    OSReport("Not yet implemented\n");
    return;
} // end of AXSetMaxVoices

} // namespace winext
} // namespace internal
} // namespace nw

