﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include "AXPrivate.h"
#include <winext/cafe/os/OSInterrupt.h>

namespace nw {
namespace internal {
namespace winext {

/*---------------------------------------------------------------------------*
    local vars
 *---------------------------------------------------------------------------*/

// These stacks are used for sorting of voices by priority. 31 is the highest
// and will not be dropped in contention during new voice allocation. These
// may be dropped as a result of DSP cycles contention during PB sync. [0] is
// the free stack, any voices put in here will be stopped during PB sync.
// User priorities are from 31 - 1 and are assigned during voice aquisition,
// in the event of voice contention, the oldest and lowest, lower priority
// voice will be given to the user. Note, in the free stack [0] only the next
// member is used in the PB and the tail is never used as there is no need to
// remove free voices from the middle of the stack and patch the link list.

// How did I come up with the number 32? Well... lets see, this system is
// designed to manage 64 voices so having say 256 or more priorities would
// leave the voices more scattered and longer to sort. Since MIDI has 16
// channels and a user may wish to support priority per channel so we should
// have at least 16. On top of these 16, there might be sound effects events
// that may require a few priorities so we add a few more... 32 seems like a
// good round number for this application.
static AXVPB *__AXStackHead[AX_PRIORITY_STACKS];
static AXVPB *__AXStackTail[AX_PRIORITY_STACKS];

// The callback stack is used by the AX during PB sync, if the allowed DSP
// cycles are exceeded, the oldest of the lowest priority voices will be
// dropped. This stack is used to store these voices such that user callback
// can be performed later with interrupts enabled. Once the callback is
// performd the voice is freed.
static AXVPB *__AXCallbackStack;

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
static u32 __AXCheckStacks(void)
{
    u32 i, voices;

    voices = 0;

    for (i = 0; i < AX_PRIORITY_STACKS; i++)
    {
        AXVPB *voice = __AXStackHead[i];

        while (voice)
        {
            voices++;

            if (voices > __AXMaxVoices)
                return 0;

            voice = reinterpret_cast<AXVPB*>(voice->next);
        }
    }

    return 1;
}


/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
AXVPB* __AXGetStackHead(u32 priority)
{
    //ASSERT(priority < AX_PRIORITY_STACKS);

    return __AXStackHead[priority];
}


/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
void __AXServiceCallbackStack(void)
{
    AXVPB *p = __AXPopCallbackStack();

    while (p)
    {
        // The user may have already free'd this voice so lets check to see
        // if we need to free it
        if (p->priority)
        {
            if (p->callback)
                (*p->callback)(p);

            __AXRemoveFromStack(p);
            __AXPushFreeStack(p);
        }

        p = __AXPopCallbackStack();
    }
}


/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
void __AXInitVoiceStacks(void)
{
    u32 i;

    __AXCallbackStack   = NULL;

    for (i = 0; i < AX_PRIORITY_STACKS; i++)
        __AXStackHead[i] = __AXStackTail[i] = NULL;
}


/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
void __AXAllocInit(void)
{
#ifdef _DEBUG
    OSReport("Initializing AXAlloc code module\n");
#endif

    __AXInitVoiceStacks();
}


/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
void __AXAllocQuit(void)
{
#ifdef _DEBUG
    OSReport("Shutting down AXAlloc code module\n");
#endif

    __AXInitVoiceStacks();
}


/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
void __AXPushFreeStack(AXVPB *p)
{
    //ASSERT(p->priority);

    p->next             = __AXStackHead[0];
    __AXStackHead[0]    = p;
    p->priority         = 0;
}


/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
AXVPB* __AXPopFreeStack(void)
{
    AXVPB *p = __AXStackHead[0];

    if (p)
        __AXStackHead[0] = reinterpret_cast<AXVPB*>(p->next);

    return p;
}


/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
void __AXPushCallbackStack(AXVPB *p)
{
    p->next1            = __AXCallbackStack;
    __AXCallbackStack   = p;
}


/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
AXVPB* __AXPopCallbackStack(void)
{
    AXVPB *p = __AXCallbackStack;

    if (p)
        __AXCallbackStack = reinterpret_cast<AXVPB*>(p->next1);

    return p;
}


/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
void __AXRemoveFromStack(AXVPB *p)
{
    u32  i;
    AXVPB  *head;
    AXVPB  *tail;

    //ASSERT(p->priority);

    i = p->priority;
    head = __AXStackHead[i];
    tail = __AXStackTail[i];

    // see if it's the one and only member
    if (head == tail)
    {
        __AXStackHead[i] = __AXStackTail[i] = NULL;
        return;
    }

    // see if it's the head
    if (p == head)
    {
        __AXStackHead[i]        = reinterpret_cast<AXVPB*>(p->next);
        __AXStackHead[i]->prev  = NULL;
        return;
    }

    // see if it's the tail
    if (p == tail)
    {
        __AXStackTail[i]        = reinterpret_cast<AXVPB*>(p->prev);
        __AXStackTail[i]->next  = NULL;
        return;
    }

    // remove from middle, reuse the head and tail vars
    head = reinterpret_cast<AXVPB*>(p->prev);
    tail = reinterpret_cast<AXVPB*>(p->next);

    head->next = tail;
    tail->prev = head;
}


/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
void __AXPushStackHead(AXVPB *p, u32 priority)
{
    //ASSERT(priority);
    //ASSERT(priority < AX_PRIORITY_STACKS);

    p->next = __AXStackHead[priority];
    p->prev = NULL;

    if (p->next)
    {
        __AXStackHead[priority]->prev   = p;
        __AXStackHead[priority]         = p;
    }
    else
    {
        __AXStackHead[priority] = __AXStackTail[priority] = p;
    }

    p->priority = priority;
}


/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
AXVPB* __AXPopStackFromBottom(u32 priority)
{
    AXVPB *p;

    //ASSERT(priority);
    //ASSERT(priority < AX_PRIORITY_STACKS);

    p = NULL;

    if (__AXStackHead[priority])
    {
        if (__AXStackHead[priority] == __AXStackTail[priority])
        {
            p = __AXStackHead[priority];
            __AXStackHead[priority] = __AXStackTail[priority] = NULL;
        }
        else if (__AXStackTail[priority])
        {
            p = __AXStackTail[priority];
            __AXStackTail[priority]         = reinterpret_cast<AXVPB*>(p->prev);
            __AXStackTail[priority]->next   = NULL;
        }
    }

    return p;
}


/*---------------------------------------------------------------------------*
    Exposed API functions
 *---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*
    Return the specified voice to the free stack. Note the voice will be depop
    if it is running.
 *---------------------------------------------------------------------------*/
void AXFreeVoice(AXVPB *p)
{
    int old;

    //ASSERT(p);
    //ASSERTMSG(
    //    p->priority,
    //    "Calling AXFreeVoice() for voice that is already free\n"
    //    );

    old = OSDisableInterrupts();

    // remove the voice from the active stack that it's in
    __AXRemoveFromStack(p);

    // if it's running mark the sync flag so we can depop it
    if (p->state == AX_PB_STATE_RUN)
        p->depop = TRUE;

    // set it to default so there's no noise
    __AXSetPBDefault(p);

    // return it to the free stack
    __AXPushFreeStack(p);

    //ASSERTMSG(__AXCheckStacks(), "Voice list is trashed!\n");

    OSRestoreInterrupts(old);
}


/*---------------------------------------------------------------------------*
    Acquire a voice for use.
 *---------------------------------------------------------------------------*/
AXVPB* AXAcquireVoice(
    u32             priority,   // user desired priority for allocation
    AXVoiceCallback callback,   // user specified callback function
    u32             userContext // user assigned context
    )
{
    int     old;
    AXVPB     *p;

    //ASSERT(priority);
    //ASSERT(priority < AX_PRIORITY_STACKS);

    old = OSDisableInterrupts();

    // see if we can get a free one
    p = __AXPopFreeStack();


    // if not get the oldest voice of the lowest, lower priority
    if (!p)
    {
        u32 i;

        for (i = 1; i < priority; i++)
        {
            p = __AXPopStackFromBottom(i);

            if (p)
            {
                // ok we are going to drop this voice...lets mark the sync
                // flag to depop
                if (p->state == AX_PB_STATE_RUN)
                    p->depop = TRUE;

                // see if we should notify the previous owner that this
                // voice has been re-acquired by another user
                if (p->callback)
                    (*p->callback)(p);

                break;
            }
        }
    }

    // alright! we got one!
    if (p)
    {
        // put it in the right stack
        __AXPushStackHead(p, priority);

        // save callback for re-aquisition
        p->callback     = callback;
        p->userContext  = userContext;

        // set the voice PB to default
        __AXSetPBDefault(p);
    }

    //ASSERTMSG(__AXCheckStacks(), "Voice list is trashed!\n");

    OSRestoreInterrupts(old);

    return p;
}

/*---------------------------------------------------------------------------*
    Change the voice priority.
 *---------------------------------------------------------------------------*/
void AXSetVoicePriority(AXVPB *p, u32 priority)
{
    int old;

    //ASSERT(p);
    //ASSERTMSG(
    //    p->priority,
    //    "Calling AXSetVoicePriority() for voice that is already free\n"
    //    );

    //ASSERTMSG(
    //    priority,
    //    "Do not set voice priority to 0, use AXFreeVoice() to free voice\n"
    //    );

    //ASSERT(priority < AX_PRIORITY_STACKS);

    old = OSDisableInterrupts();

    __AXRemoveFromStack(p);
    __AXPushStackHead(p, priority);

    //ASSERTMSG(__AXCheckStacks(), "Voice list is trashed!\n");

    OSRestoreInterrupts(old);
}

} // namespace winext
} // namespace internal
} // namespace nw

